---
title: Configuration
nav_icon: sliders
sort_rank: 1
---

# Configuration

Prometheus is configured via command-line flags and a configuration file. While
the command-line flags configure immutable system parameters (such as storage
locations, amount of data to keep on disk and in memory, etc.), the
configuration file defines everything related to scraping [jobs and their
instances](/docs/concepts/jobs_instances/), as well as which [rule files to
load](/docs/querying/rules/#configuring-rules).

To view all available command-line flags, run `prometheus -h`.

Prometheus can reload its configuration at runtime. If the new configuration
is not well-formed, the changes will not be applied.
A configuration reload is triggered by sending a `SIGHUP` to the Prometheus process.
This will also reload any configured rule files.


## Configuration file

To specify which configuration file to load, use the `-config.file` flag.

The file is written in [YAML format](http://en.wikipedia.org/wiki/YAML),
defined by the scheme described below.
Brackets indicate that a parameter is optional. For non-list parameters the
value is set to the specified default.

Generic placeholders are defined as follows:

* `<duration>`: a duration matching the regular expression `[0-9]+[smhdwy]`
* `<labelname>`: a string matching the regular expression `[a-zA-Z_][a-zA-Z0-9_]*`
* `<labelvalue>`: a string of unicode characters
* `<filename>`: a valid path in the current working directory
* `<boolean>`: a boolean that can take the values `true` or `false`

The other placeholders are specified separately.

A valid example file can be found [here](https://github.com/prometheus/prometheus/blob/master/config/testdata/conf.good.yml).

The global configuration specifies parameters that are valid in all other configuration
contexts. They also serve as defaults for other configuration sections.


```
global:
  # How frequently to scrape targets by default.
  [ scrape_interval: <duration> | default = 1m ]

  # How long until a scrape request times out.
  [ scrape_timeout: <duration> | default = 10s ]

  # How frequently to evaluate rules by default.
  [ evaluation_interval: <duration> | default = 1m ]

  # The labels to add to any time series or alerts when communicating with
  # external systems (federation, remote storage, Alertmanager).
  external_labels:
    [ <labelname>: <labelvalue> ... ]

# Rule files specifies a list of files from which rules are read.
rule_files:
  [ - <filepath> ... ]

# A list of scrape configurations.
scrape_configs:
  [ - <scrape_config> ... ]
```


### `<scrape_config>`

A `scrape_config` section specifies a set of targets and parameters describing how
to scrape them. In the general case, one scrape configuration specifies a single
job. In advanced configurations, this may change.

Targets may be statically configured via the `target_groups` parameter or
dynamically discovered using one of the supported service-discovery mechanisms.

Additionally, `relabel_configs` allow advanced modifications to any
target and its labels before scraping.

If the targets require authentication, the following options are available:

* `basic_auth` - sets the `Authorization` header on every scrape request with the
configured username and password.
* `bearer_token` - sets the `Authorization` header on every scrape request with
the configured bearer token.
* `bearer_token_file` - sets the `Authorization` header on every scrape request
with the bearer token read from the configured file.
* `tls_config` - configures the scrape request's TLS settings.

See below for the configuration of these authentication options.

```
# The job name assigned to scraped metrics by default.
job_name: <name>

# How frequently to scrape targets from this job.
[ scrape_interval: <duration> | default = <global_config.scrape_interval> ]

# Per-target timeout when scraping this job.
[ scrape_timeout: <duration> | default = <global_config.scrape_timeout> ]

# The HTTP resource path on which to fetch metrics from targets.
[ metrics_path: <path> | default = /metrics ]

# honor_labels controls how Prometheus handles conflicts between labels that are
# already present in scraped data and labels that Prometheus would attach
# server-side ("job" and "instance" labels, manually configured target
# labels, and labels generated by service discovery implementations).
#
# If honor_labels is set to "true", label conflicts are resolved by keeping label
# values from the scraped data and ignoring the conflicting server-side labels.
#
# If honor_labels is set to "false", label conflicts are resolved by renaming
# conflicting labels in the scraped data to "exported_<original-label>" (for
# example "exported_instance", "exported_job") and then attaching server-side
# labels. This is useful for use cases such as federation, where all labels
# specified in the target should be preserved.
#
# Note that any globally configured "external_labels" are unaffected by this
# setting. In communication with external systems, they are always applied only
# when a time series does not have a given label yet and are ignored otherwise.
[ honor_labels: <boolean> | default = false ]

# The URL scheme with which to fetch metrics from targets.
[ scheme: <scheme> | default = http ]

# Optional HTTP URL parameters.
params:
  [ <string>: [<string>, ...] ]

# Optional authentication information. Note that `basic_auth`, `bearer_token`
# `bearer_token_file` options are mutually exclusive.

# Optional HTTP basic authentication information.
basic_auth:
  [ username: <string> ]
  [ password: <string> ]

# Optional bearer token authentication information.
[ bearer_token: <string> ]

# Optional bearer token file authentication information.
[ bearer_token_file: /path/to/bearer/token/file ]

# Optional TLS configuration.
tls_config:
  [ <tls_config> ]

# List of DNS service discovery configurations.
dns_sd_configs:
  [ - <dns_sd_config> ... ]

# List of Consul service discovery configurations.
consul_sd_configs:
  [ - <consul_sd_config> ... ]

# List of Kubernetes service discovery configurations.
kubernetes_sd_configs:
  [ - <kubernetes_sd_config> ... ]

# List of Zookeeper Serverset service discovery configurations.
serverset_sd_configs:
  [ - <serverset_sd_config> ... ]

# List of EC2 service discovery configurations.
ec2_sd_configs:
  [ - <ec2_sd_config> ... ]

# List of file service discovery configurations.
file_sd_configs:
  [ - <file_sd_config> ... ]

# List of labeled target groups for this job.
target_groups:
  [ - <target_group> ... ]

# List of target relabel configurations.
relabel_configs:
  [ - <relabel_config> ... ]

# List of metric relabel configurations.
metric_relabel_configs:
  [ - <relabel_config> ... ]
```

Where `<scheme>` may be `http` or `https` and `<path>` is a valid URL path.
`<job_name>` must be unique across all scrape configurations and adhere to the
regex `[a-zA-Z_][a-zA-Z0-9_-]`.


### `<tls_config>`

A `tls_config` allows configuring TLS connections.

```
# CA certificate to validate API server certificate with.
[ ca_file: <filename> ]

# Certificate and key files for client cert authentication to the server.
[ cert_file: <filename> ]
[ key_file: <filename> ]

# Disable validation of the server certificate.
[ insecure_skip_verify: <boolean> ]
```


### `<target_group>`

A `target_group` allows specifying a list of targets and a common label set for them.
They are the canonical way to specify static targets in a scrape configuration.

```
# The targets specified by the target group.
targets:
  [ - '<host>' ]

# Labels assigned to all metrics scraped from the targets.
labels:
  [ <labelname>: <labelvalue> ... ]
```

Where `<host>` is a valid string consisting of a hostname or IP followed by a port
number.


### `<dns_sd_config>`

A DNS-SD configuration allows specifying a set of DNS record names which
are periodically queried to discover a list of targets (host-port pairs). The
DNS servers to be contacted are read from `/etc/resolv.conf`.

During the [relabeling phase](#target-relabeling-relabel_config), the meta
label `__meta_dns_srv_name` is available on each target and is set to the SRV
record name that produced the discovered target.

```
# A list of DNS SRV record names to be queried.
names:
  [ - <record_name> ]

# The type of DNS query to perform.
[ type: <query_type> | default = 'SRV' ]

# The port number used if the query type is not SRV.
[ port: <number>]

# The time after which the provided names are refreshed.
[ refresh_interval: <duration> | default = 30s ]
```

Where `<record_name>` is any DNS SRV record name.
Where `<query_type>` is `SRV`, `A`, or `AAAA`.

### `<consul_sd_config>`

Consul SD configurations allow retrieving scrape targets from [Consul's](https://www.consul.io)
Catalog API.

The following meta labels are available on targets during relabeling:

* `__meta_consul_address`: the address of the target
* `__meta_consul_node`: the node name defined for the target
* `__meta_consul_tags`: the list of tags of the target joined by the tag separator
* `__meta_consul_service`: the name of the service the target belongs to
* `__meta_consul_service_address`: the service address of the target
* `__meta_consul_service_port`: the service port of the target
* `__meta_consul_service_id`: the service ID of the target
* `__meta_consul_dc`: the datacenter name for the target

```
# The information to access the Consul API. It is to be defined
# as the Consul documentation requires.
server: <host>
[ token: <string> ]
[ datacenter: <string> ]
[ scheme: <string> ]
[ username: <string> ]
[ password: <string> ]

# A list of services for which targets are retrieved. If omitted, all services
# are scraped.
services:
  [ - <string> ]

# The string by which Consul tags are joined into the tag label.
[ tag_separator: <string> | default = , ]
```

Note that the IP number and port used to scrape the targets is assembled as
`<__meta_consul_address>:<__meta_consul_service_port`. However, in some
Consul setups, the relevant address is in `__meta_consul_service_address`.
In those cases, you can use the [relabel](#target-relabeling-relabel_config)
feature to replace the special `__address__` label.

### `<kubernetes_sd_config>`

CAUTION: Kubernetes SD is in beta: breaking changes to configuration are still
likely in future releases.

Kubernetes SD configurations allow retrieving scrape targets from
[Kubernetes'](http://kubernetes.io/) REST API. By default, this discovers
masters, nodes, and appropriately annotated services so that metrics from both
cluster components and deployed applications can be scraped. This will create
multiple target groups: one for all masters with each master as a target, one
for all nodes with each node as a target, and one for each service containing
each service endpoint as a target.

The following meta labels are available on targets during relabeling:

* `__meta_kubernetes_role`: the role of the target: one of `master`, `node` or
`service`
* `__meta_kubernetes_node`: the name of the node from the Kubernetes API
* `__meta_kubernetes_node_label_<labelname>`: each node label from the
Kubernetes API
* `__meta_kubernetes_service_namespace`: the namespace of the service
* `__meta_kubernetes_service_name`: the name of the service
* `__meta_kubernetes_service_label_<labelname>`: each service label from the
Kubernetes API
* `__meta_kubernetes_service_annotation_<annotationname>`: each service
annotation from the Kubernetes API

See below for the configuration options for Kubernetes discovery:

```
# The information to access the Kubernetes API.

# The master addresses. In a cluster this will normally be
# `https://kubernetes.default.svc`. Supports multiple HA masters.
masters:
  - [<host>]

# Run in cluster. This will use the automounted CA certificate and bearer
# token file at /var/run/secrets/kubernetes.io/serviceaccount/ in the pod.
[ in_cluster: <boolean> ]

# The kubelet port to scrape metrics from. This will normally be the read-only
# port of 10255 (default).
[ kubelet_port: <int> ]

# File to read the bearer token from to authenticate to the API server. If
# running in a pod, then it is best to use a service account and set in_cluster
# to true.
[ bearer_token_file: <filename> ]

# Username and password for basic authentication to the API server.
[ username: <string> ]
[ password: <string> ]

# TLS configuration. If running in a pod, then it is best to use a service
# account and set in_cluster to true.
tls_config:
  [ <tls_config> ]

# Retry interval between watches if they disconnect.
[ retry_interval: <duration> | default = 1s ]
```

### `<marathon_sd_configs>`

CAUTION: Marathon SD is in beta: breaking changes to configuration are still
likely in future releases.

Marathon SD configurations allow retrieving scrape targets using the
[Marathon](https://mesosphere.github.io/marathon/) REST API. Prometheus
will periodically check the REST endpoint for currently running tasks and
create a target group for every app that has at least one healthy task.

The following meta labels are available on targets during relabeling:

* `__meta_marathon_app`: the name of the app (with slashes replaced by dashes)
* `__meta_marathon_image`: the name of the Docker image used (if available)
* `__meta_marathon_task`: the ID of the Mesos task
* `__meta_marathon_app_label_<labelname>`: any Marathon labels attached to the app

See below for the configuration options for Marathon discovery:

```
# List of URLs to be used to contact Marathon servers.
# You need to provide at least one server URL, but should provide URLs for
# all masters you have running.
servers:
  - <string>

# Polling interval
[ refresh_interval: <duration> | default = 30s ]
```

By default every app listed in Marathon will be scraped by Prometheus. If not all
of your services provide Prometheus metrics, you can use a Marathon label and
Prometheus relabeling to control which instances will actually be scraped. Also
by default all apps will show up as a single job in Prometheus (the one specified
in the configuration file), which can also be changed using relabeling.

### `<serverset_sd_config>`

Serverset SD configurations allow retrieving scrape targets from [Serversets]
(https://github.com/twitter/finagle/tree/master/finagle-serversets) which are
stored in [Zookeeper](https://zookeeper.apache.org/). Serversets are commonly
used by [Finagle](https://twitter.github.io/finagle/) and
[Aurora](http://aurora.apache.org/).

The following meta labels are available on targets during relabeling:

* `__meta_serverset_path`: the full path to the serverset member node in Zookeeper
* `__meta_serverset_endpoint_host`: the host of the default endpoint
* `__meta_serverset_endpoint_port`: the port of the default endpoint
* `__meta_serverset_endpoint_host_<endpoint>`: the host of the given endpoint
* `__meta_serverset_endpoint_port_<endpoint>`: the port of the given endpoint
* `__meta_serverset_shard`: the shard number of the member
* `__meta_serverset_status`: the status of the member

```
# The Zookeeper servers.
servers:
  - <host>
# Paths can point to a single serverset, or the root of a tree of serversets.
paths:
  - <string>
[ timeout: <duration> | default = 10s ]
```

Serverset data must be in the JSON format, the Thrift format is not currently supported.

### `<ec2_sd_config>`

CAUTION: EC2 SD is in beta: breaking changes to configuration are still
likely in future releases.

EC2 SD configurations allow retrieving scrape targets from AWS EC2
instances. The private IP address is used by default, but may be changed to
the public IP address with relabeling.

The following meta labels are available on targets during relabeling:

* `__meta_ec2_instance_id`: the EC2 instance ID
* `__meta_ec2_public_ip`: the public IP address of the instance
* `__meta_ec2_private_ip`: the private IP address of the instance, if present
* `__meta_ec2_tag_<tagkey>`: each tag value of the instance

See below for the configuration options for EC2 discovery:

```
# The information to access the EC2 API.

# The AWS Region.
region: <string>

# The AWS API keys. If blank, the environment variables `AWS_ACCESS_KEY_ID`
# and `AWS_SECRET_ACCESS_KEY` are used.
[ access_key: <string> ]
[ secret_key: <string> ]

# Refresh interval to re-read the instance list.
[ refresh_interval: <duration> | default = 60s ]

# The port to scrape metrics from. If using the public IP address, this must
# instead be specified in the relabeling rule.
[ port: <int> | default = 80 ]
```

### `<file_sd_config>`

File-based service discovery provides a more generic way to configure static targets
and serves as an interface to plug in custom service discovery mechanisms.

It reads a set of files containing a list of zero or more `<target_group>`s. Changes to
all defined files are detected via disk watches and applied immediately. Files may be
provided in YAML or JSON format. Only changes resulting in well-formed target groups
are applied.

The JSON file must contain a list of target groups, using this format:

```
[
  {
    "targets": [ "<host>", ... ],
    "labels": {
      "<labelname>": "<labelvalue>", ...
    }
  },
  ...
]
```

As a fallback, the file contents are also re-read periodically at the specified
refresh interval.

Each target has a meta label `__meta_filepath` during the
[relabeling phase](#target-relabeling-relabel_config). Its value is set to the
filepath from which the target was extracted.

```
# Patterns for files from which target groups are extracted.
names:
  [ - <filename_pattern> ... ]

# Refresh interval to re-read the files.
[ refresh_interval: <duration> | default = 30s ]
```

Where `<filename_pattern>` may be a path ending in `.json`, `.yml` or `.yaml`. The last path segment
may contain a single `*` that matches any character sequence, e.g. `my/path/tg_*.json`.


### `<relabel_config>`

Relabeling is a powerful tool to dynamically rewrite the label set of a target before
it gets scraped. Multiple relabeling steps can be configured per scrape configuration.
They are applied to the label set of each target in order of their appearance
in the configuration file.

Initially, aside from the configured per-target labels, a target's `job`
label is set to the `job_name` value of the respective scrape configuration.
The `__address__` label is set to the `<host>:<port>` address of the target.
After relabeling, the `instance` label is set to the value of `__address__` by default if
it was not set during relabeling. The `__scheme__` and `__metrics_path__` labels
are set to the scheme and metrics path of the target respectively. The `__param_<name>`
label is set to the value of the first passed URL parameter called `<name>`.

Additional labels prefixed with `__meta_` may be available during the
relabeling phase. They are set by the service discovery mechanism that provided
the target and vary between mechanisms.

Labels starting with `__` will be removed from the label set after relabeling is completed.

If a relabeling step needs to store a label value only temporarily (as the
input to a subsequent relabeling step), use the `__tmp` label name prefix. This
prefix is guaranteed to never be used by Prometheus itself.

```
# The source labels select values from existing labels. Their content is concatenated
# using the configured separator and matched against the configured regular expression
# for the replace, keep, and drop actions.
[ source_labels: '[' <labelname> [, ...] ']' ]

# Separator placed between concatenated source label values.
[ separator: <string> | default = ; ]

# Label to which the resulting value is written in a replace action.
# It is mandatory for replace actions.
[ target_label: <labelname> ]

# Regular expression against which the extracted value is matched.
[ regex: <regex> ]

# Modulus to take of the hash of the source label values.
[ modulus: <uint64> ]

# Replacement value against which a regex replace is performed if the
# regular expression matches.
[ replacement: <string> | default = '' ]

# Action to perform based on regex matching.
[ action: <relabel_action> | default = replace ]
```

`<regex>` is any valid [RE2 regular
expression](https://github.com/google/re2/wiki/Syntax). It is required for
the `replace`, `keep`, `drop` and `labelmap` actions. The regex is fully anchored.

`<relabel_action>` determines the relabeling action to take:

* `replace`: Match `regex` against the concatenated `source_labels`. Then, set
  `target_label` to `replacement`, with match group references
  (`${1}`, `${2}`, ...) in `replacement` substituted by their value.
* `keep`: Drop targets for which `regex` does not match the concatenated `source_labels`.
* `drop`: Drop targets for which `regex` matches the concatenated `source_labels`.
* `hashmod`: Set `target_label` to the `modulus` of a hash of the concatenated `source_labels`.
* `labelmap`: Match `regex` against all label names. Then copy the values of the matching labels
   to label names given by `replacement` with match group references
  (`${1}`, `${2}`, ...) in `replacement` substituted by their value.

### `<metric_relabel_configs>`

Metric relabeling is applied to samples as the last step before ingestion. It
has the same configuration format and actions as target relabeling. Metric
relabeling does not apply to automatically generated timeseries such as `up`.

One use for this is to blacklist time series that are too expensive to ingest.
