/* BoxRefresh()
 * =========
 * Adds AJAX content control to a box.
 *
 * @Usage: $('#my-box').boxRefresh(options)
 *         or add [data-widget="box-refresh"] to the box element
 *         Pass any option as data-option="value"
 */
+function ($) {
  'use strict'

  var DataKey = 'lte.boxrefresh'

  var Default = {
    source         : '',
    params         : {},
    trigger        : '.refresh-btn',
    content        : '.box-body',
    loadInContent  : true,
    responseType   : '',
    overlayTemplate: '<div class="overlay"><div class="fa fa-refresh fa-spin"></div></div>',
    onLoadStart    : function () {
    },
    onLoadDone     : function (response) {
      return response
    }
  }

  var Selector = {
    data: '[data-widget="box-refresh"]'
  }

  // BoxRefresh Class Definition
  // =========================
  var BoxRefresh = function (element, options) {
    this.element  = element
    this.options  = options
    this.$overlay = $(options.overlay)

    if (options.source === '') {
      throw new Error('Source url was not defined. Please specify a url in your BoxRefresh source option.')
    }

    this._setUpListeners()
    this.load()
  }

  BoxRefresh.prototype.load = function () {
    this._addOverlay()
    this.options.onLoadStart.call($(this))

    $.get(this.options.source, this.options.params, function (response) {
      if (this.options.loadInContent) {
        $(this.options.content).html(response)
      }
      this.options.onLoadDone.call($(this), response)
      this._removeOverlay()
    }.bind(this), this.options.responseType !== '' && this.options.responseType)
  }

  // Private

  BoxRefresh.prototype._setUpListeners = function () {
    $(this.element).on('click', Selector.trigger, function (event) {
      if (event) event.preventDefault()
      this.load()
    }.bind(this))
  }

  BoxRefresh.prototype._addOverlay = function () {
    $(this.element).append(this.$overlay)
  }

  BoxRefresh.prototype._removeOverlay = function () {
    $(this.element).remove(this.$overlay)
  }

  // Plugin Definition
  // =================
  function Plugin(option) {
    return this.each(function () {
      var $this = $(this)
      var data  = $this.data(DataKey)

      if (!data) {
        var options = $.extend({}, Default, $this.data(), typeof option == 'object' && option)
        $this.data(DataKey, (data = new BoxRefresh($this, options)))
      }

      if (typeof data == 'string') {
        if (typeof data[option] == 'undefined') {
          throw new Error('No method named ' + option)
        }
        data[option]()
      }
    })
  }

  var old = $.fn.boxRefresh

  $.fn.boxRefresh             = Plugin
  $.fn.boxRefresh.Constructor = BoxRefresh

  // No Conflict Mode
  // ================
  $.fn.boxRefresh.noConflict = function () {
    $.fn.boxRefresh = old
    return this
  }

  // BoxRefresh Data API
  // =================
  $(window).on('load', function () {
    $(Selector.data).each(function () {
      Plugin.call($(this))
    })
  })

}(jQuery)
