/*
 * Decompiled with CFR 0.152.
 */
package com.voxeo.data;

import com.voxeo.data.DataUpdateException;
import com.voxeo.data.DatabaseUpdaterLifecycleListener;
import com.voxeo.data.IDatabaseUpdater;
import com.voxeo.data.Schema;
import com.voxeo.data.SchemaSetBuilder;
import com.voxeo.data.Script;
import com.voxeo.logging.Loggerf;
import com.voxeo.utils.Typesafe;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.sql.SQLException;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import javax.sql.DataSource;
import org.apache.commons.io.IOUtils;
import org.springframework.jdbc.BadSqlGrammarException;
import org.springframework.jdbc.core.simple.SimpleJdbcTemplate;
import org.springframework.transaction.annotation.Transactional;

public class DatabaseUpdater
implements IDatabaseUpdater {
    private String schemaName;
    private File scriptDirectory;
    private DataSource dataSource;
    private SimpleJdbcTemplate template;
    private DatabaseUpdaterLifecycleListener lifecycleListener;
    private String versionQuery = "SELECT max(number) from VERSION";
    private String versionUpdate = "UPDATE version SET number = %s";
    protected static final Loggerf log = Loggerf.getLogger(DatabaseUpdater.class);

    @Transactional
    public void executeUpdate() throws DataUpdateException {
        log.info("Beginning database upgrade", new Object[0]);
        Schema schema = SchemaSetBuilder.buildSchema(this.scriptDirectory, this.schemaName);
        boolean success = false;
        try {
            if (this.lifecycleListener != null) {
                this.lifecycleListener.before(this.dataSource, schema);
            }
            this.updateSchema(schema);
            success = true;
        }
        catch (SQLException e) {
            throw new DataUpdateException("Failed to update database", e);
        }
        finally {
            if (this.lifecycleListener != null) {
                this.lifecycleListener.after(schema, success);
            }
        }
        log.info("All databases have been upgraded", new Object[0]);
    }

    protected void updateSchema(Schema schema) throws SQLException {
        int currentVersion = this.checkVersion();
        Set<Script> scripts = schema.getScripts();
        if (scripts.size() <= currentVersion) {
            return;
        }
        for (Script script : scripts) {
            if (script.getVersion() <= currentVersion) continue;
            if (this.runScript(script)) {
                this.logUpgrade(schema, script);
                continue;
            }
            File scriptFile = script.getFile();
            throw new SQLException(String.format("Failed to perform update for script [%s]", scriptFile.getAbsolutePath()));
        }
    }

    protected boolean runScript(Script script) {
        boolean success;
        switch (script.getType()) {
            case CREATE: 
            case UPGRADE: {
                success = this.runDDL(script);
                break;
            }
            default: {
                throw new UnsupportedOperationException(String.format("Script type not implemented [%s]", new Object[]{script.getType()}));
            }
        }
        if (success) {
            this.template.update(String.format(this.versionUpdate, script.getVersion()), new Object[0]);
        }
        return success;
    }

    private boolean runDDL(Script script) {
        File file = script.getFile();
        this.runIj(file);
        return true;
    }

    private void runIj(File file) {
        try {
            FileReader reader = new FileReader(file);
            LinkedList<String> statements = new LinkedList<String>();
            List<String> lines = Typesafe.list(String.class, IOUtils.readLines((Reader)reader));
            StringBuilder builder = new StringBuilder();
            for (String line : lines) {
                if (line.startsWith("--")) continue;
                int index = line.indexOf(";");
                if (index == -1) {
                    builder.append(line);
                    continue;
                }
                builder.append(line.substring(0, index));
                statements.add(builder.toString());
                builder.setLength(0);
                if (index <= line.length() - 1) continue;
                builder.append(line.substring(index + 1, line.length()));
            }
            if (builder.length() > 0) {
                lines.add(builder.toString());
            }
            for (String sql : statements) {
                log.debug("Issuing sql %s", sql);
                this.template.update(sql, new Object[0]);
            }
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Error reading file " + file, e);
        }
    }

    protected int checkVersion() {
        int version = 0;
        try {
            version = this.template.queryForInt(this.versionQuery, new Object[0]);
        }
        catch (BadSqlGrammarException e) {
            log.info("Error stating version table, assuming schema is not initialized", new Object[0]);
        }
        return version;
    }

    private void logUpgrade(Schema schema, Script script) {
        log.info("Database upgraded [name=%s, version=%s]", schema.getName(), script.getVersion());
    }

    public void setScriptDirectory(File scriptDirectory) {
        this.scriptDirectory = scriptDirectory;
    }

    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
        this.template = new SimpleJdbcTemplate(dataSource);
    }

    public DatabaseUpdaterLifecycleListener getLifecycleListener() {
        return this.lifecycleListener;
    }

    public void setLifecycleListener(DatabaseUpdaterLifecycleListener lifecycleListener) {
        this.lifecycleListener = lifecycleListener;
    }

    public String getSchemaName() {
        return this.schemaName;
    }

    public void setSchemaName(String schemaName) {
        this.schemaName = schemaName;
    }

    public String getVersionQuery() {
        return this.versionQuery;
    }

    public void setVersionQuery(String versionQuery) {
        this.versionQuery = versionQuery;
    }

    public String getVersionUpdate() {
        return this.versionUpdate;
    }

    public void setVersionUpdate(String versionUpdate) {
        this.versionUpdate = versionUpdate;
    }
}

