/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function ArtifactController($scope, $routeParams, workspace, jolokia) {
        $scope.row = {
            groupId: $routeParams["group"] || "",
            artifactId: $routeParams["artifact"] || "",
            version: $routeParams["version"] || "",
            classifier: $routeParams["classifier"] || "",
            packaging: $routeParams["packaging"] || ""
        };
        var row = $scope.row;

        $scope.id = Maven.getName(row);

        Maven.addMavenFunctions($scope, workspace);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateTableContents, 50);
        });

        $scope.$watch('workspace.selection', function () {
            updateTableContents();
        });

        function updateTableContents() {
            var mbean = Maven.getMavenIndexerMBean(workspace);

            // lets query the name and description of the GAV
            if (mbean) {
                jolokia.execute(mbean, "search", row.groupId, row.artifactId, row.version, row.packaging, row.classifier, "", onSuccess(render));
            } else {
                console.log("No MavenIndexerMBean!");
            }
        }

        function render(response) {
            if (response && response.length) {
                var first = response[0];
                row.name = first.name;
                row.description = first.description;
            }
            Core.$apply($scope);
        }
    }
    Maven.ArtifactController = ArtifactController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function DependenciesController($scope, $routeParams, $location, workspace, jolokia) {
        $scope.artifacts = [];
        $scope.group = $routeParams["group"] || "";
        $scope.artifact = $routeParams["artifact"] || "";
        $scope.version = $routeParams["version"] || "";
        $scope.classifier = $routeParams["classifier"] || "";
        $scope.packaging = $routeParams["packaging"] || "";

        $scope.dependencyTree = null;

        Maven.addMavenFunctions($scope, workspace);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateTableContents, 50);
        });

        $scope.$watch('workspace.selection', function () {
            updateTableContents();
        });

        $scope.onSelectNode = function (node) {
            $scope.selected = node;
        };

        $scope.onRootNode = function (rootNode) {
            // process the rootNode
        };

        $scope.validSelection = function () {
            return $scope.selected && $scope.selected !== $scope.rootDependency;
        };

        $scope.viewDetails = function () {
            var dependency = Core.pathGet($scope.selected, ["dependency"]);
            var link = $scope.detailLink(dependency);
            if (link) {
                var path = Core.trimLeading(link, "#");
                console.log("going to view " + path);
                $location.path(path);
            }
        };

        function updateTableContents() {
            var mbean = Maven.getAetherMBean(workspace);
            if (mbean) {
                jolokia.execute(mbean, "resolveJson(java.lang.String,java.lang.String,java.lang.String,java.lang.String,java.lang.String)", $scope.group, $scope.artifact, $scope.version, $scope.packaging, $scope.classifier, onSuccess(render));
            } else {
                console.log("No AetherMBean!");
            }
        }

        function render(response) {
            if (response) {
                var json = JSON.parse(response);
                if (json) {
                    //console.log("Found json: " + JSON.stringify(json, null, "  "));
                    $scope.dependencyTree = new Folder("Dependencies");
                    $scope.dependencyActivations = [];
                    addChildren($scope.dependencyTree, json);
                    $scope.dependencyActivations.reverse();
                    $scope.rootDependency = $scope.dependencyTree.children[0];
                }
            }
            Core.$apply($scope);
        }

        function addChildren(folder, dependency) {
            var name = Maven.getName(dependency);
            var node = new Folder(name);
            node.key = name.replace(/\//g, '_');
            node["dependency"] = dependency;
            $scope.dependencyActivations.push(node.key);

            /*
            var imageUrl = Camel.getRouteNodeIcon(value);
            node.icon = imageUrl;
            //node.tooltip = tooltip;
            */
            folder.children.push(node);

            var children = dependency["children"];
            angular.forEach(children, function (child) {
                addChildren(node, child);
            });
        }
    }
    Maven.DependenciesController = DependenciesController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function PomXmlController($scope) {
        $scope.mavenPomXml = "\n" + "  <dependency>\n" + "    <groupId>" + orBlank($scope.row.groupId) + "</groupId>\n" + "    <artifactId>" + orBlank($scope.row.artifactId) + "</artifactId>\n" + "    <version>" + orBlank($scope.row.version) + "</version>\n" + "  </dependency>\n";

        function orBlank(text) {
            return text || "";
        }
    }
    Maven.PomXmlController = PomXmlController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function SearchController($scope, $location, workspace, jolokia) {
        var log = Logger.get("Maven");

        $scope.artifacts = [];
        $scope.selected = [];
        $scope.done = false;
        $scope.inProgress = false;
        $scope.form = {
            searchText: ""
        };
        $scope.search = "";
        $scope.searchForm = 'app/maven/html/searchForm.html';

        Maven.addMavenFunctions($scope, workspace);

        var columnDefs = [
            {
                field: 'groupId',
                displayName: 'Group'
            },
            {
                field: 'artifactId',
                displayName: 'Artifact',
                cellTemplate: '<div class="ngCellText" title="Name: {{row.entity.name}}">{{row.entity.artifactId}}</div>'
            },
            {
                field: 'version',
                displayName: 'Version',
                cellTemplate: '<div class="ngCellText" title="Name: {{row.entity.name}}"><a ng-href="{{detailLink(row.entity)}}">{{row.entity.version}}</a</div>'
            }
        ];

        $scope.gridOptions = {
            data: 'artifacts',
            displayFooter: true,
            selectedItems: $scope.selected,
            selectWithCheckboxOnly: true,
            columnDefs: columnDefs,
            rowDetailTemplateId: "artifactDetailTemplate",
            filterOptions: {
                filterText: 'search'
            }
        };

        $scope.hasAdvancedSearch = function (form) {
            return form.searchGroup || form.searchArtifact || form.searchVersion || form.searchPackaging || form.searchClassifier || form.searchClassName;
        };

        $scope.doSearch = function () {
            $scope.done = false;
            $scope.inProgress = true;
            $scope.artifacts = [];

            // ensure ui is updated with search in progress...
            setTimeout(function () {
                Core.$apply($scope);
            }, 50);

            var mbean = Maven.getMavenIndexerMBean(workspace);
            var form = $scope.form;
            if (mbean) {
                var searchText = form.searchText;
                var kind = form.artifactType;
                if (kind) {
                    if (kind === "className") {
                        log.debug("Search for: " + form.searchText + " className");
                        jolokia.execute(mbean, "searchClasses", searchText, onSuccess(render));
                    } else {
                        var paths = kind.split('/');
                        var packaging = paths[0];
                        var classifier = paths[1];
                        log.debug("Search for: " + form.searchText + " packaging " + packaging + " classifier " + classifier);
                        jolokia.execute(mbean, "searchTextAndPackaging", searchText, packaging, classifier, onSuccess(render));
                    }
                } else if (searchText) {
                    log.debug("Search text is: " + form.searchText);
                    jolokia.execute(mbean, "searchText", form.searchText, onSuccess(render));
                } else if ($scope.hasAdvancedSearch(form)) {
                    log.debug("Searching for " + form.searchGroup + "/" + form.searchArtifact + "/" + form.searchVersion + "/" + form.searchPackaging + "/" + form.searchClassifier + "/" + form.searchClassName);

                    jolokia.execute(mbean, "search", form.searchGroup || "", form.searchArtifact || "", form.searchVersion || "", form.searchPackaging || "", form.searchClassifier || "", form.searchClassName || "", onSuccess(render));
                }
            } else {
                notification("error", "Cannot find the Maven Indexer MBean!");
            }
        };

        // cap ui table at one thousand
        var RESPONSE_LIMIT = 1000;
        var SERVER_RESPONSE_LIMIT = (10 * RESPONSE_LIMIT) + 1;

        function render(response) {
            log.debug("Search done, preparing result.");
            $scope.done = true;
            $scope.inProgress = false;

            // let's limit the reponse to avoid blowing up
            // the browser until we start using a widget
            // that supports pagination
            if (response.length > RESPONSE_LIMIT) {
                var serverLimit = response.length === SERVER_RESPONSE_LIMIT;
                if (serverLimit) {
                    $scope.tooManyResponses = "This search returned more than " + (SERVER_RESPONSE_LIMIT - 1) + " artifacts, showing the first " + RESPONSE_LIMIT + ", please refine your search";
                } else {
                    $scope.tooManyResponses = "This search returned " + response.length + " artifacts, showing the first " + RESPONSE_LIMIT + ", please refine your search";
                }
            } else {
                $scope.tooManyResponses = "";
            }
            $scope.artifacts = response.first(RESPONSE_LIMIT);

            Core.$apply($scope);
        }
    }
    Maven.SearchController = SearchController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function VersionsController($scope, $routeParams, workspace, jolokia) {
        $scope.artifacts = [];
        $scope.group = $routeParams["group"] || "";
        $scope.artifact = $routeParams["artifact"] || "";
        $scope.version = "";
        $scope.classifier = $routeParams["classifier"] || "";
        $scope.packaging = $routeParams["packaging"] || "";

        var id = $scope.group + "/" + $scope.artifact;
        if ($scope.classifier) {
            id += "/" + $scope.classifier;
        }
        if ($scope.packaging) {
            id += "/" + $scope.packaging;
        }
        var columnTitle = id + " versions";

        var columnDefs = [
            {
                field: 'version',
                displayName: columnTitle,
                cellTemplate: '<div class="ngCellText"><a href="#/maven/artifact/{{row.entity.groupId}}/{{row.entity.artifactId}}/{{row.entity.version}}">{{row.entity.version}}</a></div>'
            }
        ];

        $scope.gridOptions = {
            data: 'artifacts',
            displayFooter: true,
            selectedItems: $scope.selected,
            selectWithCheckboxOnly: true,
            columnDefs: columnDefs,
            rowDetailTemplateId: "artifactDetailTemplate",
            sortInfo: { field: 'versionNumber', direction: 'DESC' },
            filterOptions: {
                filterText: 'search'
            }
        };

        Maven.addMavenFunctions($scope, workspace);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateTableContents, 50);
        });

        $scope.$watch('workspace.selection', function () {
            updateTableContents();
        });

        function updateTableContents() {
            var mbean = Maven.getMavenIndexerMBean(workspace);
            if (mbean) {
                jolokia.execute(mbean, "versionComplete", $scope.group, $scope.artifact, $scope.version, $scope.packaging, $scope.classifier, onSuccess(render));
            } else {
                console.log("No MavenIndexerMBean!");
            }
        }

        function render(response) {
            $scope.artifacts = [];
            angular.forEach(response, function (version) {
                var versionNumberArray = Core.parseVersionNumbers(version);
                var versionNumber = 0;
                for (var i = 0; i <= 4; i++) {
                    var num = (i >= versionNumberArray.length) ? 0 : versionNumberArray[i];
                    versionNumber *= 1000;
                    versionNumber += num;
                }

                $scope.artifacts.push({
                    groupId: $scope.group,
                    artifactId: $scope.artifact,
                    packaging: $scope.packaging,
                    classifier: $scope.classifier,
                    version: version,
                    versionNumber: versionNumber
                });
            });
            Core.$apply($scope);
        }
    }
    Maven.VersionsController = VersionsController;
})(Maven || (Maven = {}));
/**
* @module Maven
* @main Maven
*/
var Maven;
(function (Maven) {
    var pluginName = 'maven';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'datatable', 'tree', 'hawtioCore', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/maven/search', { templateUrl: 'app/maven/html/search.html' }).when('/maven/advancedSearch', { templateUrl: 'app/maven/html/advancedSearch.html' }).when('/maven/artifact/:group/:artifact/:version/:classifier/:packaging', { templateUrl: 'app/maven/html/artifact.html' }).when('/maven/artifact/:group/:artifact/:version/:classifier', { templateUrl: 'app/maven/html/artifact.html' }).when('/maven/artifact/:group/:artifact/:version', { templateUrl: 'app/maven/html/artifact.html' }).when('/maven/dependencies/:group/:artifact/:version/:classifier/:packaging', { templateUrl: 'app/maven/html/dependencies.html' }).when('/maven/dependencies/:group/:artifact/:version/:classifier', { templateUrl: 'app/maven/html/dependencies.html' }).when('/maven/dependencies/:group/:artifact/:version', { templateUrl: 'app/maven/html/dependencies.html' }).when('/maven/versions/:group/:artifact/:classifier/:packaging', { templateUrl: 'app/maven/html/versions.html' }).when('/maven/view/:group/:artifact/:version/:classifier/:packaging', { templateUrl: 'app/maven/html/view.html' }).when('/maven/test', { templateUrl: 'app/maven/html/test.html' });
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['maven'] = "app/maven/html/layoutMaven.html";

        workspace.topLevelTabs.push({
            id: "maven",
            content: "Maven",
            title: "Search maven repositories for artifacts",
            isValid: function (workspace) {
                return Maven.getMavenIndexerMBean(workspace);
            },
            href: function () {
                return "#/maven/search";
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("/maven");
            }
        });

        helpRegistry.addUserDoc('maven', 'app/maven/doc/help.md', function () {
            return Maven.getMavenIndexerMBean(workspace) !== null;
        });
        helpRegistry.addDevDoc("maven", 'app/maven/doc/developer.md');
    });

    hawtioPluginLoader.addModule(pluginName);
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function ViewController($scope, $location, workspace, jolokia) {
        $scope.$watch('workspace.tree', function () {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        });

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            setTimeout(loadData, 50);
        });

        function loadData() {
        }
    }
    Maven.ViewController = ViewController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    function TestController($scope, workspace, jolokia, $q, $templateCache) {
        $scope.html = "text/html";

        $scope.someUri = '';
        $scope.uriParts = [];
        $scope.mavenCompletion = $templateCache.get("mavenCompletionTemplate");

        $scope.$watch('someUri', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.uriParts = newValue.split("/");
            }
        });

        $scope.$watch('uriParts', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if (newValue.length === 1 && newValue.length < oldValue.length) {
                    if (oldValue.last() !== '' && newValue.first().has(oldValue.last())) {
                        var merged = oldValue.first(oldValue.length - 1).include(newValue.first());
                        $scope.someUri = merged.join('/');
                    }
                }
            }
        }, true);

        $scope.doCompletionMaven = function (something) {
            return Maven.completeMavenUri($q, $scope, workspace, jolokia, something);
        };
    }
    Maven.TestController = TestController;
})(Maven || (Maven = {}));
/**
* @module Maven
*/
var Maven;
(function (Maven) {
    Maven.log = Logger.get("Maven");

    /**
    * Returns the maven indexer mbean (from the hawtio-maven-indexer library)
    * @method getMavenIndexerMBean
    * @for Maven
    * @param {Core.Workspace} workspace
    * @return {String}
    */
    function getMavenIndexerMBean(workspace) {
        if (workspace) {
            var mavenStuff = workspace.mbeanTypesToDomain["Indexer"] || {};
            var object = mavenStuff["hawtio"] || {};
            return object.objectName;
        } else
            return null;
    }
    Maven.getMavenIndexerMBean = getMavenIndexerMBean;

    function getAetherMBean(workspace) {
        if (workspace) {
            var mavenStuff = workspace.mbeanTypesToDomain["AetherFacade"] || {};
            var object = mavenStuff["hawtio"] || {};
            return object.objectName;
        } else
            return null;
    }
    Maven.getAetherMBean = getAetherMBean;

    function mavenLink(url) {
        var path = null;
        if (url) {
            if (url.startsWith("mvn:")) {
                path = url.substring(4);
            } else {
                var idx = url.indexOf(":mvn:");
                if (idx > 0) {
                    path = url.substring(idx + 5);
                }
            }
        }
        return path ? "#/maven/artifact/" + path : null;
    }
    Maven.mavenLink = mavenLink;

    function getName(row) {
        var id = (row.group || row.groupId) + "/" + (row.artifact || row.artifactId);
        if (row.version) {
            id += "/" + row.version;
        }
        if (row.classifier) {
            id += "/" + row.classifier;
        }
        if (row.packaging) {
            id += "/" + row.packaging;
        }
        return id;
    }
    Maven.getName = getName;

    function completeMavenUri($q, $scope, workspace, jolokia, query) {
        var mbean = getMavenIndexerMBean(workspace);
        if (!angular.isDefined(mbean)) {
            return $q.when([]);
        }

        var parts = query.split('/');
        if (parts.length === 1) {
            // still searching the groupId
            return Maven.completeGroupId(mbean, $q, $scope, workspace, jolokia, query, null, null);
        }
        if (parts.length === 2) {
            // have the groupId, guess we're looking for the artifactId
            return Maven.completeArtifactId(mbean, $q, $scope, workspace, jolokia, parts[0], parts[1], null, null);
        }
        if (parts.length === 3) {
            // guess we're searching for the version
            return Maven.completeVersion(mbean, $q, $scope, workspace, jolokia, parts[0], parts[1], parts[2], null, null);
        }

        return $q.when([]);
    }
    Maven.completeMavenUri = completeMavenUri;

    function completeVersion(mbean, $q, $scope, workspace, jolokia, groupId, artifactId, partial, packaging, classifier) {
        /*
        if (partial.length < 5) {
        return $q.when([]);
        }
        */
        var deferred = $q.defer();

        jolokia.request({
            type: 'exec',
            mbean: mbean,
            operation: 'versionComplete(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String)',
            arguments: [groupId, artifactId, partial, packaging, classifier]
        }, {
            method: 'POST',
            success: function (response) {
                $scope.$apply(function () {
                    deferred.resolve(response.value.sortBy().first(15));
                });
            },
            error: function (response) {
                $scope.$apply(function () {
                    console.log("got back an error: ", response);
                    deferred.reject();
                });
            }
        });

        return deferred.promise;
    }
    Maven.completeVersion = completeVersion;

    function completeArtifactId(mbean, $q, $scope, workspace, jolokia, groupId, partial, packaging, classifier) {
        var deferred = $q.defer();

        jolokia.request({
            type: 'exec',
            mbean: mbean,
            operation: 'artifactIdComplete(java.lang.String, java.lang.String, java.lang.String, java.lang.String)',
            arguments: [groupId, partial, packaging, classifier]
        }, {
            method: 'POST',
            success: function (response) {
                $scope.$apply(function () {
                    deferred.resolve(response.value.sortBy().first(15));
                });
            },
            error: function (response) {
                $scope.$apply(function () {
                    console.log("got back an error: ", response);
                    deferred.reject();
                });
            }
        });

        return deferred.promise;
    }
    Maven.completeArtifactId = completeArtifactId;

    function completeGroupId(mbean, $q, $scope, workspace, jolokia, partial, packaging, classifier) {
        // let's go easy on the indexer
        if (partial.length < 5) {
            return $q.when([]);
        }

        var deferred = $q.defer();

        jolokia.request({
            type: 'exec',
            mbean: mbean,
            operation: 'groupIdComplete(java.lang.String, java.lang.String, java.lang.String)',
            arguments: [partial, packaging, classifier]
        }, {
            method: 'POST',
            success: function (response) {
                $scope.$apply(function () {
                    deferred.resolve(response.value.sortBy().first(15));
                });
            },
            error: function (response) {
                console.log("got back an error: ", response);
                $scope.$apply(function () {
                    deferred.reject();
                });
            }
        });

        return deferred.promise;
    }
    Maven.completeGroupId = completeGroupId;

    function addMavenFunctions($scope, workspace) {
        $scope.detailLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var version = row.version || "";
            var classifier = row.classifier || "";
            var packaging = row.packaging || "";
            if (group && artifact) {
                return "#/maven/artifact/" + group + "/" + artifact + "/" + version + "/" + classifier + "/" + packaging;
            }
            return "";
        };

        $scope.javadocLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var version = row.version;
            if (group && artifact && version) {
                return "javadoc/" + group + ":" + artifact + ":" + version + "/";
            }
            return "";
        };

        $scope.versionsLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var classifier = row.classifier || "";
            var packaging = row.packaging || "";
            if (group && artifact) {
                return "#/maven/versions/" + group + "/" + artifact + "/" + classifier + "/" + packaging;
            }
            return "";
        };

        $scope.dependenciesLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var classifier = row.classifier || "";
            var packaging = row.packaging || "";
            var version = row.version;
            if (group && artifact) {
                return "#/maven/dependencies/" + group + "/" + artifact + "/" + version + "/" + classifier + "/" + packaging;
            }
            return "";
        };

        $scope.hasDependencyMBean = function () {
            var mbean = Maven.getAetherMBean(workspace);
            return angular.isDefined(mbean);
        };

        $scope.sourceLink = function (row) {
            var group = row.groupId;
            var artifact = row.artifactId;
            var version = row.version;
            if (group && artifact && version) {
                return "#/source/index/" + group + ":" + artifact + ":" + version + "/";
            }
            return "";
        };
    }
    Maven.addMavenFunctions = addMavenFunctions;
})(Maven || (Maven = {}));
var Apollo;
(function (Apollo) {
    function VirtualHostController($scope, $http, $location, localStorage, workspace) {
        $scope.virtual_host = {};
        $scope.init = function (virtual_host_name) {
            $scope.ajax("GET", "/broker/virtual-hosts/" + virtual_host_name, function (host) {
                $scope.virtual_host = host;
            });
        };
    }
    Apollo.VirtualHostController = VirtualHostController;
})(Apollo || (Apollo = {}));
/**
* @module Apollo
* @main Apollo
*/
var Apollo;
(function (Apollo) {
    var pluginName = 'apollo';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/apollo', { templateUrl: 'app/apollo/html/layout-apollo.html' });
        //otherwise({templateUrl: 'app/apollo/html/layout-apollo.html'})
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['apollo'] = "app/apollo/html/layout-apollo.html";
        helpRegistry.addUserDoc('apollo', 'app/apollo/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("org.apache.apollo");
        });

        workspace.topLevelTabs.push({
            id: "apollo",
            content: "Apollo",
            title: "Manage your Apollo Broker",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("org.apache.apollo");
            },
            href: function () {
                return '#/apollo/virtual-hosts';
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("apollo");
            }
        });
    });
    hawtioPluginLoader.addModule(pluginName);
})(Apollo || (Apollo = {}));
var Apollo;
(function (Apollo) {
    function ApolloController($scope, $http, $location, localStorage, workspace) {
        var jolokia = workspace.jolokia;
        $scope.broker = {};
        $scope.online = true;
        $scope.route = function () {
            return $location.path();
        };
        $scope.apollo = {
            version: jolokia.getAttribute('org.apache.apollo:type=broker,name="default"', "Version", onSuccess(null)),
            url: jolokia.getAttribute('org.apache.apollo:type=broker,name="default"', "WebAdminUrl", onSuccess(null))
        };

        var default_error_handler = function (data, status, headers, config) {
            if (status === 401) {
                alert("Action not authorized.");
            } else {
                alert("Error: " + status);
            }
        };

        $scope.ajax = function (type, path, success, error, data, binary_options) {
            if (!error) {
                error = default_error_handler;
            }
            var username = "admin";
            var password = "password";

            var ajax_options = {
                method: type,
                url: $scope.apollo.url + "/api/json" + path,
                headers: {
                    AuthPrompt: 'false',
                    Accept: "application/json",
                    ContentType: "application/json",
                    Authorization: Core.getBasicAuthHeader(username, password)
                },
                cache: false,
                data: null
            };
            if (binary_options) {
                ajax_options.headers["Accept"] = binary_options.Accept || "application/octet-stream";
                ajax_options.headers["ContentType"] || "application/octet-stream";
                ajax_options.data = binary_options.data;
            }

            return $http(ajax_options).success(function (data, status, headers, config) {
                $scope.online = true;
                if (success) {
                    success(data, status, headers, config);
                }
            }).error(function (data, status, headers, config) {
                if (status === 0) {
                    $scope.online = false;
                } else {
                    $scope.online = true;
                    error(data, status, headers, config);
                }
            });
        };

        var reload = function () {
            if ($scope.apollo.url) {
                $scope.ajax("GET", "/broker", function (broker) {
                    $scope.broker = broker;
                    if ($scope.apollo.selected_virtual_host === undefined) {
                        $scope.apollo.selected_virtual_host = broker.virtual_hosts[0];
                    }
                }, function (error) {
                    alert("fail:" + error);
                });
            } else {
                $scope.broker = {};
            }
        };

        var schedule_refresh = function () {
        };
        schedule_refresh = function () {
            setTimeout(function () {
                reload();
                schedule_refresh();
            }, 1000);
        };
        schedule_refresh();

        $scope.$watch('apollo.url', reload);
        $scope.$watch('online', function () {
            // alert("online: "+$scope.online)
        });
    }
    Apollo.ApolloController = ApolloController;
})(Apollo || (Apollo = {}));
/**
* @module Jetty
*/
var Jetty;
(function (Jetty) {
    function iconClass(state) {
        if (state) {
            switch (state.toString().toLowerCase()) {
                case 'started':
                    return "green icon-play-circle";
                case 'true':
                    return "green icon-play-circle";
            }
        }
        return "orange icon-off";
    }
    Jetty.iconClass = iconClass;

    /**
    * Returns true if the state of the item begins with the given state - or one of the given states
    * @method isState
    * @for Jetty
    * @param {any} item the item which has a State
    * @param {any} state a value or an array of states
    * @return {Boolean}
    */
    function isState(item, state) {
        var value = (item.state || "").toLowerCase();
        if (angular.isArray(state)) {
            return state.any(function (stateText) {
                return value.startsWith(stateText);
            });
        } else {
            return value.startsWith(state);
        }
    }
    Jetty.isState = isState;
})(Jetty || (Jetty = {}));
/**
* @module Jetty
*/
var Jetty;
(function (Jetty) {
    function JettyController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}">' + '<i class="{{row.getProperty(col.field) | jettyIconClass}}"></i>' + '</div>';
        var urlTemplate = '<div class="ngCellText" title="{{row.getProperty(col.field)}}">' + '<a ng-href="{{row.getProperty(col.field)}}" target="_blank">{{row.getProperty(col.field)}}</a>' + '</div>';

        $scope.uninstallDialog = new UI.Dialog();

        $scope.httpPort;
        $scope.httpScheme = "http";

        $scope.webapps = [];
        $scope.selected = [];

        var columnDefs = [
            {
                field: 'state',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'displayName',
                displayName: 'Name',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'contextPath',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'url',
                displayName: 'Url',
                cellTemplate: urlTemplate,
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'webapps',
            displayFooter: true,
            selectedItems: $scope.selected,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            },
            title: "Web applications"
        };

        // function to control the web applications
        $scope.controlWebApps = function (op) {
            // grab id of mbean names to control
            var mbeanNames = $scope.selected.map(function (b) {
                return b.mbean;
            });
            if (!angular.isArray(mbeanNames)) {
                mbeanNames = [mbeanNames];
            }

            // execute operation on each mbean
            var lastIndex = (mbeanNames.length || 1) - 1;
            angular.forEach(mbeanNames, function (mbean, idx) {
                var onResponse = (idx >= lastIndex) ? $scope.onLastResponse : $scope.onResponse;
                jolokia.request({
                    type: 'exec',
                    mbean: mbean,
                    operation: op,
                    arguments: null
                }, onSuccess(onResponse, { error: onResponse }));
            });
        };

        $scope.stop = function () {
            $scope.controlWebApps('stop');
        };

        $scope.start = function () {
            $scope.controlWebApps('start');
        };

        $scope.uninstall = function () {
            $scope.controlWebApps('destroy');
            $scope.uninstallDialog.close();
        };

        $scope.anySelectionHasState = function (state) {
            var selected = $scope.selected || [];
            return selected.length && selected.any(function (s) {
                return Jetty.isState(s, state);
            });
        };

        $scope.everySelectionHasState = function (state) {
            var selected = $scope.selected || [];
            return selected.length && selected.every(function (s) {
                return Jetty.isState(s, state);
            });
        };

        // function to trigger reloading page
        $scope.onLastResponse = function (response) {
            $scope.onResponse(response);

            // we only want to force updating the data on the last response
            loadData();
        };

        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
        };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        // grab server information once
        $scope.jettyServerVersion = "";
        $scope.jettyServerStartupTime = "";

        var servers = jolokia.search("org.eclipse.jetty.server:type=server,*");
        if (servers && servers.length === 1) {
            $scope.jettyServerVersion = jolokia.getAttribute(servers[0], "version");
            $scope.jettyServerStartupTime = jolokia.getAttribute(servers[0], "startupTime");
        } else {
            console.log("Cannot find jetty server or there was more than one server. response is: " + servers);
        }

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading Jetty webapp data...");

            // must load connectors first, before showing applications, so we do this call synchronously
            // jetty 7/8
            var connectors = jolokia.search("org.eclipse.jetty.server.nio:type=selectchannelconnector,*");
            if (!connectors) {
                // jetty 9
                connectors = jolokia.search("org.eclipse.jetty.server:type=serverconnector,*");
            }
            if (connectors) {
                var found = false;
                angular.forEach(connectors, function (key, value) {
                    var mbean = key;
                    if (!found) {
                        var data = jolokia.request({ type: "read", mbean: mbean, attribute: ["port", "protocols"] });
                        if (data && data.value && data.value.protocols && data.value.protocols.toString().toLowerCase().startsWith("http")) {
                            found = true;
                            $scope.httpPort = data.value.port;
                            $scope.httpScheme = "http";
                        }
                    }
                });
            }

            // support embedded jetty which may use morbay mbean names
            jolokia.search("org.mortbay.jetty.plugin:type=jettywebappcontext,*", onSuccess(render));
            jolokia.search("org.eclipse.jetty.webapp:type=webappcontext,*", onSuccess(render));
            jolokia.search("org.eclipse.jetty.servlet:type=servletcontexthandler,*", onSuccess(render));
        }

        function render(response) {
            $scope.webapps = [];
            $scope.mbeanIndex = {};
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    if (!obj.state) {
                        // lets leave the state as it is if it is defined
                        obj.state = obj['running'] === undefined || obj['running'] ? "started" : "stopped";
                    }

                    // compute the url for the webapp, and we want to use http as scheme
                    var hostname = Core.extractTargetUrl($location, $scope.httpScheme, $scope.httpPort);
                    obj.url = hostname + obj['contextPath'];

                    var mbean = obj.mbean;
                    if (mbean) {
                        var idx = $scope.mbeanIndex[mbean];
                        if (angular.isDefined(idx)) {
                            $scope.webapps[mbean] = obj;
                        } else {
                            $scope.mbeanIndex[mbean] = $scope.webapps.length;
                            $scope.webapps.push(obj);
                        }
                        Core.$apply($scope);
                    }
                }
            }

            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean, attribute: [] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
    }
    Jetty.JettyController = JettyController;
})(Jetty || (Jetty = {}));
/**
* @module Jetty
* @main Jetty
*/
var Jetty;
(function (Jetty) {
    var pluginName = 'jetty';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'ui.bootstrap.dialog', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/jetty/server', { templateUrl: 'app/jetty/html/server.html' }).when('/jetty/applications', { templateUrl: 'app/jetty/html/applications.html' }).when('/jetty/connectors', { templateUrl: 'app/jetty/html/connectors.html' }).when('/jetty/threadpools', { templateUrl: 'app/jetty/html/threadpools.html' });
    }).filter('jettyIconClass', function () {
        return Jetty.iconClass;
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['jetty'] = "app/jetty/html/layoutJettyTabs.html";
        helpRegistry.addUserDoc('jetty', 'app/jetty/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("org.eclipse.jetty.server");
        });

        workspace.topLevelTabs.push({
            id: "jetty",
            content: "Jetty",
            title: "Manage your Jetty container",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("org.eclipse.jetty.server");
            },
            href: function () {
                return "#/jetty/applications";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("jetty");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Jetty || (Jetty = {}));
/**
* @module Jetty
*/
var Jetty;
(function (Jetty) {
    function ThreadPoolsController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | jettyIconClass}}"></i></div>';

        $scope.threadpools = [];

        var columnDefs = [
            {
                field: 'running',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'threads',
                displayName: 'Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'minThreads',
                displayName: 'Min Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxThreads',
                displayName: 'Max Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'idleThreads',
                displayName: 'Idle Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'idleTimeout',
                displayName: 'Idle Timeout (ms)',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'name',
                displayName: 'Name',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'threadpools',
            displayFooter: true,
            canSelectRows: false,
            columnDefs: columnDefs,
            title: "Thread Pools"
        };

        function render78(response) {
            $scope.threadpools = [];

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    // jetty78 vs jetty9 is a bit different
                    obj.running = obj['running'] !== undefined ? obj['running'] : obj['state'] == "STARTED";
                    obj.idleTimeout = obj['idleTimeout'] !== undefined ? obj['idleTimeout'] : obj['maxIdleTimeMs'];
                    $scope.threadpools.push(obj);
                }
            }

            // create structure for each response
            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean, attribute: [] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading Jetty thread pool data...");
            var tree = workspace.tree;

            jolokia.search("org.eclipse.jetty.util.thread:type=queuedthreadpool,*", onSuccess(render78));
        }
    }
    Jetty.ThreadPoolsController = ThreadPoolsController;
})(Jetty || (Jetty = {}));
/**
* @module Jetty
*/
var Jetty;
(function (Jetty) {
    function ConnectorsController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | jettyIconClass}}"></i></div>';

        $scope.connectors = [];
        $scope.selected = [];

        var columnDefs = [
            {
                field: 'running',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'port',
                displayName: 'Port',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'protocols',
                displayName: 'Protocols',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'default',
                displayName: 'Default',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'connectors',
            displayFooter: true,
            selectedItems: $scope.selected,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            },
            title: "Connectors"
        };

        // function to control the connectors
        $scope.controlConnectors = function (op) {
            // grab id of mbean names to control
            var mbeanNames = $scope.selected.map(function (b) {
                return b.mbean;
            });
            if (!angular.isArray(mbeanNames)) {
                mbeanNames = [mbeanNames];
            }

            // execute operation on each mbean
            var lastIndex = (mbeanNames.length || 1) - 1;
            angular.forEach(mbeanNames, function (mbean, idx) {
                var onResponse = (idx >= lastIndex) ? $scope.onLastResponse : $scope.onResponse;
                jolokia.request({
                    type: 'exec',
                    mbean: mbean,
                    operation: op,
                    arguments: null
                }, onSuccess(onResponse, { error: onResponse }));
            });
        };

        $scope.stop = function () {
            $scope.controlConnectors('stop');
        };

        $scope.start = function () {
            $scope.controlConnectors('start');
        };

        $scope.anySelectionIsRunning = function () {
            var selected = $scope.selected || [];
            return selected.length && selected.any(function (s) {
                return s.running;
            });
        };

        $scope.everySelectionIsRunning = function (state) {
            var selected = $scope.selected || [];
            return selected.length && selected.every(function (s) {
                return s.running;
            });
        };

        function render78(response) {
            $scope.connectors = [];
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    // split each into 2 rows as we want http and https on each row
                    obj.mbean = response.request.mbean;
                    obj.protocols = "[http]";
                    obj.default = "http";
                    obj.port = obj.port;
                    obj.running = obj['running'] !== undefined ? obj['running'] : true;
                    $scope.connectors.push(obj);
                    if (obj.confidentialPort) {
                        // create a clone of obj for https
                        var copyObj = {
                            protocols: "[https]",
                            default: "https",
                            port: obj.confidentialPort,
                            running: obj.running,
                            mbean: obj.mbean
                        };
                        $scope.connectors.push(copyObj);
                    }
                    Core.$apply($scope);
                }
            }

            // create structure for each response
            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean, attribute: [] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        function render9(response) {
            $scope.connectors = [];
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    obj.protocols = obj['protocols'];
                    obj.default = obj['defaultProtocol'];
                    obj.port = obj.port;
                    obj.running = obj['state'] == "STARTED";
                    $scope.connectors.push(obj);
                    Core.$apply($scope);
                }
            }

            // create structure for each response
            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean, attribute: [] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        // function to trigger reloading page
        $scope.onLastResponse = function (response) {
            $scope.onResponse(response);

            // we only want to force updating the data on the last response
            loadData();
        };

        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
        };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading Jetty connector data...");
            var tree = workspace.tree;

            jolokia.search("org.eclipse.jetty.server.nio:type=selectchannelconnector,*", onSuccess(render78));
            jolokia.search("org.eclipse.jetty.server:type=serverconnector,*", onSuccess(render9));
        }
    }
    Jetty.ConnectorsController = ConnectorsController;
})(Jetty || (Jetty = {}));
/**
* @module Git
* @main Git
*/
var Git;
(function (Git) {
    

    /**
    * A default implementation which uses jolokia and the
    * GitFacadeMXBean over JMX
    *
    * @class JolokiaGit
    * @uses GitRepository
    *
    */
    var JolokiaGit = (function () {
        function JolokiaGit(mbean, jolokia, localStorage, userDetails, branch) {
            if (typeof branch === "undefined") { branch = "master"; }
            this.mbean = mbean;
            this.jolokia = jolokia;
            this.localStorage = localStorage;
            this.userDetails = userDetails;
            this.branch = branch;
        }
        JolokiaGit.prototype.getRepositoryLabel = function (fn, error) {
            return this.jolokia.request({ type: "read", mbean: this.mbean, attribute: ["RepositoryLabel"] }, onSuccess(function (result) {
                fn(result.value.RepositoryLabel);
            }, { error: error }));
        };

        JolokiaGit.prototype.exists = function (branch, path, fn) {
            return this.jolokia.execute(this.mbean, "exists", branch, path, onSuccess(fn));
        };

        JolokiaGit.prototype.read = function (branch, path, fn) {
            return this.jolokia.execute(this.mbean, "read", branch, path, onSuccess(fn));
        };

        JolokiaGit.prototype.write = function (branch, path, commitMessage, contents, fn) {
            var authorName = this.getUserName();
            var authorEmail = this.getUserEmail();
            return this.jolokia.execute(this.mbean, "write", branch, path, commitMessage, authorName, authorEmail, contents, onSuccess(fn));
        };

        JolokiaGit.prototype.writeBase64 = function (branch, path, commitMessage, contents, fn) {
            var authorName = this.getUserName();
            var authorEmail = this.getUserEmail();
            return this.jolokia.execute(this.mbean, "writeBase64", branch, path, commitMessage, authorName, authorEmail, contents, onSuccess(fn));
        };

        JolokiaGit.prototype.createDirectory = function (branch, path, commitMessage, fn) {
            var authorName = this.getUserName();
            var authorEmail = this.getUserEmail();

            return this.jolokia.execute(this.mbean, "createDirectory", branch, path, commitMessage, authorName, authorEmail, onSuccess(fn));
        };

        JolokiaGit.prototype.revertTo = function (branch, objectId, blobPath, commitMessage, fn) {
            var authorName = this.getUserName();
            var authorEmail = this.getUserEmail();

            return this.jolokia.execute(this.mbean, "revertTo", branch, objectId, blobPath, commitMessage, authorName, authorEmail, onSuccess(fn));
        };

        JolokiaGit.prototype.rename = function (branch, oldPath, newPath, commitMessage, fn) {
            var authorName = this.getUserName();
            var authorEmail = this.getUserEmail();

            return this.jolokia.execute(this.mbean, "rename", branch, oldPath, newPath, commitMessage, authorName, authorEmail, onSuccess(fn));
        };

        JolokiaGit.prototype.remove = function (branch, path, commitMessage, fn) {
            var authorName = this.getUserName();
            var authorEmail = this.getUserEmail();

            return this.jolokia.execute(this.mbean, "remove", branch, path, commitMessage, authorName, authorEmail, onSuccess(fn));
        };

        JolokiaGit.prototype.completePath = function (branch, completionText, directoriesOnly, fn) {
            return this.jolokia.execute(this.mbean, "completePath", branch, completionText, directoriesOnly, onSuccess(fn));
        };

        JolokiaGit.prototype.history = function (branch, objectId, path, limit, fn) {
            return this.jolokia.execute(this.mbean, "history", branch, objectId, path, limit, onSuccess(fn));
        };

        JolokiaGit.prototype.commitTree = function (commitId, fn) {
            return this.jolokia.execute(this.mbean, "getCommitTree", commitId, onSuccess(fn));
        };

        JolokiaGit.prototype.commitInfo = function (commitId, fn) {
            return this.jolokia.execute(this.mbean, "getCommitInfo", commitId, onSuccess(fn));
        };

        JolokiaGit.prototype.diff = function (objectId, baseObjectId, path, fn) {
            return this.jolokia.execute(this.mbean, "diff", objectId, baseObjectId, path, onSuccess(fn));
        };

        JolokiaGit.prototype.getContent = function (objectId, blobPath, fn) {
            return this.jolokia.execute(this.mbean, "getContent", objectId, blobPath, onSuccess(fn));
        };

        JolokiaGit.prototype.readJsonChildContent = function (path, nameWildcard, search, fn) {
            return this.jolokia.execute(this.mbean, "readJsonChildContent", this.branch, path, nameWildcard, search, onSuccess(fn));
        };

        JolokiaGit.prototype.branches = function (fn) {
            return this.jolokia.execute(this.mbean, "branches", onSuccess(fn));
        };

        // TODO move...
        JolokiaGit.prototype.getUserName = function () {
            return this.localStorage["gitUserName"] || this.userDetails.username || "anonymous";
        };

        JolokiaGit.prototype.getUserEmail = function () {
            return this.localStorage["gitUserEmail"] || "anonymous@gmail.com";
        };
        return JolokiaGit;
    })();
    Git.JolokiaGit = JolokiaGit;
})(Git || (Git = {}));
/**
* @module Git
*/
var Git;
(function (Git) {
    function createGitRepository(workspace, jolokia, localStorage) {
        var mbean = getGitMBean(workspace);
        if (mbean && jolokia) {
            return new Git.JolokiaGit(mbean, jolokia, localStorage, workspace.userDetails);
        }

        // TODO use local storage to make a little wiki thingy?
        return null;
    }
    Git.createGitRepository = createGitRepository;

    Git.jmxDomain = "hawtio";
    Git.mbeanType = "GitFacade";

    function hasGit(workspace) {
        return getGitMBean(workspace) !== null;
    }
    Git.hasGit = hasGit;

    /**
    * Returns the JMX ObjectName of the git mbean
    * @method getGitMBean
    * @for Git
    * @param {Workspace} workspace
    * @return {String}
    */
    function getGitMBean(workspace) {
        return Core.getMBeanTypeObjectName(workspace, Git.jmxDomain, Git.mbeanType);
    }
    Git.getGitMBean = getGitMBean;

    /**
    * Returns the Folder for the git mbean if it can be found
    * @method getGitMBeanFolder
    * @for Git
    * @param {Workspace} workspace
    * @return {Folder}
    */
    function getGitMBeanFolder(workspace) {
        return Core.getMBeanTypeFolder(workspace, Git.jmxDomain, Git.mbeanType);
    }
    Git.getGitMBeanFolder = getGitMBeanFolder;

    /**
    * Returns true if the git mbean is a fabric configuration repository
    * (so we can use it for the fabric plugin)
    * @method isGitMBeanFabric
    * @for Git
    * @param {Workspace} workspace
    * @return {Boolean}
    */
    function isGitMBeanFabric(workspace) {
        var folder = getGitMBeanFolder(workspace);
        return folder && folder.entries["repo"] === "fabric";
    }
    Git.isGitMBeanFabric = isGitMBeanFabric;
})(Git || (Git = {}));
var Fabric;
(function (Fabric) {
    function CreateBrokerController($scope, localStorage, $routeParams, $location, jolokia, workspace, $compile, $templateCache) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        $scope.defaultGroup = "default";
        $scope.defaultBrokerName = "brokerName";

        $scope.groups = [];
        $scope.possibleNetworks = [];
        $scope.profiles = [];
        $scope.parentProfiles = [];
        $scope.entity = {
            // default options
            group: $scope.defaultGroup,
            ssl: true
        };
        $scope.otherEntity = {
            networkConnectAll: false
        };

        // holds all the form objects from nested child scopes
        $scope.forms = {};

        $scope.onSubmit = function (json, form) {
            $scope.message = ($scope.entity.brokerName || "unknown") + " in group " + ($scope.entity.group || "unknown");
            notification("info", "Creating broker " + $scope.message);
            var tmpJson = JSON.stringify($scope.entity, null, '  ');
            jolokia.execute(Fabric.mqManagerMBean, "saveBrokerConfigurationJSON", tmpJson, onSuccess(onSave));

            // now lets switch to the brokers view
            $location.path("/fabric/mq/brokers");
            Core.$apply($scope);
        };

        $scope.brokerNameExists = function () {
            var name = $scope.entity.brokerName;
            return name && $scope.brokerNames.indexOf(name) >= 0;
        };

        function updatePossibleNetworks() {
            var group = $scope.entity.group;
            $scope.possibleNetworks = [].concat($scope.groups);
            if (group) {
                $scope.possibleNetworks = $scope.possibleNetworks.remove(group);
            }
        }

        $scope.$watch("entity.group", updatePossibleNetworks);
        $scope.$watch("otherEntity.networkConnectAll", function () {
            if ($scope.otherEntity.networkConnectAll) {
                $scope.entity.networks = $scope.possibleNetworks;
            }
        });

        // default parameters from the URL
        angular.forEach(["group", "profile"], function (param) {
            var value = $routeParams[param];
            if (value) {
                $scope.entity[param] = value;
            }
        });
        if (!$scope.entity.kind) {
            $scope.entity.kind = "MasterSlave";
        }

        Fabric.getDtoSchema("brokerConfig", "io.fabric8.api.jmx.MQBrokerConfigDTO", jolokia, function (schema) {
            $scope.schema = schema;
            configureSchema(schema);
            jolokia.execute(Fabric.mqManagerMBean, "loadBrokerStatus()", onSuccess(onBrokerData));
            Core.$apply($scope);
        });

        function configureSchema(schema) {
            delete schema.properties['username'];
            delete schema.properties['password'];

            // avoid the properties field for now as we don't yet have a generated UI for key/value pairs...
            delete schema.properties['properties'];

            var isReplicated = "entity.kind == 'Replicated'";
            var isStandalone = "entity.kind == 'StandAlone'";

            Core.pathSet(schema.properties, ['group', 'required'], true);
            Core.pathSet(schema.properties, ['group', 'tooltip'], 'The peer group name of message brokers. The group is name is used by messaging clients to connect to a broker; so it represents a peer group of brokers used for load balancing.');
            Core.pathSet(schema.properties, ['group', 'input-attributes', 'typeahead'], 'title for title in groups | filter:$viewValue');
            Core.pathSet(schema.properties, ['group', 'input-attributes', 'typeahead-editable'], 'true');

            Core.pathSet(schema.properties, ['brokerName', 'required'], true);
            Core.pathSet(schema.properties, ['brokerName', 'tooltip'], 'The name of the broker.');
            Core.pathSet(schema.properties, ['brokerName', 'input-attributes', 'autofocus'], 'true');

            Core.pathSet(schema.properties, ['parentProfile', 'tooltip'], 'The parent profile used by the profile.');
            Core.pathSet(schema.properties, ['parentProfile', 'input-attributes', 'typeahead'], 'p.id for p in parentProfiles | filter:$viewValue');
            Core.pathSet(schema.properties, ['parentProfile', 'input-attributes', 'typeahead-editable'], 'false');
            Core.pathSet(schema.properties, ['parentProfile', 'input-attributes', "placeholder"], "{{" + isReplicated + " ? 'mq-replicated' : 'mq-base'}}");

            Core.pathSet(schema.properties, ['profile', 'tooltip'], 'The profile to create instances of this broker.');
            Core.pathSet(schema.properties, ['profile', 'input-attributes', 'typeahead'], 'title for title in profiles | filter:$viewValue');
            Core.pathSet(schema.properties, ['profile', 'input-attributes', 'typeahead-editable'], 'true');
            Core.pathSet(schema.properties, ['profile', 'input-attributes', "placeholder"], "mq-broker-{{entity.group || 'default'}}.{{entity.brokerName || 'brokerName'}}");

            Core.pathSet(schema.properties, ['clientProfile', 'tooltip'], 'The profile used by messaging clients to connect to this group of brokers.');
            Core.pathSet(schema.properties, ['clientProfile', 'input-attributes', 'typeahead'], 'title for title in profiles | filter:$viewValue');
            Core.pathSet(schema.properties, ['clientProfile', 'input-attributes', 'typeahead-editable'], 'true');
            Core.pathSet(schema.properties, ['clientProfile', 'input-attributes', "placeholder"], "mq-client-{{entity.group || 'default'}}");

            Core.pathSet(schema.properties, ['clientParentProfile', 'tooltip'], 'The parent profile used by the client profile.');
            Core.pathSet(schema.properties, ['clientParentProfile', 'input-attributes', 'typeahead'], 'p.id for p in parentProfiles | filter:$viewValue');
            Core.pathSet(schema.properties, ['clientParentProfile', 'input-attributes', 'typeahead-editable'], 'false');
            Core.pathSet(schema.properties, ['clientParentProfile', 'input-attributes', 'placeholder'], 'mq-client-base');
            Core.pathSet(schema.properties, ['parentProfile', 'input-attributes', "placeholder"], "default");

            Core.pathSet(schema.properties, ['data', 'input-attributes', "placeholder"], "${karaf.base}/data/{{entity.brokerName || 'brokerName'}}");
            Core.pathSet(schema.properties, ['configUrl', 'input-attributes', "placeholder"], "profile:broker.xml");

            Core.pathSet(schema.properties, ['replicas', 'control-group-attributes', "ng-show"], isReplicated);
            Core.pathSet(schema.properties, ['replicas', 'input-attributes', "value"], "{{3}}");
            Core.pathSet(schema.properties, ['replicas', 'input-attributes', "min"], "1");
            Core.pathSet(schema.properties, ['minimumInstances', 'control-group-attributes', "ng-hide"], isReplicated);
            Core.pathSet(schema.properties, ['minimumInstances', 'input-attributes', "value"], "{{" + isStandalone + " ? 1 : 2}}");
            Core.pathSet(schema.properties, ['minimumInstances', 'input-attributes', "min"], "1");

            Core.pathSet(schema.properties, ['networksPassword', 'type'], 'password');
            Core.pathSet(schema.properties, ['networks', 'items', 'input-attributes', 'typeahead-editable'], 'true');
            Core.pathSet(schema.properties, ['networks', 'input-attributes', "ng-hide"], "otherEntity.networkConnectAll");
            Core.pathSet(schema.properties, ['networks', 'tooltip'], 'The broker groups to create a store and forward network to');

            // add an extra property to make it easy to connect to all / none
            Core.pathSet(schema.properties, ['networkConnectAll', 'type'], 'boolean');
            Core.pathSet(schema.properties, ['networkConnectAll', 'input-attributes', 'ng-model'], "otherEntity.networkConnectAll");
            Core.pathSet(schema.properties, ['networkConnectAll', 'label'], 'Network to all groups');
            Core.pathSet(schema.properties, ['networkConnectAll', 'tooltip'], 'Should this broker create a store and forward network to all the known groups of brokers');

            schema['tabs'] = {
                'Default': ['group', 'brokerName', 'kind', 'profile', 'clientProfile', 'data', 'configUrl', 'replicas', 'minimumInstances', 'networkConnectAll', 'networks'],
                'Advanced': ['parentProfile', 'clientParentProfile', 'networksUserName', 'networksPassword', '*']
            };
        }

        function onBrokerData(brokerStatuses) {
            var networkNames = brokerStatuses.map(function (s) {
                return s.networks;
            }).flatten().unique();
            var groups = brokerStatuses.map(function (s) {
                return s.group;
            }).unique();

            $scope.groups = networkNames.concat(groups).unique().sort();
            $scope.profiles = brokerStatuses.map(function (s) {
                return s.profile;
            }).unique().sort();
            $scope.brokerNames = brokerStatuses.map(function (s) {
                return s.brokerName;
            }).unique().sort();

            updatePossibleNetworks();

            var version = brokerStatuses.map(function (s) {
                return s.version;
            }).find(function (s) {
                return s;
            }) || "1.0";
            if (version) {
                jolokia.execute(Fabric.managerMBean, "getProfiles(java.lang.String,java.util.List)", version, ["id", "abstract"], onSuccess(onProfileData));
            }
            Core.$apply($scope);
        }

        function onProfileData(profileData) {
            if (profileData) {
                $scope.parentProfiles = profileData.filter(function (p) {
                    return !p.abstract;
                }).sortBy("id");
            }
        }

        function onSave(response) {
            notification("success", "Created broker " + $scope.message);
            Core.$apply($scope);
        }
    }
    Fabric.CreateBrokerController = CreateBrokerController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function MigrateContainersController($scope, jolokia, $location) {
        $scope.versions = [];
        $scope.containers = [];
        $scope.containersResponse = [];

        $scope.selectedVersion = [];
        $scope.selectedContainers = [];

        $scope.showApply = false;

        $scope.versionGridOptions = {
            data: 'versions',
            selectedItems: $scope.selectedVersion,
            showSelectionCheckbox: true,
            multiSelect: false,
            keepLastSelected: true,
            columnDefs: [{
                    field: 'id',
                    displayName: 'Version Name',
                    width: '94%'
                }],
            filterOptions: {
                filterText: ''
            }
        };

        $scope.containerGridOptions = {
            data: 'containers',
            selectedItems: $scope.selectedContainers,
            showSelectionCheckbox: true,
            multiSelect: true,
            keepLastSelected: false,
            columnDefs: [{
                    field: 'id',
                    displayName: 'Container Name',
                    width: '94%'
                }],
            filterOptions: {
                filterText: ''
            }
        };

        $scope.canApply = function () {
            return !($scope.selectedVersion.length > 0 && $scope.selectedContainers.length > 0);
        };

        $scope.render = function (response) {
            if (response.request.operation === 'versions()') {
                if (!Object.equal($scope.versions, response.value)) {
                    $scope.versions = response.value;
                    Core.$apply($scope);
                }
            }

            if (response.request.operation === 'containerIds()') {
                if (!Object.equal($scope.containersResponse, response.value)) {
                    $scope.containersResponse = response.value;

                    $scope.containers = [];

                    $scope.containersResponse.each(function (container) {
                        $scope.containers.push({
                            id: container
                        });
                    });
                    Core.$apply($scope);
                }
            }
        };

        $scope.migrateContainers = function () {
            var containerIds = $scope.selectedContainers.map(function (container) {
                return container.id;
            });
            var versionId = $scope.selectedVersion[0].id;

            notification('info', "Moving containers to version " + versionId);
            $location.path("/fabric/containers");

            Fabric.migrateContainers(jolokia, versionId, containerIds, function () {
                notification('success', "Successfully migrated containers");
            }, function (response) {
                notification('error', "Failed to migrate containers due to " + response.error);
            });
        };

        Core.register(jolokia, $scope, [
            { type: 'exec', mbean: Fabric.managerMBean, operation: 'versions()' },
            { type: 'exec', mbean: Fabric.managerMBean, operation: 'containerIds()' }
        ], onSuccess($scope.render));
    }
    Fabric.MigrateContainersController = MigrateContainersController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function CreateFabricController($scope, jolokia, $location, workspace, branding) {
        $scope.$on('$routeChangeSuccess', function () {
            if (workspace.treeContainsDomainAndProperties(Fabric.jmxDomain, { type: "Fabric" })) {
                $location.url("/fabric/view");
            }
        });

        Fabric.getSchema('createEnsemble', 'io.fabric8.api.CreateEnsembleOptions', jolokia, function (schema) {
            $scope.schema = schema;
            Core.$apply($scope);
        });

        $scope.creating = false;

        $scope.entity = {
            zooKeeperServerPort: 2181,
            globalResolver: 'localhostname',
            resolver: 'localhostname',
            agentEnabled: true,
            autoImportEnabled: true,
            minimumPort: 0,
            maximumPort: 65535,
            profiles: ['fabric', 'hawtio']
        };

        if (branding.profile) {
            $scope.entity.profiles.push(branding.profile);
        }

        // console.log("entity: ", $scope.entity);
        $scope.forms = {};

        $scope.onSubmit = function (json, form) {
            json = Fabric.sanitizeJson(json);

            setTimeout(function () {
                jolokia.execute(Fabric.clusterBootstrapManagerMBean, 'createCluster(java.util.Map)', angular.toJson(json), {
                    method: 'post',
                    success: function (response) {
                        notification('success', "Created fabric!");
                        $location.url("/fabric/containers");
                        Core.$apply($scope);
                    },
                    error: function (response) {
                        notification('error', "Error creating fabric: " + response.error);
                        Core.$apply($scope);
                    }
                });
                notification('info', "Creating fabric, please wait...");
                $location.url("/openlogs");
                Core.$apply($scope);
            }, 30);
        };
    }
    Fabric.CreateFabricController = CreateFabricController;
})(Fabric || (Fabric = {}));
/**
* @module Fabric
*/
var Fabric;
(function (Fabric) {
    Fabric.log = Logger.get("Fabric");

    Fabric.jmxDomain = 'io.fabric8';

    Fabric.managerMBean = Fabric.jmxDomain + ":type=Fabric";
    Fabric.clusterManagerMBean = Fabric.jmxDomain + ":type=ClusterServiceManager";
    Fabric.clusterBootstrapManagerMBean = Fabric.jmxDomain + ":type=ClusterBootstrapManager";
    Fabric.openShiftFabricMBean = Fabric.jmxDomain + ":type=OpenShift";
    Fabric.mqManagerMBean = Fabric.jmxDomain + ":type=MQManager";

    var schemaLookupDomain = "hawtio";
    var schemaLookupType = "SchemaLookup";

    Fabric.schemaLookupMBean = schemaLookupDomain + ":type=" + schemaLookupType;

    Fabric.useDirectoriesInGit = true;
    Fabric.fabricTopLevel = "fabric/profiles/";
    Fabric.profileSuffix = ".profile";

    Fabric.jolokiaWebAppGroupId = Fabric.jmxDomain + ".fabric-jolokia";

    function fabricCreated(workspace) {
        return workspace.treeContainsDomainAndProperties(Fabric.jmxDomain, { type: "Fabric" });
    }
    Fabric.fabricCreated = fabricCreated;

    function canBootstrapFabric(workspace) {
        return hasClusterBootstrapManager(workspace);
    }
    Fabric.canBootstrapFabric = canBootstrapFabric;

    function hasClusterBootstrapManager(workspace) {
        return workspace.treeContainsDomainAndProperties(Fabric.jmxDomain, { type: "ClusterBootstrapManager" });
    }
    Fabric.hasClusterBootstrapManager = hasClusterBootstrapManager;

    function hasClusterServiceManager(workspace) {
        return workspace.treeContainsDomainAndProperties(Fabric.jmxDomain, { type: "ClusterServiceManager" });
    }
    Fabric.hasClusterServiceManager = hasClusterServiceManager;

    function hasZooKeeper(workspace) {
        return workspace.treeContainsDomainAndProperties(Fabric.jmxDomain, { type: "ZooKeeper" });
    }
    Fabric.hasZooKeeper = hasZooKeeper;

    function hasOpenShiftFabric(workspace) {
        return workspace.treeContainsDomainAndProperties(Fabric.jmxDomain, { type: "OpenShift" });
    }
    Fabric.hasOpenShiftFabric = hasOpenShiftFabric;

    function hasMQManager(workspace) {
        return workspace.treeContainsDomainAndProperties(Fabric.jmxDomain, { type: "MQManager" });
    }
    Fabric.hasMQManager = hasMQManager;

    function hasSchemaMBean(workspace) {
        return workspace.treeContainsDomainAndProperties(schemaLookupDomain, { type: schemaLookupType });
    }
    Fabric.hasSchemaMBean = hasSchemaMBean;

    function hasGitMBean(workspace) {
        return workspace.treeContainsDomainAndProperties(Git.jmxDomain, { type: Git.mbeanType });
    }
    Fabric.hasGitMBean = hasGitMBean;

    function isFMCContainer(workspace) {
        var hasFabric = Fabric.hasFabric(workspace);
        var hasSchemaMBean = Fabric.hasSchemaMBean(workspace);
        var hasGitMBean = Fabric.hasGitMBean(workspace);

        // Too noisy...
        // log.debug("is FMC container, hasFabric: ", hasFabric, " hasSchemaMBean:", hasSchemaMBean, " hasGitMBean:", hasGitMBean);
        return hasFabric && hasSchemaMBean && hasGitMBean;
    }
    Fabric.isFMCContainer = isFMCContainer;

    function hasFabric(workspace) {
        // lets make sure we only have a fabric if we have
        // the ClusterServiceManager or ClusterBootstrapManager available
        // so that we hide Fabric for 6.0 or earlier of JBoss Fuse
        // which doesn't have the necessary mbeans for hawtio awesomeness
        return fabricCreated(workspace) && (hasClusterServiceManager(workspace) || hasClusterBootstrapManager(workspace) || hasZooKeeper(workspace));
    }
    Fabric.hasFabric = hasFabric;

    /**
    * Adds a bunch of common helper functions to the given scope
    * @method initScope
    * @for Fabric
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @paran {*} jolokia
    * @param {Workspace} workspace
    */
    function initScope($scope, $location, jolokia, workspace) {
        // Let's avoid re-defining everything if the $scope
        // has already been initialized here
        if ($scope.fabricInitialized) {
            return;
        } else {
            $scope.fabricInitialized = true;
        }

        $scope.gotoProfile = function (versionId, profileId) {
            Fabric.gotoProfile(workspace, jolokia, workspace.localStorage, $location, versionId, profileId);
        };

        $scope.getStatusTitle = function (container) {
            return Fabric.statusTitle(container);
        };

        $scope.isCurrentContainer = function (container) {
            if (!container) {
                return false;
            }
            if (Core.isBlank(Fabric.currentContainerId)) {
                return false;
            }
            if (angular.isObject(container)) {
                return container['id'] === Fabric.currentContainerId;
            }
            if (angular.isString(container)) {
                return container === Fabric.currentContainerId;
            }

            return false;
        };

        $scope.canConnect = function (container) {
            if (!container) {
                return false;
            }
            if (Core.isBlank(container['jolokiaUrl'])) {
                return false;
            }

            if (!Core.parseBooleanValue(container['alive'])) {
                return false;
            }
            return true;
        };

        $scope.refreshProfile = function (versionId, profileId) {
            Fabric.log.debug('Refreshing profile: ' + profileId + '/' + versionId);
            if (!versionId || !profileId) {
                return;
            }
            jolokia.request({
                type: 'exec',
                mbean: Fabric.managerMBean,
                operation: 'refreshProfile',
                arguments: [versionId, profileId]
            }, {
                method: 'POST',
                success: function () {
                    notification('success', 'Triggered refresh of profile ' + profileId + '/' + versionId);
                    Core.$apply($scope);
                },
                error: function (response) {
                    Fabric.log.warn('Failed to trigger refresh for profile ' + profileId + '/' + versionId + ' due to: ', response.error);
                    Fabric.log.info("Stack trace: ", response.stacktrace);
                    Core.$apply($scope);
                }
            });
        };

        $scope.getVersionsToExclude = function () {
            if (!$scope.selectedContainers || $scope.selectedContainers.length === 0) {
                return [];
            }
            var answer = $scope.selectedContainers.map(function (c) {
                return c['versionId'];
            });
            answer = answer.unique();
            if (answer.length > 1) {
                return [];
            } else {
                return answer;
            }
        };

        $scope.hasFabricWiki = function () {
            return Git.isGitMBeanFabric(workspace);
        };

        $scope.showContainer = function (container) {
            $location.path('/fabric/container/' + container.id);
        };

        $scope.createRequiredContainers = function (profile) {
            var profileId = profile.id;
            var args = {};
            if (profileId) {
                args["profileIds"] = profileId;
            }
            var versionId = profile.versionId || profile.version;
            if (versionId) {
                args["versionId"] = versionId;
            }
            var requirements = profile.requirements;
            if (requirements) {
                var min = requirements.minimumInstances;
                if (min) {
                    var delta = min - (profile.count || 0);
                    if (delta > 1) {
                        args["number"] = delta;
                    }
                }
            }
            $location.url('/fabric/containers/createContainer').search(args);
        };

        $scope.createContainer = function () {
            var kind = null;

            // lets see if there is an openshift option
            var providers = registeredProviders(jolokia);
            angular.forEach(["openshift", "docker", "jclouds"], function (value) {
                if (!kind && providers[value]) {
                    kind = value;
                }
            });
            if (!kind) {
                kind = 'child';
            }
            $location.url('/fabric/containers/createContainer').search('tab', kind);
        };

        $scope.createChildContainer = function (container) {
            if (!container.root || !container.alive) {
                return;
            }
            $location.url('/fabric/containers/createContainer').search({ 'tab': 'child', 'parentId': container.id });
        };

        $scope.showProfile = function (profile) {
            var version = profile.versionId || profile.version || $scope.activeVersionId;
            Fabric.gotoProfile(workspace, jolokia, localStorage, $location, version, profile);
        };

        $scope.getSelectedClass = function (obj) {
            var answer = [];
            if (obj.selected) {
                answer.push('selected');
            }
            if (angular.isDefined(obj['root']) && obj['root'] === false) {
                answer.push('child-container');
            }
            return answer.join(' ');
        };

        $scope.statusIcon = function (row) {
            return Fabric.statusIcon(row);
        };

        $scope.isEnsembleContainer = function (containerId) {
            if ($scope.ensembleContainerIds) {
                return $scope.ensembleContainerIds.any(containerId);
            }
            return false;
        };

        // for connection dialog
        $scope.connect = {
            dialog: new UI.Dialog(),
            saveCredentials: false,
            userName: null,
            password: null,
            container: null,
            view: null,
            onOK: function () {
                var userName = $scope.connect.userName;
                var password = $scope.connect.password;
                var container = $scope.connect.container;
                Fabric.log.info("Logging into container " + container + " with user " + userName);

                if ($scope.connect.saveCredentials) {
                    $scope.connect.saveCredentials = false;
                    if (userName) {
                        localStorage['fabric.userName'] = userName;
                    }
                    if (password) {
                        localStorage['fabric.password'] = password;
                    }
                }
                console.log("Connecting as user " + userName);
                var options = new Core.ConnectToServerOptions();
                options.view = $scope.connect.view;
                Fabric.connect(localStorage, container, userName, password, true, options);
                $scope.connect.container = {};
                setTimeout(function () {
                    $scope.connect.dialog.close();
                    Core.$apply($scope);
                }, 100);
            }
        };

        $scope.doConnect = function (container, view) {
            if (!$scope.canConnect(container)) {
                return;
            }

            // TODO at least obfusicate this
            $scope.connect.userName = Core.username || localStorage['fabric.userName'];
            $scope.connect.password = Core.password || localStorage['fabric.password'];
            $scope.connect.container = container;
            $scope.connect.view = view || "/openlogs";

            var alwaysPrompt = localStorage['fabricAlwaysPrompt'];
            if ((alwaysPrompt && alwaysPrompt !== "false") || !$scope.connect.userName || !$scope.connect.password) {
                $scope.connect.dialog.open();
            } else {
                $scope.connect.onOK();
            }
        };

        $scope.confirmDeleteDialog = {
            dialog: new UI.Dialog(),
            onOk: function () {
                $scope.confirmDeleteDialog.dialog.close();
                if (angular.isDefined($scope.containerId)) {
                    // avoid any nasty errors that the container doesn't existing anymore
                    Core.unregister(jolokia, $scope);
                    $location.path('/fabric/containers');

                    Fabric.doDeleteContainer($scope, jolokia, $scope.containerId);
                } else if (angular.isDefined($scope.selectedContainers)) {
                    $scope.selectedContainers.each(function (c) {
                        doDeleteContainer($scope, jolokia, c.id);
                    });
                } else {
                    // bail...
                    Fabric.log.info("Asked to delete containers but no containerId or selectedContainers attributes available");
                }
            },
            open: function () {
                $scope.confirmDeleteDialog.dialog.open();
            },
            close: function () {
                $scope.confirmDeleteDialog.dialog.close();
            }
        };

        $scope.createVersionDialog = {
            dialog: new UI.Dialog(),
            newVersionName: "",
            open: function () {
                Fabric.log.debug("Opening version dialog, $scope: ", $scope);
                $scope.createVersionDialog.newVersionName = "";
                $scope.createVersionDialog.dialog.open();
            },
            onOk: function () {
                Fabric.doCreateVersion($scope, jolokia, $location, $scope.createVersionDialog.newVersionName);
                $scope.createVersionDialog.newVersionName = "";
                $scope.createVersionDialog.dialog.close();
            }
        };

        $scope.$watch('selectedContainers', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                var num = $scope.selectedContainers.length;
                $scope.versionTitle = "Migrate " + Core.maybePlural(num, "Container") + " to:";
            }
        });

        $scope.onVersionChange = function (version) {
            var containerIds = [];

            if (angular.isDefined($scope.selectedContainers)) {
                containerIds = $scope.selectedContainers.map(function (c) {
                    return c.id;
                });
            } else if (angular.isDefined($scope.row)) {
                containerIds = [$scope.row.id];
            } else {
                return;
            }

            Fabric.log.info("Setting version to " + version + " on containers: " + containerIds);

            Fabric.migrateContainers(jolokia, version, containerIds, function () {
                notification('success', "Initiated container migration to version <strong>" + version + "</strong>, changes make take some time to complete");
                Core.$apply($scope);
            }, function (response) {
                Fabric.log.error("Failed to migrate containers due to ", response.error);
                Fabric.log.info("Stack trace: ", response.stacktrace);
                Core.$apply($scope);
            });
        };

        var verbose = workspace.localStorage['fabricVerboseNotifications'];
        $scope.fabricVerboseNotifications = verbose && verbose !== "false";
    }
    Fabric.initScope = initScope;

    function doCreateVersion($scope, jolokia, $location, newVersionName) {
        var success = function (response) {
            notification('success', "Created version <strong>" + response.value.id + "</strong>, switching to this new version");

            // broadcast events to force reloads
            var $rootScope = $scope.$root || $scope.$rootScope || $scope;
            if ($rootScope) {
                $rootScope.$broadcast('wikiBranchesUpdated');
            }

            var defaultTarget = '/wiki/branch/' + response.value.id + '/view/fabric/profiles';

            var path = $location.path();
            var branch = $scope.branch || $scope.$parent.branch;

            if (!path.startsWith('/wiki/branch/') || !branch) {
                $location.path(defaultTarget);
            } else {
                path = path.replace('/branch/' + branch, '/branch/' + response.value.id);
                $location.path(path);
            }
            Core.$apply($scope);
        };

        var error = function (response) {
            Fabric.log.error("Failed to create version due to :", response.error);
            Fabric.log.info("stack trace: ", response.stacktrace);
            Core.$apply($scope);
        };

        if (!Core.isBlank(newVersionName)) {
            Fabric.createVersionWithId(jolokia, newVersionName, success, error);
        } else {
            Fabric.createVersion(jolokia, success, error);
        }
    }
    Fabric.doCreateVersion = doCreateVersion;

    function sortVersions(versions, order) {
        return (versions || []).sortBy(function (v) {
            var answer = parseFloat(v['id']);
            if (answer === NaN) {
                answer = v['id'];
            }
            return answer;
        }, order);
    }
    Fabric.sortVersions = sortVersions;

    /**
    * Converts the given path from the wiki into a profile ID
    * @method pagePathToProfileId
    * @param {String} pageId
    * @return {String}
    */
    function pagePathToProfileId(pageId) {
        var answer = null;
        if (angular.isDefined(pageId) && pageId.has(Fabric.fabricTopLevel) && pageId !== Fabric.fabricTopLevel) {
            var profileId = pageId.remove(Fabric.fabricTopLevel);
            if ((Fabric.useDirectoriesInGit || !profileId.has("/"))) {
                var profileSeparator = profileId.indexOf(Fabric.profileSuffix + "/");
                var endsWithSuffix = profileId.endsWith(Fabric.profileSuffix);
                if (!Fabric.useDirectoriesInGit || endsWithSuffix || profileSeparator > 0) {
                    if (Fabric.useDirectoriesInGit) {
                        if (endsWithSuffix) {
                            profileId = Core.trimTrailing(profileId, Fabric.profileSuffix);
                        } else if (profileSeparator > 0) {
                            profileId = profileId.substring(0, profileSeparator);
                        }
                        profileId = profileId.replace(/\//g, "-");
                    }
                    answer = profileId;
                }
            }
        }
        return answer;
    }
    Fabric.pagePathToProfileId = pagePathToProfileId;

    function profilePath(profileId) {
        if (profileId) {
            return profileId.replace(/-/g, "/") + Fabric.profileSuffix;
        } else {
            return null;
        }
    }
    Fabric.profilePath = profilePath;

    function profileLink(workspace, jolokia, localStorage, versionId, profileId) {
        var path;
        if (Wiki.isWikiEnabled(workspace, jolokia, localStorage)) {
            path = "/wiki/branch/" + versionId + "/view/fabric/profiles/" + Fabric.profilePath(profileId);
        } else {
            path = "/fabric/profile/" + versionId + "/" + profileId;
        }
        return path;
    }
    Fabric.profileLink = profileLink;

    /**
    * Returns the CSS style for the number of containers badge
    * @method containerCountBadgeStyle
    * @param {Number} min
    * @param {number} count
    * @return {string}
    */
    function containerCountBadgeStyle(min, count) {
        if (min) {
            if (!count) {
                return "badge-important";
            } else {
                return min <= count ? "badge-success" : "badge-warning";
            }
        }
        return "";
    }
    Fabric.containerCountBadgeStyle = containerCountBadgeStyle;

    function gotoProfile(workspace, jolokia, localStorage, $location, versionId, profile) {
        var path = '';
        if (angular.isString(profile)) {
            path = profileLink(workspace, jolokia, localStorage, versionId, profile);
        } else {
            path = profileLink(workspace, jolokia, localStorage, versionId, profile.id);
        }
        if (!Core.isBlank(path)) {
            $location.url(path);
        }
    }
    Fabric.gotoProfile = gotoProfile;

    function setSelect(selection, group) {
        if (!angular.isDefined(selection)) {
            return group[0];
        }
        var answer = group.findIndex(function (item) {
            return item.id === selection.id;
        });
        if (answer !== -1) {
            return group[answer];
        } else {
            return group[0];
        }
    }
    Fabric.setSelect = setSelect;

    function doDeleteContainer($scope, jolokia, name, onDelete) {
        if (typeof onDelete === "undefined") { onDelete = null; }
        notification('info', "Deleting " + name);
        destroyContainer(jolokia, name, function () {
            notification('success', "Deleted " + name);
            if (onDelete) {
                onDelete();
            }
            Core.$apply($scope);
        }, function (response) {
            notification('error', "Failed to delete " + name + " due to " + response.error);
            Core.logJolokiaStackTrace(response);
            Core.$apply($scope);
        });
    }
    Fabric.doDeleteContainer = doDeleteContainer;

    function doStartContainer($scope, jolokia, name) {
        if ($scope.fabricVerboseNotifications) {
            notification('info', "Starting " + name);
        }
        startContainer(jolokia, name, function () {
            notification('success', "Started " + name);
            Core.$apply($scope);
        }, function (response) {
            notification('error', "Failed to start " + name + " due to " + response.error);
            Core.logJolokiaStackTrace(response);
            Core.$apply($scope);
        });
    }
    Fabric.doStartContainer = doStartContainer;

    function doStopContainer($scope, jolokia, name) {
        if ($scope.fabricVerboseNotifications) {
            notification('info', "Stopping " + name);
        }
        stopContainer(jolokia, name, function () {
            notification('success', "Stopped " + name);
            Core.$apply($scope);
        }, function (response) {
            notification('error', "Failed to stop " + name + " due to " + response.error);
            Core.logJolokiaStackTrace(response);
            Core.$apply($scope);
        });
    }
    Fabric.doStopContainer = doStopContainer;

    Fabric.urlResolvers = ['http:', 'ftp:', 'mvn:'];

    function completeUri($q, $scope, workspace, jolokia, something) {
    }
    Fabric.completeUri = completeUri;

    function applyPatches(jolokia, files, targetVersion, newVersionName, proxyUser, proxyPass, success, error) {
        doAction('applyPatches(java.util.List,java.lang.String,java.lang.String,java.lang.String,java.lang.String)', jolokia, [files, targetVersion, newVersionName, proxyUser, proxyPass], success, error);
    }
    Fabric.applyPatches = applyPatches;

    function setContainerProperty(jolokia, containerId, property, value, success, error) {
        doAction('setContainerProperty(java.lang.String, java.lang.String, java.lang.Object)', jolokia, [containerId, property, value], success, error);
    }
    Fabric.setContainerProperty = setContainerProperty;

    function deleteConfigFile(jolokia, version, profile, pid, success, error) {
        doAction('deleteConfigurationFile(java.lang.String, java.lang.String, java.lang.String)', jolokia, [version, profile, pid], success, error);
    }
    Fabric.deleteConfigFile = deleteConfigFile;

    function newConfigFile(jolokia, version, profile, pid, success, error) {
        doAction('setConfigurationFile(java.lang.String, java.lang.String, java.lang.String, java.lang.String)', jolokia, [version, profile, pid, ''], success, error);
    }
    Fabric.newConfigFile = newConfigFile;

    function saveConfigFile(jolokia, version, profile, pid, data, success, error) {
        doAction('setConfigurationFile(java.lang.String, java.lang.String, java.lang.String, java.lang.String)', jolokia, [version, profile, pid, data], success, error);
    }
    Fabric.saveConfigFile = saveConfigFile;

    function addProfilesToContainer(jolokia, container, profiles, success, error) {
        doAction('addProfilesToContainer(java.lang.String, java.util.List)', jolokia, [container, profiles], success, error);
    }
    Fabric.addProfilesToContainer = addProfilesToContainer;

    function removeProfilesFromContainer(jolokia, container, profiles, success, error) {
        doAction('removeProfilesFromContainer(java.lang.String, java.util.List)', jolokia, [container, profiles], success, error);
    }
    Fabric.removeProfilesFromContainer = removeProfilesFromContainer;

    function applyProfiles(jolokia, version, profiles, containers, success, error) {
        doAction('applyProfilesToContainers(java.lang.String, java.util.List, java.util.List)', jolokia, [version, profiles, containers], success, error);
    }
    Fabric.applyProfiles = applyProfiles;

    function migrateContainers(jolokia, version, containers, success, error) {
        doAction('applyVersionToContainers(java.lang.String, java.util.List)', jolokia, [version, containers], success, error);
    }
    Fabric.migrateContainers = migrateContainers;

    function changeProfileParents(jolokia, version, id, parents, success, error) {
        doAction('changeProfileParents(java.lang.String, java.lang.String, java.util.List)', jolokia, [version, id, parents], success, error);
    }
    Fabric.changeProfileParents = changeProfileParents;

    function createProfile(jolokia, version, id, parents, success, error) {
        doAction('createProfile(java.lang.String, java.lang.String, java.util.List)', jolokia, [version, id, parents], success, error);
    }
    Fabric.createProfile = createProfile;

    function copyProfile(jolokia, version, sourceName, targetName, force, success, error) {
        doAction('copyProfile(java.lang.String, java.lang.String, java.lang.String, boolean)', jolokia, [version, sourceName, targetName, force], success, error);
    }
    Fabric.copyProfile = copyProfile;

    function createVersionWithParentAndId(jolokia, base, id, success, error) {
        doAction('createVersion(java.lang.String, java.lang.String)', jolokia, [base, id], success, error);
    }
    Fabric.createVersionWithParentAndId = createVersionWithParentAndId;

    function createVersionWithId(jolokia, id, success, error) {
        doAction('createVersion(java.lang.String)', jolokia, [id], success, error);
    }
    Fabric.createVersionWithId = createVersionWithId;

    function createVersion(jolokia, success, error) {
        doAction('createVersion()', jolokia, [], success, error);
    }
    Fabric.createVersion = createVersion;

    function deleteVersion(jolokia, id, success, error) {
        doAction('deleteVersion(java.lang.String)', jolokia, [id], success, error);
    }
    Fabric.deleteVersion = deleteVersion;

    // TODO cache the current active version? Then clear the cached value if we delete it
    function getActiveVersion($location) {
        return $location.search()['cv'] || "1.0";
    }
    Fabric.getActiveVersion = getActiveVersion;

    function getContainerIdsForProfile(jolokia, version, profileId) {
        return jolokia.execute(Fabric.managerMBean, "containerIdsForProfile", version, profileId, { method: 'POST' });
    }
    Fabric.getContainerIdsForProfile = getContainerIdsForProfile;

    function deleteProfile(jolokia, version, id, success, error) {
        doAction('deleteProfile(java.lang.String, java.lang.String)', jolokia, [version, id], success, error);
    }
    Fabric.deleteProfile = deleteProfile;

    function profileWebAppURL(jolokia, webAppId, profileId, versionId, success, error) {
        doAction('profileWebAppURL', jolokia, [webAppId, profileId, versionId], success, error);
    }
    Fabric.profileWebAppURL = profileWebAppURL;

    function onJolokiaUrlCreateJolokia(response, fn) {
        var jolokia = null;
        if (response) {
            var url = response.value;
            if (url) {
                // lets use a proxy if the URL is external
                url = Core.useProxyIfExternal(url);
                jolokia = Fabric.createJolokia(url);
            } else {
                if (response.error) {
                    Fabric.log.warn(response.error, response.stacktrace);
                }
            }
            if (fn) {
                fn(jolokia);
            }
        }
        return jolokia;
    }

    /**
    * Attempts to create a jolokia for the given profile and version passing the created object
    * into the onJolokia function
    * @method profileJolokia
    *
    * @paran {*} jolokia
    * @param {String} profileId
    * @param {String} versionId
    * @param {Function} onJolokia a function to receive the jolokia object or null if one cannot be created
    */
    function profileJolokia(jolokia, profileId, versionId, onJolokia) {
        function onJolokiaUrl(response) {
            return onJolokiaUrlCreateJolokia(response, onJolokia);
        }
        if (profileId && versionId) {
            return Fabric.profileWebAppURL(jolokia, Fabric.jolokiaWebAppGroupId, profileId, versionId, onJolokiaUrl, onJolokiaUrl);
        } else {
            onJolokia(null);
        }
    }
    Fabric.profileJolokia = profileJolokia;

    /**
    * Attempts to create a jolokia for the given container id, passing the created object
    * into the onJolokia function
    * @method containerJolokia
    * @paran {*} jolokia
    * @param {String} containerId the id of the container to connect to
    * @param {Function} onJolokia a function to receive the jolokia object or null if one cannot be created
    */
    function containerJolokia(jolokia, containerId, onJolokia) {
        function onJolokiaUrl(response) {
            return onJolokiaUrlCreateJolokia(response, onJolokia);
        }
        return Fabric.containerWebAppURL(jolokia, Fabric.jolokiaWebAppGroupId, containerId, onJolokiaUrl, onJolokiaUrl);
    }
    Fabric.containerJolokia = containerJolokia;

    function containerWebAppURL(jolokia, webAppId, containerId, success, error) {
        doAction('containerWebAppURL', jolokia, [webAppId, containerId], success, error);
    }
    Fabric.containerWebAppURL = containerWebAppURL;

    function doAction(action, jolokia, arguments, success, error) {
        jolokia.request({
            type: 'exec', mbean: Fabric.managerMBean,
            operation: action,
            arguments: arguments
        }, {
            method: 'POST',
            success: success,
            error: error
        });
    }
    Fabric.doAction = doAction;

    function stopContainer(jolokia, id, success, error) {
        doAction('stopContainer(java.lang.String)', jolokia, [id], success, error);
    }
    Fabric.stopContainer = stopContainer;

    function destroyContainer(jolokia, id, success, error) {
        doAction('destroyContainer(java.lang.String)', jolokia, [id], success, error);
    }
    Fabric.destroyContainer = destroyContainer;

    function startContainer(jolokia, id, success, error) {
        doAction('startContainer(java.lang.String)', jolokia, [id], success, error);
    }
    Fabric.startContainer = startContainer;

    function getServiceList(container) {
        var answer = [];
        var javaContainer = true;
        if (angular.isDefined(container) && angular.isDefined(container.jmxDomains) && angular.isArray(container.jmxDomains) && container.alive) {
            answer = Fabric.serviceIconRegistry.getIcons(container.jmxDomains);
        }
        return answer;
    }
    Fabric.getServiceList = getServiceList;

    function getTypeIcon(container) {
        var type = container.type;

        // use the type in the metadata if it's there...
        if (container.metadata && container.metadata.containerType) {
            type = container.metadata.containerType;
        }
        var answer = Fabric.containerIconRegistry.getIcon(type);
        Fabric.log.debug("Icon for ", container, " : ", answer);
        if (!answer) {
            return Fabric.javaIcon;
        } else {
            return answer;
        }
    }
    Fabric.getTypeIcon = getTypeIcon;

    /**
    * Returns the default version ID for the current fabric
    * @param jolokia
    * @returns the version ID as a string; or defaults to 1.0 if not available
    */
    function getDefaultVersionId(jolokia) {
        return (getDefaultVersion(jolokia) || {})["id"] || "1.0";
    }
    Fabric.getDefaultVersionId = getDefaultVersionId;

    /**
    * Returns the default version object for the current fabric
    * @param jolokia
    * @returns the version object
    */
    function getDefaultVersion(jolokia) {
        return jolokia.execute(Fabric.managerMBean, "defaultVersion()");
    }
    Fabric.getDefaultVersion = getDefaultVersion;

    /**
    * Default the values that are missing in the returned JSON
    * @method defaultContainerValues
    * @param {Workspace} workspace
    * @param {any} $scope
    * @param {Array} values
    */
    function defaultContainerValues(workspace, $scope, values) {
        var map = {};
        angular.forEach(values, function (row) {
            var profileIds = row["profileIds"];
            if (profileIds) {
                angular.forEach(profileIds, function (profileId) {
                    var containers = map[profileId];
                    if (!containers) {
                        containers = [];
                        map[profileId] = containers;
                    }
                    containers.push(row);
                });
            }
            $scope.profileMap = map;
            row["link"] = containerLinks(workspace, row["id"]);
            row["profileLinks"] = profileLinks(workspace, row["versionId"], profileIds);

            var versionId = row["versionId"];
            var versionHref = url("#/fabric/profiles?v=" + versionId);
            var versionLink = "<a href='" + versionHref + "'>" + versionId + "</a>";
            row["versionHref"] = versionHref;
            row["versionLink"] = versionLink;

            var id = row['id'] || "";
            var title = "container " + id + " ";
            var img = "red-dot.png";
            if (row['managed'] === false) {
                img = "spacer.gif";
            } else if (!row['alive']) {
                img = "gray-dot.png";
            } else if (row['provisionPending']) {
                img = "pending.gif";
            } else if (row['provisionStatus'] === 'success') {
                img = "green-dot.png";
            }
            img = "img/dots/" + img;
            row["statusImageHref"] = img;
            row["link"] = "<img src='" + img + "' title='" + title + "'/> " + (row["link"] || id);
        });
        return values;
    }
    Fabric.defaultContainerValues = defaultContainerValues;

    // TODO move to core?
    function toCollection(values) {
        var collection = values;
        if (!angular.isArray(values)) {
            collection = [values];
        }
        return collection;
    }
    Fabric.toCollection = toCollection;

    function containerLinks(workspace, values) {
        var answer = "";
        angular.forEach(toCollection(values), function (value, key) {
            var prefix = "";
            if (answer.length > 0) {
                prefix = " ";
            }
            answer += prefix + "<a href='" + url("#/fabric/container/" + value + workspace.hash()) + "'>" + value + "</a>";
        });
        return answer;
    }
    Fabric.containerLinks = containerLinks;

    function profileLinks(workspace, versionId, values) {
        var answer = "";
        angular.forEach(toCollection(values), function (value, key) {
            var prefix = "";
            if (answer.length > 0) {
                prefix = " ";
            }
            answer += prefix + "<a href='" + url("#/fabric/profile/" + versionId + "/" + value + workspace.hash()) + "'>" + value + "</a>";
        });
        return answer;
    }
    Fabric.profileLinks = profileLinks;

    /**
    * Default the values that are missing in the returned JSON
    * @method defaultProfileValues
    * @param {Workspace} workspace
    * @param {String} versionId
    * @param {Array} values
    */
    function defaultProfileValues(workspace, versionId, values) {
        angular.forEach(values, function (row) {
            var id = row["id"];
            row["link"] = profileLinks(workspace, versionId, id);
            row["parentLinks"] = profileLinks(workspace, versionId, row["parentIds"]);
            var containersHref = url("#/fabric/containers?p=" + id);
            var containerCount = row["containerCount"];
            var containersLink = "";
            if (containerCount) {
                containersLink = "<a href='" + containersHref + "'>" + containerCount + "</a>";
            }
            row["containersCountLink"] = containersLink;
            row["containersHref"] = containersHref;
        });
        return values;
    }
    Fabric.defaultProfileValues = defaultProfileValues;

    function getZooKeeperFacadeMBean(workspace) {
        var folder = workspace.findMBeanWithProperties(Fabric.jmxDomain, { type: "ZooKeeper" });
        return Core.pathGet(folder, "objectName");
    }
    Fabric.getZooKeeperFacadeMBean = getZooKeeperFacadeMBean;

    function statusTitle(container) {
        var answer = 'Alive';
        if (!container.alive) {
            answer = 'Not Running';
        } else {
            answer += ' - ' + humanizeValue(container.provisionResult);
        }
        return answer;
    }
    Fabric.statusTitle = statusTitle;

    function statusIcon(row) {
        if (row) {
            if (row.alive) {
                switch (row.provisionResult) {
                    case 'success':
                        return "green icon-play-circle";
                    case 'downloading':
                        return "icon-download-alt";
                    case 'installing':
                        return "icon-hdd";
                    case 'analyzing':
                    case 'finalizing':
                        return "icon-refresh icon-spin";
                    case 'resolving':
                        return "icon-sitemap";
                    case 'error':
                        return "red icon-warning-sign";
                }
            } else {
                return "orange icon-off";
            }
        }
        return "icon-refresh icon-spin";
    }
    Fabric.statusIcon = statusIcon;

    /**
    * Opens a window connecting to the given container row details if the jolokiaUrl is available
    * @method connect
    * @param {any} localStorage
    * @param {any} row
    * @param {String} userName
    * @param {String} password
    * @param {Boolean} useProxy
    * @param {ConnectToServerOptions} options
    */
    function connect(localStorage, row, userName, password, useProxy, options) {
        if (typeof userName === "undefined") { userName = ""; }
        if (typeof password === "undefined") { password = ""; }
        if (typeof useProxy === "undefined") { useProxy = true; }
        if (typeof options === "undefined") { options = new Core.ConnectToServerOptions(); }
        options.name = row.id;
        options.jolokiaUrl = row.jolokiaUrl;
        options.userName = userName;
        options.password = password;
        options.useProxy = useProxy;

        Core.connectToServer(localStorage, options);
    }
    Fabric.connect = connect;

    /**
    * Creates a jolokia object for connecting to the container with the given remote jolokia URL
    * @method createJolokia
    * @param {String} url
    */
    function createJolokia(url) {
        // lets default to the user/pwd for the login
        // TODO maybe allow these to be configured to other values?
        var username = Core.username;
        var password = Core.password;
        if (!username) {
            // lets try reverse engineer the user/pwd from the stored user/pwd
            var jsonText = localStorage[url];
            if (jsonText) {
                var obj = Wiki.parseJson(jsonText);
                if (obj) {
                    username = obj["username"];
                    password = obj["password"];
                }
            }
        }
        Fabric.log.info("Logging into remote jolokia " + url + " using username: " + username);
        return Core.createJolokia(url, username, password);
    }
    Fabric.createJolokia = createJolokia;

    function registeredProviders(jolokia) {
        var providers = jolokia.execute(Fabric.managerMBean, 'registeredProviders()');
        var answer = {};
        angular.forEach(providers, function (value, key) {
            answer[key] = {
                id: key,
                className: value
            };
        });
        return answer;
    }
    Fabric.registeredProviders = registeredProviders;

    function getSchema(id, className, jolokia, cb) {
        jolokia.execute(Fabric.schemaLookupMBean, 'getSchemaForClass(java.lang.String)', className, {
            method: 'POST',
            success: function (value) {
                cb(Fabric.customizeSchema(id, angular.fromJson(value)));
            }
        });
    }
    Fabric.getSchema = getSchema;

    function getDtoSchema(id, className, jolokia, cb) {
        jolokia.execute(Fabric.schemaLookupMBean, 'getSchemaForClass(java.lang.String)', className, {
            method: 'POST',
            success: function (value) {
                cb(angular.fromJson(value));
            }
        });
    }
    Fabric.getDtoSchema = getDtoSchema;

    function getCurrentContainer(jolokia, fields) {
        var name = jolokia.getAttribute(Fabric.managerMBean, 'CurrentContainerName', { method: 'POST' });
        return jolokia.execute(Fabric.managerMBean, "getContainer(java.lang.String, java.util.List)", name, fields, { method: 'POST' });
    }
    Fabric.getCurrentContainer = getCurrentContainer;

    function getContainerFields(jolokia, name, fields) {
        return jolokia.execute(Fabric.managerMBean, "getContainer(java.lang.String, java.util.List)", name, fields, { method: 'POST' });
    }
    Fabric.getContainerFields = getContainerFields;

    function getRootContainers(jolokia) {
        var fields = ["id", "root"];
        var answer = jolokia.execute(Fabric.managerMBean, "containers(java.util.List)", fields, { method: 'POST' });
        return answer.filter({ root: true }).map(function (v) {
            return v["id"];
        });
    }
    Fabric.getRootContainers = getRootContainers;

    /**
    * Queries the given fields on the contianers in the fabric invoking the given function or returning the results if the fn is null
    * @param jolokia
    * @param fields
    * @param fn
    * @return the result if fn is null
    */
    function getContainersFields(jolokia, fields, fn) {
        if (typeof fn === "undefined") { fn = null; }
        return jolokia.execute(Fabric.managerMBean, "containers(java.util.List)", fields, onSuccess(fn));
    }
    Fabric.getContainersFields = getContainersFields;

    function getOpenShiftDomains(workspace, jolokia, serverUrl, login, password, fn, onError) {
        if (typeof fn === "undefined") { fn = null; }
        if (typeof onError === "undefined") { onError = null; }
        if (hasOpenShiftFabric(workspace) && serverUrl && login && password) {
            var options = onSuccess(fn, { error: onError });
            return jolokia.execute(Fabric.openShiftFabricMBean, "getDomains", serverUrl, login, password, options);
        } else {
            if (fn) {
                fn([]);
            }
            return [];
        }
    }
    Fabric.getOpenShiftDomains = getOpenShiftDomains;

    function getOpenShiftGearProfiles(workspace, jolokia, serverUrl, login, password, fn) {
        if (typeof fn === "undefined") { fn = null; }
        if (hasOpenShiftFabric(workspace) && serverUrl && login && password) {
            return jolokia.execute(Fabric.openShiftFabricMBean, "getGearProfiles", serverUrl, login, password, onSuccess(fn));
        } else {
            if (fn) {
                fn([]);
            }
            return [];
        }
    }
    Fabric.getOpenShiftGearProfiles = getOpenShiftGearProfiles;

    function filterProfiles(jolokia, versionId, profileIds) {
        var profiles = [];
        if (versionId) {
            profiles = jolokia.execute(Fabric.managerMBean, "getProfiles(java.lang.String, java.util.List)", versionId, ['id', 'hidden', 'abstract'], { method: 'POST' });
        }

        profiles = profiles.filter(function (profile) {
            return profileIds.some(function (id) {
                return profile.id === id;
            });
        });
        profiles = profiles.filter((function (profile) {
            return !profile.abstract && !profile.hidden;
        }));

        return profiles.map(function (p) {
            return p.id;
        });
    }
    Fabric.filterProfiles = filterProfiles;

    function getProfileData(jolokia, versionId, profileId, fields) {
        return jolokia.execute(Fabric.managerMBean, "getProfile(java.lang.String, java.lang.String, java.util.List)", versionId, profileId, fields, { method: 'POST' });
    }
    Fabric.getProfileData = getProfileData;

    function getConfigFile(jolokia, versionId, profileId, fileName, fn) {
        if (typeof fn === "undefined") { fn = null; }
        function onResults(answer) {
            return answer ? answer.decodeBase64() : null;
        }

        var callback = !fn ? null : function (result) {
            fn(onResults(result));
        };
        var answer = jolokia.execute(Fabric.managerMBean, "getConfigurationFile(java.lang.String, java.lang.String, java.lang.String)", versionId, profileId, fileName, onSuccess(callback));
        return fn ? answer : onResults(answer);
    }
    Fabric.getConfigFile = getConfigFile;

    /**
    * Creates a link to the given broker configuration so we can connect in the UI
    * @param workspace
    * @param jolokia
    * @param localStorage
    * @param brokerVersion
    * @param brokerProfile
    * @param brokerId
    * @return the link to the broker page
    */
    function brokerConfigLink(workspace, jolokia, localStorage, brokerVersion, brokerProfile, brokerId) {
        var path = Fabric.profileLink(workspace, jolokia, localStorage, brokerVersion, brokerProfile);
        path += "/org.fusesource.mq.fabric.server-" + brokerId + ".properties";
        return path;
    }
    Fabric.brokerConfigLink = brokerConfigLink;

    /**
    * Connects to the broker in a new window
    */
    function connectToBroker($scope, container, postfix) {
        if (typeof postfix === "undefined") { postfix = null; }
        var view = "/jmx/attributes?tab=activemq";
        if (postfix) {
            view += "&" + postfix;
        }
        $scope.doConnect(container, view);
    }
    Fabric.connectToBroker = connectToBroker;

    /**
    * Removes any attributes from the object that are set to an empty string.
    *
    * @method sanitizeJson
    * @for Fabric
    * @param {Object} json
    * @return {Object}
    */
    function sanitizeJson(json) {
        angular.forEach(json, function (value, key) {
            if (value === "") {
                delete json[key];
            }
        });
        return json;
    }
    Fabric.sanitizeJson = sanitizeJson;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function FabricBrokersController($scope, localStorage, $routeParams, $location, jolokia, workspace, $compile, $templateCache) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        $scope.maps = {
            group: {},
            profile: {},
            broker: {},
            container: {}
        };

        $scope.showBroker = function (broker) {
            var brokerVersion = broker.version;
            var brokerProfile = broker.profile;
            var brokerId = broker.id;
            var path = Fabric.brokerConfigLink(workspace, jolokia, localStorage, brokerVersion, brokerProfile, brokerId);
            $location.path(path);
        };

        $scope.connectToBroker = function (container, broker) {
            Fabric.connectToBroker($scope, container);
        };

        $scope.createBroker = function (group, profile) {
            var args = {};
            if (group) {
                var groupId = group["id"];
                if (groupId) {
                    args["group"] = groupId;
                }
            }
            if (profile) {
                var profileId = profile["id"];
                if (profileId) {
                    args["profile"] = profileId;
                }
            }
            $location.url("/fabric/mq/createBroker").search(args);
        };

        function matchesFilter(text) {
            var filter = $scope.searchFilter;
            return !filter || (text && text.toLowerCase().has(filter.toLowerCase()));
        }

        $scope.groupMatchesFilter = function (group) {
            return matchesFilter(group.id) || group.profiles.find(function (item) {
                return $scope.profileMatchesFilter(item);
            });
        };

        $scope.profileMatchesFilter = function (profile) {
            return matchesFilter(profile.id) || matchesFilter(profile.group) || matchesFilter(profile.version) || profile.brokers.find(function (item) {
                return $scope.brokerMatchesFilter(item);
            });
        };

        $scope.brokerMatchesFilter = function (broker) {
            return matchesFilter(broker.id) || matchesFilter(broker.group) || matchesFilter(broker.version) || broker.containers.find(function (item) {
                return $scope.containerMatchesFilter(item);
            });
        };

        $scope.containerMatchesFilter = function (container) {
            return matchesFilter(container.id) || matchesFilter(container.group) || matchesFilter(container.profile) || matchesFilter(container.version) || matchesFilter(container.brokerName) || (container.master && $scope.searchFilter && $scope.searchFilter.has("master"));
        };

        if (Fabric.hasMQManager) {
            Core.register(jolokia, $scope, { type: 'exec', mbean: Fabric.mqManagerMBean, operation: "loadBrokerStatus()" }, onSuccess(onBrokerData));
        }

        function onBrokerData(response) {
            if (response) {
                var responseJson = angular.toJson(response.value);
                if ($scope.responseJson === responseJson) {
                    return;
                }

                $scope.responseJson = responseJson;

                var brokers = response.value;

                function findByIdOrCreate(collection, id, map, fn) {
                    var value = collection.find(function (v) {
                        return v && v['id'] === id;
                    });
                    if (!value) {
                        value = fn();
                        value["id"] = id;
                        collection.push(value);

                        var old = map[id];

                        // copy any view related across
                        value["expanded"] = old ? old["expanded"] : true;
                        map[id] = value;
                    }
                    return value;
                }

                $scope.groups = [];
                var maps = $scope.maps;

                angular.forEach(brokers, function (brokerStatus) {
                    var groupId = brokerStatus.group || "Unknown";
                    var profileId = brokerStatus.profile || "Unknown";
                    var brokerId = brokerStatus.brokerName || "Unknown";
                    var containerId = brokerStatus.container;
                    var versionId = brokerStatus.version || "1.0";

                    var group = findByIdOrCreate($scope.groups, groupId, maps.group, function () {
                        return {
                            profiles: []
                        };
                    });
                    var profile = findByIdOrCreate(group.profiles, profileId, maps.profile, function () {
                        return {
                            group: groupId,
                            version: versionId,
                            requirements: {
                                minimumInstances: brokerStatus.minimumInstances
                            },
                            brokers: [],
                            containers: {}
                        };
                    });
                    var connectTo = (brokerStatus.networks || []).join(",");
                    var broker = findByIdOrCreate(profile.brokers, brokerId, maps.broker, function () {
                        return {
                            group: groupId,
                            profile: profileId,
                            version: versionId,
                            containers: [],
                            connectTo: connectTo
                        };
                    });
                    if (containerId) {
                        // lets create a container object per broker for the N+1 case
                        var container = findByIdOrCreate(broker.containers, containerId, maps.container, function () {
                            return brokerStatus;
                        });
                        if (container.master) {
                            container.masterTooltip = " is the master for broker: " + brokerId;
                        }
                        profile.containers[containerId] = container;
                    }
                });

                // update the stats
                angular.forEach($scope.groups, function (group) {
                    angular.forEach(group.profiles, function (profile) {
                        angular.forEach(profile.brokers, function (broker) {
                            broker.containers = broker.containers.sortBy(function (c) {
                                return c.id;
                            });
                        });
                        var count = Object.values(profile.containers).length;
                        var required = profile.requirements.minimumInstances || 0;
                        profile.requireStyle = Fabric.containerCountBadgeStyle(required, count);
                        profile.count = count;
                        profile.requiredToolTip = "this profile requires " + Core.maybePlural(required, "container") + " to be running but is currently running " + Core.maybePlural(count, "container");
                        if (required > count) {
                            profile.requiredToolTip += ". click here to create more containers";
                        }
                    });
                });

                Core.$apply($scope);
            }
        }
    }
    Fabric.FabricBrokersController = FabricBrokersController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    var ProfileSelector = (function () {
        function ProfileSelector() {
            this.restrict = 'A';
            this.replace = true;
            this.templateUrl = Fabric.templatePath + "profileSelector.html";
            this.scope = {
                selectedProfiles: '=fabricProfileSelector',
                versionId: '=',
                filterWatch: '@',
                selectedWatch: '@',
                clearOnVersionChange: '@',
                noLinks: '@',
                showHeader: '@',
                useCircles: '@',
                expanded: '@',
                excludedProfiles: '=',
                includedProfiles: '='
            };
            this.controller = function ($scope, $element, $attrs, workspace, jolokia, localStorage, $location) {
                $scope.profiles = [];
                $scope.responseJson = '';
                $scope.filterText = '';
                $scope.clearOnVersionChange = false;
                $scope.noLinks = false;
                $scope.selectedAll = false;
                $scope.indeterminate = false;
                $scope.showFilter = true;
                $scope.useCircles = false;
                $scope.expanded = false;
                $scope.tree = [];

                $scope.showProfile = function (profile) {
                    return Core.matchFilterIgnoreCase(profile.id, $scope.filterText);
                };

                $scope.showBranch = function (branch) {
                    return $scope.filterText.isBlank() || branch.profiles.some(function (profile) {
                        return Core.matchFilterIgnoreCase(profile.id, $scope.filterText);
                    });
                };

                $scope.goto = function (profile) {
                    Fabric.gotoProfile(workspace, jolokia, localStorage, $location, $scope.versionId, profile);
                };

                $scope.render = function (response) {
                    var responseJson = angular.toJson(response.value);
                    if ($scope.responseJson !== responseJson) {
                        $scope.responseJson = responseJson;
                        var selected = $scope.selectedProfiles;
                        $scope.profiles = response.value.sortBy(function (profile) {
                            return profile.id;
                        });
                        angular.forEach(selected, function (profile) {
                            var p = $scope.profiles.find(function (p) {
                                return p.id === profile.id;
                            });
                            if (p && profile) {
                                p.selected = profile.selected;
                            }
                        });

                        $scope.profiles = $scope.profiles.exclude(function (p) {
                            return p.hidden;
                        });

                        if ($scope.excludedProfiles) {
                            $scope.profiles = $scope.excluded();
                        }

                        if ($scope.includedProfiles) {
                            $scope.profiles = $scope.included();
                        }

                        var paths = [];

                        $scope.profiles.each(function (profile) {
                            var path = profile.id.split('-');
                            profile.name = path.last();
                            profile.path = path.exclude(profile.name).join(' / ');
                            paths.push(profile.path);
                        });

                        paths = paths.unique().sortBy('length').sortBy(function (n) {
                            return n;
                        });
                        var tree = [];
                        paths.forEach(function (path) {
                            var branch = {
                                expanded: $scope.expanded,
                                path: path,
                                profiles: $scope.profiles.filter(function (profile) {
                                    return profile.path === path;
                                })
                            };
                            tree.push(branch);
                        });
                        $scope.tree = tree;

                        Core.$apply($scope);
                    }
                };

                $scope.excluded = function () {
                    return $scope.profiles.exclude(function (p) {
                        return $scope.excludedProfiles.some(function (e) {
                            return e === p.id;
                        });
                    });
                };

                $scope.included = function () {
                    return $scope.profiles.exclude(function (p) {
                        return $scope.includedProfiles.none(function (e) {
                            return e === p.id;
                        });
                    });
                };

                $scope.isOpen = function (branch) {
                    if ($scope.filterText !== '') {
                        return "opened";
                    }
                    if (branch.expanded) {
                        return "opened";
                    }
                    return "closed";
                };

                $scope.isOpenIcon = function (branch) {
                    if (branch.expanded) {
                        return "icon-folder-open";
                    }
                    return "icon-folder-closed";
                };

                $scope.$watch('includedProfiles', function (newValue, oldValue) {
                    if (newValue !== oldValue) {
                        $scope.init();
                    }
                }, true);

                $scope.$watch('excludedProfiles', function (newValue, oldValue) {
                    if (newValue !== oldValue) {
                        $scope.init();
                    }
                }, true);

                $scope.abstract = function () {
                    return $scope.profiles.filter(function (profile) {
                        return profile['abstract'];
                    });
                };

                $scope.selected = function () {
                    return $scope.profiles.filter(function (profile) {
                        return profile['selected'];
                    });
                };

                $scope.selectAll = function () {
                    $scope.profiles.each(function (profile) {
                        profile.selected = true;
                    });
                };

                $scope.selectNone = function () {
                    $scope.profiles.each(function (profile) {
                        delete profile.selected;
                    });
                };

                $scope.$parent.profileSelectAll = $scope.selectAll;
                $scope.$parent.profileSelectNone = $scope.selectNone;

                $scope.getSelectedClass = function (profile) {
                    if (profile.abstract) {
                        return "abstract";
                    }
                    if (profile.selected) {
                        return "selected";
                    }
                    return "";
                };

                $scope.$watch('selectedAll', function (newValue, oldValue) {
                    if (!$scope.indeterminate && newValue !== oldValue) {
                        if (newValue) {
                            $scope.selectAll();
                        } else {
                            $scope.selectNone();
                        }
                    }
                });

                $scope.$watch('profiles', function (newValue, oldValue) {
                    if (newValue !== oldValue) {
                        $scope.selectedProfiles = $scope.selected();
                    }
                }, true);

                $scope.$on("fabricProfileRefresh", function () {
                    if ($scope.versionId) {
                        jolokia.request({
                            type: 'exec', mbean: Fabric.managerMBean,
                            operation: 'getProfiles(java.lang.String, java.util.List)',
                            arguments: [$scope.versionId, ['id', 'hidden']]
                        }, {
                            method: 'POST',
                            success: function (response) {
                                $scope.render(response);
                            }
                        });
                    }
                });

                $scope.init = function () {
                    Fabric.log.debug("Initializing profile selector, version: ", $scope.versionId);
                    $scope.responseJson = null;
                    Core.unregister(jolokia, $scope);
                    if ($scope.versionId !== '') {
                        if ($scope.clearOnVersionChange) {
                            $scope.selectNone();
                        }
                        if ($scope.versionId) {
                            Core.register(jolokia, $scope, {
                                type: 'exec',
                                mbean: Fabric.managerMBean,
                                operation: 'getProfiles(java.lang.String, java.util.List)',
                                arguments: [$scope.versionId, ['id', 'hidden', 'abstract']]
                            }, onSuccess($scope.render, {
                                error: function (response) {
                                    // TODO somewhere this directive is kinda getting leaked, need to track down
                                    Fabric.log.debug("Error fetching profiles: ", response.error, " unregistering poller");
                                    Core.unregister(jolokia, $scope);
                                }
                            }));
                        }
                    }
                };

                $scope.$watch('versionId', function (newValue, oldValue) {
                    Fabric.log.debug("versionId, newValue: ", newValue, " oldValue: ", oldValue);
                    if ($scope.versionId && $scope.versionId !== '') {
                        Fabric.log.debug("Unregistering old poller");
                        Core.unregister(jolokia, $scope);
                        jolokia.request({
                            type: 'exec',
                            mbean: Fabric.managerMBean,
                            operation: 'versions()',
                            arguments: []
                        }, {
                            method: 'POST',
                            success: function (response) {
                                if (response.value.some(function (version) {
                                    return version.id === newValue;
                                })) {
                                    Fabric.log.debug("registering new poller");
                                    $scope.init();
                                    Core.$apply($scope);
                                }
                            },
                            error: function (response) {
                                Core.$apply($scope);
                            }
                        });
                    }
                });
            };
            this.link = function ($scope, $element, $attrs) {
                var selector = $element.find('#selector');

                if (!angular.isDefined($attrs['showHeader'])) {
                    $scope.showFilter = true;
                } else {
                    $scope.showFilter = $attrs['showHeader'];
                }

                if (angular.isDefined($attrs['filterWatch'])) {
                    $scope.$parent.$watch($attrs['filterWatch'], function (newValue, oldValue) {
                        if (newValue !== oldValue) {
                            $scope.filterText = newValue;
                        }
                    });
                }

                $scope.$watch('indeterminate', function (newValue, oldValue) {
                    if (newValue !== oldValue) {
                        selector.prop('indeterminate', $scope.indeterminate);
                    }
                });

                $scope.$watch('selectedProfiles', function (newValue, oldValue) {
                    if (newValue !== oldValue) {
                        if ($scope.selectedProfiles.length > 0) {
                            if ($scope.selectedProfiles.length !== $scope.profiles.length) {
                                $scope.indeterminate = true;
                                $scope.selectedAll = false;

                                $scope.$parent.profileSomeSelected = true;
                                $scope.$parent.profileNoneSelected = false;
                                $scope.$parent.profileAllSelected = false;
                            } else {
                                $scope.indeterminate = false;
                                $scope.selectedAll = true;

                                $scope.$parent.profileSomeSelected = false;
                                $scope.$parent.profileNoneSelected = false;
                                $scope.$parent.profileAllSelected = true;
                            }
                        } else {
                            $scope.indeterminate = false;
                            $scope.selectedAll = false;

                            $scope.$parent.profileSomeSelected = false;
                            $scope.$parent.profileNoneSelected = true;
                            $scope.$parent.profileAllSelected = false;
                        }
                    }
                }, true);
            };
        }
        return ProfileSelector;
    })();
    Fabric.ProfileSelector = ProfileSelector;
})(Fabric || (Fabric = {}));
/**
* @module Fabric
* @main Fabric
*/
var Fabric;
(function (Fabric) {
    Fabric.templatePath = 'app/fabric/html/';
    Fabric.activeMQTemplatePath = 'app/activemq/html/';

    Fabric.currentContainerId = '';
    Fabric.currentContainer = {};

    angular.module('fabric', ['bootstrap', 'ui.bootstrap', 'ui.bootstrap.dialog', 'ngResource', 'ngGrid', 'hawtio-forms', 'hawtioCore', 'ngDragDrop', 'wiki']).config(function ($routeProvider) {
        $routeProvider.when('/fabric/containers/createContainer', { templateUrl: Fabric.templatePath + 'createContainer.html', reloadOnSearch: false }).when('/fabric/map', { templateUrl: Fabric.templatePath + 'map.html' }).when('/fabric/clusters/*page', { templateUrl: Fabric.templatePath + 'clusters.html' }).when('/fabric/containers', { templateUrl: Fabric.templatePath + 'containers.html', reloadOnSearch: false }).when('/fabric/container/:containerId', { templateUrl: Fabric.templatePath + 'container.html', reloadOnSearch: false }).when('/fabric/assignProfile', { templateUrl: Fabric.templatePath + 'assignProfile.html' }).when('/fabric/activeProfiles', { templateUrl: Fabric.templatePath + 'activeProfiles.html' }).when('/wiki/profile/:versionId/:profileId/editFeatures', { templateUrl: Fabric.templatePath + 'editFeatures.html' }).when('/fabric/profile/:versionId/:profileId/:fname', { templateUrl: Fabric.templatePath + 'pid.html' }).when('/fabric/view', { templateUrl: Fabric.templatePath + 'fabricView.html', reloadOnSearch: false }).when('/fabric/migrate', { templateUrl: Fabric.templatePath + 'migrateVersions.html' }).when('/fabric/patching', { templateUrl: Fabric.templatePath + 'patching.html' }).when('/fabric/configurations/:versionId/:profileId', { templateUrl: 'app/osgi/html/configurations.html' }).when('/fabric/configuration/:versionId/:profileId/:pid', { templateUrl: 'app/osgi/html/pid.html' }).when('/fabric/configuration/:versionId/:profileId/:pid/:factoryPid', { templateUrl: 'app/osgi/html/pid.html' }).when('/fabric/mq/brokers', { templateUrl: Fabric.templatePath + 'brokers.html' }).when('/fabric/mq/brokerDiagram', { templateUrl: Fabric.activeMQTemplatePath + 'brokerDiagram.html', reloadOnSearch: false }).when('/fabric/mq/brokerNetwork', { templateUrl: Fabric.templatePath + 'brokerNetwork.html' }).when('/fabric/mq/createBroker', { templateUrl: Fabric.templatePath + 'createBroker.html' }).when('/fabric/camel/diagram', { templateUrl: 'app/camel/html/fabricDiagram.html', reloadOnSearch: false }).when('/fabric/api', { templateUrl: Fabric.templatePath + 'apis.html' }).when('/fabric/api/wsdl', { templateUrl: 'app/api/html/wsdl.html' }).when('/fabric/api/wadl', { templateUrl: 'app/api/html/wadl.html' }).when('/fabric/test', { templateUrl: Fabric.templatePath + 'test.html' });
    }).directive('fabricVersionSelector', function ($templateCache) {
        return Fabric.VersionSelector($templateCache);
    }).directive('fabricProfileSelector', function () {
        return new Fabric.ProfileSelector();
    }).directive('fabricContainerList', function () {
        return new Fabric.ContainerList();
    }).directive('fabricProfileDetails', function () {
        return new Fabric.ProfileDetails();
    }).directive('fabricActiveProfileList', function () {
        return new Fabric.ActiveProfileList();
    }).directive('fabricProfileLink', function (workspace, jolokia, localStorage) {
        return {
            restrict: 'A',
            link: function ($scope, $element, $attrs) {
                var profileId = $attrs['fabricProfileLink'];

                if (profileId && !profileId.isBlank() && Fabric.fabricCreated(workspace)) {
                    var container = Fabric.getCurrentContainer(jolokia, ['versionId']);
                    var versionId = container['versionId'];
                    if (versionId && !versionId.isBlank()) {
                        var url = '#' + Fabric.profileLink(workspace, jolokia, localStorage, versionId, profileId);
                        if (angular.isDefined($attrs['file'])) {
                            url = url + "/" + $attrs['file'];
                        }

                        $element.attr('href', url);
                    }
                }
            }
        };
    }).directive('fabricContainers', function ($location, jolokia, workspace, $compile) {
        return {
            restrict: 'A',
            link: function ($scope, $element, $attrs) {
                var model = $attrs['fabricContainers'];
                var profileId = $attrs['profile'];
                var version = $scope.versionId || $scope.version || "1.0";
                if (model && !model.isBlank() && profileId && !profileId.isBlank()) {
                    // lets expose the $scope.connect object!
                    Fabric.initScope($scope, $location, jolokia, workspace);
                    var containerIds = Fabric.getContainerIdsForProfile(jolokia, version, profileId);
                    Fabric.log.info("Searching for containers for profile: " + profileId + " version " + version + ". Found: " + containerIds);
                    $scope[model] = containerIds;

                    $scope["onCancel"] = function () {
                        console.log("In our new cancel thingy!");
                    };

                    // now lets add the connect dialog
                    var dialog = $("<div ng-include=\"'app/fabric/html/connectToContainerDialog.html'\"></div>");
                    var answer = $compile(dialog)($scope);
                    $element.append(answer);
                }
            }
        };
    }).directive('fabricContainerLink', function ($location, jolokia, workspace) {
        return {
            restrict: 'A',
            scope: { containerModel: '@fabricContainerLink' },
            link: function ($scope, $element, $attrs) {
                $scope.$watch("containerModel", function (nv) {
                    var modelName = $scope.containerModel;
                    var containerId = modelName;
                    var container = null;
                    if (modelName && !modelName.isBlank()) {
                        // lets check if the value is a model object containing the container details
                        var modelValue = Core.pathGet($scope, modelName);
                        if (angular.isObject(modelValue)) {
                            var id = modelValue["container"] || modelValue["containerId"] || modelValue["id"];
                            if (id && modelValue["provisionResult"]) {
                                container = modelValue;
                                containerId = id;
                            }
                        }
                        if (!container) {
                            var fields = ["alive", "provisionResult", "versionId", "jmxDomains"];
                            container = Fabric.getContainerFields(jolokia, containerId, fields);
                        }

                        var link = "#/fabric/container/" + containerId;
                        var title = Fabric.statusTitle(container) || "container " + containerId;
                        var icon = Fabric.statusIcon(container) || "";

                        var html = "<a href='" + link + "' title='" + title + "'><i class='" + icon + "'></i> " + containerId + "</a>";
                        $element.html(html);
                    } else {
                        $element.html(" ");
                    }
                });
            }
        };
    }).directive('fabricContainerConnect', function ($location, jolokia) {
        return {
            restrict: 'A',
            link: function ($scope, $element, $attrs) {
                var containerId = $attrs['fabricContainerConnect'];
                var view = $attrs['view'];
                if (containerId && !containerId.isBlank()) {
                    //var fields = ["parentId", "profileIds", "versionId", "provisionResult", "jolokiaUrl", "root", 'jmxDomains'];
                    var fields = ["jolokiaUrl"];

                    //Fabric.initScope($scope, $location, jolokia, workspace);
                    var connectFn = function () {
                        var container = Fabric.getContainerFields(jolokia, containerId, fields);
                        Fabric.log.info("Connecting to container id " + containerId + " view + " + view);
                        container["id"] = containerId;
                        $scope.doConnect(container, view);
                        Core.$apply($scope);
                    };
                    $element.on("click", connectFn);
                }
            }
        };
    }).directive('fabricVersionLink', function (workspace, jolokia, localStorage) {
        return {
            restrict: 'A',
            link: function ($scope, $element, $attrs) {
                var versionLink = $attrs['fabricVersionLink'];

                if (versionLink && !versionLink.isBlank() && Fabric.fabricCreated(workspace)) {
                    var container = Fabric.getCurrentContainer(jolokia, ['versionId']);
                    var versionId = container['versionId'] || "1.0";
                    if (versionId && !versionId.isBlank()) {
                        var url = "#/wiki/branch/" + versionId + "/" + Core.trimLeading(versionLink, "/");
                        $element.attr('href', url);
                    }
                }
            }
        };
    }).factory('serviceIconRegistry', function () {
        return Fabric.serviceIconRegistry;
    }).factory('containerIconRegistry', function () {
        return Fabric.containerIconRegistry;
    }).run(function ($location, workspace, jolokia, viewRegistry, pageTitle, helpRegistry, $rootScope, postLoginTasks, preferencesRegistry) {
        viewRegistry['fabric'] = Fabric.templatePath + 'layoutFabric.html';

        pageTitle.addTitleElement(function () {
            return Fabric.currentContainerId;
        });

        postLoginTasks.addTask('fabricFetchContainerName', function () {
            if (Fabric.currentContainerId === '' && Fabric.fabricCreated(workspace)) {
                jolokia.request({
                    type: 'exec',
                    mbean: Fabric.managerMBean,
                    operation: 'currentContainer()',
                    arguments: []
                }, onSuccess(function (response) {
                    if (!response.value) {
                        return;
                    }
                    Fabric.currentContainer = response.value;

                    Fabric.currentContainerId = Fabric.currentContainer['id'];
                    if ('container' in Perspective.metadata) {
                        Core.pathSet(Perspective.metadata, ['container', 'label'], Fabric.currentContainerId);
                        Core.pathSet(Perspective.metadata, ['container', 'icon'], Fabric.getTypeIcon(Fabric.currentContainer));
                    }
                    Core.$apply($rootScope);
                }));
            }
        });

        preferencesRegistry.addTab("Fabric", "app/fabric/html/preferences.html", function () {
            return Fabric.isFMCContainer(workspace);
        });

        workspace.topLevelTabs.push({
            id: "fabric.runtime",
            content: "Runtime",
            title: "Manage your containers in this fabric",
            isValid: function (workspace) {
                return Fabric.isFMCContainer(workspace);
            },
            href: function () {
                return "#/fabric/containers";
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("fabric");
            }
        });
        workspace.topLevelTabs.push({
            id: "fabric.configuration",
            content: "Wiki",
            title: "View the documentation and configuration of your profiles in Fabric",
            isValid: function (workspace) {
                var answer = Fabric.isFMCContainer(workspace);
                if (answer) {
                    // must be in fabric perspective as we have wiki in container perspective as well which is not this plugin
                    var currentId = Perspective.currentPerspectiveId($location, workspace, jolokia, localStorage);
                    answer = "fabric" === currentId;
                }
                return answer;
            },
            href: function () {
                return "#/wiki/branch/" + Fabric.getActiveVersion($location) + "/view/fabric/profiles";
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("/wiki") && (workspace.linkContains("fabric", "profiles") || workspace.linkContains("editFeatures"));
            }
        });
        workspace.topLevelTabs.push({
            id: "fabric.insight",
            content: "Insight",
            title: "View insight into your fabric looking at logs, metrics and messages across the fabric",
            isValid: function (workspace) {
                return Fabric.isFMCContainer(workspace) && Insight.hasInsight(workspace);
            },
            href: function () {
                return "#/insight/all?p=insight";
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("/insight");
            }
        });

        helpRegistry.addUserDoc('fabric', 'app/fabric/doc/help.md', function () {
            return Fabric.isFMCContainer(workspace);
        });

        // don't need to pass the isValid parameter in subsequent calls...
        helpRegistry.addDevDoc("fabric", 'app/fabric/doc/developer.md');
    });

    hawtioPluginLoader.addModule('fabric');
})(Fabric || (Fabric = {}));
/**
* @module Fabric
*/
var Fabric;
(function (Fabric) {
    function PreferencesController($scope, localStorage) {
        Core.initPreferenceScope($scope, localStorage, {
            'fabricAlwaysPrompt': {
                'value': false,
                'converter': Core.parseBooleanValue
            },
            'fabricEnableMaps': {
                'value': true,
                'converter': Core.parseBooleanValue
            },
            'fabricVerboseNotifications': {
                'value': true,
                'converter': Core.parseBooleanValue
            }
        });
    }
    Fabric.PreferencesController = PreferencesController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    /**
    * Returns the resolvers for the given schema id (child, ssh, jclouds, openshift, docker)
    */
    function getResolvers(id) {
        var answer;
        switch (id) {
            case 'child':
                answer = [];
                break;
            case 'ssh':
                answer = ['localip', 'localhostname', 'publicip', 'publichostname', 'manualip'];
                break;
            case 'jclouds':
                answer = ['localip', 'localhostname', 'publicip', 'publichostname', 'manualip'];
                break;
            case 'openshift':
                answer = [];
                break;
            case 'docker':
                answer = [];
                break;
        }
        return answer;
    }
    Fabric.getResolvers = getResolvers;

    function customizeSchema(id, schema) {
        // console.log("Schema: ", schema);
        Core.pathSet(schema, ["properties", "name", "required"], true);
        Core.pathSet(schema, ['properties', 'name', 'input-attributes', 'ng-pattern'], "/^[a-zA-Z0-9_-]*$/");

        delete schema.properties['metadataMap'];
        delete schema.properties['zookeeperUrl'];
        delete schema.properties['zookeeperPassword'];
        delete schema.properties['globalResolver'];
        delete schema.properties['zooKeeperServerPort'];
        delete schema.properties['zooKeeperServerConnectionPort'];
        delete schema.properties['agentEnabled'];
        delete schema.properties['autoImportEnabled'];
        delete schema.properties['importPath'];
        delete schema.properties['users'];

        // we cannot pass in system properties (you can use jvmOpt instead)
        delete schema.properties['systemProperties'];

        [
            'zooKeeperServerInitLimit',
            'zooKeeperServerTickTime',
            'zooKeeperServerSyncLimit',
            'zooKeeperServerDataDir',
            'waitForProvision',
            'ensembleStart',
            'migrationTimeout',
            'dataStoreProperties'].forEach(function (attr) {
            Core.pathSet(schema, ['properties', attr, 'control-attributes', 'ng-show'], 'entity.ensembleServer');
        });

        Core.pathSet(schema, ['properties', 'providerType', 'type'], 'hidden');
        Core.pathSet(schema, ['properties', 'profiles', 'type'], 'hidden');
        Core.pathSet(schema, ['properties', 'version', 'type'], 'hidden');

        Core.pathSet(schema.properties, ['name', 'label'], 'Container Name');
        Core.pathSet(schema.properties, ['name', 'tooltip'], 'Name of the container to create (or prefix of the container name if you create multiple containers)');

        Core.pathSet(schema.properties, ['number', 'label'], 'Number of containers');
        Core.pathSet(schema.properties, ['number', 'tooltip'], 'The number of containers to create; when set higher than 1 a number will be appended to each container name. Max value: 99');
        Core.pathSet(schema.properties, ['number', 'input-attributes', 'min'], '1');
        Core.pathSet(schema.properties, ['number', 'input-attributes', 'max'], '99');
        Core.pathSet(schema.properties, ['number', 'input-attributes', 'ng-pattern'], "/^[1-9][0-9]?$/");
        Core.pathSet(schema.properties, ['number', 'required'], true);

        // mark properties as autofill to avoid issues with angular missing autofill events
        Core.pathSet(schema.properties, ['login', 'input-attributes', "autofill"], "true");
        Core.pathSet(schema.properties, ['password', 'input-attributes', "autofill"], "true");

        Core.pathSet(schema.properties, ['jmxUser', 'input-attributes', "autofill"], "true");
        Core.pathSet(schema.properties, ['jmxUser', 'tooltip'], 'The username for connecting to the container using JMX');

        Core.pathSet(schema.properties, ['jmxPassword', 'input-attributes', "autofill"], "true");
        Core.pathSet(schema.properties, ['jmxPassword', 'tooltip'], 'The password for connecting to the container using JMX');

        Core.pathSet(schema.properties, ['resolver', 'input-element'], "select");
        Core.pathSet(schema.properties, ['resolver', 'input-attributes', "ng-options"], "r for r in resolvers");

        switch (id) {
            case 'child':
                delete schema.properties['manualIp'];
                delete schema.properties['preferredAddress'];
                delete schema.properties['resolver'];
                delete schema.properties['ensembleServer'];
                delete schema.properties['proxyUri'];
                delete schema.properties['adminAccess'];
                delete schema.properties['minimumPort'];
                delete schema.properties['maximumPort'];
                schema.properties['jmxPassword']['type'] = 'password';
                schema.properties['saveJmxCredentials'] = {
                    'type': 'boolean'
                };
                Core.pathSet(schema.properties, ['saveJmxCredentials', 'tooltip'], 'Remember credentials when connecting to container (avoid prompting user to enter credentials)');

                Core.pathSet(schema.properties, ['parent', 'label'], 'Parent Container');
                Core.pathSet(schema.properties, ['parent', 'tooltip'], 'The name of the parent container used to create the child container');
                Core.pathSet(schema.properties, ['parent', 'input-element'], "select");
                Core.pathSet(schema.properties, ['parent', 'input-attributes', "ng-options"], "c for c in child.rootContainers");

                bulkSet(schema, ["jmxUser", "jmxPassword", "parent"], 'required', true);
                schema['tabs'] = {
                    'Common': ['name', 'parent', 'jmxUser', 'jmxPassword', 'saveJmxCredentials', 'number'],
                    'Advanced': ['*']
                };
                break;

            case 'ssh':
                delete schema.properties['jmxUser'];
                delete schema.properties['jmxPassword'];
                delete schema.properties['parent'];

                bulkSet(schema, ['host'], 'required', true);
                Core.pathSet(schema.properties, ['password', 'type'], 'password');

                schema['tabs'] = {
                    'Common': ['name', 'host', 'port', 'username', 'password', 'privateKeyFile', 'passPhrase'],
                    'Advanced': ['*']
                };
                break;

            case 'jclouds':
                delete schema.properties['jmxUser'];
                delete schema.properties['jmxPassword'];
                delete schema.properties['parent'];

                schema['tabs'] = {
                    'Common': ['name', 'owner', 'credential', 'providerName', 'imageId', 'hardwareId', 'locationId', 'number', 'instanceType'],
                    'Advanced': ['*']
                };
                break;

            case 'openshift':
                delete schema.properties['jmxUser'];
                delete schema.properties['jmxPassword'];
                delete schema.properties['parent'];
                delete schema.properties['manualIp'];
                delete schema.properties['preferredAddress'];
                delete schema.properties['ensembleServer'];
                delete schema.properties['proxyUri'];
                delete schema.properties['adminAccess'];
                delete schema.properties['path'];
                delete schema.properties['bindAddress'];
                delete schema.properties['hostNameContext'];
                delete schema.properties['resolver'];

                schema.properties['serverUrl']['default'] = 'openshift.redhat.com';

                // openshift must select publichostname as the resolver
                Core.pathSet(schema.properties, ['resolver', 'default'], 'publichostname');
                Core.pathSet(schema.properties, ['serverUrl', 'label'], 'OpenShift Broker');
                Core.pathSet(schema.properties, ['serverUrl', 'tooltip'], 'The OpenShift broker host name of the cloud to create the container inside. This is either the URL for your local OpenShift Enterprise installation, or its the public OpenShift online URL: openshift.redhat.com');
                Core.pathSet(schema.properties, ['login', 'label'], 'OpenShift Login');
                Core.pathSet(schema.properties, ['login', 'tooltip'], 'Your personal login to the OpenShift portal');
                Core.pathSet(schema.properties, ['login', 'input-attributes', "autofill"], "true");
                Core.pathSet(schema.properties, ['password', 'label'], 'OpenShift Password');
                Core.pathSet(schema.properties, ['password', 'tooltip'], 'Your personal password on the OpenShift portal');
                Core.pathSet(schema.properties, ['password', 'type'], 'password');

                // openshift only allows lowercase a-z and numbers
                Core.pathSet(schema.properties, ['name', 'input-attributes', 'ng-pattern'], "/^[a-z0-9]*$/");

                // add an extra property to make it easy to login
                /*
                */
                /*
                Core.pathSet(schema.properties, ['tryLogin', 'label'], 'Try');
                */
                /*
                Core.pathSet(schema.properties, ['tryLogin', 'input-element'], "button");
                Core.pathSet(schema.properties, ['tryLogin', 'input-attributes', "class"], "btn");
                Core.pathSet(schema.properties, ['tryLogin', 'input-attributes', "ng-click"], "openShift.login()");
                */
                //Core.pathSet(schema.properties, ['tryLogin', 'input-attributes', "ng-disable"], "!entity.login || !entity.password || !entity.serverUrl");
                /*
                */
                Core.pathSet(schema.properties, ['tryLogin', 'type'], 'string');
                Core.pathSet(schema.properties, ['tryLogin', 'input-attributes', "ng-model"], "openShift.tryLogin");
                Core.pathSet(schema.properties, ['tryLogin', 'label'], 'Authenticate');
                Core.pathSet(schema.properties, ['tryLogin', 'tooltip'], 'Authenticate with the OpenShift Broker using your login and password');
                Core.pathSet(schema.properties, ['tryLogin', 'formTemplate'], '<a ng-click="openShift.login()" ng-disabled="!entity.login || !entity.password || !entity.serverUrl" ' + 'title="Test you entered the correct OpenShift Broker, login and password" class="btn btn-primary">Login to OpenShift</a>' + '<div class="alert" ng-show="openShift.loginFailed" ' + 'title="Are you sure you correctly entered the OpenShift Broker, login and password correctly?">Login failed</div>');

                /*
                Core.pathSet(schema.properties, ['tryLogin', 'formTemplate'], '<button ng-click="openShift.login()" title="Test you entered the correct OpenShift Broker, login and password">Try Login</button>');
                */
                Core.pathSet(schema.properties, ['domain', 'label'], 'OpenShift Domain');
                Core.pathSet(schema.properties, ['domain', 'tooltip'], 'What is your unique domain name used for applications you create on OpenShift. Often this is your own user name or group name');
                Core.pathSet(schema.properties, ['domain', 'input-element'], "select");
                Core.pathSet(schema.properties, ['domain', 'input-attributes', "ng-options"], "c for c in openShift.domains");

                Core.pathSet(schema.properties, ['gearProfile', 'tooltip'], 'Which kind of gear to create');
                Core.pathSet(schema.properties, ['gearProfile', 'input-element'], "select");
                Core.pathSet(schema.properties, ['gearProfile', 'input-attributes', "ng-options"], "c for c in openShift.gearProfiles");

                bulkSet(schema, ['serverUrl', 'login', 'password', 'domain'], 'required', true);
                schema['tabs'] = {
                    'Common': ['name', 'serverUrl', 'login', 'password', 'tryLogin', 'domain', 'gearProfile', 'number'],
                    'Advanced': ['environmentalVariables', 'jvmOpts', '*']
                };
                break;

            case 'docker':
                delete schema.properties['jmxUser'];
                delete schema.properties['jmxPassword'];
                delete schema.properties['parent'];
                delete schema.properties['manualIp'];
                delete schema.properties['preferredAddress'];
                delete schema.properties['resolver'];
                delete schema.properties['ensembleServer'];
                delete schema.properties['proxyUri'];
                delete schema.properties['adminAccess'];
                delete schema.properties['path'];
                delete schema.properties['bindAddress'];
                delete schema.properties['hostNameContext'];

                schema['tabs'] = {
                    'Common': ['name', 'number'],
                    'Advanced': ['environmentalVariables', 'jvmOpts', '*']
                };
                break;

            default:
        }

        return schema;
    }
    Fabric.customizeSchema = customizeSchema;

    function bulkSet(schema, properties, field, value) {
        properties.each(function (name) {
            Core.pathSet(schema, ['properties', name, field], value);
        });
    }

    function setGlobalResolverEnum(schema) {
        var globalResolverEnum = ['localip', 'localhostname', 'publicip', 'publichostname'];
        Core.pathSet(schema, ['properties', 'globalResolver', 'enum'], globalResolverEnum);
    }
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function ContainerController($scope, $routeParams, $location, jolokia, workspace, userDetails) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        if ($scope.inDashboard) {
            $scope.operation = 'getContainer(java.lang.String, java.util.List)';
        } else {
            $scope.operation = 'getContainer(java.lang.String)';
        }

        $scope.mavenRepoUploadUri = "upload";
        $scope.mavenRepoDownloadUri = "download";

        $scope.username = userDetails.username;
        $scope.password = userDetails.password;

        $scope.tab = $routeParams['tab'];
        if (!$scope.tab) {
            $scope.tab = 'Status';
        }

        $scope.$watch('tab', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $location.search({ tab: newValue }).replace();
            }
        });

        /*
        // handy for working around any randomly added fields that won't marshal
        $scope.fields = jolokia.execute(Fabric.managerMBean, 'getFields(java.lang.String)', 'io.fabric8.api.Container');
        $scope.fields.remove('fabricService');
        $scope.operation = 'getContainer(java.lang.String, java.util.List)'
        */
        $scope.loading = true;

        $scope.containerId = $routeParams.containerId;

        $scope.addToDashboardLink = function () {
            var href = "#/fabric/container/:containerId";
            var routeParams = angular.toJson($routeParams);
            var title = $scope.containerId;
            return "#/dashboard/add?tab=dashboard&href=" + encodeURIComponent(href) + "&routeParams=" + encodeURIComponent(routeParams) + "&title=" + encodeURIComponent(title);
        };

        $scope.versionTitle = "Migrate to:";

        $scope.selectedProfiles = [];
        $scope.selectedProfilesDialog = [];
        $scope.selectedProfilesString = '';

        $scope.addProfileDialog = new UI.Dialog();
        $scope.deleteProfileDialog = new UI.Dialog();
        $scope.deleteContainerDialog = new UI.Dialog();

        $scope.$watch('selectedProfiles', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.selectedProfilesString = '';
                $scope.selectedProfiles.each(function (p) {
                    $scope.selectedProfilesString += '<li>' + p.id + '</li>\n';
                });
            }
        }, true);

        $scope.stop = function () {
            Fabric.doStopContainer($scope, jolokia, $scope.containerId);
        };

        $scope.start = function () {
            Fabric.doStartContainer($scope, jolokia, $scope.containerId);
        };

        $scope.statusIcon = function () {
            return Fabric.statusIcon($scope.row);
        };

        $scope.getGitURL = function (jolokiaUrl) {
            var answer = jolokiaUrl ? jolokiaUrl.replace("jolokia", "git/fabric") : null;
            if (answer !== null) {
                var command = "git clone -b 1.0 " + answer;
                if ($scope.username !== null && $scope.password !== null) {
                    command = command.replace("://", "://" + $scope.username + ":" + $scope.password + "@");
                }
                answer = command;
            }
            return answer;
        };

        $scope.getSshURL = function (sshUrl) {
            if (!sshUrl) {
                return '';
            }
            var answer = sshUrl;
            if ($scope.username !== null && $scope.password !== null) {
                var parts = sshUrl.split(":");
                if (parts.length === 2) {
                    answer = "ssh -p " + parts[1] + " " + $scope.username + "@" + parts[0];
                }
            }
            return answer;
        };

        $scope.getJmxURL = function (jmxUrl) {
            return jmxUrl;
        };

        $scope.getType = function () {
            if ($scope.row) {
                if ($scope.row.ensembleServer) {
                    return "Fabric Server";
                } else if ($scope.row.managed) {
                    return "Managed Container";
                } else {
                    return "Unmanaged Container";
                }
            }
            return "";
        };

        $scope.updateContainerProperty = function (propertyName, row) {
            Fabric.setContainerProperty(jolokia, row.id, propertyName, row[propertyName], function () {
                Core.$apply($scope);
            }, function (response) {
                notification('error', 'Failed to set container property due to : ' + response.error);
                Core.$apply($scope);
            });
        };

        $scope.getClass = function (item) {
            if (!$scope.provisionListFilter) {
                return 'no-filter';
            } else if (item.has($scope.provisionListFilter)) {
                return 'match-filter';
            } else {
                return 'no-match-filter';
            }
        };

        /*
        $scope.$watch('selectedProfilesDialog', (newValue, oldValue) => {
        if (newValue !== oldValue) {
        console.log("Selected profiles: ", $scope.selectedProfilesDialog);
        }
        }, true);
        */
        $scope.addProfiles = function () {
            $scope.addProfileDialog.close();
            var addedProfiles = $scope.selectedProfilesDialog.map(function (p) {
                return p.id;
            });
            var text = Core.maybePlural(addedProfiles.length, "profile");
            Fabric.addProfilesToContainer(jolokia, $scope.row.id, addedProfiles, function () {
                notification('success', "Successfully added " + text);
                $scope.selectedProfilesDialog = [];
                $scope.$broadcast('fabricProfileRefresh');
                Core.$apply($scope);
            }, function (response) {
                notification('error', "Failed to add " + text + " due to " + response.error);
                $scope.selectedProfilesDialog = [];
                Core.$apply($scope);
            });
        };

        $scope.getArguments = function () {
            if ($scope.inDashboard) {
                return [$scope.containerId, ['id', 'versionId', 'profileIds', 'provisionResult', 'jolokiaUrl', 'alive', 'jmxDomains', 'ensembleServer', 'debugPort']];
            }
            return [$scope.containerId];
        };

        $scope.deleteProfiles = function () {
            $scope.deleteProfileDialog.close();
            var removedProfiles = $scope.selectedProfiles.map(function (p) {
                return p.id;
            });
            var text = Core.maybePlural(removedProfiles.length, "profile");
            Fabric.removeProfilesFromContainer(jolokia, $scope.row.id, removedProfiles, function () {
                notification('success', "Successfully removed " + text);
                $scope.selectedProfiles = [];
                $scope.$broadcast('fabricProfileRefresh');
                Core.$apply($scope);
            }, function (response) {
                notification('error', "Failed to remove " + text + " due to " + response.error);
                $scope.selectedProfiles = [];
                Core.$apply($scope);
            });
        };

        $scope.$on("fabricProfileRefresh", function () {
            setTimeout(function () {
                jolokia.request({
                    type: 'exec', mbean: Fabric.managerMBean,
                    operation: $scope.operation,
                    arguments: $scope.getArguments()
                }, {
                    method: 'POST',
                    success: function (response) {
                        render(response);
                    }
                });
            }, 500);
        });

        if (angular.isDefined($scope.containerId)) {
            Core.register(jolokia, $scope, {
                type: 'read', mbean: Fabric.managerMBean,
                attribute: ["MavenRepoURI", "MavenRepoUploadURI"]
            }, onSuccess(mavenUris));

            Core.register(jolokia, $scope, {
                type: 'exec', mbean: Fabric.managerMBean,
                operation: $scope.operation,
                arguments: $scope.getArguments()
            }, onSuccess(render));
        }

        $scope.formatStackTrace = function (exception) {
            return Log.formatStackTrace(exception);
        };

        function mavenUris(response) {
            var obj = response.value;
            if (obj) {
                $scope.mavenRepoUploadUri = obj.MavenRepoUploadURI;
                $scope.mavenRepoDownloadUri = obj.MavenRepoURI;
            }
        }

        function render(response) {
            if (!angular.isDefined($scope.responseJson)) {
                $scope.loading = false;
            }
            var responseJson = angular.toJson(response.value);
            if ($scope.responseJson !== responseJson) {
                $scope.responseJson = responseJson;
                $scope.row = response.value;
                $scope.container = $scope.row;
                if ($scope.row) {
                    var row = $scope.row;
                    row.debugHost = row.publicIp || row.localHostname || row.localIp || row.ip || row.manualIp;
                    if (angular.isDefined($scope.row.provisionException) && angular.isString($scope.row.provisionException)) {
                        $scope.row.provisionExceptionArray = $scope.row.provisionException.lines();
                    }
                    $scope.services = Fabric.getServiceList($scope.row);
                    if (angular.isDefined($scope.resolverWatch) && angular.isFunction($scope.resolverWatch)) {
                        $scope.resolverWatch();
                    }
                    $scope.resolverWatch = $scope.$watch('row.resolver', function (newValue, oldValue) {
                        if (newValue !== oldValue) {
                            $scope.updateContainerProperty('resolver', $scope.row);
                        }
                    });
                }
                Core.$apply($scope);
            }
        }
    }
    Fabric.ContainerController = ContainerController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function FeatureEditController($scope, $routeParams, $location, jolokia, xml2json, workspace) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        $scope.getProfileFeaturesOp = "getProfileFeatures(java.lang.String, java.lang.String)";
        $scope.versionId = $routeParams.versionId;
        $scope.profileId = $routeParams.profileId;

        $scope.response = {};

        $scope.features = [];

        $scope.selectedRepoFeatures = [];

        $scope.deletingFeatures = [];
        $scope.addingFeatures = [];

        $scope.selectedRepoSelectedFeatures = [];

        $scope.featureGridOptions = {
            data: 'selectedRepoFeatures',
            selectedItems: $scope.selectedRepoSelectedFeatures,
            displayFooter: false,
            showFilter: false,
            keepLastSelected: true,
            showSelectionCheckbox: true,
            filterOptions: {
                filterText: ''
            },
            columnDefs: [
                {
                    field: 'name',
                    displayName: 'Name'
                },
                {
                    field: 'version',
                    displayName: 'Version'
                }
            ]
        };

        $scope.$watch('features', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.parentFeatures = $scope.features.filter(function (f) {
                    return f.isParentFeature;
                });
                $scope.profileFeatures = $scope.features.filter(function (f) {
                    return !f.isParentFeature;
                });
                $scope.addingFeatures = $scope.features.filter(function (f) {
                    return f.adding;
                });
                $scope.deletingFeatures = $scope.features.filter(function (f) {
                    return f.deleting;
                });
            }
        }, true);

        $scope.$watch('addingFeatures', function (newValue, oldValue) {
            if (newValue !== oldValue) {
            }
        }, true);

        $scope.dispatch = function (response) {
            var responseJson = angular.toJson(response.value);
            if (responseJson !== $scope.responseJson) {
                if (angular.isDefined($scope.responseJson)) {
                    notification('info', "Profile feature definitions updated");
                }
                $scope.responseJson = responseJson;
                $scope.features = response.value.featureDefinitions;
                var repositories = response.value.repositoryDefinitions;

                $scope.selectedRepoFeatures = [];

                repositories.forEach(function (repo) {
                    var repoJson = xml2json(repo['data']);
                    if ('feature' in repoJson) {
                        var features = repoJson['feature'];
                        if (!angular.isArray(features)) {
                            features = [features];
                        }
                        $scope.selectedRepoFeatures.add(features);
                    }
                });

                $scope.selectedRepoFeatures = $scope.selectedRepoFeatures.sortBy('name');

                Core.$apply($scope);
            }
        };

        $scope.getClass = function (feature) {
            if (feature.adding) {
                return "adding";
            }
            if (feature.deleting) {
                return "deleting";
            }
            return "";
        };

        $scope.removeFeature = function (feature) {
            if (feature.adding) {
                $scope.features.remove(function (f) {
                    return f.id === feature.id;
                });
            } else {
                feature.deleting = !feature.deleting;
            }
        };

        $scope.addSelectedFeatures = function (withVersion) {
            $scope.selectedRepoSelectedFeatures.each(function (feature) {
                var id = feature.name;

                if (withVersion) {
                    id = id + "/" + feature.version;
                }

                $scope.features.push({
                    id: id,
                    adding: true
                });
            });
        };

        $scope.save = function () {
            jolokia.request({
                type: 'exec', mbean: Fabric.managerMBean,
                operation: 'getConfigurationFile(java.lang.String, java.lang.String, java.lang.String)',
                arguments: [$scope.versionId, $scope.profileId, 'io.fabric8.agent.properties']
            }, onSuccess($scope.doSave));
        };

        $scope.doSave = function (response) {
            var configFile = response.value.decodeBase64();
            var lines = configFile.lines();

            if ($scope.deletingFeatures.length > 0) {
                $scope.deletingFeatures.each(function (feature) {
                    lines.remove(function (line) {
                        return line.startsWith("feature." + feature.id);
                    });
                });
            }

            if ($scope.addingFeatures.length > 0) {
                $scope.addingFeatures.each(function (feature) {
                    lines.add("feature." + feature.id + " = " + feature.id);
                });
            }

            configFile = lines.join('\n');

            Fabric.saveConfigFile(jolokia, $scope.versionId, $scope.profileId, 'io.fabric8.agent.properties', configFile.encodeBase64(), function () {
                notification('success', "Updated feature definitions...");
                Core.$apply($scope);
            }, function (response) {
                notification('error', "Failed to save feature definitions due to " + response.error);
                Core.$apply($scope);
            });
        };

        Core.register(jolokia, $scope, [{
                type: 'exec', mbean: Fabric.managerMBean, operation: $scope.getProfileFeaturesOp,
                arguments: [$scope.versionId, $scope.profileId]
            }], onSuccess($scope.dispatch));
    }
    Fabric.FeatureEditController = FeatureEditController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function NavBarController($scope, $location, jolokia, workspace, localStorage) {
        $scope.activeVersion = "1.0";

        $scope.mapsEnabled = localStorage['fabricEnableMaps'];

        // update the active version whenever query parameters change
        $scope.$on('$routeUpdate', reloadVersion);

        $scope.isActive = function (href) {
            return workspace.isLinkActive(href);
        };

        $scope.clusterLink = function () {
            // TODO move to use /fabric/clusters by default maybe?
            return Core.createHref($location, "#/fabric/clusters/fabric/registry/clusters", ["cv", "cp", "pv"]);
        };

        function reloadVersion() {
            $scope.activeVersion = Fabric.getActiveVersion($location);
        }

        function reloadData() {
            // TODO should probably load the default version here
            reloadVersion();
            $scope.hasFabric = Fabric.hasFabric(workspace);
            $scope.hasMQManager = Fabric.hasMQManager(workspace);
            if ($scope.hasFabric) {
                var containerId = null;
                Fabric.containerWebAppURL(jolokia, "drools-wb-distribution-wars", containerId, onDroolsUrl, onDroolsUrl);
                $scope.canUpload = workspace.treeContainsDomainAndProperties('hawtio', { type: 'UploadManager' });
            }
        }

        reloadData();

        function onDroolsUrl(response) {
            var url = response ? response.value : null;
            console.log("========== onDroolsUrl: " + url);
            $scope.droolsHref = url;
            Core.$apply($scope);
        }
    }
    Fabric.NavBarController = NavBarController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function AssignProfileController($scope, jolokia, $location, $routeParams, workspace) {
        $scope.profileId = $routeParams['pid'];
        $scope.versionId = $routeParams['vid'];

        Fabric.initScope($scope, $location, jolokia, workspace);

        $scope.containerIdFilter = '';

        var valid = true;

        if (Core.isBlank($scope.profileId)) {
            Fabric.log.warn("No profile ID specified, redirecting to Fabric management view");
            valid = false;
        }

        if (Core.isBlank($scope.versionId)) {
            Fabric.log.warn("No version ID specified, redirecting to Fabric management view");
            valid = false;
        }

        if (!valid) {
            $location.path("/fabric/view");
        }

        $scope.gotoCreate = function () {
            $location.path('/fabric/containers/createContainer').search({
                versionId: $scope.versionId,
                profileIds: $scope.profileId
            });
        };

        $scope.$on('$routeChangeSuccess', function () {
            Fabric.log.debug("RouteParams: ", $routeParams);
            Fabric.log.debug("Scope: ", $scope);
        });

        $scope.$watch('containers', function (newValue, oldValue) {
            if (newValue !== oldValue && newValue) {
                $scope.selected = newValue.filter(function (c) {
                    return c['selected'];
                });
            }
        }, true);

        $scope.assignProfiles = function () {
            var requests = [];
            $scope.selected.forEach(function (c) {
                requests.push({
                    type: 'exec', mbean: Fabric.managerMBean,
                    operation: 'addProfilesToContainer',
                    arguments: [c.id, [$scope.profileId]]
                });
            });
            notification('info', "Applying " + $scope.profileId + " to the selected containers");
            var outstanding = requests.length;
            jolokia.request(requests, onSuccess(function () {
                outstanding = outstanding - 1;
                if (outstanding === 0) {
                    notification('success', "Applied " + $scope.profileId);
                    Core.$apply($scope);
                }
            }));
            setTimeout(function () {
                $location.path("/fabric/activeProfiles");
                Core.$apply($scope);
            }, 30);
        };
    }
    Fabric.AssignProfileController = AssignProfileController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    var ProfileDetails = (function () {
        function ProfileDetails() {
            this.restrict = 'A';
            this.replace = true;
            this.templateUrl = Fabric.templatePath + "profileDetailsDirective.html";
            this.scope = {
                versionId: '=',
                profileId: '='
            };
            this.controller = function ($scope, $element, $attrs, $routeParams, jolokia, $location, workspace, $q) {
                $scope.inDirective = true;

                Fabric.ProfileController($scope, $routeParams, jolokia, $location, workspace, $q);
            };
        }
        return ProfileDetails;
    })();
    Fabric.ProfileDetails = ProfileDetails;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function ClusterController($scope, $location, $routeParams, workspace, jolokia) {
        $scope.path = $routeParams["page"] || "/";
        if (!$scope.path.startsWith("/")) {
            $scope.path = "/" + $scope.path;
        }

        $scope.gridOptions = {
            data: 'children',
            displayFooter: false,
            sortInfo: { fields: ['name'], directions: ['asc'] },
            columnDefs: [
                {
                    field: 'name',
                    displayName: 'Name',
                    cellTemplate: '<div class="ngCellText"><a href="{{childLink(row.entity)}}"><i class="{{row | fileIconClass}}"></i> {{row.getProperty(col.field)}}</a></div>',
                    cellFilter: ""
                }
            ]
        };

        $scope.isTabActive = function (href) {
            var tidy = Core.trimLeading(href, "#");
            var loc = $location.path();
            return loc === tidy;
        };

        $scope.childLink = function (child) {
            var prefix = "#/fabric/clusters/" + Core.trimLeading($scope.path, "/") + "/";
            var postFix = "";
            var path = child.name;
            return Core.createHref($location, prefix + path + postFix);
        };

        $scope.$watch('workspace.tree', function () {
            setTimeout(updateView, 50);
        });

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateView, 50);
        });

        updateView();

        function updateView() {
            loadBreadcrumbs();

            var mbean = Fabric.getZooKeeperFacadeMBean(workspace);
            if (mbean) {
                jolokia.execute(mbean, "read", $scope.path, onSuccess(onContents));
            }
        }

        function onContents(contents) {
            // for now it returns just lists of names
            $scope.children = [];
            $scope.stringData = null;
            $scope.html = null;
            if (contents) {
                angular.forEach(contents.children, function (childName) {
                    $scope.children.push({ name: childName });
                });
                if (!$scope.children.length) {
                    var stringData = contents.stringData;
                    if (stringData) {
                        $scope.stringData = stringData;
                        var json = Core.tryParseJson(stringData);
                        if (json) {
                            $scope.html = Core.valueToHtml(json);
                        } else {
                            // TODO detect properties files
                            $scope.html = stringData;
                        }
                    }
                }
            }
            Core.$apply($scope);
        }

        function loadBreadcrumbs() {
            var href = "#/fabric/clusters";
            $scope.breadcrumbs = [
                { href: href + "/", name: "/" }
            ];
            var path = $scope.path;
            var array = path ? path.split("/") : [];
            angular.forEach(array, function (name) {
                if (name) {
                    if (!name.startsWith("/") && !href.endsWith("/")) {
                        href += "/";
                    }
                    href += name;
                    $scope.breadcrumbs.push({ href: href, name: name });
                }
            });
        }
    }
    Fabric.ClusterController = ClusterController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function PIDController($scope, $routeParams, jolokia, $location) {
        $scope.versionId = $routeParams.versionId;
        $scope.profileId = $routeParams.profileId;
        $scope.fname = $routeParams.fname;
        $scope.response = undefined;
        $scope.data = "";
        $scope.dirty = false;

        $scope.getMode = function () {
            var parts = $scope.fname.split('.');
            var mode = parts[parts.length - 1];
            if (!mode) {
                return 'text';
            }
            switch (mode) {
                case 'cfg':
                    mode = "properties";
                    break;
            }
            return mode;
        };

        $scope.mode = $scope.getMode();

        if (angular.isDefined($scope.versionId) && angular.isDefined($scope.profileId) && angular.isDefined($scope.fname)) {
            Core.register(jolokia, $scope, {
                type: 'exec', mbean: Fabric.managerMBean,
                operation: 'getConfigurationFile(java.lang.String,java.lang.String,java.lang.String)',
                arguments: [$scope.versionId, $scope.profileId, $scope.fname]
            }, onSuccess(render));
        }

        $scope.save = function () {
            Fabric.saveConfigFile(jolokia, $scope.versionId, $scope.profileId, $scope.fname, $scope.data.encodeBase64(), function () {
                $scope.dirty = false;
                notification('success', "Saved " + $scope.fname);
                $location.path("/fabric/profile/" + $scope.versionId + "/" + $scope.profileId);
            }, function (response) {
                notification('error', "Failed to save " + $scope.fname + " due to " + response.error);
            });
        };

        function stringToBytes(s) {
            return s.codes();
        }

        function bytesToString(b) {
            var answer = [];
            b.forEach(function (b) {
                answer.push(String.fromCharCode(b));
            });
            return answer.join('');
        }

        function render(response) {
            if (!Object.equal($scope.response, response.value)) {
                $scope.response = response.value;
                $scope.data = $scope.response.decodeBase64();
                $scope.mode = $scope.getMode();
                Core.$apply($scope);
            }
        }
    }
    Fabric.PIDController = PIDController;
})(Fabric || (Fabric = {}));
var __extends = this.__extends || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    __.prototype = b.prototype;
    d.prototype = new __();
};
var Fabric;
(function (Fabric) {
    var ContainerList = (function () {
        function ContainerList() {
            this.restrict = 'A';
            this.replace = true;
            this.templateUrl = Fabric.templatePath + "containerList.html";
            this.scope = false;
            this.link = function ($scope, $element, $attrs) {
                $scope.showSelect = Core.parseBooleanValue(UI.getIfSet('showSelect', $attrs, 'true'));

                var atVersion = UI.getIfSet('atVersion', $attrs, null);
                var withoutProfile = UI.getIfSet('withoutProfile', $attrs, null);

                if (atVersion !== null) {
                    $scope.atVersion = $scope.$eval(atVersion);
                }

                if (withoutProfile !== null) {
                    $scope.withoutProfile = $scope.$eval(withoutProfile);
                }

                Fabric.log.debug("atVersion: ", $scope.atVersion);
                Fabric.log.debug("withoutProfile: ", $scope.withoutProfile);

                Fabric.log.debug("container list attributes: ", $attrs);
            };
        }
        ContainerList.prototype.controller = function ($scope, $element, $attrs, jolokia, $location, workspace, $templateCache) {
            $scope.containerArgs = ["id", "alive", "parentId", "profileIds", "versionId", "provisionResult", "jolokiaUrl", "root", 'jmxDomains', "type", "metadata", "location"];
            $scope.profileFields = ["id", "hidden"];
            $scope.containersOp = 'containers(java.util.List, java.util.List)';
            $scope.ensembleContainerIdListOp = 'EnsembleContainers';

            $scope.containers = [];
            $scope.activeProfiles = [];
            $scope.selectedContainers = [];
            $scope.selectedContainerIds = [];
            $scope.showSelect = true;
            $scope.requirements = null;

            Fabric.initScope($scope, $location, jolokia, workspace);

            $scope.currentPage = $templateCache.get("addProfileRequirements");

            // for editing container requirements
            $scope.editRequirements = {
                dialog: new UI.Dialog(),
                excludeProfiles: [],
                selectedProfiles: [],
                excludeDependentProfiles: [],
                selectedDependentProfiles: [],
                addDependentProfileDialog: new UI.Dialog(),
                versionId: null,
                addProfileSelectShow: false,
                dialogOpen: function (profile) {
                    // lets make sure the requirements are pre-populated with values
                    var editRequirementsEntity = {
                        profileRequirements: []
                    };

                    // initially the requirements stored in ZK look like this:
                    // > zk:get /fabric/configs/io.fabric8.requirements.json
                    // {"profileRequirements":[],"version":"1.0"}
                    if ($scope.requirements) {
                        angular.copy($scope.requirements, editRequirementsEntity);
                    }
                    var profileRequirements = editRequirementsEntity.profileRequirements;
                    if (profileRequirements) {
                        angular.forEach($scope.activeProfiles, function (profile) {
                            var currentRequirements = profile.requirements;
                            if (!currentRequirements) {
                                currentRequirements = {
                                    profile: profile.id
                                };
                                profile.requirements = currentRequirements;
                            }
                            if (!profileRequirements.find(function (p) {
                                return p.profile === currentRequirements.profile;
                            })) {
                                profileRequirements.push(currentRequirements);
                            }
                        });
                    }
                    if (!profile && $scope.activeProfiles.length) {
                        // lets pick the first one - its just to default a version
                        profile = $scope.activeProfiles[0];
                    }
                    if (profile) {
                        $scope.editRequirements.versionId = profile.versionId;
                    }
                    $scope.editRequirements.entity = editRequirementsEntity;
                    $scope.editRequirements.dialog.open();
                },
                // show / hide the new dependent profiles on a profile requirement
                addDependentProfileDialogOpen: function (requirement) {
                    $scope.editRequirements.addDependentProfileDialogProfile = requirement.profile;
                    $scope.editRequirements.selectedDependentProfiles.splice(0, $scope.editRequirements.selectedDependentProfiles.length);
                    $scope.editRequirements.excludeDependentProfiles = [requirement.profile].concat(requirement.dependentProfiles || []);
                    $scope.editRequirements.addDependentProfilesToRequirement = requirement;
                    $scope.editRequirements.addDependentProfileDialogShow = true;
                },
                addDependentProfileDialogHide: function () {
                    $scope.editRequirements.addDependentProfileDialogShow = false;
                },
                addDependentProfileDialogApply: function () {
                    var requirement = $scope.editRequirements.addDependentProfilesToRequirement;
                    angular.forEach($scope.editRequirements.selectedDependentProfiles, function (profile) {
                        var id = profile.id;
                        if (id && requirement) {
                            if (!requirement.dependentProfiles)
                                requirement.dependentProfiles = [];
                            if (!requirement.dependentProfiles.find(function (el) {
                                return el === id;
                            })) {
                                requirement.dependentProfiles.push(id);
                            }
                        }
                    });
                    $scope.editRequirements.addDependentProfileDialogHide();
                },
                // how / hide / add a requirement on new profile
                addProfileRequirementOpen: function () {
                    $scope.editRequirements.selectedProfiles.splice(0, $scope.editRequirements.selectedProfiles.length);
                    $scope.editRequirements.excludeProfiles = $scope.activeProfiles.map(function (p) {
                        return p.id;
                    });
                    $scope.editRequirements.addProfileRequirementShow = true;
                },
                addProfileRequirementHide: function () {
                    $scope.editRequirements.addProfileRequirementShow = false;
                },
                addProfileRequirementApply: function () {
                    var entity = $scope.editRequirements.entity;
                    var profileRequirements = entity.profileRequirements;
                    if (!profileRequirements) {
                        profileRequirements = [];
                        entity.profileRequirements = profileRequirements;
                    }
                    angular.forEach($scope.editRequirements.selectedProfiles, function (profile) {
                        var id = profile.id;
                        if (id) {
                            profileRequirements.push({ profile: id });
                        }
                    });
                    $scope.editRequirements.addProfileRequirementHide();
                }
            };

            $scope.getFilteredName = function (item) {
                return item.versionId + " / " + item.id;
            };

            $scope.filterContainer = function (container) {
                var filterText = $scope.containerIdFilter;
                var filterName = $scope.getFilteredName(container);

                if (!Core.matchFilterIgnoreCase(filterName, filterText)) {
                    // we did not match the container name, then try to see if we match any of its profiles
                    var profileIds = container.profileIds;
                    if (profileIds) {
                        return profileIds.any(function (id) {
                            return Core.matchFilterIgnoreCase(id, filterText);
                        });
                    }
                    return false;
                }
                return true;
            };

            $scope.$watch('editRequirements.addDependentProfileDialogShow', function (newValue, oldValue) {
                if (newValue !== oldValue) {
                    if (newValue) {
                        $scope.currentPage = $templateCache.get("addDependentProfile");
                    } else {
                        $scope.currentPage = $templateCache.get("addProfileRequirements");
                    }
                }
            });

            $scope.$watch('editRequirements.addProfileRequirementShow', function (newValue, oldValue) {
                if (newValue !== oldValue) {
                    if (newValue) {
                        $scope.currentPage = $templateCache.get("addProfileRequirement");
                    } else {
                        $scope.currentPage = $templateCache.get("addProfileRequirements");
                    }
                }
            });

            // invoked regularly by Jolokia after detecting new response from requirements()
            // from object io.fabric:type=Fabric
            $scope.updateActiveContainers = function () {
                var activeProfiles = $scope.activeProfiles;
                $scope.activeProfiles = $scope.currentActiveProfiles();
                $scope.activeProfiles.each(function (activeProfile) {
                    var ap = activeProfiles.find(function (ap) {
                        return ap.id === activeProfile.id && ap.versionId === activeProfile.versionId;
                    });
                    if (ap) {
                        activeProfile['selected'] = ap.selected;
                        activeProfile['expanded'] = ap.expanded;
                    } else {
                        activeProfile['selected'] = false;
                        activeProfile['expanded'] = false;
                    }
                });
            };

            // invoked regularly by Jolokia wth the result of containers(List, List)
            // from object io.fabric:type=Fabric
            $scope.updateContainers = function (newContainers) {
                var response = angular.toJson(newContainers);
                if ($scope.containersResponse !== response) {
                    $scope.containersResponse = response;

                    newContainers = newContainers.sortBy('id');

                    var rootContainers = newContainers.exclude(function (c) {
                        return !c.root;
                    });
                    var childContainers = newContainers.exclude(function (c) {
                        return c.root;
                    });

                    if (childContainers.length > 0) {
                        var tmp = [];
                        rootContainers.each(function (c) {
                            tmp.add(c);
                            var children = childContainers.exclude(function (child) {
                                return child.parentId !== c.id;
                            });
                            tmp.add(children);
                        });
                        newContainers = tmp;
                    }

                    if (angular.isDefined($scope.atVersion)) {
                        newContainers = newContainers.filter(function (c) {
                            return c.versionId === $scope.atVersion;
                        });
                    }

                    if (angular.isDefined($scope.withoutProfile)) {
                        newContainers = newContainers.filter(function (c) {
                            return !c.profileIds.any(function (p) {
                                return p === $scope.withoutProfile;
                            });
                        });
                    }

                    newContainers.each(function (container) {
                        container.services = Fabric.getServiceList(container);
                        container.icon = Fabric.getTypeIcon(container);
                        var c = $scope.containers.find(function (c) {
                            return c.id === container.id;
                        });
                        if (c) {
                            container['selected'] = c.selected;
                        } else {
                            container['selected'] = false;
                        }
                        if ($scope.selectedContainerIds.any(container.id)) {
                            container.selected = true;
                        }
                    });

                    $scope.containers = newContainers;
                    $scope.updateActiveContainers();
                    Core.$apply($scope);
                }
            };

            $scope.currentActiveProfiles = function () {
                var answer = [];

                $scope.containers.each(function (container) {
                    container.profileIds.each(function (profile) {
                        var p = container.profiles.find(function (p) {
                            return p.id === profile;
                        });
                        if (p && p.hidden) {
                            return;
                        }

                        var activeProfile = answer.find(function (o) {
                            return o.versionId === container.versionId && o.id === profile;
                        });

                        if (activeProfile) {
                            activeProfile['containers'] = activeProfile['containers'].include(container.id).unique();

                            activeProfile.count = activeProfile['containers'].length;
                        } else {
                            answer.push({
                                id: profile,
                                count: 1,
                                versionId: container.versionId,
                                containers: [container.id],
                                selected: false,
                                requirements: null,
                                requireStyle: null
                            });
                        }
                    });
                });

                if ($scope.requirements) {
                    angular.forEach($scope.requirements.profileRequirements, function (profileRequirement) {
                        var id = profileRequirement.profile;
                        var min = profileRequirement.minimumInstances;
                        if (id) {
                            var profile = answer.find(function (p) {
                                return (p.id === id);
                            });

                            function requireStyle() {
                                var count = 0;
                                if (profile) {
                                    count = profile['count'];
                                }
                                return Fabric.containerCountBadgeStyle(min, count);
                            }

                            if (profile) {
                                profile["requirements"] = profileRequirement;
                                profile["requireStyle"] = requireStyle();
                            } else {
                                // lets add the profile with no containers
                                answer.push({
                                    id: id,
                                    count: 0,
                                    versionId: $scope.requirements.version || "1.0",
                                    containers: [],
                                    selected: false,
                                    requirements: profileRequirement,
                                    requireStyle: requireStyle()
                                });
                            }
                        }
                    });
                }

                return answer;
            };

            $scope.updateEnsembleContainerIdList = function (ids) {
                var response = angular.toJson(ids);
                if ($scope.ensembleContainerIdsResponse !== response) {
                    $scope.ensembleContainerIdsResponse = response;
                    $scope.ensembleContainerIds = ids;
                    Core.$apply($scope);
                }
            };

            $scope.dispatch = function (response) {
                switch (response.request.operation) {
                    case ($scope.containersOp):
                        $scope.updateContainers(response.value);
                        return;
                }
                switch (response.request.attribute) {
                    case ($scope.ensembleContainerIdListOp):
                        $scope.updateEnsembleContainerIdList(response.value);
                        return;
                }
            };

            $scope.clearSelection = function (group) {
                group.each(function (item) {
                    item.selected = false;
                });
            };

            $scope.setActiveProfile = function (profile) {
                $scope.clearSelection($scope.activeProfiles);
                if (!profile || profile === null) {
                    return;
                }
                profile.selected = true;
            };

            $scope.selectAllContainers = function () {
                $scope.containers.each(function (container) {
                    if ($scope.filterContainer(container)) {
                        container.selected = true;
                    }
                });
            };

            $scope.setActiveContainer = function (container) {
                $scope.clearSelection($scope.containers);
                if (!container || container === null) {
                    return;
                }
                container.selected = true;
            };

            $scope.startSelectedContainers = function () {
                $scope.selectedContainers.each(function (c) {
                    $scope.startContainer(c.id);
                });
            };

            $scope.stopSelectedContainers = function () {
                $scope.selectedContainers.each(function (c) {
                    $scope.stopContainer(c.id);
                });
            };

            $scope.startContainer = function (name) {
                Fabric.doStartContainer($scope, jolokia, name);
            };

            $scope.stopContainer = function (name) {
                Fabric.doStopContainer($scope, jolokia, name);
            };

            $scope.anySelectionAlive = function (state) {
                var selected = $scope.selectedContainers;
                return selected.length > 0 && selected.any(function (s) {
                    return s.alive === state;
                });
            };

            $scope.everySelectionAlive = function (state) {
                var selected = $scope.selectedContainers;
                return selected.length > 0 && selected.every(function (s) {
                    return s.alive === state;
                });
            };

            Core.register(jolokia, $scope, [
                { type: 'exec', mbean: Fabric.managerMBean, operation: $scope.containersOp, arguments: [$scope.containerArgs, $scope.profileFields] },
                { type: 'read', mbean: Fabric.clusterManagerMBean, attribute: $scope.ensembleContainerIdListOp }
            ], onSuccess($scope.dispatch, { silent: true }));
        };
        return ContainerList;
    })();
    Fabric.ContainerList = ContainerList;

    var ActiveProfileList = (function (_super) {
        __extends(ActiveProfileList, _super);
        function ActiveProfileList() {
            _super.apply(this, arguments);
            this.templateUrl = Fabric.templatePath + "activeProfileList.html";
        }
        ActiveProfileList.prototype.controller = function ($scope, $element, $attrs, jolokia, $location, workspace, $templateCache) {
            _super.prototype.controller.call(this, $scope, $element, $attrs, jolokia, $location, workspace, $templateCache);

            $scope.searchFilter = '';

            $scope.isOpen = function (profile) {
                if ($scope.searchFilter !== '') {
                    return "opened";
                }
                return "closed";
            };

            $scope.containersForProfile = function (id) {
                return $scope.containers.filter(function (container) {
                    return container.profileIds.some(id);
                });
            };

            $scope.profileMatchesFilter = function (profile) {
                var filterText = $scope.searchFilter;

                return Core.matchFilterIgnoreCase(profile.id, filterText) || !profile.containers.filter(function (id) {
                    return Core.matchFilterIgnoreCase(id, filterText);
                }).isEmpty();
            };

            $scope.containerMatchesFilter = function (container) {
                var filterText = $scope.searchFilter;
                return Core.matchFilterIgnoreCase(container.id, filterText) || !container.profileIds.filter(function (id) {
                    return Core.matchFilterIgnoreCase(id, filterText);
                }).isEmpty;
            };

            $scope.updateRequirements = function (requirements) {
                function onRequirementsSaved(response) {
                    $scope.requirements = requirements;
                    notification("success", "Updated the requirements");
                    $scope.updateActiveContainers();
                    Core.$apply($scope);
                }
                ;

                if (requirements) {
                    $scope.editRequirements.dialog.close();

                    var json = JSON.stringify(requirements);
                    jolokia.execute(Fabric.managerMBean, "requirementsJson", json, onSuccess(onRequirementsSaved));
                }
            };

            function onRequirements(response) {
                var responseJson = angular.toJson(response.value);

                if (responseJson !== $scope.requirementsResponse) {
                    $scope.requirementsResponse = responseJson;
                    $scope.requirements = response.value;
                    $scope.updateActiveContainers();
                    Core.$apply($scope);
                }
            }

            Core.register(jolokia, $scope, { type: 'exec', mbean: Fabric.managerMBean, operation: "requirements()" }, onSuccess(onRequirements));
        };
        return ActiveProfileList;
    })(Fabric.ContainerList);
    Fabric.ActiveProfileList = ActiveProfileList;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    Fabric.startMaps = function () {
    };

    function MapController($scope, $templateCache, jolokia) {
        $scope.myMarkers = [];
        $scope.containers = {};
        $scope.template = "";
        $scope.first = true;
        $scope.myMap = null;

        $scope.start = function () {
            // must have initial map options
            $scope.mapOptions = {
                center: new google.maps.LatLng(35.784, -78.670),
                zoom: 15,
                mapTypeId: google.maps.MapTypeId.ROADMAP
            };

            Core.register(jolokia, $scope, {
                type: 'exec', mbean: Fabric.managerMBean,
                operation: 'containers()',
                arguments: []
            }, onSuccess(render));
        };

        Fabric.startMaps = $scope.start;

        $('body').append('<script type="text/javascript" src="//maps.google.com/maps/api/js?sensor=false&async=2&callback=Fabric.startMaps"></script>');

        // TODO: adding markers on map to place containers is not (yet) supported
        /*
        $scope.addMarker = function ($event) {
        $scope.myMarkers.push(new google.maps.Marker({
        map: $scope.myMap,
        position: $event.latLng
        }));
        };
        
        $scope.setZoomMessage = function (zoom) {
        //$scope.zoomMessage = 'You just zoomed to ' + zoom + '!';
        console.log(zoom, 'zoomed')
        };
        
        $scope.openMarkerInfo = function (marker) {
        $scope.currentMarker = marker;
        $scope.currentMarkerLat = marker.getPosition().lat();
        $scope.currentMarkerLng = marker.getPosition().lng();
        $scope.myInfoWindow.open($scope.myMap, marker);
        };
        
        $scope.setMarkerPosition = function (marker, lat, lng) {
        marker.setPosition(new google.maps.LatLng(lat, lng));
        };
        */
        function render(response) {
            if (response && response.value) {
                response.value.forEach(function (container) {
                    var addMarker = false;
                    var id = container.id;
                    var containerData = $scope.containers[id];
                    if (!containerData) {
                        containerData = {
                            name: id
                        };

                        $scope.containers[id] = containerData;
                        addMarker = true;
                    }
                    containerData.alive = container.alive;
                    containerData.version = container.versionId;
                    containerData.profileIds = container.profileIds;

                    var geoLocation = container["geoLocation"];
                    if (geoLocation) {
                        var values = geoLocation.split(",");
                        if (values.length >= 2) {
                            var lattitude = Core.parseFloatValue(values[0], "lattitude");
                            var longitude = Core.parseFloatValue(values[1], "longitude");
                            if (lattitude && longitude) {
                                // only add marker if we got the map initialized
                                if ($scope.myMap) {
                                    var marker = containerData.marker;
                                    if (addMarker || !marker) {
                                        Fabric.log.info("Adding marker as we have map " + $scope.myMap);
                                        marker = new google.maps.Marker({
                                            position: new google.maps.LatLng(lattitude, longitude),
                                            map: $scope.myMap,
                                            title: container.id,
                                            tooltip: "(lattitude: " + lattitude + ", longitude: " + longitude + ")"
                                        });
                                        containerData.marker = marker;
                                        $scope.myMarkers.push(marker);
                                    }
                                } else {
                                    // lets update the marker in case the container moved ;)
                                    if (containerData.marker) {
                                        containerData.marker.position = new google.maps.LatLng(lattitude, longitude);
                                    }
                                }

                                // lets update the container data
                                containerData.lattitude = lattitude;
                                containerData.longitude = longitude;
                            }
                        }
                    }
                });

                // if there is only 1 container then jump to it asap
                if ($scope.myMarkers.length > 0 && $scope.first) {
                    // lets jump to this as the centre
                    if ($scope.myMap) {
                        var marker = $scope.myMarkers[0];
                        Fabric.log.info("Auto selecting first container on map: " + marker.title);
                        $scope.myMap.panTo(marker.getPosition());
                        $scope.first = false;
                    }
                }

                // only assign template to scope so we only draw map when we are ready
                $scope.template = $templateCache.get("pageTemplate");

                Core.$apply($scope);
            }
        }
    }
    Fabric.MapController = MapController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function ProfileController($scope, $routeParams, jolokia, $location, workspace, $q) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        $scope.loading = true;

        $scope.mavenMBean = Maven.getMavenIndexerMBean(workspace);

        if (!angular.isDefined($scope.versionId)) {
            $scope.versionId = $routeParams.versionId;
        }
        if (!angular.isDefined($scope.profileId)) {
            $scope.profileId = $routeParams.profileId;
        }

        $scope.newFileDialog = false;
        $scope.deleteFileDialog = false;
        $scope.newFileName = '';
        $scope.markedForDeletion = '';

        $scope.newProfileName = '';
        $scope.deleteThingDialog = false;
        $scope.changeParentsDialog = false;
        $scope.removeParentDialog = false;
        $scope.newThingName = '';
        $scope.selectedParents = [];

        $scope.profilePath = Fabric.profilePath;
        $scope.pageId = Fabric.fabricTopLevel + Fabric.profilePath($scope.profileId);

        var versionId = $scope.versionId;
        var profileId = $scope.profileId;
        if (versionId && versionId) {
            Fabric.profileJolokia(jolokia, profileId, versionId, function (profileJolokia) {
                $scope.profileJolokia = profileJolokia;
            });
        }

        if ($scope.inDirective && angular.isDefined($scope.$parent.childActions) && $scope.versionId) {
            var actions = $scope.$parent.childActions;

            if ($scope.profileId) {
                actions.push({
                    doAction: function () {
                        $scope.showChangeParentsDialog();
                    },
                    title: "Edit parent profiles",
                    icon: "icon-edit",
                    name: "Change Parents"
                });
                actions.push({
                    doAction: function () {
                        $scope.copyProfileDialog = true;
                    },
                    title: "Copy Profile",
                    icon: "icon-copy",
                    name: "Copy Profile"
                });
                actions.push({
                    doAction: function () {
                        $scope.goto('/wiki/profile/' + $scope.versionId + '/' + $scope.profileId + '/editFeatures');
                    },
                    title: "Edit the features defined in this profile",
                    icon: "icon-edit",
                    name: "Edit Features"
                });
                actions.push({
                    doAction: function () {
                        $location.url('/fabric/assignProfile').search({
                            vid: $scope.versionId,
                            pid: $scope.profileId
                        });
                    },
                    title: "Assign profile to existing containers",
                    icon: "icon-truck",
                    name: "Assign to Container"
                });
                actions.push({
                    doAction: function () {
                        $location.url('/fabric/containers/createContainer').search({
                            versionId: $scope.versionId,
                            profileIds: $scope.profileId
                        });
                    },
                    title: "Create a new container with this profile",
                    icon: "icon-truck",
                    name: "New Container"
                });
            }
            /*
            var createVersionDialog = $scope.createVersionDialog;
            if (createVersionDialog) {
            actions.push({
            doAction: () => {
            $scope.createVersionDialog.open();
            },
            title: "Create a new version of this configuration so you can edit it and then perform rolling upgrades",
            icon: "icon-plus",
            name: "New Version"
            });
            }
            */
        }

        $scope.$watch('activeTab', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.newThingName = '';
            }
        });

        $scope.$watch('versionId', function (newValue, oldValue) {
            if (angular.isDefined($scope.versionId) && angular.isDefined($scope.profileId)) {
                $scope.doRegister();
            }
        });

        $scope.$watch('profileId', function (newValue, oldValue) {
            if (angular.isDefined($scope.versionId) && angular.isDefined($scope.profileId)) {
                $scope.doRegister();
            }
        });

        // TODO, should complete URL handlers too
        $scope.doCompletionFabric = function (something) {
            if (something.startsWith("mvn:")) {
                $scope.prefix = "mvn:";
                return Maven.completeMavenUri($q, $scope, workspace, jolokia, something.from(4));
            }
            $scope.prefix = "";
            return $q.when([]);
        };

        $scope.uriParts = [];

        $scope.$watch('newThingName', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.uriParts = newValue.split("/");
            }
        });

        $scope.$watch('uriParts', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if (!$scope.prefix || $scope.prefix === '') {
                    return;
                }
                if (newValue && newValue.length > 0 && !newValue.first().startsWith($scope.prefix)) {
                    /*
                    console.log("newValue: ", newValue);
                    console.log("oldValue: ", oldValue);
                    console.log("prefix: ", $scope.prefix);
                    */
                    if (newValue.first() === "" || newValue.first().length < $scope.prefix.length) {
                        return;
                    }
                    if (oldValue.length === 0) {
                        return;
                    }

                    // a completion occurred...
                    if (oldValue.length === 1) {
                        $scope.newThingName = $scope.prefix + newValue.first();
                    } else {
                        var merged = oldValue.first(oldValue.length - 1).include(newValue.first());
                        $scope.newThingName = merged.join('/');
                    }
                }
            }
        }, true);

        $scope.doRegister = function () {
            Core.unregister(jolokia, $scope);
            if ($scope.versionId && $scope.profileId && !$scope.versionId.isBlank() && !$scope.profileId.isBlank()) {
                Core.register(jolokia, $scope, {
                    type: 'exec', mbean: Fabric.managerMBean,
                    operation: 'getProfile(java.lang.String, java.lang.String)',
                    arguments: [$scope.versionId, $scope.profileId]
                }, onSuccess(render));
            }
        };

        $scope.showChangeParentsDialog = function () {
            $scope.selectedParents = $scope.row.parentIds.map(function (parent) {
                return {
                    id: parent,
                    selected: true
                };
            });
            $scope.changeParentsDialog = true;
        };

        $scope.removeParentProfile = function (parent) {
            $scope.markedForDeletion = parent;
            $scope.removeParentDialog = true;
        };

        $scope.doRemoveParentProfile = function () {
            var parents = $scope.row.parentIds.exclude($scope.markedForDeletion);
            Fabric.changeProfileParents(jolokia, $scope.versionId, $scope.profileId, parents, function () {
                notification('success', 'Removed parent profile ' + $scope.markedForDeletion + ' from ' + $scope.profileId);
                Core.$apply($scope);
            }, function (response) {
                notification('error', 'Failed to change parent profiles of ' + $scope.profileId + ' due to ' + response.error);
                Core.$apply($scope);
            });
        };

        $scope.changeAttribute = function (attribute, value) {
            jolokia.request({
                type: 'exec',
                method: 'post',
                mbean: Fabric.managerMBean,
                operation: 'setProfileAttribute',
                arguments: [$scope.versionId, $scope.profileId, attribute, value]
            }, {
                success: function () {
                    // TODO - we're secretly hiding that the ng-click event is firing twice...
                    // notification('success', "Set attribute " + attribute + " to " + value);
                    Core.$apply($scope);
                },
                error: function (response) {
                    console.log("Failed to set attribute " + attribute + " to " + value + " due to " + response.error);

                    // notification('error', "Failed to set attribute " + attribute + " to " + value + " due to " + response.error);
                    Core.$apply($scope);
                }
            });
        };

        $scope.doChangeParents = function () {
            $scope.changeParentsDialog = false;
            var parents = $scope.selectedParents.map(function (parent) {
                return parent.id;
            });
            Fabric.changeProfileParents(jolokia, $scope.versionId, $scope.profileId, parents, function () {
                notification('success', 'Successfully changed parent profiles of ' + $scope.profileId);
                Core.$apply($scope);
            }, function (response) {
                notification('error', 'Failed to change parent profiles of ' + $scope.profileId + ' due to ' + response.error);
                Core.$apply($scope);
            });
        };

        $scope.goto = function (location) {
            $location.url(location);
        };

        $scope.addNewThing = function (title, type, current) {
            if (Core.isBlank($scope.newThingName)) {
                return;
            }
            $scope.thingName = title;
            $scope.currentThing = current;
            $scope.currentThingType = type;
            $scope.doAddThing();
        };

        $scope.deleteThing = function (title, type, current, item) {
            $scope.thingName = title;
            $scope.currentThing = current;
            $scope.currentThingType = type;
            $scope.currentThingItem = item;
            $scope.deleteThingDialog = true;
        };

        $scope.updateThing = function (title, type, current) {
            $scope.thingName = title;
            $scope.currentThing = current;
            $scope.currentThingType = type;
            $scope.callSetProfileThing("Changed", "change", title);
        };

        $scope.mavenLink = function (url) {
            return Maven.mavenLink(url);
        };

        $scope.callSetProfileThing = function (success, error, thing) {
            jolokia.request({
                type: 'exec',
                mbean: Fabric.managerMBean,
                operation: "setProfile" + $scope.currentThingType + "(java.lang.String, java.lang.String, java.util.List)",
                arguments: [$scope.versionId, $scope.profileId, $scope.currentThing]
            }, {
                method: 'POST',
                success: function () {
                    notification('success', success + ' ' + thing);
                    $scope.newThingName = '';
                    Core.$apply($scope);
                },
                error: function (response) {
                    notification('error', 'Failed to ' + error + ' ' + thing + ' due to ' + response.error);
                    Core.$apply($scope);
                }
            });
        };

        $scope.doDeleteThing = function () {
            $scope.currentThing.remove($scope.currentThingItem);
            $scope.callSetProfileThing('Deleted', 'delete', $scope.currentThingItem);
        };

        $scope.doAddThing = function () {
            if (!$scope.currentThing.any($scope.newThingName)) {
                $scope.currentThing.push($scope.newThingName);
                $scope.addThingDialog = false;
                $scope.callSetProfileThing('Added', 'add', $scope.newThingName);
            } else {
                notification('error', 'There is already a ' + $scope.thingName + ' with the name ' + $scope.newThingName);
            }
        };

        $scope.deleteFile = function (file) {
            $scope.markedForDeletion = file;
            $scope.deleteFileDialog = true;
        };

        $scope.doDeleteFile = function () {
            $scope.deleteFileDialog = false;
            Fabric.deleteConfigFile(jolokia, $scope.versionId, $scope.profileId, $scope.markedForDeletion, function () {
                notification('success', 'Deleted file ' + $scope.markedForDeletion);
                $scope.markedForDeletion = '';
                Core.$apply($scope);
            }, function (response) {
                notification('error', 'Failed to delete file ' + $scope.markedForDeletion + ' due to ' + response.error);
                $scope.markedForDeletion = '';
                Core.$apply($scope);
            });
        };

        $scope.doCreateFile = function () {
            $scope.newFileDialog = false;
            Fabric.newConfigFile(jolokia, $scope.versionId, $scope.profileId, $scope.newFileName, function () {
                notification('success', 'Created new configuration file ' + $scope.newFileName);
                $location.path("/fabric/profile/" + $scope.versionId + "/" + $scope.profileId + "/" + $scope.newFileName);
            }, function (response) {
                notification('error', 'Failed to create ' + $scope.newFileName + ' due to ' + response.error);
            });
        };

        $scope.copyProfile = function () {
            $scope.copyProfileDialog = false;

            if ($scope.profileId.has('-') && !$scope.newProfileName.has('-')) {
                var parts = $scope.profileId.split('-');
                parts.pop();
                parts.push($scope.newProfileName);
                $scope.newProfileName = parts.join('-');
            }

            notification('info', 'Copying ' + $scope.profileId + ' to ' + $scope.newProfileName);

            Fabric.copyProfile(jolokia, $scope.versionId, $scope.profileId, $scope.newProfileName, true, function () {
                notification('success', 'Created new profile ' + $scope.newProfileName);
                Fabric.gotoProfile(workspace, jolokia, localStorage, $location, $scope.versionId, { id: $scope.newProfileName });
                Core.$apply($scope);
            }, function (response) {
                notification('error', 'Failed to create new profile ' + $scope.newProfileName + ' due to ' + response.error);
                Core.$apply($scope);
            });
        };

        function render(response) {
            if (!angular.isDefined($scope.row)) {
                $scope.loading = false;
            }
            var responseJson = angular.toJson(response.value);

            if ($scope.profileResponseJson !== responseJson) {
                if (!$scope.activeTab) {
                    $scope.activeTab = "features";
                }
                $scope.profileResponseJson = responseJson;
                $scope.row = response.value;
                var id = $scope.row.id;
                var version = $scope.row.version;
                $scope.configFolderLink = null;
                if ($scope.hasFabricWiki() && id && version) {
                    $scope.configFolderLink = "#/wiki/branch/" + version + "/view/fabric/profiles/" + Fabric.profilePath(id);
                }
                Core.$apply($scope);
            }
        }
    }
    Fabric.ProfileController = ProfileController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function TestController($scope, jolokia, $q, workspace, $templateCache) {
        $scope.mavenMBean = Maven.getMavenIndexerMBean(workspace);

        $scope.html = "text/html";
        $scope.versionSelector = $templateCache.get("versionSelectorTemplate");
        $scope.profileIncludes = $templateCache.get("profile-includes");
        $scope.profileExcludes = $templateCache.get("profile-excludes");
        $scope.containerList = $templateCache.get("containerListTemplate");
        $scope.profileLink = $templateCache.get("profileLinkTemplate");

        $scope.version = {};
        $scope.versionId = '';
        $scope.someUri = '';
        $scope.uriParts = [];

        $scope.version = {};

        $scope.osp = [];
        $scope.vid = '1.0';
        $scope.someProfiles = ['a-mq', 'aws-ec2'];

        $scope.selectedProfiles = [
            {
                id: '1-dot-0',
                selected: true
            }, {
                id: 'a-mq',
                selected: true
            }];

        $scope.selectedProfilesString = "";

        $scope.$watch('version', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if ($scope.version && !Object.equal($scope.version, {})) {
                    $scope.versionId = $scope.version.id;
                }
            }
        });

        $scope.$watch('osp', function (newValue, oldValue) {
            $scope.selectedProfilesString = angular.toJson($scope.osp);
        });

        $scope.$watch('someUri', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.uriParts = newValue.split("/");
            }
        });

        $scope.$watch('uriParts', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if (newValue.length === 1 && newValue.length < oldValue.length) {
                    if (oldValue.last() !== '' && newValue.first().has(oldValue.last())) {
                        var merged = oldValue.first(oldValue.length - 1).include(newValue.first());
                        $scope.someUri = merged.join('/');
                    }
                }
            }
        }, true);

        $scope.doCompletionMaven = function (something) {
            return Maven.completeMavenUri($q, $scope, workspace, jolokia, something);
        };

        $scope.doCompletionFabric = function (something) {
            return Fabric.completeUri($q, $scope, workspace, jolokia, something);
        };
    }
    Fabric.TestController = TestController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function PatchingController($scope, jolokia, localStorage, $location) {
        $scope.files = [];
        $scope.targetVersion = $location.search()['versionId'];
        $scope.newVersionName = '';
        $scope.proxyUser = localStorage['fabric.userName'];
        $scope.proxyPassword = localStorage['fabric.password'];
        $scope.saveJmxCredentials = false;

        $scope.cancel = function () {
            $location.url('/fabric/view').search({ cv: $scope.targetVersion });
        };

        $scope.valid = function () {
            return $scope.files && $scope.files.length > 0 && $scope.targetVersion !== null && $scope.proxyUser && $scope.proxyPassword;
        };

        $scope.go = function () {
            var message = $scope.files.length + ' patches';

            if ($scope.files.length === 1) {
                message = "patch: " + $scope.files[0].fileName;
            }

            notification('info', "Applying " + message);

            if ($scope.saveJmxCredentials) {
                localStorage['fabric.userName'] = $scope.proxyUser;
                localStorage['fabric.password'] = $scope.proxyPassword;
            }

            var files = $scope.files.map(function (file) {
                return file.absolutePath;
            });

            Fabric.applyPatches(jolokia, files, $scope.targetVersion, $scope.newVersionName, $scope.proxyUser, $scope.proxyPassword, function () {
                notification('success', "Successfully applied " + message);
                $location.url("/fabric/view");
                Core.$apply($scope);
            }, function (response) {
                Fabric.log.error("Failed to apply ", message, " due to ", response.error);
                Fabric.log.info("Stack trace: ", response.stacktrace);
                Core.$apply($scope);
            });
        };
    }
    Fabric.PatchingController = PatchingController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function ProfilesController($scope, $location, workspace, jolokia) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        $scope.defaultVersion = Fabric.getDefaultVersion(jolokia);
        $scope.version = { id: $scope.defaultVersion.id };

        $scope.selected = [];
        $scope.selectedParents = [];
        $scope.selectedParentVersion = [];

        $scope.deleteVersionDialog = false;
        $scope.deleteProfileDialog = false;

        $scope.createProfileDialog = false;
        $scope.createVersionDialog = false;

        $scope.triggerResize = function () {
            setTimeout(function () {
                $('.dialogGrid').trigger('resize');
            }, 10);
        };

        $scope.$watch('createProfileDialog', function () {
            if ($scope.createProfileDialog) {
                $scope.triggerResize();
            }
        });

        $scope.$watch('createVersionDialog', function () {
            if ($scope.createVersionDialog) {
                $scope.triggerResize();
            }
        });

        $scope.newProfileName = '';
        $scope.newVersionName = '';

        var key = $location.search()['pv'];
        if (key) {
            $scope.version = { id: key };
        }

        key = $location.search()['ao'];

        // lets default to activeOnly if no query parameter used
        $scope.activeOnly = !angular.isDefined(key) || key === 'true';

        $scope.versions = [];
        $scope.profiles = [];

        $scope.versionResponse = [];
        $scope.profilesResponse = [];

        $scope.$watch('activeOnly', function (oldValue, newValue) {
            if (oldValue === newValue) {
                return;
            }
            var q = $location.search();
            q['ao'] = "" + $scope.activeOnly;
            $location.search(q);
        });

        $scope.$watch('version', function (oldValue, newValue) {
            var q = $location.search();
            q['pv'] = $scope.version.id;
            $location.search(q);

            if (oldValue === newValue) {
                notification('info', "Please wait, fetching profile data for version " + $scope.version.id);
            }

            Core.unregister(jolokia, $scope);
            var versionId = $scope.version.id;
            if (versionId) {
                Core.register(jolokia, $scope, [
                    { type: 'exec', mbean: Fabric.managerMBean, operation: 'versions()' },
                    { type: 'exec', mbean: Fabric.managerMBean, operation: 'getProfiles(java.lang.String, java.util.List)', arguments: [versionId, ["id", "parentIds", "childIds", "containerCount", "locked", "abstract"]] }], onSuccess(render));
            }
        });

        $scope.selectedHasContainers = function () {
            return $scope.selected.findAll(function (item) {
                return item.containerCount > 0;
            }).length > 0;
        };

        $scope.versionCanBeDeleted = function () {
            if ($scope.version.id === $scope.defaultVersion.id) {
                return true;
            }
            if ($scope.versions.length === 0) {
                return true;
            }
            return $scope.profiles.findAll(function (item) {
                return item.containerCount > 0;
            }).length > 0;
        };

        $scope.createProfileGridOptions = {
            data: 'profiles',
            selectedItems: $scope.selectedParents,
            showSelectionCheckbox: true,
            multiSelect: true,
            selectWithCheckboxOnly: false,
            keepLastSelected: false,
            columnDefs: [{
                    field: 'id',
                    displayName: 'Name'
                }]
        };

        $scope.createVersionGridOptions = {
            data: 'versions',
            selectedItems: $scope.selectedParentVersion,
            showSelectionCheckbox: true,
            multiSelect: false,
            selectWithCheckboxOnly: false,
            keepLastSelected: false,
            columnDefs: [{
                    field: 'id',
                    displayName: 'Name'
                }]
        };

        $scope.gridOptions = {
            data: 'profiles',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                filterText: ''
            },
            selectedItems: $scope.selected,
            showSelectionCheckbox: true,
            multiSelect: true,
            selectWithCheckboxOnly: true,
            keepLastSelected: false,
            checkboxCellTemplate: '<div class="ngSelectionCell"><input tabindex="-1" class="ngSelectionCheckbox" type="checkbox" ng-checked="row.selected" ng-disabled="row.entity.containerCount > 0 || row.entity.childIds.length > 0"/></div>',
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'Name',
                    cellTemplate: '<div class="ngCellText"><a ng-href="#/fabric/profile/{{$parent.version.id}}/{{row.getProperty(col.field)}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 300
                },
                {
                    field: 'attributes',
                    displayName: 'A',
                    headerCellTemplate: '<div ng-click="col.sort()" class="ngHeaderSortColumn {{col.headerClass}}" ng-style="{\'cursor\': col.cursor}" ng-class="{ \'ngSorted\': !noSortVisible }"><div class="ngHeaderText colt{{$index}} pagination-centered" title="Attributes"><i class="icon-cogs"></i></div><div class="ngSortButtonDown" ng-show="col.showSortButtonDown()"></div><div class="ngSortButtonUp" ng-show="col.showSortButtonUp()"></div></div>',
                    cellTemplate: '<div class="ngCellText"><ul class="unstyled inline"><li class="attr-column"><i ng-show="row.entity.locked" title="Locked" class="icon-lock"></i></li><li class="attr-column"><i ng-show="row.entity.abstract" title="Abstract" class="icon-font"></i></li></ul></div>',
                    width: 52
                },
                {
                    field: 'containerCount',
                    displayName: 'C',
                    headerCellTemplate: '<div ng-click="col.sort()" class="ngHeaderSortColumn {{col.headerClass}}" ng-style="{\'cursor\': col.cursor}" ng-class="{ \'ngSorted\': !noSortVisible }"><div class="ngHeaderText colt{{$index}} pagination-centered" title="Containers"><i class="icon-truck"></i></div><div class="ngSortButtonDown" ng-show="col.showSortButtonDown()"></div><div class="ngSortButtonUp" ng-show="col.showSortButtonUp()"></div></div>',
                    cellTemplate: '<div class="ngCellText pagination-centered"><a ng-show="row.getProperty(col.field) > 0" title="{{row.entity.containers.sortBy().join(\'\n\')}}" href="#/fabric/containers?cv={{$parent.version.id}}&cp={{row.entity.id}}{{hash}}">{{row.getProperty(col.field)}}</a></div>',
                    width: 28
                },
                {
                    field: 'parentIds',
                    displayName: 'Parent Profiles',
                    cellTemplate: '<div class="ngCellText"><ul class="unstyled inline"><li ng-repeat="profile in row.entity.parentIds.sortBy()"><a href="#/fabric/profile/{{$parent.version.id}}/{{profile}}">{{profile}}</a></li></ul></div>',
                    width: 400
                },
                {
                    field: 'childIds',
                    displayName: 'Child Profiles',
                    cellTemplate: '<div class="ngCellText"><ul class="unstyled inline"><li ng-repeat="profile in row.entity.childIds.sortBy()"><a href="#/fabric/profile/{{$parent.version.id}}/{{profile}}">{{profile}}</a></li></ul></div>',
                    width: 800
                }
            ]
        };

        $scope.doCreateProfile = function () {
            $scope.createProfileDialog = false;
            var parents = $scope.selectedParents.map(function (profile) {
                return profile.id;
            });
            Fabric.createProfile(jolokia, $scope.version.id, $scope.newProfileName, parents, function () {
                notification('success', "Created profile " + $scope.newProfileName);
                $scope.newProfileName = "";
                Core.$apply($scope);
            }, function (response) {
                notification('error', "Failed to create profile " + $scope.newProfileName + " due to " + response.error);
            });
        };

        $scope.doCreateVersion = function () {
            $scope.createVersionDialog = false;

            var success = function (response) {
                notification('success', "Created version " + response.value.id);
                $scope.newVersionName = '';
                $scope.version = response.value;
                Core.$apply($scope);
            };

            var error = function (response) {
                var msg = "Error creating new version: " + response.error;
                if ($scope.newVersionName !== '') {
                    msg = "Error creating " + $scope.newVersionName + " : " + response.error;
                }
                notification('error', msg);
            };

            if ($scope.selectedParentVersion.length > 0 && $scope.newVersionName !== '') {
                Fabric.createVersionWithParentAndId(jolokia, $scope.selectedParentVersion[0].id, $scope.newVersionName, success, error);
            } else if ($scope.newVersionName !== '') {
                Fabric.createVersionWithId(jolokia, $scope.newVersionName, success, error);
            } else {
                Fabric.createVersion(jolokia, success, error);
            }
        };

        $scope.deleteVersion = function () {
            // avoid getting any not found errors while deleting the version
            Core.unregister(jolokia, $scope);

            Fabric.deleteVersion(jolokia, $scope.version.id, function () {
                notification('success', "Deleted version " + $scope.version.id);
                $scope.version = $scope.defaultVersion;
                Core.$apply($scope);
            }, function (response) {
                notification('error', "Failed to delete version " + $scope.version.id + " due to " + response.error);
                $scope.version = $scope.defaultVersion;
                Core.$apply($scope);
            });
        };

        $scope.deleteSelected = function () {
            $scope.selected.each(function (profile) {
                Fabric.deleteProfile(jolokia, $scope.version.id, profile.id, function () {
                    notification('success', "Deleted profile " + profile.id);
                }, function (response) {
                    notification('error', "Failed to delete profile " + profile.id + ' due to ' + response.error);
                });
            });
        };

        function filterActive(data) {
            var rc = data;
            if ($scope.activeOnly) {
                rc = data.filter(function (item) {
                    return item.containerCount > 0;
                });
            }
            return rc;
        }

        function render(response) {
            clearNotifications();

            if (response.request.operation === 'versions()') {
                if (!Object.equal($scope.versionResponse, response.value)) {
                    $scope.versionResponse = response.value;
                    $scope.versions = response.value.map(function (version) {
                        var v = {
                            id: version.id,
                            'defaultVersion': version.defaultVersion
                        };

                        if (v['defaultVersion']) {
                            $scope.defaultVersion = v;
                        }

                        return v;
                    });
                    $scope.version = Fabric.setSelect($scope.version, $scope.versions);

                    Core.$apply($scope);
                }
            } else {
                if (!Object.equal($scope.profilesResponse, response.value)) {
                    $scope.profilesResponse = response.value;
                    $scope.profiles = [];

                    $scope.profilesResponse.forEach(function (profile) {
                        $scope.profiles.push({
                            id: profile.id,
                            parentIds: profile.parentIds,
                            childIds: profile.childIds,
                            containerCount: profile.containerCount,
                            containers: profile.containers,
                            locked: profile.locked,
                            abstract: profile['abstract']
                        });
                    });

                    $scope.profiles = filterActive($scope.profiles);
                    Core.$apply($scope);
                }
            }
        }
    }
    Fabric.ProfilesController = ProfilesController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function CreateContainerController($scope, $element, $compile, $location, workspace, jolokia, localStorage, userDetails) {
        var log = Logger.get("Fabric");

        if (!('fabric.userName' in localStorage)) {
            localStorage['fabric.userName'] = userDetails.username;
            localStorage['fabric.password'] = userDetails.password;
        }

        $scope.versionsOp = 'versions()';

        $scope.entity = {
            // default options
            number: 1,
            saveJmxCredentials: true
        };

        // the form properties stored in local storage
        // which we then default when creating a new container
        var localStorageProperties = {
            child: {
                jmxUser: 'fabric.userName',
                jmxPassword: 'fabric.password'
            },
            openshift: {
                serverUrl: 'openshift.serverUrl',
                login: 'openshift.login',
                password: 'openshift.password',
                domain: 'openshift.domain',
                gearProfile: 'openshift.gearProfile'
            },
            jclouds: {
                owner: 'jclouds.owner',
                credential: 'jclouds.credential',
                providerName: 'jclouds.providerName',
                imageId: 'jclouds.imageId',
                hardwareId: 'jclouds.hardwareId',
                locationId: 'jclouds.locationId',
                group: 'jclouds.group',
                instanceType: 'jclouds.instanceType'
            }
        };

        $scope.providers = Fabric.registeredProviders(jolokia);

        //console.log("providers: ", $scope.providers);
        $scope.selectedProvider = $scope.providers[Object.extended($scope.providers).keys().first()];
        $scope.resolvers = [];
        $scope.schema = {};

        $scope.response = {};

        $scope.versions = [];
        $scope.profiles = [];

        $scope.selectedVersion = {};

        $scope.selectedProfiles = [];
        $scope.selectedProfileIds = '';
        $scope.selectedVersionId = '';
        $scope.profileIdFilter = '';

        // referenced static data for child
        $scope.child = {
            rootContainers: []
        };

        // referenced static data for openshift
        $scope.openShift = {
            loginDataKey: "openshift.loginData",
            params: null,
            domains: [],
            gearProfiles: [],
            tryLogin: "",
            login: function () {
                var entity = $scope.entity;
                var serverUrl = Core.pathGet(entity, ["serverUrl"]) || "openshift.redhat.com";
                var login = Core.pathGet(entity, ["login"]);
                var password = Core.pathGet(entity, ["password"]);

                log.debug("Invoking login to server " + serverUrl + " user " + login);
                $scope.openShift.loginFailed = false;
                if (serverUrl && login && password) {
                    $scope.openShift.domains = [];
                    Fabric.getOpenShiftDomains(workspace, jolokia, serverUrl, login, password, function (results) {
                        $scope.openShift.domains = results;
                        log.debug("found openshift domains: " + results);

                        // lets default the value if there's only 1
                        if (results.length === 1) {
                            $scope.entity.domain = results[0];
                        }
                        Core.$apply($scope);

                        Fabric.getOpenShiftGearProfiles(workspace, jolokia, serverUrl, login, password, function (results) {
                            $scope.openShift.gearProfiles = results;
                            log.debug("found openshift gears: " + $scope.openShift.gearProfiles);

                            // now lets store the current settings so they can be defaulted next time without a login
                            savePropertiesInLocalStorage();
                            var loginData = {
                                domains: $scope.openShift.domains,
                                gearProfiles: $scope.openShift.gearProfiles
                            };
                            localStorage[$scope.openShift.loginDataKey] = angular.toJson(loginData);
                            Core.$apply($scope);
                        });
                    }, function (error) {
                        $scope.openShift.loginFailed = true;
                        Core.$apply($scope);
                    });
                }
            }
        };

        // referenced static data for jclouds
        $scope.jclouds = {};

        // holds all the form objects from nested child scopes
        $scope.forms = {};

        $scope.showAddProfileDialog = false;

        $scope.$watch('selectedProvider', function (newValue, oldValue) {
            if ($scope.selectedProvider) {
                Fabric.getSchema($scope.selectedProvider.id, $scope.selectedProvider.className, jolokia, function (schema) {
                    $scope.schema = schema;
                    $scope.resolvers = Fabric.getResolvers($scope.selectedProvider.id);
                    Core.$apply($scope);
                });
            }
        }, true);

        $scope.$watch('schema', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.entity['providerType'] = $scope.selectedProvider.id;
                $location.search('tab', $scope.selectedProvider.id);

                var providerId = $scope.entity['providerType'];
                var properties = localStorageProperties[providerId];

                // e.g. key = jmxUser, value = fabric.userName
                //
                //    $scope.entity['jmxUser'] = localStorage['fabric.userName'];
                //    $scope.entity['jmxPassword'] = localStorage['fabric.password'];
                angular.forEach(properties, function (value, key) {
                    var localValue = localStorage[value];
                    if (localValue) {
                        $scope.entity[key] = localValue;
                        log.debug("Defaulted entity " + key + " to " + localValue + " from localStorage");
                    }
                });

                if (providerId === "openshift") {
                    var loginDataText = localStorage[$scope.openShift.loginDataKey];
                    if (loginDataText) {
                        log.debug("Loaded openshift login details: " + loginDataText);
                        var loginData = Wiki.parseJson(loginDataText);
                        if (loginData) {
                            angular.forEach(["domains", "gearProfiles"], function (key) {
                                var value = loginData[key];

                                // assume all non-empty arrays for n ow
                                if (value && angular.isArray(value) && value.length) {
                                    $scope.openShift[key] = value;
                                }
                            });
                        }
                    }
                }

                Forms.defaultValues($scope.entity, $scope.schema);

                if ($scope.selectedProvider.id === 'child') {
                    // load the root containers and default the parent if its not set
                    var rootContainers = Fabric.getRootContainers(jolokia);
                    $scope.child.rootContainers = rootContainers;
                    if (rootContainers && rootContainers.length === 1 && !$scope.entity["parent"]) {
                        $scope.entity["parent"] = rootContainers[0];
                    }
                } else {
                    if ('parent' in $scope.entity) {
                        delete $scope.entity["parent"];
                    }
                }

                // updates autofilled fields
                window.setTimeout(function () {
                    $('input[ng-model]').trigger('input');
                }, 100);
            }
        }, true);

        $scope.$watch('versions', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if (!$scope.selectedVersion) {
                    if ($scope.selectedVersionId !== '') {
                        $scope.selectedVersion = $scope.versions.find(function (v) {
                            return v.id === $scope.selectedVersionId;
                        });
                    } else {
                        $scope.selectedVersion = $scope.versions.find(function (v) {
                            return v.defaultVersion;
                        });
                    }
                }
            }
        });

        $scope.$watch('selectedVersion', function (newValue, oldValue) {
            if (oldValue !== newValue) {
                if (newValue && 'id' in newValue) {
                    $scope.selectedVersionId = newValue['id'];
                    $location.search('versionId', $scope.selectedVersionId);
                }
            }
        }, true);

        $scope.deselect = function (profile) {
            profile.selected = false;
            $scope.selectedProfiles.remove(function (p) {
                return p.id === profile.id;
            });
        };

        $scope.$watch('selectedProfiles', function (newValue, oldValue) {
            if (oldValue !== newValue) {
                log.debug("selectedProfiles: ", $scope.selectedProfiles);
                $scope.selectedProfileIds = $scope.selectedProfiles.map(function (p) {
                    return p.id;
                }).join(',');
            }
        }, true);

        $scope.$watch('selectedProfileIds', function (newValue, oldValue) {
            var profileIds = $scope.selectedProfileIds.split(',');
            var selected = [];
            profileIds.each(function (id) {
                selected.push({
                    id: id,
                    selected: true
                });
            });
            $scope.selectedProfiles = selected;
            $location.search('profileIds', $scope.selectedProfileIds);
        });

        $scope.massage = function (str) {
            if (str === 'name') {
                return 'containerName';
            }
            return str;
        };

        $scope.rootContainers = function () {
            return Fabric.getRootContainers(jolokia);
        };

        $scope.init = function () {
            var tab = $location.search()['tab'];
            if (tab) {
                $scope.selectedProvider = $scope.providers[tab];
            }

            var parentId = $location.search()['parentId'];
            if (parentId) {
                $scope.entity['parent'] = parentId;
            }

            var versionId = $location.search()['versionId'];
            if (versionId) {
                $scope.selectedVersion = {
                    id: versionId
                };
            }

            var profileIds = $location.search()['profileIds'];
            if (profileIds) {
                $scope.selectedProfileIds = profileIds;
            }

            var count = $location.search()['number'];
            if (count) {
                $scope.entity.number = count;
            }
        };

        $scope.init();

        $scope.$on('$routeUpdate', $scope.init);

        /**
        * Saves the provider specific properties into localStorage; called on a succesful submit
        * or on a Login in the form so we remember the last successful login attempt.
        */
        function savePropertiesInLocalStorage() {
            var providerId = $scope.entity['providerType'];

            // e.g. key = jmxUser, value = fabric.userName
            //    localStorage['fabric.userName'] = $scope.entity.jmxUser;
            //    localStorage['fabric.password'] = $scope.entity.jmxPassword;
            var properties = localStorageProperties[providerId];

            angular.forEach(properties, function (value, key) {
                var entityValue = $scope.entity[key];
                if (entityValue) {
                    localStorage[value] = entityValue;
                }
            });
        }

        $scope.onSubmit = function (json, form) {
            var providerId = $scope.entity['providerType'];
            if (json.saveJmxCredentials || 'child' !== providerId) {
                savePropertiesInLocalStorage();
            }

            // remove possibly dodgy values if they are blank
            json = Fabric.sanitizeJson(json);
            delete json.saveJmxCredentials;

            if (json.number === 1) {
                delete json.number;
            }

            var selectedVersion = $scope.selectedVersion;
            if (selectedVersion) {
                json['version'] = selectedVersion.id;
            }
            if ($scope.selectedProfiles.length > 0) {
                json['profiles'] = $scope.selectedProfiles.map(function (p) {
                    return p.id;
                });
            }

            var createJson = angular.toJson(json);

            log.debug("createContainers json:\n" + createJson);

            setTimeout(function () {
                jolokia.execute(Fabric.managerMBean, 'createContainers(java.util.Map)', createJson, {
                    method: "post",
                    success: function (response) {
                        log.debug("Response from creating container(s): ", response);
                        var error = false;
                        if ('<not available>' in response) {
                            var message = response['<not available>'];
                            if (message.toLowerCase().has('exception')) {
                                error = true;
                                var cont = "container";
                                if (json.number) {
                                    cont = Core.maybePlural(json.number, "container");
                                }
                                notification('error', "Creating " + cont + " failed: " + message);
                            }
                        }

                        // check for error if a container already exists with that name
                        var text = response[json.name];
                        if (text && text.toLowerCase().has('already exists')) {
                            error = true;
                            notification('error', "Creating container " + json.name + " failed as a container with that name already exists.");
                        }

                        angular.forEach(response.value, function (value, key) {
                            error = true;
                            notification('error', "Creating container " + key + " failed: " + value);
                        });
                        if (!error) {
                            notification('success', "Successfully created containers");
                        }
                        Core.$apply($scope);
                    },
                    error: function (response) {
                        notification('error', "Error creating containers: " + response.error);
                        Core.$apply($scope);
                    }
                });
                Core.$apply($scope);
            }, 10);

            //notification('info', "Requesting that new container(s) be created");
            $location.url('/fabric/containers');
        };
    }
    Fabric.CreateContainerController = CreateContainerController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function ContainersController($scope, $location, $route, jolokia, workspace) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        // bind model values to search params...
        Core.bindModelToSearchParam($scope, $location, "containerIdFilter", "q", "");

        // only reload the page if certain search parameters change
        Core.reloadWhenParametersChange($route, $scope, $location);

        $scope.locationMenu = {
            icon: 'icon-beer',
            title: 'Set Location',
            items: []
        };

        $scope.noLocation = "(No Location)";
        $scope.newLocationDialog = {
            dialog: new UI.Dialog(),
            onOk: function () {
                $scope.newLocationDialog.close();
                $scope.selectedContainers.each(function (container) {
                    Fabric.setContainerProperty(jolokia, container.id, 'location', $scope.newLocationName, function () {
                        Core.$apply($scope);
                    }, function () {
                        Core.$apply($scope);
                    });
                });
            },
            open: function () {
                $scope.newLocationDialog.dialog.open();
            },
            close: function () {
                $scope.newLocationDialog.dialog.close();
            }
        };
        $scope.newLocationName = "";

        $scope.addToDashboardLink = function () {
            var href = "#/fabric/containers";
            var title = "Containers";
            var size = angular.toJson({ size_y: 1, size_x: 4 });

            return "#/dashboard/add?tab=dashboard" + "&href=" + encodeURIComponent(href) + "&size=" + encodeURIComponent(size) + "&title=" + encodeURIComponent(title);
        };

        $scope.$watch('containers', function (oldValue, newValue) {
            if (oldValue !== newValue) {
                $scope.selectedContainers = $scope.containers.filter(function (c) {
                    return c.selected;
                });

                var menuItems = [];

                var locations = $scope.containers.map(function (container) {
                    if (Core.isBlank(container['location'])) {
                        return $scope.noLocation;
                    } else {
                        return container['location'];
                    }
                });
                locations.push($scope.noLocation);
                locations = locations.unique().sortBy();
                locations = locations.exclude(function (location) {
                    return Core.isBlank(location);
                });

                locations.forEach(function (location) {
                    menuItems.push({
                        title: location,
                        action: function () {
                            $scope.selectedContainers.each(function (container) {
                                var arg = location;
                                if (arg === $scope.noLocation) {
                                    arg = "";
                                }
                                Fabric.setContainerProperty(jolokia, container.id, 'location', arg, function () {
                                    Core.$apply($scope);
                                }, function () {
                                    Core.$apply($scope);
                                });
                            });
                        }
                    });
                });

                menuItems.push({
                    title: "New...",
                    action: function () {
                        $scope.newLocationName = "";
                        $scope.newLocationDialog.open();
                    }
                });

                $scope.locationMenu.items = menuItems;

                if ($scope.selectedContainers.length > 0) {
                    $scope.activeContainerId = '';
                }
            }
        }, true);
    }
    Fabric.ContainersController = ContainersController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function ActiveProfileController($scope, jolokia) {
        $scope.addToDashboardLink = function () {
            var href = "#/fabric/activeProfiles";
            var title = "Active Profiles";
            var size = angular.toJson({
                size_y: 1,
                size_x: 5
            });

            return "#/dashboard/add?tab=dashboard" + "&href=" + encodeURIComponent(href) + "&size=" + encodeURIComponent(size) + "&title=" + encodeURIComponent(title);
        };
    }
    Fabric.ActiveProfileController = ActiveProfileController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function FabricApisController($scope, localStorage, $routeParams, $location, jolokia, workspace, $compile, $templateCache) {
        $scope.path = "apis";

        Fabric.initScope($scope, $location, jolokia, workspace);

        $scope.apis = null;
        $scope.selectedApis = [];

        $scope.versionId = Fabric.getDefaultVersionId(jolokia);

        $scope.apiOptions = {
            //plugins: [searchProvider],
            data: 'apis',
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                filterText: "",
                useExternalFilter: false
            },
            selectedItems: $scope.selectedApis,
            rowHeight: 32,
            selectWithCheckboxOnly: true,
            columnDefs: [
                {
                    field: 'serviceName',
                    displayName: 'Service',
                    cellTemplate: '<div class="ngCellText">{{row.entity.serviceName}}</div>',
                    //width: 400
                    width: "***"
                },
                {
                    field: 'wadlHref',
                    displayName: 'APIs',
                    cellTemplate: '<div class="ngCellText">' + '<a ng-show="row.entity.apidocsHref" ng-href="{{row.entity.apidocsHref}}"><i class="icon-puzzle-piece"></i> Swagger</a> ' + '<a ng-show="row.entity.wadlHref" ng-href="{{row.entity.wadlHref}}"><i class="icon-puzzle-piece"></i> WADL</a> ' + '<a ng-show="row.entity.wsdlHref" ng-href="{{row.entity.wsdlHref}}"><i class="icon-puzzle-piece"></i> WSDL</a>' + '</div>',
                    //width: 100
                    width: "*"
                },
                {
                    field: 'container',
                    displayName: 'Container',
                    cellTemplate: '<div class="ngCellText"><span fabric-container-link="{{row.entity.container}}"/></div>',
                    //width: 100
                    width: "*"
                },
                {
                    field: 'version',
                    displayName: 'Version',
                    cellTemplate: '<div class="ngCellText">{{row.entity.version}}</div>',
                    //width: 100
                    width: "*"
                },
                {
                    field: 'endpoint',
                    displayName: 'Location',
                    cellTemplate: '<div class="ngCellText"><a target="endpoint" href="{{row.entity.endpoint}}">{{row.entity.endpoint}}</a></div>',
                    width: "***"
                }
            ]
        };

        function matchesFilter(text) {
            var filter = $scope.searchFilter;
            return !filter || (text && text.has(filter));
        }

        if (Fabric.fabricCreated(workspace)) {
            Core.register(jolokia, $scope, {
                type: 'exec',
                mbean: Fabric.managerMBean,
                operation: "clusterJson",
                arguments: [$scope.path] }, onSuccess(onClusterData, { error: onClusterDataError }));
        }

        /*
        * Pulls all the properties out of the objectName and adds them to the object
        */
        function addObjectNameProperties(object) {
            var objectName = object["objectName"];
            if (objectName) {
                var properties = Core.objectNameProperties(objectName);
                if (properties) {
                    angular.forEach(properties, function (value, key) {
                        if (!object[key]) {
                            object[key] = value;
                        }
                    });
                }
            }
            return null;
        }

        function createFlatList(array, json, path) {
            if (typeof path === "undefined") { path = ""; }
            angular.forEach(json, function (value, key) {
                var childPath = path + "/" + key;

                function addParameters(href) {
                    angular.forEach(["container", "objectName"], function (name) {
                        var param = value[name];
                        if (param) {
                            href += "&" + name + "=" + encodeURIComponent(param);
                        }
                    });
                    return href;
                }

                // lets check if we are a services object or a folder
                var services = value["services"];
                if (services && angular.isArray(services) && value["id"]) {
                    value["path"] = childPath;
                    if (services.length) {
                        var url = services[0];
                        value["endpoint"] = url;
                        addObjectNameProperties(value);

                        // lets use proxy if external URL
                        url = Core.useProxyIfExternal(url);
                        value["serviceName"] = trimQuotes(value["service"]);
                        var apidocs = value["apidocs"];
                        var wadl = value["wadl"];
                        var wsdl = value["wsdl"];
                        if (apidocs) {
                            value["apidocsHref"] = addParameters("/hawtio-swagger/index.html?baseUri=" + url + apidocs);
                        }
                        if (wadl) {
                            value["wadlHref"] = addParameters("#/fabric/api/wadl?wadl=" + encodeURIComponent(url + wadl));
                        }
                        if (wsdl) {
                            value["wsdlHref"] = addParameters("#/fabric/api/wsdl?wsdl=" + encodeURIComponent(url + wsdl));
                        }
                    }
                    array.push(value);
                } else {
                    createFlatList(array, value, childPath);
                }
            });
        }

        function onClusterData(response) {
            var responseJson = null;
            if (response) {
                responseJson = response.value;
            }
            if ($scope.responseJson === responseJson) {
                return;
            }
            $scope.apis = [];
            $scope.responseJson = responseJson;

            try  {
                var json = JSON.parse(responseJson);
                createFlatList($scope.apis, json);
                Core.$apply($scope);
            } catch (e) {
                console.log("Failed to parse JSON " + e);
                console.log("JSON: " + responseJson);
            }
        }

        function onClusterDataError(response) {
            // make sure we initialise the apis so we know to show the warning of no
            // APIs available yet
            $scope.apis = [];
            Core.$apply($scope);
            Core.defaultJolokiaErrorHandler(response);
        }
    }
    Fabric.FabricApisController = FabricApisController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function VersionSelector($templateCache) {
        return {
            restrict: 'A',
            replace: true,
            templateUrl: Fabric.templatePath + "versionSelector.html",
            scope: {
                selectedVersion: '=fabricVersionSelector',
                availableVersions: '=?',
                menuBind: '=?',
                exclude: '@'
            },
            controller: function ($scope, $element, $attrs, jolokia) {
                $scope.versions = [];
                $scope.responseJson = '';

                $scope.$watch('selectedVersion', function (newValue, oldValue) {
                    if (newValue !== oldValue) {
                        if (newValue && 'id' in newValue) {
                            $scope.selectedVersion = $scope.versions.find(function (version) {
                                return version.id === newValue['id'];
                            });
                        } else {
                            $scope.selectedVersion = $scope.versions.find(function (version) {
                                return version.defaultVersion;
                            });
                        }
                    }
                });

                $scope.$watch('versions', function (newValue, oldValue) {
                    if (newValue !== oldValue) {
                        if ($scope.selectedVersion && 'id' in $scope.selectedVersion) {
                            $scope.selectedVersion = $scope.versions.find(function (version) {
                                return version.id === $scope.selectedVersion['id'];
                            });
                        } else {
                            $scope.selectedVersion = $scope.versions.find(function (version) {
                                return version.defaultVersion;
                            });
                        }
                    }
                }, true);

                function excludeVersions(versions, exclude) {
                    if (angular.isString(exclude)) {
                        if (exclude.has("[") && exclude.has("]")) {
                            exclude = angular.fromJson(exclude);
                        } else {
                            exclude = [exclude];
                        }
                    }

                    //log.debug("exclude: ", exclude);
                    if (!exclude || exclude.length === 0) {
                        return versions;
                    }
                    return versions.exclude(function (v) {
                        return exclude.some(function (e) {
                            return e === v.id;
                        });
                    });
                }

                function generateMenu(versions) {
                    return $scope.versions.map(function (v) {
                        return {
                            title: v.id,
                            action: function () {
                                $scope.selectedVersion = v;
                                if (!Core.isBlank($scope.onPick)) {
                                    $scope.$parent.$eval($scope.onPick, {
                                        version: v['id']
                                    });
                                }
                            }
                        };
                    });
                }

                $scope.$watch('exclude', function (newValue, oldValue) {
                    if (newValue !== oldValue) {
                        // need to rebuild the original version list
                        if ($scope.responseJson) {
                            var versions = angular.fromJson($scope.responseJson);
                            buildArray(versions);
                        }
                    }
                });

                function buildArray(versions) {
                    //log.debug("Building array from: ", versions);
                    $scope.versions = Fabric.sortVersions(versions, $scope.desc);
                    $scope.versions = excludeVersions($scope.versions, $scope.exclude);
                    if ($scope.config) {
                        $scope.config.items = generateMenu($scope.versions);
                    }
                    $scope.availableVersions = $scope.versions;
                }

                $scope.render = function (response) {
                    var responseJson = angular.toJson(response.value);
                    if ($scope.responseJson !== responseJson) {
                        $scope.responseJson = responseJson;
                        buildArray(response.value);
                        Core.$apply($scope);
                    }
                };

                Core.register(jolokia, $scope, {
                    type: 'exec',
                    mbean: Fabric.managerMBean,
                    operation: 'versions(java.util.List)',
                    arguments: [
                        ['id', 'defaultVersion']
                    ]
                }, onSuccess($scope.render));
            },
            link: function ($scope, $element, $attrs) {
                $scope.template = $templateCache.get('withSelect');
                if (Core.parseBooleanValue($attrs['useMenu'])) {
                    $scope.config = {
                        title: 'Version'
                    };
                    if (!Core.isBlank($attrs['menuTitle'])) {
                        $scope.config.title = $attrs['menuTitle'];
                    }
                    if (!Core.isBlank($attrs['menuBind'])) {
                        $scope.$watch('menuBind', function (newValue, oldValue) {
                            if (!Core.isBlank(newValue)) {
                                $scope.config.title = newValue;
                            }
                        });
                    }
                    if (!Core.isBlank($attrs['onPick'])) {
                        $scope.onPick = $attrs['onPick'];
                    }
                    if (!Core.isBlank($attrs['useIcon'])) {
                        $scope.config.icon = $attrs['useIcon'];
                    }
                    $scope.desc = 'desc' in $attrs;
                    $scope.template = $templateCache.get('withMenu');
                }
            }
        };
    }
    Fabric.VersionSelector = VersionSelector;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    function FabricViewController($scope, $location, jolokia, localStorage, workspace) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        $scope.containerArgs = ["id", "alive", "parentId", "profileIds", "versionId", "provisionResult", "jolokiaUrl", "root"];
        $scope.containersOp = 'containers(java.util.List)';
        $scope.ensembleContainerIdListOp = 'EnsembleContainers';

        $scope.init = function () {
            var activeVersionId = $location.search()['cv'];
            if (activeVersionId) {
                $scope.activeVersionId = activeVersionId;
                $scope.activeVersion = {
                    id: $scope.activeVersionId
                };
            }

            var profiles = $location.search()['sp'];
            $scope.selectedProfileIds = [];
            if (profiles) {
                $scope.selectedProfileIds = profiles.split(',');
            }

            var containers = $location.search()['sc'];
            $scope.selectedContainerIds = [];
            if (containers) {
                $scope.selectedContainerIds = containers.split(',');
            }
        };

        $scope.versions = [];
        $scope.profiles = [];
        $scope.containers = [];
        $scope.activeProfiles = [];

        $scope.activeVersion = {};
        $scope.activeVersionId = '';
        $scope.selectedContainers = [];
        $scope.selectedProfiles = [];
        $scope.selectedActiveProfiles = [];

        $scope.dialogProfiles = [];

        $scope.profileIdFilter = '';
        $scope.activeProfileIdFilter = '';
        $scope.containerIdFilter = '';

        $scope.filterActiveVersion = false;
        $scope.filterActiveProfile = false;

        $scope.deleteVersionDialog = new UI.Dialog();
        $scope.deleteProfileDialog = new UI.Dialog();
        $scope.createProfileDialog = new UI.Dialog();

        $scope.ensembleContainerIds = [];
        $scope.profileSelectedAll = false;
        $scope.profileNoneSelected = true;

        $scope.targetContainer = {};

        // Tweaks to ensure ng-grid displays on dialogs
        $scope.triggerResize = function () {
            setTimeout(function () {
                $('.dialogGrid').trigger('resize');
            }, 10);
        };

        /*
        $scope.$watch('createProfileDialog', function() {
        if ($scope.createProfileDialog) {
        $scope.triggerResize();
        }
        });
        
        $scope.$watch('createVersionDialog', function() {
        if ($scope.createVersionDialog) {
        $scope.triggerResize();
        }
        });
        */
        // holders for dialog data
        $scope.newProfileName = '';
        $scope.selectedParents = [];
        $scope.selectedParentVersion = [];

        $scope.$on('$routeUpdate', $scope.init);

        // watchers for selection handling
        $scope.$watch('activeVersionId', function (oldValue, newValue) {
            $location.search('cv', $scope.activeVersionId);
        });

        $scope.$watch('activeVersion', function (newValue, oldValue) {
            if (newValue !== oldValue && $scope.activeVersion && $scope.activeVersion.id !== $scope.activeVersionId) {
                $scope.activeVersionId = $scope.activeVersion.id;
            }
        });

        $scope.$watch('containers', function (oldValue, newValue) {
            if (oldValue !== newValue) {
                $scope.selectedContainers = $scope.containers.filter(function (c) {
                    return c.selected;
                });

                if ($scope.selectedContainers.length > 0) {
                    $scope.activeContainerId = '';
                }
            }
        }, true);

        $scope.$watch('activeProfiles', function (oldValue, newValue) {
            if (oldValue !== newValue) {
                $scope.selectedActiveProfiles = $scope.activeProfiles.filter(function (ap) {
                    return ap.selected;
                });
            }
        }, true);

        $scope.$watch('selectedProfiles', function (oldValue, newValue) {
            if (oldValue !== newValue) {
                var ids = $scope.getSelectedProfileIds().join(',');
                $location.search('sp', ids);
            }
        }, true);

        $scope.$watch('selectedContainers', function (oldValue, newValue) {
            if (oldValue !== newValue) {
                var ids = $scope.getSelectedContainerIds().join(',');
                $location.search('sc', ids);
            }
        }, true);

        // initialize the scope after we set all our watches
        $scope.init();

        // create profile dialog action
        $scope.doCreateProfile = function (newProfileName, selectedParents) {
            $scope.newProfileName = newProfileName;
            $scope.createProfileDialog.close();
            var parents = selectedParents.map(function (profile) {
                return profile.id;
            });
            Fabric.createProfile(jolokia, $scope.activeVersionId, $scope.newProfileName, parents, function () {
                notification('success', "Created profile " + $scope.newProfileName);
                $scope.profileIdFilter = $scope.newProfileName;
                $scope.newProfileName = "";
                Core.$apply($scope);
            }, function (response) {
                notification('error', "Failed to create profile " + $scope.newProfileName + " due to " + response.error);
                Core.$apply($scope);
            });
        };

        // delete version dialog action
        $scope.deleteVersion = function () {
            var id = $scope.activeVersionId;

            jolokia.request({
                type: 'read',
                mbean: Fabric.managerMBean,
                attribute: 'DefaultVersion'
            }, onSuccess(function (response) {
                $scope.activeVersionId = response.value;
                Core.$apply($scope);
                setTimeout(function () {
                    Fabric.deleteVersion(jolokia, id, function () {
                        notification('success', "Deleted version " + id);
                        Core.$apply($scope);
                    }, function (response) {
                        notification('error', "Failed to delete version " + id + " due to " + response.error);
                        Core.$apply($scope);
                    });
                }, 100);
            }));
        };

        $scope.deleteSelectedProfiles = function () {
            $scope.selectedProfiles.each(function (profile) {
                var profileId = profile.id;
                Fabric.deleteProfile(jolokia, $scope.activeVersionId, profileId, function () {
                    notification('success', "Deleted profile " + profileId);
                }, function (response) {
                    notification('error', "Failed to delete profile " + profileId + ' due to ' + response.error);
                });
            });
        };

        $scope.patchVersion = function (versionId) {
            $location.url('/fabric/patching').search({ versionId: versionId });
        };

        $scope.migrateVersion = function (targetName, sourceName) {
            notification('info', "Moving " + targetName + " to " + sourceName);

            Fabric.migrateContainers(jolokia, sourceName, [targetName], function () {
                notification('success', "Moved " + targetName + " to version " + sourceName);
            }, function (response) {
                notification('error', "Failed to move " + targetName + " to version " + sourceName + " due to " + response.error);
            });
        };

        $scope.addProfiles = function (targetName, profiles) {
            notification('info', "Adding " + profiles.join(', ') + " to " + targetName);

            Fabric.addProfilesToContainer(jolokia, targetName, profiles, function () {
                notification('success', "Added " + profiles.join(', ') + " to " + targetName);
            }, function (response) {
                notification('error', "Failed to add " + profiles.join(', ') + " to " + targetName + " due to " + response.error);
            });
        };

        $scope.removeActiveProfiles = function () {
            $scope.selectedActiveProfiles.each(function (profile) {
                $scope.removeActiveProfile(profile);
            });
        };

        $scope.removeActiveProfile = function (profile) {
            if ($scope.selectedContainers.length > 0) {
                $scope.selectedContainers.each(function (container) {
                    if (container.profileIds.some(profile.id) && container.versionId === profile.versionId) {
                        $scope.removeProfile(container.id, profile.id);
                    }
                });
            } else {
                $scope.removeProfile($scope.activeContainerId, profile.id);
            }
        };

        $scope.removeProfile = function (containerId, profileId) {
            notification('info', "Removing " + profileId + " from " + containerId);

            Fabric.removeProfilesFromContainer(jolokia, containerId, [profileId], function () {
                notification('success', "Removed " + profileId + " from " + containerId);
            }, function (response) {
                notification('error', "Failed to remove " + profileId + " from " + containerId + " due to " + response.error);
            });
        };

        $scope.getFilteredName = function (item) {
            return (item.versionId + " / " + item.id);
        };

        $scope.filterContainer = function (container) {
            if ($scope.containerIdFilter) {
                var filterName = $scope.getFilteredName(container);
                var filterText = $scope.containerIdFilter;
                if (!Core.matchFilterIgnoreCase(filterName, filterText)) {
                    return false;
                }
            }

            if ($scope.selectedActiveProfiles.length > 0) {
                if ($scope.selectedActiveProfiles.none(function (ap) {
                    return ap.versionId === container.versionId && container.profileIds.any(function (id) {
                        return Core.matchFilterIgnoreCase(id, ap.id);
                    });
                })) {
                    return false;
                }
            }

            return true;
        };

        $scope.filterActiveProfile = function (profile) {
            var filterText = $scope.activeProfileIdFilter;
            var filterName = $scope.getFilteredName(profile);
            if (!Core.matchFilterIgnoreCase(filterName, filterText)) {
                return false;
            }

            if ($scope.filterActiveVersion && $scope.activeVersionId && $scope.activeVersionId !== '' && profile.versionId !== $scope.activeVersionId) {
                return false;
            }

            if ($scope.selectedContainers.length > 0) {
                if ($scope.selectedContainers.none(function (c) {
                    return c.versionId === profile.versionId && c.profileIds.some(profile.id);
                })) {
                    return false;
                }
            }

            if ($scope.activeContainerId && $scope.activeContainerId !== '') {
                if ($scope.activeContainerVersion && $scope.activeContainerVersion !== '' && $scope.activeContainerVersion !== profile.versionId) {
                    return false;
                }
                if (!profile.containers.some($scope.activeContainerId)) {
                    return false;
                }
            }
            return true;
        };

        $scope.showMigrateButton = function () {
            return $scope.selectedContainers.length > 0 && $scope.activeVersionId && $scope.activeVersionId !== '';
        };

        $scope.applyVersionToContainers = function () {
            $scope.selectedContainers.each(function (c) {
                $scope.migrateVersion(c.id, $scope.activeVersionId);
            });
        };

        $scope.showProfileAddButton = function () {
            return $scope.selectedProfiles.length > 0 && $scope.selectedContainers.length > 0 && $scope.selectedContainers.every(function (c) {
                return c.versionId === $scope.activeVersionId;
            });
        };

        $scope.addProfilesToContainers = function () {
            var profileIds = $scope.selectedProfiles.map(function (p) {
                return p.id;
            });

            $scope.selectedContainers.each(function (c) {
                $scope.addProfiles(c.id, profileIds);
            });
        };

        $scope.versionCanBeDeleted = function () {
            return $scope.containers.none(function (c) {
                return c.versionId === $scope.activeVersionId;
            });
        };

        $scope.profilesCanBeDeleted = function () {
            var possibleMatches = $scope.containers.filter(function (c) {
                return c.versionId === $scope.activeVersionId;
            });

            if (possibleMatches.length === 0) {
                return true;
            }

            possibleMatches = possibleMatches.filter(function (c) {
                return $scope.selectedProfiles.some(function (p) {
                    return c.profileIds.some(p.id);
                });
            });

            if (possibleMatches.length === 0) {
                return true;
            }
            return false;
        };

        $scope.getSelectedProfileIds = function () {
            return $scope.getIds($scope.selectedProfiles);
        };

        $scope.getSelectedContainerIds = function () {
            return $scope.getIds($scope.selectedContainers);
        };

        $scope.getIds = function (arr) {
            return arr.map(function (o) {
                return o.id;
            });
        };

        $scope.containersForVersion = function (id) {
            var count = $scope.containers.findAll(function (container) {
                return container.versionId === id;
            }).length;
            if (count === 0) {
                return '';
            }
            return "(" + count + ")";
        };

        $scope.containersForProfile = function (id) {
            var profile = $scope.currentActiveProfiles().find(function (profile) {
                return profile.versionId === $scope.activeVersionId && profile.id === id;
            });
            if (profile) {
                return "(" + profile.count + ")";
            } else {
                return "";
            }
        };

        $scope.isSelectedVersion = function (id) {
            if ($scope.activeVersionId === id) {
                return 'selected';
            }
            return '';
        };

        $scope.getSelectedClass = function (obj) {
            var answer = [];
            if (obj.selected) {
                answer.push('selected');
            }
            if (angular.isDefined(obj['root']) && obj['root'] === false) {
                answer.push('child-container');
            }
            return answer.join(' ');
        };

        $scope.setActiveVersionId = function (id) {
            $scope.activeVersionId = id;
        };

        $scope.showProfile = function (profile) {
            if (angular.isDefined(profile.versionId)) {
                Fabric.gotoProfile(workspace, jolokia, localStorage, $location, profile.versionId, profile);
            } else {
                Fabric.gotoProfile(workspace, jolokia, localStorage, $location, $scope.activeVersionId, profile);
            }
        };
    }
    Fabric.FabricViewController = FabricViewController;
})(Fabric || (Fabric = {}));
var Fabric;
(function (Fabric) {
    ;

    var IconRegistry = (function () {
        function IconRegistry() {
            this.icons = {};
        }
        IconRegistry.prototype.addIcons = function (icon, domain) {
            var _this = this;
            var domains = [];
            for (var _i = 0; _i < (arguments.length - 2); _i++) {
                domains[_i] = arguments[_i + 2];
            }
            this.addIcon(icon, domain);
            if (domains && angular.isArray(domains)) {
                domains.forEach(function (domain) {
                    _this.addIcon(icon, domain);
                });
            }
        };

        IconRegistry.prototype.addIcon = function (icon, domain) {
            this.icons[domain] = icon;
        };

        IconRegistry.prototype.getIcons = function (things) {
            var _this = this;
            var answer = [];
            if (things && angular.isArray(things)) {
                things.forEach(function (thing) {
                    if (_this.icons[thing]) {
                        answer.push(_this.icons[thing]);
                    }
                });
            }
            return answer.unique();
        };

        IconRegistry.prototype.getIcon = function (thing) {
            Fabric.log.debug("Returning icon for: ", thing);
            return this.icons[thing];
        };
        return IconRegistry;
    })();
    Fabric.IconRegistry = IconRegistry;

    // Common icons that functions could return directly
    Fabric.javaIcon = {
        title: "Java",
        type: "img",
        src: "img/icons/java.svg"
    };

    // Service Icon Registry, maps icons to JMX domains
    Fabric.serviceIconRegistry = new IconRegistry();

    Fabric.serviceIconRegistry.addIcons({
        title: "Fabric8",
        type: "img",
        src: "img/icons/fabric8_icon.svg"
    }, "io.fabric8", "org.fusesource.fabric");

    Fabric.serviceIconRegistry.addIcons({
        title: "Fabric8 Insight",
        type: "icon",
        src: "icon-eye-open"
    }, "org.fusesource.insight", "io.fabric8.insight");

    Fabric.serviceIconRegistry.addIcons({
        title: "hawtio",
        type: "img",
        src: "img/hawtio_icon.svg"
    }, "hawtio");

    Fabric.serviceIconRegistry.addIcons({
        title: "Apache ActiveMQ",
        type: "img",
        src: "img/icons/messagebroker.svg"
    }, "org.apache.activemq");

    Fabric.serviceIconRegistry.addIcons({
        title: "Apache Camel",
        type: "img",
        src: "img/icons/camel.svg"
    }, "org.apache.camel");

    Fabric.serviceIconRegistry.addIcons({
        title: "Apache CXF",
        type: "icon",
        src: "icon-puzzle-piece"
    }, "org.apache.cxf");

    Fabric.serviceIconRegistry.addIcons({
        title: "Apache Karaf",
        type: "icon",
        src: "icon-beaker"
    }, "org.apache.karaf");

    Fabric.serviceIconRegistry.addIcons({
        title: "Apache Zookeeper",
        type: "icon",
        src: "icon-group"
    }, "org.apache.zookeeper");

    Fabric.serviceIconRegistry.addIcons({
        title: "Jetty",
        type: "img",
        src: "img/icons/jetty.svg"
    }, "org.eclipse.jetty.server");

    Fabric.serviceIconRegistry.addIcons({
        title: "Apache Tomcat",
        type: "img",
        src: "img/icons/tomcat.svg"
    }, "Catalina", "Tomcat");

    Fabric.serviceIconRegistry.addIcons({
        title: "Apache Cassandra",
        type: "img",
        src: "img/icons/cassandra.svg",
        "class": "girthy"
    }, "org.apache.cassandra.db", "org.apache.cassandra.metrics", "org.apache.cassandra.net", "org.apache.cassandra.request");

    // Container Icon Registry, maps icons to container types
    Fabric.containerIconRegistry = new IconRegistry();

    Fabric.containerIconRegistry.addIcons({
        title: "Apache Karaf",
        type: "icon",
        src: "icon-beaker"
    }, "karaf");

    Fabric.containerIconRegistry.addIcons({
        title: "Apache Cassandra",
        type: "img",
        src: "img/icons/cassandra.svg",
        "class": "girthy"
    }, "Cassandra");

    Fabric.containerIconRegistry.addIcons({
        title: "Apache Tomcat",
        type: "img",
        src: "img/icons/tomcat.svg"
    }, "Tomcat");

    // TODO could use a TomEE specific icon really
    Fabric.containerIconRegistry.addIcons({
        title: "Apache TomEE",
        type: "img",
        src: "img/icons/tomcat.svg"
    }, "TomEE");

    Fabric.containerIconRegistry.addIcons({
        title: "Jetty",
        type: "img",
        src: "img/icons/jetty.svg"
    }, "Jetty");

    // TODO - placeholder for Java containers
    Fabric.containerIconRegistry.addIcons(Fabric.javaIcon, "java");
})(Fabric || (Fabric = {}));
/**
* @module ElasticSearch
* @main ElasticSearch
*/
var ES;
(function (ES) {
    var pluginName = 'elasticsearch';
    var base_url = 'app/elasticsearch/html';

    /* Application level module which depends on filters, controllers, and services */
    angular.module(pluginName, ['bootstrap', 'ngResource', 'elasticjs.service', 'dangle']).config([
        '$routeProvider', function ($routeProvider) {
            $routeProvider.when('/elasticsearch', { templateUrl: base_url + '/es.html' });
        }]).run(function ($location, workspace, viewRegistry, helpRegistry) {
        // Use Full Layout of Hawtio
        viewRegistry[pluginName] = 'app/elasticsearch/html/es.html';

        helpRegistry.addUserDoc(pluginName, 'app/elasticsearch/doc/help.md', function () {
            // TODO not sure how this plugin actually shows up in the toolbar
            return false;
        });
        /*
        // Set up top-level link to our plugin
        workspace.topLevelTabs.push({
        content: "ElasticSearch",
        title: "ElasticSearch",
        isValid: (workspace) => true,
        href: () => '#/elasticsearch',
        isActive: (workspace:Workspace) => workspace.isLinkActive("elasticsearch")
        });
        */
    });

    hawtioPluginLoader.addModule(pluginName);
})(ES || (ES = {}));
var ES;
(function (ES) {
    // Function to test if a property is empty, not null
    function isEmptyObject(value) {
        return $.isEmptyObject(value);
    }
    ES.isEmptyObject = isEmptyObject;

    // Search Angular Controller used by ES
    function SearchCtrl($scope, $location, $log, ejsResource) {
        // Retrieve by default parameters from config.js
        //var defaultEsServer = $scope.defaultEsServer = "http://localhost:9200";
        var esServer = $scope.esServer = ES.config["elasticsearch"];
        var query = $scope.queryTerm = ES.config["query"];
        var facetField = $scope.facetField = "tags";
        var facetType = $scope.facetType = "terms";
        var index = $scope.indice = ES.config["indice"];
        var type = $scope.docType = ES.config["doctype"];
        var ejs;
        var request;
        $scope.log = $log;

        /* Define search function that will be called when a user
        submits a Query String search
        Query syntax : *
        Query syntax : field: 'value'
        Query syntax : field: 'value' AND field: 'value'
        Query syntax : field: 'value' OR field: 'value'
        where value corresponds to text to search ortext + * symbol
        */
        $scope.search = function () {
            if (isEmptyObject(ejs)) {
                console.log("Init EJS server");
                ejs = initElasticsearchServer(esServer);
            }

            // Initialize ES Server to send request
            setupEsRequest();

            // Setup Query String
            request = request.query(ejs.QueryStringQuery(query));

            // Run query
            var results = request.doSearch();

            console.log("Do Elastic Search");

            results.then(function (results) {
                //$location.path("/elasticjs");
                // Reset field after search
                $scope.queryTerm = "";

                if (typeof results.error != 'undefined') {
                    // Message should be displayed in the web page as a modal window
                    console.error("ES error : " + results.error);

                    // Solution proposed by kibana3
                    // $scope.panel.error = $scope.parse_error(results.error);
                    return;
                }

                console.log(results.hits.total + " : results retrieved");
                $scope.results = results;
            });
        };

        $scope.facetTermsSearch = function () {
            if (isEmptyObject(ejs)) {
                console.log("Init EJS server");
                ejs = initElasticsearchServer(esServer);
            }

            // Initialize ES Server to send request
            setupEsRequest();

            if (!isEmptyObject($scope.facetField)) {
                facetField = $scope.facetField;
            }

            if (!isEmptyObject($scope.facetType)) {
                facetType = $scope.facetType;
            }

            // Setup QueryString and Facets
            request = request.query(ejs.QueryStringQuery(query)).facet(ejs.TermsFacet("termFacet").field(facetField).size(50));

            // Run query
            var results = request.doSearch();

            console.log("Do Elastic Search");

            results.then(function (results) {
                //$location.path("/elasticjs");
                // Reset field after search
                $scope.queryTerm = "";

                if (typeof results.error != 'undefined') {
                    // Message should be displayed in the web page as a modal window
                    console.error("ES error : " + results.error);

                    // Solution proposed by kibana3
                    // $scope.panel.error = $scope.parse_error(results.error);
                    return;
                }

                console.log(results.hits.total + " : results retrieved");
                $scope.results = results;
            });
        };

        $scope.facetDateHistogramSearch = function () {
            if (isEmptyObject(ejs)) {
                console.log("Init EJS server");
                ejs = initElasticsearchServer(esServer);
            }

            // Initialize ES Server to send request
            setupEsRequest();

            if (!isEmptyObject($scope.facetField)) {
                facetField = $scope.facetField;
            }

            if (!isEmptyObject($scope.facetType)) {
                facetType = $scope.facetType;
            }

            // Setup QueryString and Facets
            request = request.query(ejs.QueryStringQuery(query)).facet(ejs.DateHistogramFacet("dateHistoFacet").field(facetField).interval("minute"));

            // Run query
            var results = request.doSearch();

            console.log("Do Elastic Search");

            results.then(function (results) {
                //$location.path("/elasticjs");
                // Reset field after search
                $scope.queryTerm = "";

                if (typeof results.error != 'undefined') {
                    // Message should be displayed in the web page as a modal window
                    console.error("ES error : " + results.error);

                    // Solution proposed by kibana3
                    // $scope.panel.error = $scope.parse_error(results.error);
                    return;
                }

                console.log(results.hits.total + " : results retrieved");
                $scope.results = results;
            });
        };

        // index the sample documents using data
        // coming from json file
        $scope.indexSampleDocs = function () {
            var host = "http://" + location.host;

            if (isEmptyObject(ejs)) {
                console.log("EJS object is not defined - create it - setupEsRequest");
                ejs = initElasticsearchServer(esServer);
            }

            // Load json records from JSON file
            // & create elastcsearch document
            var docs = [];
            $.getJSON(host + "/hawtio/app/elasticsearch/js/data.json", function (result) {
                $.each(result, function (i, field) {
                    console.log("Field : " + field);
                    docs[i] = ejs.Document(index, type, i).source(field);
                    docs[i].refresh(true).doIndex();
                });
            });
            // Using sugarjs & ECMA5 forEach
            /*
            var doSearch = ( $scope.search ).after(docs.length);
            docs.forEach(function (doc) {
            console.log("Do Index called");
            doc.refresh(true).doIndex(doSearch);
            });
            */
        };

        function setupEsRequest() {
            console.log("ES Server = " + $scope.esServer);
            console.log("Indice = " + $scope.indice);
            console.log("Type = " + $scope.docType);
            console.log("Query = " + $scope.queryTerm);

            if (!isEmptyObject($scope.indice)) {
                index = $scope.indice;
            }

            if (!isEmptyObject($scope.esServer)) {
                esServer = $scope.esServer;
            }

            if (!isEmptyObject($scope.docType)) {
                type = $scope.docType;
            }

            if (!isEmptyObject($scope.queryTerm)) {
                query = $scope.queryTerm;
            }

            var ejs = ejsResource($scope.esServer);

            // Define Request to call ES
            request = ejs.Request().indices(index).types(type);

            console.log("Request to call ElasticSearch defined");
        }

        function initElasticsearchServer(esServer) {
            return ejsResource(esServer);
        }

        $scope.parse_error = function (data) {
            var _error = data.match("nested: (.*?);");
            return _error == null ? data : _error[1];
        };
    }
    ES.SearchCtrl = SearchCtrl;
})(ES || (ES = {}));
var ES;
(function (ES) {
    ES.config = {
        elasticsearch: "http://" + window.location.hostname + ":9200",
        indice: "twitter",
        doctype: "tweet",
        query: "*"
    };
})(ES || (ES = {}));
/**
* @module JVM
*/
var JVM;
(function (JVM) {
    function JVMsController($scope, $window, $location, workspace, jolokia, mbeanName) {
        JVM.configureScope($scope, $location, workspace);
        $scope.data = [];
        $scope.deploying = false;
        $scope.status = '';

        $scope.fetch = function () {
            notification('info', 'Discovering local JVM processes, please wait...');
            jolokia.request({
                type: 'exec', mbean: mbeanName,
                operation: 'listLocalJVMs()',
                arguments: []
            }, {
                success: render,
                error: function (response) {
                    $scope.data = [];
                    $scope.status = 'Could not discover local JVM processes: ' + response.error;
                    Core.$apply($scope);
                }
            });
        };

        $scope.stopAgent = function (pid) {
            notification('info', "Attempting to detach agent from PID " + pid);
            jolokia.request({
                type: 'exec', mbean: mbeanName,
                operation: 'stopAgent(java.lang.String)',
                arguments: [pid]
            }, onSuccess(function () {
                notification('success', "Detached agent from PID " + pid);
                $scope.fetch();
            }));
        };

        $scope.startAgent = function (pid) {
            notification('info', "Attempting to attach agent to PID " + pid);
            jolokia.request({
                type: 'exec', mbean: mbeanName,
                operation: 'startAgent(java.lang.String)',
                arguments: [pid]
            }, onSuccess(function () {
                notification('success', "Attached agent to PID " + pid);
                $scope.fetch();
            }));
        };

        $scope.connectTo = function (url) {
            $window.open("?url=" + encodeURIComponent(url));
        };

        function render(response) {
            $scope.data = response.value;
            if ($scope.data.length === 0) {
                $scope.status = 'Could not discover local JVM processes';
            }
            Core.$apply($scope);
        }

        $scope.fetch();
    }
    JVM.JVMsController = JVMsController;
})(JVM || (JVM = {}));
/**
* @module JVM
*/
var JVM;
(function (JVM) {
    JVM.log = Logger.get("JVM");

    JVM.connectControllerKey = "jvmConnectSettings";
    JVM.connectionSettingsKey = "jvmConnect";

    JVM.logoPath = 'img/icons/jvm/';

    JVM.logoRegistry = {
        'jetty': JVM.logoPath + 'jetty-logo-80x22.png',
        'tomcat': JVM.logoPath + 'tomcat-logo.gif',
        'generic': JVM.logoPath + 'java-logo.svg'
    };

    /**
    * Adds common properties and functions to the scope
    * @method configureScope
    * @for Jvm
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @param {Core.Workspace} workspace
    */
    function configureScope($scope, $location, workspace) {
        $scope.isActive = function (href) {
            var tidy = Core.trimLeading(href, "#");
            var loc = $location.path();
            return loc === tidy;
        };

        $scope.isValid = function (link) {
            return link && link.isValid(workspace);
        };

        $scope.hasLocalMBean = function () {
            return JVM.hasLocalMBean(workspace);
        };

        $scope.breadcrumbs = [
            {
                content: '<i class=" icon-signin"></i> Remote',
                title: "Connect to a remote JVM running Jolokia",
                isValid: function (workspace) {
                    return true;
                },
                href: "#/jvm/connect"
            },
            {
                content: '<i class="icon-list-ul"></i> Local',
                title: "View a diagram of the route",
                isValid: function (workspace) {
                    return hasLocalMBean(workspace);
                },
                href: "#/jvm/local"
            },
            {
                content: '<i class="icon-signin"></i> Discovery',
                title: "Discover",
                isValid: function (workspace) {
                    return hasDiscoveryMBean(workspace);
                },
                href: "#/jvm/discover"
            }
        ];
    }
    JVM.configureScope = configureScope;

    function hasLocalMBean(workspace) {
        return workspace.treeContainsDomainAndProperties('hawtio', { type: 'JVMList' });
    }
    JVM.hasLocalMBean = hasLocalMBean;

    function hasDiscoveryMBean(workspace) {
        return workspace.treeContainsDomainAndProperties('jolokia', { type: 'Discovery' });
    }
    JVM.hasDiscoveryMBean = hasDiscoveryMBean;
})(JVM || (JVM = {}));
/**
* @module JVM
*/
var JVM;
(function (JVM) {
    function ResetController($scope, localStorage) {
        $scope.doClearConnectSettings = function () {
            var doReset = function () {
                delete localStorage[JVM.connectControllerKey];
                delete localStorage[JVM.connectionSettingsKey];
                setTimeout(function () {
                    window.location.reload();
                }, 10);
            };
            doReset();
        };
    }
    JVM.ResetController = ResetController;
})(JVM || (JVM = {}));
/**
* @module JVM
*/
var JVM;
(function (JVM) {
    function DiscoveryController($scope, localStorage, jolokia) {
        $scope.discovering = true;

        $scope.$watch('agents', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                $scope.selectedAgent = $scope.agents.find(function (a) {
                    return a['selected'];
                });
            }
        }, true);

        $scope.closePopover = function ($event) {
            $($event.currentTarget).parents('.popover').prev().popover('hide');
        };

        function doConnect(agent) {
            if (!agent.url) {
                notification('warning', 'No URL available to connect to agent');
                return;
            }
            var options = new Core.ConnectToServerOptions();

            var urlObject = Core.parseUrl(agent.url);
            angular.extend(options, urlObject);
            options.userName = agent.username;
            options.password = agent.password;

            Core.connectToServer(localStorage, options);
        }
        ;

        $scope.connectWithCredentials = function ($event, agent) {
            $scope.closePopover($event);
            doConnect(agent);
        };

        $scope.gotoServer = function ($event, agent) {
            if (agent.secured) {
                $($event.currentTarget).popover('show');
            } else {
                doConnect(agent);
            }
        };

        $scope.getElementId = function (agent) {
            return agent.agent_id.dasherize().replace(/\./g, "-");
        };

        $scope.getLogo = function (agent) {
            if (agent.server_product) {
                return JVM.logoRegistry[agent.server_product];
            }
            return JVM.logoRegistry['generic'];
        };

        $scope.filterMatches = function (agent) {
            if (Core.isBlank($scope.filter)) {
                return true;
            } else {
                return angular.toJson(agent).toLowerCase().has($scope.filter.toLowerCase());
            }
        };

        $scope.getAgentIdClass = function (agent) {
            if ($scope.hasName(agent)) {
                return "";
            }
            return "strong";
        };

        $scope.hasName = function (agent) {
            if (agent.server_vendor && agent.server_product && agent.server_version) {
                return true;
            }
            return false;
        };

        function render(response) {
            if (!response.value) {
                return;
            }
            var responseJson = angular.toJson(response.value.sortBy(function (agent) {
                return agent['agent_id'];
            }), true);
            if ($scope.responseJson !== responseJson) {
                if ($scope.discovering) {
                    $scope.discovering = false;
                }
                $scope.responseJson = responseJson;
                JVM.log.debug("agents: ", $scope.agents);
                $scope.agents = response.value;
                Core.$apply($scope);
            }
        }

        var updateRate = localStorage['updateRate'];
        if (updateRate > 0) {
            Core.register(jolokia, $scope, {
                type: 'exec', mbean: 'jolokia:type=Discovery',
                operation: 'lookupAgentsWithTimeout',
                arguments: [updateRate]
            }, onSuccess(render));
        } else {
            Core.register(jolokia, $scope, {
                type: 'exec', mbean: 'jolokia:type=Discovery',
                operation: 'lookupAgents',
                arguments: []
            }, onSuccess(render));
        }
    }
    JVM.DiscoveryController = DiscoveryController;
})(JVM || (JVM = {}));
/**
* @module JVM
*/
var JVM;
(function (JVM) {
    function NavController($scope, $location, workspace) {
        JVM.configureScope($scope, $location, workspace);
    }
    JVM.NavController = NavController;
})(JVM || (JVM = {}));
/**
* @module JVM
* @main JVM
*/
/*
var JVM;
(function (JVM) {
    JVM.rootPath = 'app/jvm';
    JVM.templatePath = JVM.rootPath + '/html/';
    JVM.pluginName = 'jvm';

    angular.module(JVM.pluginName, ['bootstrap', 'ngResource', 'datatable', 'hawtioCore', 'hawtio-forms', 'ui']).config(function ($routeProvider) {
        $routeProvider.when('/jvm/discover', { templateUrl: JVM.templatePath + 'discover.html' }).when('/jvm/connect', { templateUrl: JVM.templatePath + 'connect.html' }).when('/jvm/local', { templateUrl: JVM.templatePath + 'local.html' });
    }).constant('mbeanName', 'hawtio:type=JVMList').run(function ($location, workspace, viewRegistry, layoutFull, helpRegistry, preferencesRegistry) {
        viewRegistry[JVM.pluginName] = JVM.templatePath + 'layoutConnect.html';
        helpRegistry.addUserDoc('jvm', 'app/jvm/doc/help.md');

        preferencesRegistry.addTab("Connect", 'app/jvm/html/reset.html');

        workspace.topLevelTabs.push({
            id: "connect",
            content: "Connect",
            title: "Connect to other JVMs",
            isValid: function (workspace) {
                return true;
            },
            href: function () {
                return '#/jvm/connect';
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("jvm");
            }
        });
    });

    hawtioPluginLoader.addModule(JVM.pluginName);
})(JVM || (JVM = {}));
*/
/**
* @module JVM
*/
var JVM;
(function (JVM) {
    function ConnectController($scope, $location, localStorage, workspace) {
        JVM.configureScope($scope, $location, workspace);

        $scope.forms = {};

        $scope.chromeApp = Core.isChromeApp();
        $scope.useProxy = $scope.chromeApp ? false : true;

        $scope.settings = {
            last: 1,
            lastConnection: ''
        };

        // load settings like current tab, last used connection
        if (JVM.connectControllerKey in localStorage) {
            try  {
                $scope.settings = angular.fromJson(localStorage[JVM.connectControllerKey]);
            } catch (e) {
                // corrupt config
                delete localStorage[JVM.connectControllerKey];
            }
        }

        // load connection settings
        // TODO add known default configurations here...
        $scope.connectionConfigs = {};

        if (JVM.connectionSettingsKey in localStorage) {
            try  {
                $scope.connectionConfigs = angular.fromJson(localStorage[JVM.connectionSettingsKey]);
            } catch (e) {
                // corrupt config
                delete localStorage[JVM.connectionSettingsKey];
            }
        }

        /*
        log.debug("Controller settings: ", $scope.settings);
        log.debug("Current config: ", $scope.currentConfig);
        log.debug("All connection settings: ", $scope.connectionConfigs);
        */
        $scope.formConfig = {
            properties: {
                connectionName: {
                    type: 'java.lang.String',
                    tooltip: 'Name for this connection',
                    'input-attributes': {
                        'placeholder': 'Unnamed...'
                    }
                },
                scheme: {
                    type: 'java.lang.String',
                    tooltip: 'HTTP or HTTPS',
                    required: true
                },
                host: {
                    type: 'java.lang.String',
                    tooltip: 'Target host to connect to',
                    required: true
                },
                port: {
                    type: 'java.lang.Integer',
                    tooltip: 'The HTTP port used to connect to the server',
                    'input-attributes': {
                        'min': '0'
                    },
                    required: true
                },
                path: {
                    type: 'java.lang.String',
                    tooltip: "The URL path used to connect to Jolokia on the remote server"
                },
                userName: {
                    type: 'java.lang.String',
                    tooltip: "The user name to be used when connecting to Jolokia"
                },
                password: {
                    type: 'password',
                    tooltip: 'The password to be used when connecting to Jolokia'
                },
                useProxy: {
                    type: 'java.lang.Boolean',
                    tooltip: 'Whether or not we should use a proxy. See more information in the panel to the left.',
                    'control-attributes': {
                        'ng-hide': 'chromeApp'
                    }
                }
            },
            type: 'void'
        };

        function newConfig() {
            var answer = {
                scheme: 'http',
                host: 'localhost',
                path: 'jolokia',
                port: '8181',
                userName: '',
                password: ''
            };

            if ($scope.chromeApp) {
                answer['useProxy'] = false;
            } else {
                answer['useProxy'] = true;
            }
            return answer;
        }

        $scope.clearSettings = function () {
            delete localStorage[JVM.connectControllerKey];
            delete localStorage[JVM.connectionSettingsKey];
            window.location.reload();
        };

        $scope.newConnection = function () {
            $scope.settings.lastConnection = '';
        };

        $scope.deleteConnection = function () {
            Core.removeRegex($scope.settings.lastConnection);
            delete $scope.connectionConfigs[$scope.settings.lastConnection];
            var tmp = Object.extended($scope.connectionConfigs);
            if (tmp.size() === 0) {
                $scope.settings.lastConnection = '';
            } else {
                $scope.settings.lastConnection = tmp.keys().first();
            }
            localStorage[JVM.connectionSettingsKey] = angular.toJson($scope.connectionConfigs);
        };

        $scope.$watch('settings', function (newValue, oldValue) {
            if (Core.isBlank($scope.settings['lastConnection'])) {
                $scope.currentConfig = newConfig();
            } else {
                $scope.currentConfig = Object.extended($scope.connectionConfigs[$scope.settings['lastConnection']]).clone();
            }

            if (newValue !== oldValue) {
                localStorage[JVM.connectControllerKey] = angular.toJson(newValue);
            }
        }, true);

        $scope.save = function () {
            $scope.gotoServer($scope.currentConfig, null, true);
        };

        $scope.gotoServer = function (json, form, saveOnly) {
            if (json) {
                var jsonCloned = Object.extended(json).clone(true);

                JVM.log.debug("json: ", jsonCloned);

                // new connection created via the form, let's save it
                var connectionName = jsonCloned['connectionName'];
                if (Core.isBlank(connectionName)) {
                    connectionName = "Unnamed" + $scope.settings.last++;
                    jsonCloned['connectionName'] = connectionName;
                }

                var regexs = Core.getRegexs();

                var hasFunc = function (r) {
                    return r['name'] === $scope.settings.lastConnection;
                };

                if ($scope.settings.lastConnection !== connectionName && !Core.isBlank($scope.settings.lastConnection)) {
                    //we're updating an existing connection...
                    delete $scope.connectionConfigs[$scope.settings.lastConnection];

                    // clean up any similarly named regex
                    if (regexs) {
                        regexs = regexs.exclude(hasFunc);
                    }
                }

                $scope.connectionConfigs[connectionName] = jsonCloned;
                localStorage[JVM.connectionSettingsKey] = angular.toJson($scope.connectionConfigs);
                if (regexs && !regexs.any(hasFunc)) {
                    Core.storeConnectionRegex(regexs, connectionName, jsonCloned);
                }

                // let's default to saved connections now that we've a new connection
                $scope.currentConfig = jsonCloned;
                $scope.settings.lastConnection = connectionName;
            }

            if (saveOnly === true) {
                Core.$apply($scope);
                return;
            }

            var options = new Core.ConnectToServerOptions();
            var host = $scope.currentConfig['host'] || 'localhost';

            JVM.log.info("using scheme: " + $scope.currentConfig['scheme'] + " and host name: " + host + " and user: " + $scope.currentConfig['userName'] + " and password: " + ($scope.currentConfig['password'] ? "********" : $scope.currentConfig['password']));
            options.name = $scope.currentConfig['connectionName'];
            options.scheme = $scope.currentConfig['scheme'];
            options.host = host;
            options.port = $scope.currentConfig['port'];
            options.path = $scope.currentConfig['path'];
            options.userName = $scope.currentConfig['userName'];
            options.password = $scope.currentConfig['password'];
            options.useProxy = $scope.currentConfig['useProxy'];

            Core.$apply($scope);

            Core.connectToServer(localStorage, options);
        };

        function init() {
            JVM.log.debug("Initializing");
            var schemeEnum = ['http', 'https'];
            Core.pathSet($scope.formConfig, ['properties', 'scheme', 'enum'], schemeEnum);
        }

        init();
    }
    JVM.ConnectController = ConnectController;
})(JVM || (JVM = {}));
/**
* @module OpenEJB
*/
var OpenEJB;
(function (OpenEJB) {
    function TreeController($scope, $location, workspace) {
        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        $scope.$watch('workspace.tree', function () {
            if (workspace.moveIfViewInvalid())
                return;

            var children = [];
            var tree = workspace.tree;
            if (tree) {
                var nodes = tree.children;
                angular.forEach(nodes, function (node) {
                    var nodeChildren = node.children;
                    if (node.title.startsWith("openejb") && nodeChildren) {
                        children = children.concat(nodeChildren);
                    }
                });
            }
            var treeElement = $("#openejbTree");
            Jmx.enableTree($scope, $location, workspace, treeElement, children, true);

            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        function updateSelectionFromURL() {
            Jmx.updateTreeSelectionFromURL($location, $("#openejbTree"), true);
        }
    }
    OpenEJB.TreeController = TreeController;
})(OpenEJB || (OpenEJB = {}));
/**
* @module OpenEJB
* @main OpenEJB
*/
var OpenEJB;
(function (OpenEJB) {
    var pluginName = 'openejb';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'hawtioCore']).config(function ($routeProvider) {
        // TODO custom tomcat views go here...
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['openojb'] = "app/openejb/html/layoutOpenEJBTree.html";
        helpRegistry.addUserDoc('openejb', 'app/openejb/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("openejb");
        });

        workspace.topLevelTabs.push({
            id: "openejb",
            content: "OpenEJB",
            title: "Manage your OpenEJB resources",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("openejb");
            },
            href: function () {
                return "#/jmx/attributes?tab=openejb";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("openejb");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(OpenEJB || (OpenEJB = {}));
/**
* @module Dashboard
* @main Dashboard
*/
var Dashboard;
(function (Dashboard) {
    Dashboard.templatePath = 'app/dashboard/html/';
    Dashboard.pluginName = 'dashboard';

    angular.module(Dashboard.pluginName, ['bootstrap', 'ngResource', 'hawtioCore', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/dashboard/add', { templateUrl: Dashboard.templatePath + 'addToDashboard.html' }).when('/dashboard/edit', { templateUrl: Dashboard.templatePath + 'editDashboards.html' }).when('/dashboard/idx/:dashboardIndex', { templateUrl: Dashboard.templatePath + 'dashboard.html' }).when('/dashboard/id/:dashboardId', { templateUrl: Dashboard.templatePath + 'dashboard.html' }).when('/dashboard/id/:dashboardId/share', { templateUrl: Dashboard.templatePath + 'share.html' }).when('/dashboard/import', { templateUrl: Dashboard.templatePath + 'import.html' });
    }).value('ui.config', {
        // The ui-jq directive namespace
        jq: {
            gridster: {
                widget_margins: [10, 10],
                widget_base_dimensions: [140, 140]
            }
        }
    }).factory('dashboardRepository', function (workspace, jolokia, localStorage) {
        return new Dashboard.DefaultDashboardRepository(workspace, jolokia, localStorage);
    }).directive('hawtioDashboard', function () {
        return new Dashboard.GridsterDirective();
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['dashboard'] = 'app/dashboard/html/layoutDashboard.html';
        helpRegistry.addUserDoc('dashboard', 'app/dashboard/doc/help.md');

        workspace.topLevelTabs.push({
            id: "dashboard",
            content: "Dashboard",
            title: "View and edit your own custom dashboards",
            isValid: function (workspace) {
                return workspace.hasMBeans();
            },
            href: function () {
                return "#/dashboard/idx/0?tab=dashboard";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("dashboard");
            }
        });
    });

    hawtioPluginLoader.addModule(Dashboard.pluginName);
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    var FabricDashboardRepository = (function () {
        function FabricDashboardRepository(workspace, jolokia, localStorage) {
            this.workspace = workspace;
            this.jolokia = jolokia;
            this.localStorage = localStorage;
            this.details = this.getBranchAndProfiles();
        }
        FabricDashboardRepository.prototype.getBranchAndProfiles = function () {
            if (Fabric.fabricCreated(this.workspace)) {
                var container = Fabric.getCurrentContainer(this.jolokia, ['id', 'versionId', 'profiles']);
                var profiles = [];
                if (container.profiles) {
                    profiles = container.profiles.unique();
                    profiles = Fabric.filterProfiles(this.jolokia, container.versionId, profiles);
                }

                return {
                    branch: container.versionId,
                    profiles: profiles
                };
            } else {
                return {
                    branch: "1.0",
                    profiles: []
                };
            }
        };

        FabricDashboardRepository.prototype.putDashboards = function (array, commitMessage, fn) {
            var _this = this;
            var jolokia = this.jolokia;
            var details = this.details;

            var toPut = array.length;

            var maybeCallback = function () {
                toPut = toPut - 1;
                if (toPut === 0) {
                    _this.getDashboards(fn);
                }
            };

            array.forEach(function (dashboard) {
                // console.log("Saving dash: ", dashboard);
                var data = angular.toJson(dashboard, true);
                var profileId = dashboard.profileId;
                if (!profileId) {
                    // TODO maybe not just pick the first one :-)
                    profileId = details.profiles.first();
                }
                var fileName = dashboard.fileName;
                if (!fileName) {
                    fileName = Core.getUUID() + ".dashboard";
                }
                Fabric.saveConfigFile(jolokia, details.branch, profileId, fileName, data.encodeBase64(), function () {
                    maybeCallback();
                    //notification('success', "Saved dashboard " + dashboard.title);
                }, function (response) {
                    Dashboard.log.error("Failed to store dashboard: ", dashboard.title, " due to: ", response.error, " stack trace: ", response.stacktrace);
                    maybeCallback();
                });
            });
        };

        FabricDashboardRepository.prototype.deleteDashboards = function (array, fn) {
            var _this = this;
            var jolokia = this.jolokia;
            var details = this.details;

            var toDelete = array.length;

            var maybeCallback = function () {
                toDelete = toDelete - 1;
                if (toDelete === 0) {
                    _this.getDashboards(fn);
                }
            };

            array.forEach(function (dashboard) {
                var profileId = dashboard.profileId;
                var fileName = dashboard.fileName;
                if (profileId && fileName) {
                    Fabric.deleteConfigFile(jolokia, details.branch, profileId, fileName, function () {
                        maybeCallback();
                    }, function (response) {
                        Dashboard.log.error("Failed to delete dashboard: ", dashboard.title, " due to: ", response.error, " stack trace: ", response.stacktrace);
                        maybeCallback();
                    });
                }
            });
        };

        FabricDashboardRepository.prototype.createDashboard = function (options) {
            var answer = {
                title: "New Dashboard",
                group: "Fabric",
                versionId: this.details.branch,
                profileId: this.details.profiles.first(),
                widgets: []
            };
            answer = angular.extend(answer, options);
            var uuid = Core.getUUID();
            answer['id'] = uuid;
            answer['fileName'] = uuid + ".dashboard";
            return answer;
        };

        FabricDashboardRepository.prototype.cloneDashboard = function (dashboard) {
            var newDashboard = Object.clone(dashboard);
            var uuid = Core.getUUID();
            newDashboard['id'] = uuid;
            newDashboard['fileName'] = uuid + ".dashboard";
            newDashboard['title'] = "Copy of " + dashboard.title;
            return newDashboard;
        };

        FabricDashboardRepository.prototype.getType = function () {
            return 'fabric';
        };

        FabricDashboardRepository.prototype.isValid = function () {
            return Fabric.hasFabric(this.workspace);
        };

        FabricDashboardRepository.prototype.getDashboards = function (fn) {
            var _this = this;
            var jolokia = this.jolokia;
            var details = this.details;
            var dashboards = [];

            jolokia.request({
                type: 'exec',
                mbean: Fabric.managerMBean,
                operation: 'getConfigurationFiles',
                arguments: [details.branch, details.profiles, ".*dashboard"]
            }, {
                method: 'POST',
                success: function (response) {
                    angular.forEach(response.value, function (value, profile) {
                        angular.forEach(value, function (value, fileName) {
                            var dashboard = angular.fromJson(value.decodeBase64());
                            dashboard['versionId'] = details.branch;
                            dashboard['profileId'] = profile;
                            dashboard['fileName'] = fileName;
                            dashboards.push(dashboard);
                        });
                    });

                    if (dashboards.isEmpty()) {
                        dashboards.push(_this.createDashboard({}));
                    }

                    // sort dash boards by title, so they dont appear in random order
                    dashboards = dashboards.sort(function (d1, d2) {
                        var title1 = d1.title;
                        var title2 = d2.title;
                        return title1.localeCompare(title2);
                    });

                    fn(dashboards);
                },
                error: function (response) {
                    Dashboard.log.error("Failed to load dashboard data: error: ", response.error, " stack trace: ", response.stacktrace);
                    fn([]);
                }
            });
        };

        FabricDashboardRepository.prototype.getDashboard = function (id, fn) {
            this.getDashboards(function (dashboards) {
                var dashboard = dashboards.find(function (dashboard) {
                    return dashboard.id === id;
                });
                fn(dashboard);
            });
        };
        return FabricDashboardRepository;
    })();
    Dashboard.FabricDashboardRepository = FabricDashboardRepository;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    function ShareController($scope, $location, $routeParams, workspace, dashboardRepository) {
        var id = $routeParams["dashboardId"];
        dashboardRepository.getDashboard(id, onDashboardLoad);

        var options = {
            mode: {
                name: "javascript"
            }
        };
        $scope.codeMirrorOptions = CodeEditor.createEditorSettings(options);

        function onDashboardLoad(dashboard) {
            $scope.dashboard = Dashboard.cleanDashboardData(dashboard);

            $scope.json = {
                "description": "hawtio dashboards",
                "public": true,
                "files": {
                    "dashboards.json": {
                        "content": JSON.stringify($scope.dashboard, null, "  ")
                    }
                }
            };

            $scope.source = JSON.stringify($scope.dashboard, null, "  ");
            Core.$applyNowOrLater($scope);
        }
    }
    Dashboard.ShareController = ShareController;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    /**
    * Implements the ng.ILocationService interface and is used by the dashboard to supply
    * controllers with a saved URL location
    *
    * @class RectangleLocation
    */
    var RectangleLocation = (function () {
        function RectangleLocation(delegate, path, search, hash) {
            this.delegate = delegate;
            this._path = path;
            this._search = search;
            this._hash = hash;
        }
        RectangleLocation.prototype.absUrl = function () {
            return this.protocol() + this.host() + ":" + this.port() + this.path() + this.search();
        };

        RectangleLocation.prototype.hash = function (newHash) {
            if (typeof newHash === "undefined") { newHash = null; }
            if (newHash) {
                return this.delegate.hash(newHash).search('tab', null);
                //this._hash = newHash;
            }
            return this._hash;
        };

        RectangleLocation.prototype.host = function () {
            return this.delegate.host();
        };

        RectangleLocation.prototype.path = function (newPath) {
            if (typeof newPath === "undefined") { newPath = null; }
            if (newPath) {
                return this.delegate.path(newPath).search('tab', null);
            }
            return this._path;
        };

        RectangleLocation.prototype.port = function () {
            return this.delegate.port();
        };

        RectangleLocation.prototype.protocol = function () {
            return this.delegate.port();
        };

        RectangleLocation.prototype.replace = function () {
            // TODO
            return this;
        };

        RectangleLocation.prototype.search = function (parametersMap) {
            if (typeof parametersMap === "undefined") { parametersMap = null; }
            if (parametersMap) {
                return this.delegate.search(parametersMap);
            }
            return this._search;
        };

        RectangleLocation.prototype.url = function (newValue) {
            if (typeof newValue === "undefined") { newValue = null; }
            if (newValue) {
                return this.delegate.url(newValue).search('tab', null);
            }
            return this.absUrl();
        };
        return RectangleLocation;
    })();
    Dashboard.RectangleLocation = RectangleLocation;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    /**
    * The default dashboard definition if no saved dashboards are available
    *
    * @property defaultDashboards
    * @for Dashboard
    * @type {any}
    */
    var defaultDashboards = [
        {
            "title": "Monitor",
            "group": "Personal",
            "widgets": [
                {
                    "id": "w1",
                    "title": "Operating System",
                    "row": 1,
                    "col": 1,
                    "size_x": 3,
                    "size_y": 4,
                    "path": "jmx/attributes",
                    "include": "app/jmx/html/attributes.html",
                    "search": {
                        "nid": "root-java.lang-OperatingSystem"
                    },
                    "hash": ""
                },
                {
                    "id": "w3",
                    "title": "Java Heap Memory",
                    "row": 1,
                    "col": 6,
                    "size_x": 2,
                    "size_y": 2,
                    "path": "jmx/widget/donut",
                    "include": "app/jmx/html/donutChart.html",
                    "search": {},
                    "hash": "",
                    "routeParams": "{\"type\":\"donut\",\"title\":\"Java Heap Memory\",\"mbean\":\"java.lang:type=Memory\",\"attribute\":\"HeapMemoryUsage\",\"total\":\"Max\",\"terms\":\"Used\",\"remaining\":\"Free\"}"
                },
                {
                    "id": "w4",
                    "title": "Java Non Heap Memory",
                    "row": 1,
                    "col": 8,
                    "size_x": 2,
                    "size_y": 2,
                    "path": "jmx/widget/donut",
                    "include": "app/jmx/html/donutChart.html",
                    "search": {},
                    "hash": "",
                    "routeParams": "{\"type\":\"donut\",\"title\":\"Java Non Heap Memory\",\"mbean\":\"java.lang:type=Memory\",\"attribute\":\"NonHeapMemoryUsage\",\"total\":\"Max\",\"terms\":\"Used\",\"remaining\":\"Free\"}"
                },
                {
                    "id": "w5",
                    "title": "",
                    "row": 3,
                    "col": 4,
                    "size_x": 6,
                    "size_y": 2,
                    "path": "jmx/charts",
                    "include": "app/jmx/html/charts.html",
                    "search": {
                        "size": "%7B%22size_x%22%3A2%2C%22size_y%22%3A2%7D",
                        "title": "Java%20Non%20Heap%20Memory",
                        "routeParams": "%7B%22type%22%3A%22donut%22%2C%22title%22%3A%22Java%20Non%20Heap%20Memory%22%2C%22mbean%22%3A%22java.lang%3Atype",
                        "nid": "root-java.lang-Threading"
                    },
                    "hash": ""
                },
                {
                    "id": "w6",
                    "title": "System CPU Load",
                    "row": 1,
                    "col": 4,
                    "size_x": 2,
                    "size_y": 2,
                    "path": "jmx/widget/area",
                    "include": "app/jmx/html/areaChart.html",
                    "search": {},
                    "hash": "",
                    "routeParams": "{\"type\":\"area\",\"title\":\"System CPU Load\",\"mbean\":\"java.lang:type=OperatingSystem\",\"attribute\":\"SystemCpuLoad\"}"
                }
            ],
            "id": "4e9d116173ca41767e"
        }
    ];

    

    /**
    * Registry of dashboard repositories that delegates to the current effective
    * dashboard repository
    *
    * @class DefaultDashboardRepository
    * @uses DashboardRepository
    */
    var DefaultDashboardRepository = (function () {
        function DefaultDashboardRepository(workspace, jolokia, localStorage) {
            this.workspace = workspace;
            this.jolokia = jolokia;
            this.localStorage = localStorage;
            this.repository = null;
        }
        DefaultDashboardRepository.prototype.putDashboards = function (array, commitMessage, fn) {
            this.getRepository().putDashboards(array, commitMessage, fn);
        };

        DefaultDashboardRepository.prototype.deleteDashboards = function (array, fn) {
            this.getRepository().deleteDashboards(array, fn);
        };

        /**
        * Loads the dashboards then asynchronously calls the function with the data
        * @method getDashboards
        * @param {Function} fn
        */
        DefaultDashboardRepository.prototype.getDashboards = function (fn) {
            this.getRepository().getDashboards(function (values) {
                fn(values);
            });
        };

        /**
        * Loads the given dashboard and invokes the given function with the result
        * @method getDashboard
        * @param {String} id
        * @param {Function} onLoad
        */
        DefaultDashboardRepository.prototype.getDashboard = function (id, onLoad) {
            this.getRepository().getDashboard(id, onLoad);
        };

        DefaultDashboardRepository.prototype.createDashboard = function (options) {
            return this.getRepository().createDashboard(options);
        };

        DefaultDashboardRepository.prototype.cloneDashboard = function (dashboard) {
            return this.getRepository().cloneDashboard(dashboard);
        };

        DefaultDashboardRepository.prototype.getType = function () {
            return this.getRepository().getType();
        };

        DefaultDashboardRepository.prototype.isValid = function () {
            return this.getRepository().isValid();
        };

        /**
        * Looks up the MBean in the JMX tree
        * @method getRepository
        * @return {DashboardRepository}
        */
        DefaultDashboardRepository.prototype.getRepository = function () {
            if (this.repository && this.repository.isValid()) {
                return this.repository;
            }
            if (Fabric.hasFabric(this.workspace)) {
                this.repository = new Dashboard.FabricDashboardRepository(this.workspace, this.jolokia, this.localStorage);
                return this.repository;
            }
            var git = Git.createGitRepository(this.workspace, this.jolokia, this.localStorage);
            if (git) {
                this.repository = new GitDashboardRepository(this.workspace, git);
                return this.repository;
            }
            this.repository = new LocalDashboardRepository(this.workspace);
            return this.repository;
        };
        return DefaultDashboardRepository;
    })();
    Dashboard.DefaultDashboardRepository = DefaultDashboardRepository;

    /**
    * @class LocalDashboardRepository
    * @uses DashboardRepository
    */
    var LocalDashboardRepository = (function () {
        function LocalDashboardRepository(workspace) {
            this.workspace = workspace;
            this.localStorage = null;
            this.localStorage = workspace.localStorage;

            if ('userDashboards' in this.localStorage) {
                // log.info("Found previously saved dashboards");
            } else {
                this.storeDashboards(defaultDashboards);
            }
        }
        LocalDashboardRepository.prototype.loadDashboards = function () {
            var answer = angular.fromJson(localStorage['userDashboards']);
            if (answer.length === 0) {
                answer.push(this.createDashboard({}));
            }
            Dashboard.log.debug("returning dashboards: ", answer);
            return answer;
        };

        LocalDashboardRepository.prototype.storeDashboards = function (dashboards) {
            Dashboard.log.debug("storing dashboards: ", dashboards);
            localStorage['userDashboards'] = angular.toJson(dashboards);
            return this.loadDashboards();
        };

        LocalDashboardRepository.prototype.putDashboards = function (array, commitMessage, fn) {
            var dashboards = this.loadDashboards();

            array.forEach(function (dash) {
                var existing = dashboards.findIndex(function (d) {
                    return d.id === dash.id;
                });
                if (existing >= 0) {
                    dashboards[existing] = dash;
                } else {
                    dashboards.push(dash);
                }
            });
            fn(this.storeDashboards(dashboards));
        };

        LocalDashboardRepository.prototype.deleteDashboards = function (array, fn) {
            var dashboards = this.loadDashboards();
            angular.forEach(array, function (item) {
                dashboards.remove(function (i) {
                    return i.id === item.id;
                });
            });
            fn(this.storeDashboards(dashboards));
        };

        LocalDashboardRepository.prototype.getDashboards = function (fn) {
            fn(this.loadDashboards());
        };

        LocalDashboardRepository.prototype.getDashboard = function (id, fn) {
            var dashboards = this.loadDashboards();
            var dashboard = dashboards.find(function (dashboard) {
                return dashboard.id === id;
            });
            fn(dashboard);
        };

        LocalDashboardRepository.prototype.createDashboard = function (options) {
            var answer = {
                title: "New Dashboard",
                group: "Personal",
                widgets: []
            };
            answer = angular.extend(answer, options);
            answer['id'] = Core.getUUID();
            return answer;
        };

        LocalDashboardRepository.prototype.cloneDashboard = function (dashboard) {
            var newDashboard = Object.clone(dashboard);
            newDashboard['id'] = Core.getUUID();
            newDashboard['title'] = "Copy of " + dashboard.title;
            return newDashboard;
        };

        LocalDashboardRepository.prototype.getType = function () {
            return 'container';
        };

        LocalDashboardRepository.prototype.isValid = function () {
            return !Fabric.hasFabric(this.workspace) && !Git.hasGit(this.workspace);
        };
        return LocalDashboardRepository;
    })();
    Dashboard.LocalDashboardRepository = LocalDashboardRepository;

    /**
    * @class GitDashboardRepository
    * @uses DashboardRepository
    */
    var GitDashboardRepository = (function () {
        function GitDashboardRepository(workspace, git) {
            this.workspace = workspace;
            this.git = git;
            this.branch = null;
        }
        GitDashboardRepository.prototype.putDashboards = function (array, commitMessage, fn) {
            var _this = this;
            var toPut = array.length;
            var maybeCallback = function () {
                toPut = toPut - 1;
                if (toPut === 0) {
                    _this.getDashboards(fn);
                }
            };

            angular.forEach(array, function (dash) {
                var path = _this.getDashboardPath(dash);
                var contents = JSON.stringify(dash, null, "  ");
                _this.git.write(_this.branch, path, commitMessage, contents, function () {
                    maybeCallback();
                });
            });
        };

        GitDashboardRepository.prototype.deleteDashboards = function (array, fn) {
            var _this = this;
            var toDelete = array.length;
            var maybeCallback = function () {
                toDelete = toDelete - 1;
                if (toDelete === 0) {
                    _this.getDashboards(fn);
                }
            };
            angular.forEach(array, function (dash) {
                var path = _this.getDashboardPath(dash);
                var commitMessage = "Removing dashboard " + path;
                _this.git.remove(_this.branch, path, commitMessage, function () {
                    maybeCallback();
                });
            });
        };

        GitDashboardRepository.prototype.createDashboard = function (options) {
            var answer = {
                title: "New Dashboard",
                group: "Personal",
                widgets: []
            };
            answer = angular.extend(answer, options);
            answer['id'] = Core.getUUID();
            return answer;
        };

        GitDashboardRepository.prototype.cloneDashboard = function (dashboard) {
            var newDashboard = Object.clone(dashboard);
            newDashboard['id'] = Core.getUUID();
            newDashboard['title'] = "Copy of " + dashboard.title;
            return newDashboard;
        };

        GitDashboardRepository.prototype.getType = function () {
            return 'git';
        };

        GitDashboardRepository.prototype.isValid = function () {
            return Git.hasGit(this.workspace);
        };

        GitDashboardRepository.prototype.getDashboardPath = function (dash) {
            // TODO assume a user dashboard for now
            // ideally we'd look up the teams path based on the group
            var id = dash.id || Core.getUUID();
            var path = this.getUserDashboardPath(id);
            return path;
        };

        GitDashboardRepository.prototype.getDashboards = function (fn) {
            var _this = this;
            // TODO lets look in each team directory as well and combine the results...
            var path = this.getUserDashboardDirectory();
            var dashboards = [];
            this.git.read(this.branch, path, function (details) {
                var files = details.children;

                var toRead = files.length;

                var maybeCallback = function () {
                    toRead = toRead - 1;
                    if (toRead === 0) {
                        // sort dash boards by title, so they dont appear in random order
                        dashboards = dashboards.sort(function (d1, d2) {
                            var title1 = d1.title;
                            var title2 = d2.title;
                            return title1.localeCompare(title2);
                        });

                        fn(dashboards);
                    }
                };

                if (files.length === 0) {
                    dashboards.push(_this.createDashboard({}));
                    fn(dashboards);
                    return;
                }

                // we now have all the files we need; lets read all their contents
                angular.forEach(files, function (file, idx) {
                    var path = file.path;
                    if (!file.directory && path.endsWith(".json")) {
                        _this.git.read(_this.branch, path, function (details) {
                            // lets parse the contents
                            var content = details.text;
                            if (content) {
                                try  {
                                    var json = JSON.parse(content);
                                    json.uri = path;
                                    dashboards.push(json);
                                } catch (e) {
                                    console.log("Failed to parse: " + content + " due to: " + e);
                                }
                            }
                            Dashboard.log.debug("git - read ", idx, " files, total: ", files.length);
                            maybeCallback();
                        });
                    }
                });
            });
        };

        GitDashboardRepository.prototype.getDashboard = function (id, fn) {
            var path = this.getUserDashboardPath(id);
            this.git.read(this.branch, path, function (details) {
                var dashboard = null;
                var content = details.text;
                if (content) {
                    try  {
                        dashboard = JSON.parse(content);
                    } catch (e) {
                        console.log("Failed to parse: " + content + " due to: " + e);
                    }
                }
                fn(dashboard);
            });
        };

        GitDashboardRepository.prototype.getUserDashboardDirectory = function () {
            // TODO until we fix #96 lets default to a common user name so
            // all the dashboards are shared for all users for now
            //return "/dashboards/user/" + this.git.getUserName();
            return "/dashboards/team/all";
        };

        GitDashboardRepository.prototype.getUserDashboardPath = function (id) {
            return this.getUserDashboardDirectory() + "/" + id + ".json";
        };
        return GitDashboardRepository;
    })();
    Dashboard.GitDashboardRepository = GitDashboardRepository;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    var GridsterDirective = (function () {
        function GridsterDirective() {
            this.restrict = 'A';
            this.replace = true;
            this.controller = function ($scope, $element, $attrs, $location, $routeParams, $injector, $route, $templateCache, workspace, dashboardRepository, $compile) {
                $scope.route = $route;
                $scope.injector = $injector;

                var gridSize = 150;
                var gridMargin = 6;
                var gridHeight;

                $scope.gridX = gridSize;
                $scope.gridY = gridSize;

                $scope.widgetMap = {};

                $scope.$on('$destroy', function () {
                    angular.forEach($scope.widgetMap, function (value, key) {
                        if ('scope' in value) {
                            var scope = value['scope'];
                            scope.$destroy();
                        }
                    });
                });

                updateWidgets();

                $scope.removeWidget = function (widget) {
                    var gridster = getGridster();
                    var widgetElem = null;

                    // lets destroy the widgets's scope
                    var widgetData = $scope.widgetMap[widget.id];
                    if (widgetData) {
                        delete $scope.widgetMap[widget.id];
                        var scope = widgetData.scope;
                        widgetElem = widgetData.widget;
                        if (scope) {
                            scope.$destroy();
                        }
                    }
                    if (!widgetElem) {
                        // lets get the li parent element of the template
                        widgetElem = $("div").find("[data-widgetId='" + widget.id + "']").parent();
                    }
                    if (gridster && widgetElem) {
                        gridster.remove_widget(widgetElem);
                    }

                    // no need to remove it...
                    //widgetElem.remove();
                    // lets trash the JSON metadata
                    if ($scope.dashboard) {
                        var widgets = $scope.dashboard.widgets;
                        if (widgets) {
                            widgets.remove(widget);
                        }
                    }

                    updateDashboardRepository("Removed widget " + widget.title);
                };

                function changeWidgetSize(widget, sizefunc, savefunc) {
                    var gridster = getGridster();
                    var entry = $scope.widgetMap[widget.id];
                    var w = entry.widget;
                    var scope = entry.scope;
                    sizefunc(entry);
                    gridster.resize_widget(w, entry.size_x, entry.size_y);
                    gridster.set_dom_grid_height();

                    setTimeout(function () {
                        var template = $templateCache.get("widgetTemplate");
                        var div = $('<div></div>');
                        div.html(template);
                        w.html($compile(div.contents())(scope));

                        makeResizable();
                        Core.$apply($scope);

                        setTimeout(function () {
                            savefunc(widget);
                        }, 50);
                    }, 30);
                }

                $scope.onWidgetRenamed = function (widget) {
                    updateDashboardRepository("Renamed widget to " + widget.title);
                };

                function updateWidgets() {
                    $scope.id = $routeParams["dashboardId"];
                    $scope.idx = $routeParams["dashboardIndex"];
                    if ($scope.id) {
                        $scope.$emit('loadDashboards');
                        dashboardRepository.getDashboard($scope.id, onDashboardLoad);
                    } else {
                        dashboardRepository.getDashboards(function (dashboards) {
                            $scope.$emit('dashboardsUpdated', dashboards);

                            var idx = $scope.idx ? parseInt($scope.idx) : 0;
                            var id = null;
                            if (dashboards.length > 0) {
                                var dashboard = dashboards.length > idx ? dashboards[idx] : dashboard[0];
                                id = dashboard.id;
                            }
                            if (id) {
                                $location.path("/dashboard/id/" + id);
                            } else {
                                $location.path("/dashboard/edit?tab=dashboard");
                            }
                            Core.$apply($scope);
                        });
                    }
                }

                function onDashboardLoad(dashboard) {
                    $scope.dashboard = dashboard;
                    var widgets = ((dashboard) ? dashboard.widgets : null) || [];

                    var minHeight = 10;
                    var minWidth = 6;

                    angular.forEach(widgets, function (widget) {
                        if (angular.isDefined(widget.row) && minHeight < widget.row) {
                            minHeight = widget.row + 1;
                        }
                        if (angular.isDefined(widget.size_x && angular.isDefined(widget.col))) {
                            var rightEdge = widget.col + widget.size_x;
                            if (rightEdge > minWidth) {
                                minWidth = rightEdge + 1;
                            }
                        }
                    });

                    var gridster = $element.gridster({
                        widget_margins: [gridMargin, gridMargin],
                        widget_base_dimensions: [$scope.gridX, $scope.gridY],
                        extra_rows: minHeight,
                        extra_cols: minWidth,
                        max_size_x: minWidth,
                        max_size_y: minHeight,
                        draggable: {
                            stop: function (event, ui) {
                                if (serializeDashboard()) {
                                    updateDashboardRepository("Changing dashboard layout");
                                }
                            }
                        }
                    }).data('gridster');

                    var template = $templateCache.get("widgetTemplate");
                    angular.forEach(widgets, function (widget) {
                        var childScope = $scope.$new(false);
                        childScope.widget = widget;
                        var path = widget.path;
                        var search = null;
                        if (widget.search) {
                            search = Dashboard.decodeURIComponentProperties(widget.search);
                        }
                        var hash = widget.hash;
                        var location = new Dashboard.RectangleLocation($location, path, search, hash);
                        var routeParams = null;
                        if (widget.routeParams) {
                            routeParams = angular.fromJson(widget.routeParams);
                        }

                        var childWorkspace = workspace.createChildWorkspace(location);

                        //var childWorkspace = workspace;
                        childWorkspace.$location = location;

                        // now we need to update the selection from the location search()
                        if (search) {
                            var key = location.search()['nid'];
                            if (key && workspace.tree) {
                                // lets find the node for this key...
                                childWorkspace.selection = workspace.keyToNodeMap[key];
                                if (!childWorkspace.selection) {
                                    var decodedKey = decodeURIComponent(key);
                                    childWorkspace.selection = workspace.keyToNodeMap[decodedKey];
                                }
                            }
                        }

                        var $$scopeInjections = {
                            workspace: childWorkspace,
                            location: location,
                            $location: location,
                            $routeParams: routeParams
                        };
                        childScope.$$scopeInjections = $$scopeInjections;
                        childScope.inDashboard = true;

                        if (!widget.size_x || widget.size_x < 1) {
                            widget.size_x = 1;
                        }
                        if (!widget.size_y || widget.size_y < 1) {
                            widget.size_y = 1;
                        }
                        var div = $('<div></div>');
                        div.html(template);

                        var outerDiv = $('<li class="grid-block" style="display: list-item; position: absolute"></li>');
                        outerDiv.html($compile(div.contents())(childScope));
                        var w = gridster.add_widget(outerDiv, widget.size_x, widget.size_y, widget.col, widget.row);

                        $scope.widgetMap[widget.id] = {
                            widget: w,
                            scope: childScope
                        };
                    });

                    makeResizable();
                    getGridster().enable();

                    Core.$apply($scope);
                }

                function serializeDashboard() {
                    var gridster = getGridster();
                    if (gridster) {
                        var data = gridster.serialize();

                        //console.log("got data: " + JSON.stringify(data));
                        var widgets = $scope.dashboard.widgets || [];

                        // console.log("Widgets: ", widgets);
                        // lets assume the data is in the order of the widgets...
                        angular.forEach(widgets, function (widget, idx) {
                            var value = data[idx];
                            if (value && widget) {
                                // lets copy the values across
                                angular.forEach(value, function (attr, key) {
                                    return widget[key] = attr;
                                });
                            }
                        });
                        return true;
                    }
                    return false;
                }

                function makeResizable() {
                    var blocks = $('.grid-block');
                    blocks.resizable('destroy');

                    blocks.resizable({
                        grid: [gridSize + (gridMargin * 2), gridSize + (gridMargin * 2)],
                        animate: false,
                        minWidth: gridSize,
                        minHeight: gridSize,
                        autoHide: false,
                        start: function (event, ui) {
                            gridHeight = getGridster().$el.height();
                        },
                        resize: function (event, ui) {
                            //set new grid height along the dragging period
                            var g = getGridster();
                            var delta = gridSize + gridMargin * 2;
                            if (event.offsetY > g.$el.height()) {
                                var extra = Math.floor((event.offsetY - gridHeight) / delta + 1);
                                var newHeight = gridHeight + extra * delta;
                                g.$el.css('height', newHeight);
                            }
                        },
                        stop: function (event, ui) {
                            var resized = $(this);
                            setTimeout(function () {
                                resizeBlock(resized);
                            }, 300);
                        }
                    });

                    $('.ui-resizable-handle').hover(function () {
                        getGridster().disable();
                    }, function () {
                        getGridster().enable();
                    });
                }

                function resizeBlock(elmObj) {
                    //var elmObj = $(elmObj);
                    var area = elmObj.find('.widget-area');
                    var w = elmObj.width() - gridSize;
                    var h = elmObj.height() - gridSize;

                    for (var grid_w = 1; w > 0; w -= (gridSize + (gridMargin * 2))) {
                        grid_w++;
                    }

                    for (var grid_h = 1; h > 0; h -= (gridSize + (gridMargin * 2))) {
                        grid_h++;
                    }

                    var widget = {
                        id: area.attr('data-widgetId')
                    };

                    changeWidgetSize(widget, function (widget) {
                        widget.size_x = grid_w;
                        widget.size_y = grid_h;
                    }, function (widget) {
                        if (serializeDashboard()) {
                            updateDashboardRepository("Changed size of widget: " + widget.id);
                        }
                    });
                }

                function updateDashboardRepository(message) {
                    if ($scope.dashboard) {
                        var commitMessage = message;
                        if ($scope.dashboard && $scope.dashboard.title) {
                            commitMessage += " on dashboard " + $scope.dashboard.title;
                        }
                        dashboardRepository.putDashboards([$scope.dashboard], commitMessage, Dashboard.onOperationComplete);
                    }
                }

                function getGridster() {
                    return $element.gridster().data('gridster');
                }
            };
        }
        return GridsterDirective;
    })();
    Dashboard.GridsterDirective = GridsterDirective;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    function NavBarController($scope, $routeParams, $rootScope, workspace, dashboardRepository) {
        $scope.hash = workspace.hash();
        $scope._dashboards = [];

        $scope.activeDashboard = $routeParams['dashboardId'];

        $rootScope.$on('loadDashboards', loadDashboards);

        $rootScope.$on('dashboardsUpdated', dashboardLoaded);

        $scope.dashboards = function () {
            return $scope._dashboards;
        };

        $scope.isActive = function (dash) {
            return workspace.isLinkActive("#/dashboard/id/" + dash.id);
        };

        $scope.isEditing = function () {
            return workspace.isLinkActive("#/dashboard/edit");
        };

        $scope.onTabRenamed = function (dash) {
            dashboardRepository.putDashboards([dash], "Renamed dashboard", function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        };

        function dashboardLoaded(event, dashboards) {
            Dashboard.log.debug("navbar dashboardLoaded: ", dashboards);
            $scope._dashboards = dashboards;
            if (event === null) {
                $rootScope.$broadcast('dashboardsUpdated', dashboards);
                Core.$apply($scope);
            }
        }

        function loadDashboards(event) {
            dashboardRepository.getDashboards(function (dashboards) {
                // prevent the broadcast from happening...
                dashboardLoaded(event, dashboards);
                Core.$apply($scope);
            });
        }
    }
    Dashboard.NavBarController = NavBarController;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    Dashboard.log = Logger.get('Dashboard');

    /**
    * Returns the cleaned up version of the dashboard data without any UI selection state
    * @method cleanDashboardData
    * @static
    * @for Dashboard
    * @param {any} item
    * @return {any}
    */
    function cleanDashboardData(item) {
        var cleanItem = {};
        angular.forEach(item, function (value, key) {
            if (!angular.isString(key) || (!key.startsWith("$") && !key.startsWith("_"))) {
                cleanItem[key] = value;
            }
        });
        return cleanItem;
    }
    Dashboard.cleanDashboardData = cleanDashboardData;

    /**
    * Runs decodeURIComponent() on each value in the object
    * @method decodeURIComponentProperties
    * @static
    * @for Dashboard
    * @param {any} hash
    * @return {any}
    */
    function decodeURIComponentProperties(hash) {
        if (!hash) {
            return hash;
        }
        var decodeHash = {};
        angular.forEach(hash, function (value, key) {
            decodeHash[key] = value ? decodeURIComponent(value) : value;
        });
        return decodeHash;
    }
    Dashboard.decodeURIComponentProperties = decodeURIComponentProperties;

    function onOperationComplete(result) {
        console.log("Completed adding the dashboard with response " + JSON.stringify(result));
    }
    Dashboard.onOperationComplete = onOperationComplete;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    function EditDashboardsController($scope, $routeParams, $route, $location, $rootScope, dashboardRepository, jolokia, workspace) {
        $scope.hash = workspace.hash();
        $scope.selectedItems = [];
        $scope.repository = dashboardRepository;
        $scope.duplicateDashboards = new UI.Dialog();
        $scope.selectedProfilesDialog = [];
        $scope._dashboards = [];

        $rootScope.$on('dashboardsUpdated', dashboardLoaded);

        $scope.hasUrl = function () {
            return ($scope.url) ? true : false;
        };

        $scope.hasSelection = function () {
            return $scope.selectedItems.length !== 0;
        };

        $scope.gridOptions = {
            selectedItems: $scope.selectedItems,
            showFilter: false,
            showColumnMenu: false,
            filterOptions: {
                filterText: ''
            },
            data: '_dashboards',
            selectWithCheckboxOnly: true,
            showSelectionCheckbox: true,
            columnDefs: [
                {
                    field: 'title',
                    displayName: 'Dashboard',
                    cellTemplate: '<div class="ngCellText"><a ng-href="#/dashboard/id/{{row.getProperty(' + "'id'" + ')}}{{hash}}"><editable-property class="inline-block" on-save="onDashRenamed(row.entity)" property="title" ng-model="row.entity"></editable-property></a></div>'
                },
                {
                    field: 'group',
                    displayName: 'Group'
                }
            ]
        };

        $scope.onDashRenamed = function (dash) {
            dashboardRepository.putDashboards([dash], "Renamed dashboard", function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        };

        // helpers so we can enable/disable parts of the UI depending on how
        // dashboard data is stored
        $scope.usingGit = function () {
            return dashboardRepository.getType() === 'git';
        };

        $scope.usingFabric = function () {
            return dashboardRepository.getType() === 'fabric';
        };

        $scope.usingLocal = function () {
            return dashboardRepository.getType() === 'container';
        };

        if ($scope.usingFabric()) {
            $scope.container = Fabric.getCurrentContainer(jolokia, ['versionId', 'profileIds']);

            $scope.gridOptions.columnDefs.add([
                {
                    field: 'versionId',
                    displayName: 'Version'
                }, {
                    field: 'profileId',
                    displayName: 'Profile'
                }, {
                    field: 'fileName',
                    displayName: 'File Name'
                }]);
        }

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateData, 100);
        });

        $scope.goBack = function () {
            var href = Core.trimLeading($scope.url, "#");
            if (href) {
                $location.url(href);
            }
        };

        $scope.duplicateToProfiles = function () {
            if ($scope.hasSelection()) {
                $scope.duplicateDashboards.open();
            }
        };

        $scope.doDuplicateToProfiles = function () {
            $scope.duplicateDashboards.close();

            var newDashboards = [];

            $scope.selectedItems.forEach(function (dashboard) {
                $scope.selectedProfilesDialog.forEach(function (profile) {
                    var newDash = dashboardRepository.cloneDashboard(dashboard);
                    newDash['profileId'] = profile.id;
                    newDash['title'] = dashboard.title;
                    newDashboards.push(newDash);
                });
            });

            var commitMessage = "Duplicating " + $scope.selectedItems.length + " dashboards to " + $scope.selectedProfilesDialog.length + " profiles";

            dashboardRepository.putDashboards(newDashboards, commitMessage, function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        };

        $scope.addViewToDashboard = function () {
            var nextHref = null;
            angular.forEach($scope.selectedItems, function (selectedItem) {
                // TODO this could be a helper function
                var text = $scope.url;
                var query = null;
                if (text) {
                    var idx = text.indexOf('?');
                    if (idx && idx > 0) {
                        query = text.substring(idx + 1);
                        text = text.substring(0, idx);
                    }
                    text = Core.trimLeading(text, "#");
                }
                var search = {};
                if (query) {
                    var expressions = query.split("&");
                    angular.forEach(expressions, function (expression) {
                        if (expression) {
                            var names = expression.split("=");
                            var key = names[0];
                            var value = names.length > 1 ? names[1] : null;
                            if (value) {
                                value = encodeURIComponent(value);
                            }
                            var old = search[key];
                            if (old) {
                                if (!angular.isArray(old)) {
                                    old = [old];
                                    search[key] = old;
                                }
                                old.push(value);
                            } else {
                                search[key] = value;
                            }
                        }
                    });
                }

                //console.log("path is: " + text + " the search is " + JSON.stringify(search));
                if ($route && $route.routes) {
                    var value = $route.routes[text];
                    if (value) {
                        var templateUrl = value["templateUrl"];
                        if (templateUrl) {
                            if (!selectedItem.widgets) {
                                selectedItem.widgets = [];
                            }
                            var nextNumber = selectedItem.widgets.length + 1;
                            var widget = {
                                id: "w" + nextNumber, title: "",
                                row: 1,
                                col: 1,
                                size_x: 1,
                                size_y: 1,
                                path: Core.trimLeading(text, "/"),
                                include: templateUrl,
                                search: search,
                                hash: ""
                            };

                            if ($scope.widgetTitle) {
                                widget.title = $scope.widgetTitle;
                            }

                            // figure out the width of the dash
                            var gridWidth = 0;

                            selectedItem.widgets.forEach(function (w) {
                                var rightSide = w.col + w.size_x;
                                if (rightSide > gridWidth) {
                                    gridWidth = rightSide;
                                }
                            });

                            if ($scope.preferredSize) {
                                widget.size_x = parseInt($scope.preferredSize['size_x']);
                                widget.size_y = parseInt($scope.preferredSize['size_y']);
                            }

                            var found = false;

                            var left = function (w) {
                                return w.col;
                            };

                            var right = function (w) {
                                return w.col + w.size_x - 1;
                            };

                            var top = function (w) {
                                return w.row;
                            };

                            var bottom = function (w) {
                                return w.row + w.size_y - 1;
                            };

                            var collision = function (w1, w2) {
                                return !(left(w2) > right(w1) || right(w2) < left(w1) || top(w2) > bottom(w1) || bottom(w2) < top(w1));
                            };

                            if (selectedItem.widgets.isEmpty()) {
                                found = true;
                            }

                            while (!found) {
                                widget.col = 1;
                                for (; (widget.col + widget.size_x) <= gridWidth; widget.col++) {
                                    if (!selectedItem.widgets.any(function (w) {
                                        var c = collision(w, widget);
                                        return c;
                                    })) {
                                        found = true;
                                        break;
                                    }
                                }
                                if (!found) {
                                    widget.row = widget.row + 1;
                                }

                                // just in case, keep the script from running away...
                                if (widget.row > 50) {
                                    found = true;
                                }
                            }

                            if ($scope.routeParams) {
                                widget['routeParams'] = $scope.routeParams;
                            }
                            selectedItem.widgets.push(widget);

                            if (!nextHref && selectedItem.id) {
                                nextHref = "/dashboard/id/" + selectedItem.id;
                            }
                        }
                    } else {
                        // TODO we need to be able to match URI templates...
                    }
                }
            });

            // now lets update the actual dashboard config
            var commitMessage = "Add widget";
            dashboardRepository.putDashboards($scope.selectedItems, commitMessage, function (dashboards) {
                if (nextHref) {
                    // remove any dodgy query
                    delete $location.search()["href"];
                    $location.path(nextHref);
                    Core.$apply($scope);
                }
            });
        };

        $scope.create = function () {
            var counter = dashboards().length + 1;
            var title = "Untitled" + counter;
            var newDash = dashboardRepository.createDashboard({ title: title });

            dashboardRepository.putDashboards([newDash], "Created new dashboard: " + title, function (dashboards) {
                // let's just be safe and ensure there's no selections
                $scope.selectedItems.splice(0);
                dashboardLoaded(null, dashboards);
            });
        };

        $scope.duplicate = function () {
            var newDashboards = [];
            var commitMessage = "Duplicated dashboard(s) ";
            angular.forEach($scope.selectedItems, function (item, idx) {
                // lets unselect this item
                var commitMessage = "Duplicated dashboard " + item.title;
                var newDash = dashboardRepository.cloneDashboard(item);
                newDashboards.push(newDash);
            });

            // let's just be safe and ensure there's no selections
            $scope.selectedItems.splice(0);

            commitMessage = commitMessage + newDashboards.map(function (d) {
                return d.title;
            }).join(',');
            dashboardRepository.putDashboards(newDashboards, commitMessage, function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        };

        $scope.delete = function () {
            if ($scope.hasSelection()) {
                dashboardRepository.deleteDashboards($scope.selectedItems, function (dashboards) {
                    // let's just be safe and ensure there's no selections
                    $scope.selectedItems.splice(0);
                    dashboardLoaded(null, dashboards);
                });
            }
        };

        $scope.gist = function () {
            if ($scope.selectedItems.length > 0) {
                var id = $scope.selectedItems[0].id;
                $location.path("/dashboard/id/" + id + "/share");
            }
        };

        function updateData() {
            var url = $routeParams["href"];
            if (url) {
                $scope.url = decodeURIComponent(url);
            }

            var routeParams = $routeParams["routeParams"];
            if (routeParams) {
                $scope.routeParams = decodeURIComponent(routeParams);
            }
            var size = $routeParams["size"];
            if (size) {
                size = decodeURIComponent(size);
                $scope.preferredSize = angular.fromJson(size);
            }
            var title = $routeParams["title"];
            if (title) {
                title = decodeURIComponent(title);
                $scope.widgetTitle = title;
            }

            dashboardRepository.getDashboards(function (dashboards) {
                dashboardLoaded(null, dashboards);
            });
        }

        function dashboardLoaded(event, dashboards) {
            $scope._dashboards = dashboards;
            if (event === null) {
                $scope.$emit('dashboardsUpdated', dashboards);
            }
            Core.$apply($scope);
        }

        function dashboards() {
            return $scope._dashboards;
        }

        updateData();
        /*
        // TODO for case where we navigate to the add view
        // for some reason the route update event isn't enough...
        // and we need to do this async to avoid the size calculation being wrong
        // bit of a hack - would love to remove! :)
        setTimeout(updateData, 100);
        */
    }
    Dashboard.EditDashboardsController = EditDashboardsController;
})(Dashboard || (Dashboard = {}));
/**
* @module Dashboard
*/
var Dashboard;
(function (Dashboard) {
    function ImportController($scope, $location, $routeParams, workspace, dashboardRepository) {
        $scope.placeholder = "Paste the JSON here for the dashboard configuration to import...";
        $scope.source = $scope.placeholder;

        var options = {
            mode: {
                name: "javascript"
            }
        };
        $scope.codeMirrorOptions = CodeEditor.createEditorSettings(options);

        $scope.isValid = function () {
            return $scope.source && $scope.source !== $scope.placeholder;
        };

        $scope.importJSON = function () {
            var json = [];

            try  {
                json = JSON.parse($scope.source);
            } catch (e) {
                notification("error", "Could not parse the JSON\n" + e);
                json = [];
            }
            var array = [];
            if (angular.isArray(json)) {
                array = json;
            } else if (angular.isObject(json)) {
                array.push(json);
            }

            if (array.length) {
                // lets ensure we have some valid ids and stuff...
                angular.forEach(array, function (dash, index) {
                    angular.copy(dash, dashboardRepository.createDashboard(dash));
                });
                dashboardRepository.putDashboards(array, "Imported dashboard JSON", Dashboard.onOperationComplete);
                $location.path("/dashboard/edit");
            }
        };
    }
    Dashboard.ImportController = ImportController;
})(Dashboard || (Dashboard = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    var PreferencesRegistry = (function () {
        function PreferencesRegistry() {
            this.tabs = {};
        }
        PreferencesRegistry.prototype.addTab = function (name, template, isValid) {
            if (typeof isValid === "undefined") { isValid = undefined; }
            if (!isValid) {
                isValid = function () {
                    return true;
                };
            }
            this.tabs[name] = {
                template: template,
                isValid: isValid
            };
        };

        PreferencesRegistry.prototype.getTab = function (name) {
            return this.tabs[name];
        };

        PreferencesRegistry.prototype.getTabs = function () {
            var answer = {};
            angular.forEach(this.tabs, function (value, key) {
                if (value.isValid()) {
                    answer[key] = value;
                }
            });
            return answer;
        };
        return PreferencesRegistry;
    })();
    Core.PreferencesRegistry = PreferencesRegistry;
    ;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    /**
    * Parsers the given value as JSON if it is define
    */
    function parsePreferencesJson(value, key) {
        var answer = null;
        if (angular.isDefined(value)) {
            answer = Core.parseJsonText(value, "localStorage for " + key);
        }
        return answer;
    }
    Core.parsePreferencesJson = parsePreferencesJson;

    /**
    * Function to return the configured plugin for the given perspective. The returned
    * list is sorted in the configured order.
    * Notice the list contains plugins which may have been configured as disabled.
    */
    function configuredPluginsForPerspectiveId(perspectiveId, workspace, jolokia, localStorage) {
        // grab the top level tabs which is the plugins we can select as our default plugin
        var topLevelTabs = Perspective.topLevelTabsForPerspectiveId(workspace, perspectiveId);
        if (topLevelTabs && topLevelTabs.length > 0) {
            Core.log.debug("Found " + topLevelTabs.length + " plugins");

            // exclude invalid tabs at first
            topLevelTabs = topLevelTabs.filter(function (tab) {
                var href = tab.href();
                return href && isValidFunction(workspace, tab.isValid);
            });
            Core.log.debug("After filtering there are " + topLevelTabs.length + " plugins");

            var id = "plugins-" + perspectiveId;
            var initPlugins = parsePreferencesJson(localStorage[id], id);
            if (initPlugins) {
                // remove plugins which we cannot find active currently
                initPlugins = initPlugins.filter(function (p) {
                    return topLevelTabs.some(function (tab) {
                        return tab.id === p.id;
                    });
                });

                // add new active plugins which we didn't know about before
                topLevelTabs.forEach(function (tab) {
                    var knownPlugin = initPlugins.some(function (p) {
                        return p.id === tab.id;
                    });
                    if (!knownPlugin) {
                        Core.log.info("Discovered new plugin in JVM since loading configuration: " + tab.id);
                        initPlugins.push({ id: tab.id, index: -1, displayName: tab.content, enabled: true, isDefault: false });
                    }
                });
            } else {
                // okay no configured saved yet, so use what is active
                initPlugins = topLevelTabs;
            }
        }

        // okay push plugins to scope so we can see them in the UI
        var answer = safeTabsToPlugins(initPlugins);
        return answer;
    }
    Core.configuredPluginsForPerspectiveId = configuredPluginsForPerspectiveId;

    /**
    * Function which safely can turn tabs/plugins to plugins
    */
    function safeTabsToPlugins(tabs) {
        var answer = [];
        if (tabs) {
            tabs.forEach(function (tab, idx) {
                var name;
                if (angular.isUndefined(tab.displayName)) {
                    name = tab.content;
                } else {
                    name = tab.displayName;
                }
                var enabled;
                if (angular.isUndefined(tab.enabled)) {
                    enabled = true;
                } else {
                    enabled = tab.enabled;
                }
                var isDefault;
                if (angular.isUndefined(tab.isDefault)) {
                    isDefault = false;
                } else {
                    isDefault = tab.isDefault;
                }
                answer.push({ id: tab.id, index: idx, displayName: name, enabled: enabled, isDefault: isDefault });
            });
        }
        return answer;
    }
    Core.safeTabsToPlugins = safeTabsToPlugins;

    function filterTopLevelTabs(perspective, workspace, configuredPlugins) {
        var topLevelTabs = Perspective.topLevelTabsForPerspectiveId(workspace, perspective);
        if (perspective === "website")
            return topLevelTabs;

        // only include the tabs accordingly to configured
        var result = [];
        configuredPlugins.forEach(function (p) {
            if (p.enabled) {
                var pid = p.id;
                var tab = null;
                if (pid) {
                    tab = topLevelTabs.find(function (t) {
                        return t.id === pid;
                    });
                }
                if (tab) {
                    result.push(tab);
                }
            }
        });
        return result;
    }
    Core.filterTopLevelTabs = filterTopLevelTabs;

    function initPreferenceScope($scope, localStorage, defaults) {
        angular.forEach(defaults, function (_default, key) {
            $scope[key] = _default['value'];
            var converter = _default['converter'];
            var formatter = _default['formatter'];
            if (!formatter) {
                formatter = function (value) {
                    return value;
                };
            }
            if (!converter) {
                converter = function (value) {
                    return value;
                };
            }
            if (key in localStorage) {
                var value = converter(localStorage[key]);
                Core.log.debug("from local storage, setting ", key, " to ", value);
                $scope[key] = value;
            } else {
                var value = _default['value'];
                Core.log.debug("from default, setting ", key, " to ", value);
                localStorage[key] = value;
            }

            var watchFunc = _default['override'];
            if (!watchFunc) {
                watchFunc = function (newValue, oldValue) {
                    if (newValue !== oldValue) {
                        if (angular.isFunction(_default['pre'])) {
                            _default.pre(newValue);
                        }

                        var value = formatter(newValue);
                        Core.log.debug("to local storage, setting ", key, " to ", value);
                        localStorage[key] = value;

                        if (angular.isFunction(_default['post'])) {
                            _default.post(newValue);
                        }
                    }
                };
            }

            if (_default['compareAsObject']) {
                $scope.$watch(key, watchFunc, true);
            } else {
                $scope.$watch(key, watchFunc);
            }
        });
    }
    Core.initPreferenceScope = initPreferenceScope;

    /**
    * Returns true if there is no validFn defined or if its defined
    * then the function returns true.
    *
    * @method isValidFunction
    * @for Perspective
    * @param {Core.Workspace} workspace
    * @param {Function} validFn
    * @return {Boolean}
    */
    function isValidFunction(workspace, validFn) {
        return !validFn || validFn(workspace);
    }
    Core.isValidFunction = isValidFunction;

    /**
    * Gets the default configured plugin for the given perspective, or <tt>null</tt> if no default has been configured.
    */
    function getDefaultPlugin(perspectiveId, workspace, jolokia, localStorage) {
        var plugins = Core.configuredPluginsForPerspectiveId(perspectiveId, workspace, jolokia, localStorage);

        // find the default plugins
        var defaultPlugin = null;
        plugins.forEach(function (p) {
            if (p.isDefault) {
                defaultPlugin = p;
            }
        });
        return defaultPlugin;
    }
    Core.getDefaultPlugin = getDefaultPlugin;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    var HelpRegistry = (function () {
        function HelpRegistry($rootScope) {
            this.$rootScope = $rootScope;
            this.discoverableDocTypes = {
                user: 'help.md'
            };
            this.topicNameMappings = {
                activemq: 'ActiveMQ',
                camel: 'Camel',
                jboss: 'JBoss',
                jclouds: 'jclouds',
                jmx: 'JMX',
                jvm: 'Connect',
                log: 'Logs',
                openejb: 'OpenEJB'
            };
            this.subTopicNameMappings = {
                user: 'For Users',
                developer: 'For Developers',
                faq: 'FAQ'
            };
            // map plugin names to their path in the app
            this.pluginNameMappings = {
                hawtioCore: 'core',
                'hawtio-branding': 'branding',
                forceGraph: 'forcegraph',
                'hawtio-ui': 'ui',
                'hawtio-forms': 'forms',
                elasticjs: 'elasticsearch'
            };
            // let's not auto-discover help files in these plugins
            this.ignoredPlugins = [
                'core',
                'branding',
                'datatable',
                'forcegraph',
                'forms',
                'perspective',
                'tree',
                'ui'
            ];
            this.topics = {};
        }
        HelpRegistry.prototype.addUserDoc = function (topic, path, isValid) {
            if (typeof isValid === "undefined") { isValid = null; }
            this.addSubTopic(topic, 'user', path, isValid);
        };

        HelpRegistry.prototype.addDevDoc = function (topic, path, isValid) {
            if (typeof isValid === "undefined") { isValid = null; }
            this.addSubTopic(topic, 'developer', path, isValid);
        };

        HelpRegistry.prototype.addSubTopic = function (topic, subtopic, path, isValid) {
            if (typeof isValid === "undefined") { isValid = null; }
            this.getOrCreateTopic(topic, isValid)[subtopic] = path;
        };

        HelpRegistry.prototype.getOrCreateTopic = function (topic, isValid) {
            if (typeof isValid === "undefined") { isValid = null; }
            if (!angular.isDefined(this.topics[topic])) {
                if (isValid === null) {
                    isValid = function () {
                        return true;
                    };
                }

                this.topics[topic] = {
                    isValid: isValid
                };
                this.$rootScope.$broadcast('hawtioNewHelpTopic');
            }
            return this.topics[topic];
        };

        HelpRegistry.prototype.mapTopicName = function (name) {
            if (angular.isDefined(this.topicNameMappings[name])) {
                return this.topicNameMappings[name];
            }
            return name.capitalize();
        };

        HelpRegistry.prototype.mapSubTopicName = function (name) {
            if (angular.isDefined(this.subTopicNameMappings[name])) {
                return this.subTopicNameMappings[name];
            }
            return name.capitalize();
        };

        HelpRegistry.prototype.getTopics = function () {
            var answer = {};

            angular.forEach(this.topics, function (value, key) {
                if (value.isValid()) {
                    Core.log.debug(key, " is available");

                    // strip out any functions...
                    answer[key] = angular.fromJson(angular.toJson(value));
                } else {
                    Core.log.debug(key, " is not available");
                }
            });

            return answer;
        };

        HelpRegistry.prototype.disableAutodiscover = function (name) {
            this.ignoredPlugins.push(name);
        };

        HelpRegistry.prototype.discoverHelpFiles = function (plugins) {
            var self = this;

            console.log("Ignored plugins: ", self.ignoredPlugins);

            plugins.forEach(function (plugin) {
                var pluginName = self.pluginNameMappings[plugin];
                if (!angular.isDefined(pluginName)) {
                    pluginName = plugin;
                }

                if (!self.ignoredPlugins.any(function (p) {
                    return p === pluginName;
                })) {
                    angular.forEach(self.discoverableDocTypes, function (value, key) {
                        // avoid trying to discover these if plugins register them
                        if (!angular.isDefined(self[pluginName]) || !angular.isDefined(self[pluginName][key])) {
                            var target = 'app/' + pluginName + '/doc/' + value;
                            console.log("checking: ", target);

                            $.ajax(target, {
                                type: 'HEAD',
                                statusCode: {
                                    200: function () {
                                        self.getOrCreateTopic(plugin)[key] = target;
                                    }
                                }
                            });
                        }
                    });
                }
            });
        };
        return HelpRegistry;
    })();
    Core.HelpRegistry = HelpRegistry;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function PreferencesController($scope, $location, workspace, preferencesRegistry, $element) {
        Core.bindModelToSearchParam($scope, $location, "pref", "pref", "Core");
        $scope.panels = {};

        $scope.$watch(function () {
            return $element.is(':visible');
        }, function (newValue, oldValue) {
            if (newValue) {
                setTimeout(function () {
                    $scope.panels = preferencesRegistry.getTabs();
                    Core.log.debug("Panels: ", $scope.panels);
                    Core.$apply($scope);
                }, 50);
            }
        });
    }
    Core.PreferencesController = PreferencesController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function CorePreferences($scope, localStorage) {
        Core.initPreferenceScope($scope, localStorage, {
            'updateRate': {
                'value': 5000,
                'post': function (newValue) {
                    $scope.$emit('UpdateRate', newValue);
                }
            },
            'showWelcomePage': {
                'value': true,
                'converter': Core.parseBooleanValue
            },
            'regexs': {
                'value': "",
                'converter': function (value) {
                    if (angular.isArray(value)) {
                        return value;
                    } else if (Core.isBlank(value)) {
                        return [];
                    }
                    return angular.fromJson(value);
                },
                'formatter': function (value) {
                    return angular.toJson(value);
                },
                'compareAsObject': true
            }
        });

        $scope.newHost = {};
        $scope.forms = {};

        $scope.addRegexDialog = new UI.Dialog();

        $scope.onOk = function (json, form) {
            $scope.addRegexDialog.close();
            json['color'] = UI.colors.sample();
            if (!angular.isArray($scope.regexs)) {
                $scope.regexs = [json];
            } else {
                $scope.regexs.push(json);
            }
            $scope.newHost = {};
            Core.$apply($scope);
        };

        // used by add dialog in preferences.html
        $scope.hostSchema = {
            properties: {
                'name': {
                    description: 'Indicator name',
                    type: 'string',
                    required: true
                },
                'regex': {
                    description: 'Indicator regex',
                    type: 'string',
                    required: true
                }
            }
        };

        $scope.delete = function (index) {
            $scope.regexs.removeAt(index);
        };

        $scope.moveUp = function (index) {
            var tmp = $scope.hosts[index];
            $scope.regexs[index] = $scope.regexs[index - 1];
            $scope.regexs[index - 1] = tmp;
        };

        $scope.moveDown = function (index) {
            var tmp = $scope.regexs[index];
            $scope.regexs[index] = $scope.regexs[index + 1];
            $scope.regexs[index + 1] = tmp;
        };
    }
    Core.CorePreferences = CorePreferences;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    

    /**
    * @class Workspace
    */
    var Workspace = (function () {
        function Workspace(jolokia, jolokiaStatus, jmxTreeLazyLoadRegistry, $location, $compile, $templateCache, localStorage, $rootScope, userDetails) {
            this.jolokia = jolokia;
            this.jolokiaStatus = jolokiaStatus;
            this.jmxTreeLazyLoadRegistry = jmxTreeLazyLoadRegistry;
            this.$location = $location;
            this.$compile = $compile;
            this.$templateCache = $templateCache;
            this.localStorage = localStorage;
            this.$rootScope = $rootScope;
            this.userDetails = userDetails;
            this.operationCounter = 0;
            this.tree = new Core.Folder('MBeans');
            this.treeResponse = {};
            this.mbeanTypesToDomain = {};
            this.mbeanServicesToDomain = {};
            this.attributeColumnDefs = {};
            this.treePostProcessors = [];
            this.topLevelTabs = [];
            this.subLevelTabs = [];
            this.keyToNodeMap = {};
            this.pluginRegisterHandle = null;
            this.pluginUpdateCounter = null;
            this.treeWatchRegisterHandle = null;
            this.treeWatcherCounter = null;
            this.treeElement = null;
            // mapData allows to store arbitrary data on the workspace
            this.mapData = {};
            // set defaults
            if (!('autoRefresh' in localStorage)) {
                localStorage['autoRefresh'] = true;
            }
            if (!('updateRate' in localStorage)) {
                localStorage['updateRate'] = 5000;
            }
        }
        /**
        * Creates a shallow copy child workspace with its own selection and location
        * @method createChildWorkspace
        * @param {ng.ILocationService} location
        * @return {Workspace}
        */
        Workspace.prototype.createChildWorkspace = function (location) {
            var child = new Workspace(this.jolokia, this.jolokiaStatus, this.jmxTreeLazyLoadRegistry, this.$location, this.$compile, this.$templateCache, this.localStorage, this.$rootScope, this.userDetails);

            // lets copy across all the properties just in case
            angular.forEach(this, function (value, key) {
                return child[key] = value;
            });
            child.$location = location;
            return child;
        };

        Workspace.prototype.getLocalStorage = function (key) {
            return this.localStorage[key];
        };

        Workspace.prototype.setLocalStorage = function (key, value) {
            this.localStorage[key] = value;
        };

        Workspace.prototype.loadTree = function () {
            // Make an initial blocking call to ensure the JMX tree is populated while the
            // app is initializing...
            //var flags = {error: initialLoadError, ajaxError: initialLoadError, maxDepth: 2};
            var flags = { ignoreErrors: true, maxDepth: 2 };
            var data = this.jolokia.list(null, onSuccess(null, flags));

            if (data) {
                this.jolokiaStatus.xhr = null;
            }
            this.populateTree({
                value: data
            });
            // we now only reload the tree if the TreeWatcher mbean is present...
            // Core.register(this.jolokia, this, {type: 'list', maxDepth: 2}, onSuccess(angular.bind(this, this.populateTree), {maxDepth: 2}));
        };

        /**
        * Adds a post processor of the tree to swizzle the tree metadata after loading
        * such as correcting any typeName values or CSS styles by hand
        * @method addTreePostProcessor
        * @param {Function} processor
        */
        Workspace.prototype.addTreePostProcessor = function (processor) {
            this.treePostProcessors.push(processor);

            var tree = this.tree;
            if (tree) {
                // the tree is loaded already so lets process it now :)
                processor(tree);
            }
        };

        Workspace.prototype.maybeMonitorPlugins = function () {
            if (this.treeContainsDomainAndProperties("hawtio", { type: "Registry" })) {
                if (this.pluginRegisterHandle === null) {
                    this.pluginRegisterHandle = this.jolokia.register(angular.bind(this, this.maybeUpdatePlugins), {
                        type: "read",
                        mbean: "hawtio:type=Registry",
                        attribute: "UpdateCounter"
                    });
                }
            } else {
                if (this.pluginRegisterHandle !== null) {
                    this.jolokia.unregister(this.pluginRegisterHandle);
                    this.pluginRegisterHandle = null;
                    this.pluginUpdateCounter = null;
                }
            }

            // lets also listen to see if we have a JMX tree watcher
            if (this.treeContainsDomainAndProperties("hawtio", { type: "TreeWatcher" })) {
                if (this.treeWatchRegisterHandle === null) {
                    this.treeWatchRegisterHandle = this.jolokia.register(angular.bind(this, this.maybeReloadTree), {
                        type: "read",
                        mbean: "hawtio:type=TreeWatcher",
                        attribute: "Counter"
                    });
                }
            }
        };

        Workspace.prototype.maybeUpdatePlugins = function (response) {
            if (this.pluginUpdateCounter === null) {
                this.pluginUpdateCounter = response.value;
                return;
            }
            if (this.pluginUpdateCounter !== response.value) {
                if (Core.parseBooleanValue(localStorage['autoRefresh'])) {
                    window.location.reload();
                }
            }
        };

        Workspace.prototype.maybeReloadTree = function (response) {
            var counter = response.value;
            if (this.treeWatcherCounter === null) {
                this.treeWatcherCounter = counter;
                return;
            }
            if (this.treeWatcherCounter !== counter) {
                this.treeWatcherCounter = counter;
                var workspace = this;
                function wrapInValue(response) {
                    var wrapper = {
                        value: response
                    };
                    workspace.populateTree(wrapper);
                }
                this.jolokia.list(null, onSuccess(wrapInValue, { ignoreErrors: true, maxDepth: 2 }));
            }
        };

        Workspace.prototype.folderGetOrElse = function (folder, value) {
            if (folder) {
                try  {
                    return folder.getOrElse(value);
                } catch (e) {
                    Core.log.warn("Failed to find value " + value + " on folder " + folder);
                }
            }
            return null;
        };

        Workspace.prototype.populateTree = function (response) {
            if (!Object.equal(this.treeResponse, response.value)) {
                this.treeResponse = response.value;
                Core.log.debug("JMX tree has been loaded!");

                var rootId = 'root';
                var separator = '-';
                this.mbeanTypesToDomain = {};
                this.mbeanServicesToDomain = {};
                this.keyToNodeMap = {};
                var tree = new Core.Folder('MBeans');
                tree.key = rootId;
                var domains = response.value;
                for (var domain in domains) {
                    var domainClass = escapeDots(domain);
                    var mbeans = domains[domain];
                    for (var path in mbeans) {
                        var entries = {};
                        var folder = this.folderGetOrElse(tree, domain);

                        //if (!folder) continue;
                        folder.domain = domain;
                        if (!folder.key) {
                            folder.key = rootId + separator + domain;
                        }
                        var folderNames = [domain];
                        folder.folderNames = folderNames;
                        folderNames = folderNames.clone();
                        var items = path.split(',');
                        var paths = [];
                        var typeName = null;
                        var serviceName = null;
                        items.forEach(function (item) {
                            var kv = item.split('=');
                            var key = kv[0];
                            var value = kv[1] || key;
                            entries[key] = value;
                            var moveToFront = false;
                            var lowerKey = key.toLowerCase();
                            if (lowerKey === "type") {
                                typeName = value;

                                // if the type name value already exists in the root node
                                // of the domain then lets move this property around too
                                if (folder.map[value]) {
                                    moveToFront = true;
                                }
                            }
                            if (lowerKey === "service") {
                                serviceName = value;
                            }
                            if (moveToFront) {
                                paths.splice(0, 0, value);
                            } else {
                                paths.push(value);
                            }
                        });

                        var configureFolder = function (folder, name) {
                            folder.domain = domain;
                            if (!folder.key) {
                                folder.key = rootId + separator + folderNames.join(separator);
                            }
                            this.keyToNodeMap[folder.key] = folder;
                            folder.folderNames = folderNames.clone();

                            //var classes = escapeDots(folder.key);
                            var classes = "";
                            var entries = folder.entries;
                            var entryKeys = Object.keys(entries).filter(function (n) {
                                return n.toLowerCase().indexOf("type") >= 0;
                            });
                            if (entryKeys.length) {
                                angular.forEach(entryKeys, function (entryKey) {
                                    var entryValue = entries[entryKey];
                                    if (!folder.ancestorHasEntry(entryKey, entryValue)) {
                                        classes += " " + domainClass + separator + entryValue;
                                    }
                                });
                            } else {
                                var kindName = folderNames.last();

                                /*if (folder.parent && folder.parent.title === typeName) {
                                kindName = typeName;
                                } else */
                                if (kindName === name) {
                                    kindName += "-folder";
                                }
                                if (kindName) {
                                    classes += " " + domainClass + separator + kindName;
                                }
                            }
                            folder.addClass = escapeTreeCssStyles(classes);
                            return folder;
                        };

                        var lastPath = paths.pop();
                        var ws = this;
                        paths.forEach(function (value) {
                            folder = ws.folderGetOrElse(folder, value);
                            if (folder) {
                                folderNames.push(value);
                                angular.bind(ws, configureFolder, folder, value)();
                            }
                        });
                        var key = rootId + separator + folderNames.join(separator) + separator + lastPath;
                        var objectName = domain + ":" + path;

                        if (folder) {
                            folder = this.folderGetOrElse(folder, lastPath);
                            if (folder) {
                                // lets add the various data into the folder
                                folder.entries = entries;
                                folder.key = key;
                                angular.bind(this, configureFolder, folder, lastPath)();
                                folder.title = Core.trimQuotes(lastPath);
                                folder.objectName = objectName;
                                folder.typeName = typeName;

                                var addFolderByDomain = function (owner, typeName) {
                                    var map = owner[typeName];
                                    if (!map) {
                                        map = {};
                                        owner[typeName] = map;
                                    }
                                    var value = map[domain];
                                    if (!value) {
                                        map[domain] = folder;
                                    } else {
                                        var array = null;
                                        if (angular.isArray(value)) {
                                            array = value;
                                        } else {
                                            array = [value];
                                            map[domain] = array;
                                        }
                                        array.push(folder);
                                    }
                                };

                                if (serviceName) {
                                    angular.bind(this, addFolderByDomain, this.mbeanServicesToDomain, serviceName)();
                                }
                                if (typeName) {
                                    angular.bind(this, addFolderByDomain, this.mbeanTypesToDomain, typeName)();
                                }
                            }
                        } else {
                            Core.log.info("No folder found for lastPath: " + lastPath);
                        }
                    }
                }

                tree.sortChildren(true);

                // now lets mark the nodes with no children as lazy loading...
                this.enableLazyLoading(tree);
                this.tree = tree;

                var processors = this.treePostProcessors;
                angular.forEach(processors, function (processor) {
                    return processor(tree);
                });

                this.maybeMonitorPlugins();

                var rootScope = this.$rootScope;
                if (rootScope) {
                    rootScope.$broadcast('jmxTreeUpdated');
                }
            }
        };

        Workspace.prototype.enableLazyLoading = function (folder) {
            var _this = this;
            var children = folder.children;
            if (children && children.length) {
                angular.forEach(children, function (child) {
                    _this.enableLazyLoading(child);
                });
            } else {
                // we have no children so enable lazy loading if we have a custom loader registered
                var lazyFunction = Jmx.findLazyLoadingFunction(this, folder);
                if (lazyFunction) {
                    folder.isLazy = true;
                }
            }
        };

        /**
        * Returns the hash query argument to append to URL links
        * @method hash
        * @return {String}
        */
        Workspace.prototype.hash = function () {
            var hash = this.$location.search();
            var params = Core.hashToString(hash);
            if (params) {
                return "?" + params;
            }
            return "";
        };

        /**
        * Returns the currently active tab
        * @method getActiveTab
        * @return {Boolean}
        */
        Workspace.prototype.getActiveTab = function () {
            var workspace = this;
            return this.topLevelTabs.find(function (tab) {
                if (!angular.isDefined(tab.isActive)) {
                    return workspace.isLinkActive(tab.href());
                } else {
                    return tab.isActive(workspace);
                }
            });
        };

        Workspace.prototype.getStrippedPathName = function () {
            var pathName = Core.trimLeading((this.$location.path() || '/'), "#");
            pathName = Core.trimLeading(pathName, "/");
            return pathName;
        };

        Workspace.prototype.linkContains = function () {
            var words = [];
            for (var _i = 0; _i < (arguments.length - 0); _i++) {
                words[_i] = arguments[_i + 0];
            }
            var pathName = this.getStrippedPathName();
            return words.all(function (word) {
                return pathName.has(word);
            });
        };

        /**
        * Returns true if the given link is active. The link can omit the leading # or / if necessary.
        * The query parameters of the URL are ignored in the comparison.
        * @method isLinkActive
        * @param {String} href
        * @return {Boolean} true if the given link is active
        */
        Workspace.prototype.isLinkActive = function (href) {
            // lets trim the leading slash
            var pathName = this.getStrippedPathName();

            var link = Core.trimLeading(href, "#");
            link = Core.trimLeading(link, "/");

            // strip any query arguments
            var idx = link.indexOf('?');
            if (idx >= 0) {
                link = link.substring(0, idx);
            }
            if (!pathName.length) {
                return link === pathName;
            } else {
                return pathName.startsWith(link);
            }
        };

        /**
        * Returns true if the given link is active. The link can omit the leading # or / if necessary.
        * The query parameters of the URL are ignored in the comparison.
        * @method isLinkActive
        * @param {String} href
        * @return {Boolean} true if the given link is active
        */
        Workspace.prototype.isLinkPrefixActive = function (href) {
            // lets trim the leading slash
            var pathName = this.getStrippedPathName();

            var link = Core.trimLeading(href, "#");
            link = Core.trimLeading(link, "/");

            // strip any query arguments
            var idx = link.indexOf('?');
            if (idx >= 0) {
                link = link.substring(0, idx);
            }
            return pathName.startsWith(link);
        };

        /**
        * Returns true if the tab query parameter is active or the URL starts with the given path
        * @method isTopTabActive
        * @param {String} path
        * @return {Boolean}
        */
        Workspace.prototype.isTopTabActive = function (path) {
            var tab = this.$location.search()['tab'];
            if (angular.isString(tab)) {
                return tab.startsWith(path);
            }
            return this.isLinkActive(path);
        };

        /**
        * Returns the selected mbean name if there is one
        * @method getSelectedMBeanName
        * @return {String}
        */
        Workspace.prototype.getSelectedMBeanName = function () {
            var selection = this.selection;
            if (selection) {
                return selection.objectName;
            }
            return null;
        };

        /**
        * Returns true if the path is valid for the current selection
        * @method validSelection
        * @param {String} uri
        * @return {Boolean}
        */
        Workspace.prototype.validSelection = function (uri) {
            var workspace = this;
            var filter = function (t) {
                var fn = t.href;
                if (fn) {
                    var href = fn();
                    if (href) {
                        if (href.startsWith("#/")) {
                            href = href.substring(2);
                        }
                        return href === uri;
                    }
                }
                return false;
            };
            var tab = this.subLevelTabs.find(filter);
            if (!tab) {
                tab = this.topLevelTabs.find(filter);
            }
            if (tab) {
                //console.log("Found tab " + JSON.stringify(tab));
                var validFn = tab['isValid'];
                return !angular.isDefined(validFn) || validFn(workspace);
            } else {
                Core.log.info("Could not find tab for " + uri);
                return false;
            }
            /*
            var value = this.uriValidations[uri];
            if (value) {
            if (angular.isFunction(value)) {
            return value();
            }
            }
            return true;
            */
        };

        /**
        * In cases where we have just deleted something we typically want to change
        * the selection to the parent node
        * @method removeAndSelectParentNode
        */
        Workspace.prototype.removeAndSelectParentNode = function () {
            var selection = this.selection;
            if (selection) {
                var parent = selection.parent;
                if (parent) {
                    // lets remove the selection from the parent so we don't do any more JMX attribute queries on the children
                    // or include it in table views etc
                    // would be nice to eagerly remove the tree node too?
                    var idx = parent.children.indexOf(selection);
                    if (idx < 0) {
                        idx = parent.children.findIndex(function (n) {
                            return n.key === selection.key;
                        });
                    }
                    if (idx >= 0) {
                        parent.children.splice(idx, 1);
                    }
                    this.updateSelectionNode(parent);
                }
            }
        };

        Workspace.prototype.selectParentNode = function () {
            var selection = this.selection;
            if (selection) {
                var parent = selection.parent;
                if (parent) {
                    this.updateSelectionNode(parent);
                }
            }
        };

        /**
        * Returns the view configuration key for the kind of selection
        * for example based on the domain and the node type
        * @method selectionViewConfigKey
        * @return {String}
        */
        Workspace.prototype.selectionViewConfigKey = function () {
            return this.selectionConfigKey("view/");
        };

        /**
        * Returns a configuration key for a node which is usually of the form
        * domain/typeName or for folders with no type, domain/name/folder
        * @method selectionConfigKey
        * @param {String} prefix
        * @return {String}
        */
        Workspace.prototype.selectionConfigKey = function (prefix) {
            if (typeof prefix === "undefined") { prefix = ""; }
            var key = null;
            var selection = this.selection;
            if (selection) {
                // lets make a unique string for the kind of select
                key = prefix + selection.domain;
                var typeName = selection.typeName;
                if (!typeName) {
                    typeName = selection.title;
                }
                key += "/" + typeName;
                if (selection.isFolder()) {
                    key += "/folder";
                }
            }
            return key;
        };

        Workspace.prototype.moveIfViewInvalid = function () {
            var workspace = this;
            var uri = Core.trimLeading(this.$location.path(), "/");
            if (this.selection) {
                var key = this.selectionViewConfigKey();
                if (this.validSelection(uri)) {
                    // lets remember the previous selection
                    this.setLocalStorage(key, uri);
                    return false;
                } else {
                    Core.log.info("the uri '" + uri + "' is not valid for this selection");

                    // lets look up the previous preferred value for this type
                    var defaultPath = this.getLocalStorage(key);
                    if (!defaultPath || !this.validSelection(defaultPath)) {
                        // lets find the first path we can find which is valid
                        defaultPath = null;
                        angular.forEach(this.subLevelTabs, function (tab) {
                            var fn = tab.isValid;
                            if (!defaultPath && tab.href && angular.isDefined(fn) && fn(workspace)) {
                                defaultPath = tab.href();
                            }
                        });
                    }
                    if (!defaultPath) {
                        defaultPath = "#/jmx/help";
                    }
                    Core.log.info("moving the URL to be " + defaultPath);
                    if (defaultPath.startsWith("#")) {
                        defaultPath = defaultPath.substring(1);
                    }
                    this.$location.path(defaultPath);
                    return true;
                }
            } else {
                return false;
            }
        };

        Workspace.prototype.updateSelectionNode = function (node) {
            var originalSelection = this.selection;
            this.selection = node;
            var key = null;
            if (node) {
                key = node['key'];
            }
            var $location = this.$location;
            var q = $location.search();
            if (key) {
                q['nid'] = key;
            }
            $location.search(q);

            // if we have updated the selection (rather than just loaded a page)
            // lets use the previous preferred view - otherwise we may be loading
            // a page from a bookmark so lets not change the view :)
            if (originalSelection) {
                key = this.selectionViewConfigKey();
                if (key) {
                    var defaultPath = this.getLocalStorage(key);
                    if (defaultPath) {
                        this.$location.path(defaultPath);
                    }
                }
            }
        };

        /**
        * Redraws the tree widget
        * @method redrawTree
        */
        Workspace.prototype.redrawTree = function () {
            var treeElement = this.treeElement;
            if (treeElement) {
                treeElement.dynatree("getTree").reload();
            }
        };

        /**
        * Expand / collapse the current active node
        * @method expandSelection
        * @param {Boolean} flag
        */
        Workspace.prototype.expandSelection = function (flag) {
            var treeElement = this.treeElement;
            if (treeElement) {
                var node = treeElement.dynatree("getActiveNode");
                if (node) {
                    node.expand(flag);
                }
            }
        };

        Workspace.prototype.matchesProperties = function (entries, properties) {
            if (!entries)
                return false;
            for (var key in properties) {
                var value = properties[key];
                if (!value || entries[key] !== value) {
                    return false;
                }
            }
            return true;
        };

        Workspace.prototype.treeContainsDomainAndProperties = function (domainName, properties) {
            var _this = this;
            if (typeof properties === "undefined") { properties = null; }
            var workspace = this;
            var tree = workspace.tree;
            if (tree) {
                var folder = tree.get(domainName);
                if (folder) {
                    if (properties) {
                        var children = folder.children || [];
                        var checkProperties = function (node) {
                            if (!_this.matchesProperties(node.entries, properties)) {
                                if (node.domain === domainName && node.children && node.children.length > 0) {
                                    return node.children.some(checkProperties);
                                } else {
                                    return false;
                                }
                            } else {
                                return true;
                            }
                        };
                        return children.some(checkProperties);
                    }
                    return true;
                } else {
                    // console.log("no hasMBean for " + objectName + " in tree " + tree);
                }
            } else {
                // console.log("workspace has no tree! returning false for hasMBean " + objectName);
            }
            return false;
        };

        Workspace.prototype.matches = function (folder, properties, propertiesCount) {
            if (folder) {
                var entries = folder.entries;
                if (properties) {
                    if (!entries)
                        return false;
                    for (var key in properties) {
                        var value = properties[key];
                        if (!value || entries[key] !== value) {
                            return false;
                        }
                    }
                }
                if (propertiesCount) {
                    return entries && Object.keys(entries).length === propertiesCount;
                }
                return true;
            }
            return false;
        };

        // only display stuff if we have an mbean with the given properties
        Workspace.prototype.hasDomainAndProperties = function (domainName, properties, propertiesCount) {
            if (typeof properties === "undefined") { properties = null; }
            if (typeof propertiesCount === "undefined") { propertiesCount = null; }
            var node = this.selection;
            if (node) {
                return this.matches(node, properties, propertiesCount) && node.domain === domainName;
            }
            return false;
        };

        // only display stuff if we have an mbean with the given properties
        Workspace.prototype.findMBeanWithProperties = function (domainName, properties, propertiesCount) {
            if (typeof properties === "undefined") { properties = null; }
            if (typeof propertiesCount === "undefined") { propertiesCount = null; }
            var tree = this.tree;
            if (tree) {
                return this.findChildMBeanWithProperties(tree.get(domainName), properties, propertiesCount);
            }
            return null;
        };

        Workspace.prototype.findChildMBeanWithProperties = function (folder, properties, propertiesCount) {
            var _this = this;
            if (typeof properties === "undefined") { properties = null; }
            if (typeof propertiesCount === "undefined") { propertiesCount = null; }
            var workspace = this;
            if (folder) {
                var children = folder.children;
                if (children) {
                    var answer = children.find(function (node) {
                        return _this.matches(node, properties, propertiesCount);
                    });
                    if (answer) {
                        return answer;
                    }
                    return children.map(function (node) {
                        return workspace.findChildMBeanWithProperties(node, properties, propertiesCount);
                    }).find(function (node) {
                        return node;
                    });
                }
            }
            return null;
        };

        Workspace.prototype.selectionHasDomainAndLastFolderName = function (objectName, lastName) {
            var lastNameLower = (lastName || "").toLowerCase();
            function isName(name) {
                return (name || "").toLowerCase() === lastNameLower;
            }
            var node = this.selection;
            if (node) {
                if (objectName === node.domain) {
                    var folders = node.folderNames;
                    if (folders) {
                        var last = folders.last();
                        return (isName(last) || isName(node.title)) && node.isFolder() && !node.objectName;
                    }
                }
            }
            return false;
        };

        Workspace.prototype.selectionHasDomain = function (domainName) {
            var node = this.selection;
            if (node) {
                return domainName === node.domain;
            }
            return false;
        };

        Workspace.prototype.selectionHasDomainAndType = function (objectName, typeName) {
            var node = this.selection;
            if (node) {
                return objectName === node.domain && typeName === node.typeName;
            }
            return false;
        };

        /**
        * Returns true if this workspace has any mbeans at all
        */
        Workspace.prototype.hasMBeans = function () {
            var answer = false;
            var tree = this.tree;
            if (tree) {
                var children = tree.children;
                if (angular.isArray(children) && children.length > 0) {
                    answer = true;
                }
            }
            return answer;
        };
        Workspace.prototype.hasFabricMBean = function () {
            return this.hasDomainAndProperties('io.fabric8', { type: 'Fabric' });
        };
        Workspace.prototype.isFabricFolder = function () {
            return this.hasDomainAndProperties('io.fabric8');
        };

        Workspace.prototype.isCamelContext = function () {
            return this.hasDomainAndProperties('org.apache.camel', { type: 'context' });
        };
        Workspace.prototype.isCamelFolder = function () {
            return this.hasDomainAndProperties('org.apache.camel');
        };
        Workspace.prototype.isEndpointsFolder = function () {
            return this.selectionHasDomainAndLastFolderName('org.apache.camel', 'endpoints');
        };
        Workspace.prototype.isEndpoint = function () {
            return this.hasDomainAndProperties('org.apache.camel', { type: 'endpoints' });
        };
        Workspace.prototype.isRoutesFolder = function () {
            return this.selectionHasDomainAndLastFolderName('org.apache.camel', 'routes');
        };
        Workspace.prototype.isRoute = function () {
            return this.hasDomainAndProperties('org.apache.camel', { type: 'routes' });
        };

        Workspace.prototype.isOsgiFolder = function () {
            return this.hasDomainAndProperties('osgi.core');
        };
        Workspace.prototype.isKarafFolder = function () {
            return this.hasDomainAndProperties('org.apache.karaf');
        };
        Workspace.prototype.isOsgiCompendiumFolder = function () {
            return this.hasDomainAndProperties('osgi.compendium');
        };
        return Workspace;
    })();
    Core.Workspace = Workspace;
})(Core || (Core = {}));

// TODO refactor other code to use Core.Workspace
var Workspace = (function (_super) {
    __extends(Workspace, _super);
    function Workspace() {
        _super.apply(this, arguments);
    }
    return Workspace;
})(Core.Workspace);
;
;
/**
* @module Core
*/
var Core;
(function (Core) {
    Core.fileUploadMBean = "hawtio:type=UploadManager";

    var FileUpload = (function () {
        function FileUpload() {
            this.restrict = 'A';
            this.replace = true;
            this.templateUrl = Core.templatePath + "fileUpload.html";
            this.scope = {
                files: '=hawtioFileUpload',
                target: '@',
                showFiles: '@'
            };
            this.controller = function ($scope, $element, $attrs, jolokia) {
                $scope.target = '';
                $scope.response = '';
                $scope.percentComplete = 0;

                UI.observe($scope, $attrs, 'target', '');
                UI.observe($scope, $attrs, 'showFiles', true);

                $scope.update = function (response) {
                    var responseJson = angular.toJson(response.value);
                    if ($scope.responseJson !== responseJson) {
                        $scope.responseJson = responseJson;
                        $scope.files = response.value;
                        Core.$applyNowOrLater($scope);
                    }
                };

                $scope.delete = function (fileName) {
                    //notification('info', 'Deleting ' + fileName);
                    jolokia.request({
                        type: 'exec', mbean: Core.fileUploadMBean,
                        operation: 'delete(java.lang.String, java.lang.String)',
                        arguments: [$scope.target, fileName] }, {
                        success: function () {
                            //notification('success', 'Deleted ' + fileName);
                            Core.$apply($scope);
                        },
                        error: function (response) {
                            Core.notification('error', "Failed to delete " + fileName + " due to: " + response.error);
                            Core.$apply($scope);
                        }
                    });
                };

                $scope.$watch('target', function (newValue, oldValue) {
                    if (oldValue !== newValue) {
                        Core.unregister(jolokia, $scope);
                    }
                    Core.register(jolokia, $scope, {
                        type: 'exec', mbean: Core.fileUploadMBean,
                        operation: 'list(java.lang.String)',
                        arguments: [$scope.target]
                    }, onSuccess($scope.update));
                });
            };
            this.link = function ($scope, $element, $attrs) {
                var fileInput = $element.find('input[type=file]');
                var form = $element.find('form[name=file-upload]');
                var button = $element.find('input[type=button]');

                var onFileChange = function () {
                    button.prop('disabled', true);

                    var files = fileInput.get(0).files;

                    var fileName = files.length + " files";
                    if (files.length === 1) {
                        fileName = files[0].name;
                    }

                    form.ajaxSubmit({
                        beforeSubmit: function (arr, $form, options) {
                            Core.notification('info', "Uploading " + fileName);
                            $scope.percentComplete = 0;
                            Core.$apply($scope);
                        },
                        success: function (response, statusText, xhr, $form) {
                            Core.notification('success', "Uploaded " + fileName);
                            setTimeout(function () {
                                button.prop('disabled', false);
                                $scope.percentComplete = 0;
                                Core.$apply($scope);
                            }, 1000);
                            Core.$apply($scope);
                        },
                        error: function (response, statusText, xhr, $form) {
                            Core.notification('error', "Failed to upload " + fileName + " due to " + statusText);
                            setTimeout(function () {
                                button.prop('disabled', false);
                                $scope.percentComplete = 0;
                                Core.$apply($scope);
                            }, 1000);
                            Core.$apply($scope);
                        },
                        uploadProgress: function (event, position, total, percentComplete) {
                            $scope.percentComplete = percentComplete;
                            Core.$apply($scope);
                        }
                    });
                    return false;
                };

                button.click(function () {
                    if (!button.prop('disabled')) {
                        fileInput.click();
                    }
                    return false;
                });

                form.submit(function () {
                    return false;
                });

                if ($.browser.msie) {
                    fileInput.click(function (event) {
                        setTimeout(function () {
                            if (fileInput.val().length > 0) {
                                onFileChange();
                            }
                        }, 0);
                    });
                } else {
                    fileInput.change(onFileChange);
                }
            };
        }
        return FileUpload;
    })();
    Core.FileUpload = FileUpload;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    /**
    * Directive that's used to ensure an ng-grid expands it's height to fit the viewport height
    * @class GridStyle
    */
    var GridStyle = (function () {
        function GridStyle($window) {
            var _this = this;
            this.$window = $window;
            this.restrict = 'C';
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        GridStyle.prototype.doLink = function (scope, element, attrs) {
            var lastHeight = 0;

            var resizeFunc = angular.bind(this, function (scope) {
                var top = element.position().top;
                var windowHeight = $(this.$window).height();
                var height = windowHeight - top - 15;
                var heightStr = height + 'px';

                element.css({
                    'min-height': heightStr,
                    'height': heightStr
                });

                if (lastHeight !== height) {
                    lastHeight = height;
                    element.trigger('resize');
                }
            });

            resizeFunc();

            scope.$watch(resizeFunc);

            $(this.$window).resize(function () {
                resizeFunc();
                Core.$apply(scope);
                return false;
            });
        };
        return GridStyle;
    })();
    Core.GridStyle = GridStyle;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function HelpController($scope, $routeParams, marked, helpRegistry, branding) {
        $scope.branding = branding;
        $scope.topics = helpRegistry.getTopics();

        if ('topic' in $routeParams) {
            $scope.topic = $routeParams['topic'];
        } else {
            $scope.topic = 'index';
        }

        if ('subtopic' in $routeParams) {
            $scope.subTopic = $routeParams['subtopic'];
        } else {
            $scope.subTopic = Object.extended($scope.topics[$scope.topic]).keys().first();
        }

        Core.log.debug("topic: ", $scope.topic, " subtopic: ", $scope.subTopic);

        // when on the index pages, filter the user subTopic unless on the dev page
        var isIndex = $scope.topic === "index";
        var filterSubTopic = $scope.subTopic;
        if (isIndex && filterSubTopic !== "developer") {
            filterSubTopic = "user";
        }

        $scope.breadcrumbs = [
            {
                topic: "index",
                subTopic: "user",
                label: "User Guide"
            },
            {
                topic: "index",
                subTopic: "faq",
                label: "FAQ"
            },
            {
                topic: "index",
                subTopic: "changes",
                label: "Changes"
            },
            {
                topic: "index",
                subTopic: "developer",
                label: "Developers"
            }
        ];

        $scope.sectionLink = function (section) {
            var topic = section.topic || "";
            var subTopic = section.subTopic || "";
            var link = Core.pathGet(helpRegistry.topics, [topic, subTopic]);
            if (link && link.indexOf("#") >= 0) {
                return link;
            } else {
                return "#/help/" + topic + "/" + subTopic;
            }
        };

        // lets select the active tab
        var activeBreadcrumb = $scope.breadcrumbs.find(function (b) {
            return b.topic === $scope.topic && b.subTopic === $scope.subTopic;
        });
        if (activeBreadcrumb)
            activeBreadcrumb.active = true;

        $scope.sections = [];
        angular.forEach($scope.topics, function (details, topic) {
            // lets hide any index topics or any topics which don't have a filter sub topic
            if (topic !== "index" && details[filterSubTopic]) {
                $scope.sections.push({
                    topic: topic,
                    subTopic: filterSubTopic,
                    label: helpRegistry.mapTopicName(topic),
                    active: topic === $scope.topic
                });
            }
        });
        $scope.sections = $scope.sections.sortBy("label");

        $scope.$on('hawtioNewHelpTopic', function () {
            $scope.topics = helpRegistry.getTopics();
        });

        $scope.$watch('topics', function (newValue, oldValue) {
            Core.log.debug("Topics: ", $scope.topics);
        });

        if (!angular.isDefined($scope.topics[$scope.topic])) {
            $scope.html = "Unable to download help data for " + $scope.topic;
        } else {
            $.ajax({
                url: $scope.topics[$scope.topic][$scope.subTopic],
                dataType: 'html',
                cache: false,
                success: function (data, textStatus, jqXHR) {
                    $scope.html = "Unable to download help data for " + $scope.topic;
                    if (angular.isDefined(data)) {
                        $scope.html = marked(data);
                    }
                    Core.$apply($scope);
                },
                error: function (jqXHR, textStatus, errorThrown) {
                    $scope.html = "Unable to download help data for " + $scope.topic;
                    Core.$apply($scope);
                }
            });
        }
    }
    Core.HelpController = HelpController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function LoggingPreferences($scope) {
        Core.initPreferenceScope($scope, localStorage, {
            'logBuffer': {
                'value': 100,
                'converter': parseInt,
                'formatter': parseInt,
                'post': function (newValue) {
                    window['LogBuffer'] = newValue;
                }
            },
            'logLevel': {
                'value': '{"value": 2, "name": "INFO"}',
                'post': function (value) {
                    var level = angular.fromJson(value);
                    Logger.setLevel(level);
                }
            }
        });
    }
    Core.LoggingPreferences = LoggingPreferences;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function d3ForceGraph(scope, nodes, links, canvasElement) {
        // lets remove the old graph first
        if (scope.graphForce) {
            scope.graphForce.stop();
        }
        if (!canvasElement) {
            canvasElement = $("#canvas")[0];
        }
        var canvasDiv = $(canvasElement);
        canvasDiv.children("svg").remove();

        if (nodes.length) {
            var width = canvasDiv.parent().width();
            var height = canvasDiv.parent().height();

            if (height < 100) {
                //console.log("browse thinks the height is only " + height + " so calculating offset from doc height");
                var offset = canvasDiv.offset();
                height = $(document).height() - 5;
                if (offset) {
                    height -= offset['top'];
                }
            }

            //console.log("Using width " + width + " and height " + height);
            var svg = d3.select(canvasDiv[0]).append("svg").attr("width", width).attr("height", height);

            var force = d3.layout.force().distance(100).charge(-120 * 10).linkDistance(50).size([width, height]);

            scope.graphForce = force;

            /*
            var force = d3.layout.force()
            .gravity(.05)
            .distance(100)
            .charge(-100)
            .size([width, height]);
            */
            // prepare the arrows
            svg.append("svg:defs").selectAll("marker").data(["from"]).enter().append("svg:marker").attr("id", String).attr("viewBox", "0 -5 10 10").attr("refX", 25).attr("refY", -1.5).attr("markerWidth", 6).attr("markerHeight", 6).attr("orient", "auto").append("svg:path").attr("d", "M0,-5L10,0L0,5");

            force.nodes(nodes).links(links).start();

            var link = svg.selectAll(".link").data(links).enter().append("line").attr("class", "link");

            // draw the arrow
            link.attr("class", "link from");

            // end marker
            link.attr("marker-end", "url(#from)");

            var node = svg.selectAll(".node").data(nodes).enter().append("g").attr("class", "node").call(force.drag);

            node.append("image").attr("xlink:href", function (d) {
                return d.imageUrl;
            }).attr("x", -15).attr("y", -15).attr("width", 30).attr("height", 30);

            node.append("text").attr("dx", 20).attr("dy", ".35em").text(function (d) {
                return d.label;
            });

            force.on("tick", function () {
                link.attr("x1", function (d) {
                    return d.source.x;
                }).attr("y1", function (d) {
                    return d.source.y;
                }).attr("x2", function (d) {
                    return d.target.x;
                }).attr("y2", function (d) {
                    return d.target.y;
                });

                node.attr("transform", function (d) {
                    return "translate(" + d.x + "," + d.y + ")";
                });
            });
        }
    }
    Core.d3ForceGraph = d3ForceGraph;

    function createGraphStates(nodes, links, transitions) {
        var stateKeys = {};
        nodes.forEach(function (node) {
            var idx = node.id;
            if (idx === undefined) {
                console.log("No node found for node " + JSON.stringify(node));
            } else {
                if (node.edges === undefined)
                    node.edges = [];
                if (!node.label)
                    node.label = "node " + idx;
                stateKeys[idx] = node;
            }
        });
        var states = d3.values(stateKeys);
        links.forEach(function (d) {
            var source = stateKeys[d.source];
            var target = stateKeys[d.target];
            if (source === undefined || target === undefined) {
                console.log("Bad link!  " + source + " target " + target + " for " + d);
            } else {
                var edge = { source: source, target: target };
                transitions.push(edge);
                source.edges.push(edge);
                target.edges.push(edge);
                // TODO should we add the edge to the target?
            }
        });
        return states;
    }
    Core.createGraphStates = createGraphStates;

    // TODO Export as a service
    function dagreLayoutGraph(nodes, links, width, height, svgElement) {
        var nodePadding = 10;
        var transitions = [];
        var states = Core.createGraphStates(nodes, links, transitions);
        function spline(e) {
            var points = e.dagre.points.slice(0);
            var source = dagre.util.intersectRect(e.source.dagre, points.length > 0 ? points[0] : e.source.dagre);
            var target = dagre.util.intersectRect(e.target.dagre, points.length > 0 ? points[points.length - 1] : e.source.dagre);
            points.unshift(source);
            points.push(target);
            return d3.svg.line().x(function (d) {
                return d.x;
            }).y(function (d) {
                return d.y;
            }).interpolate("linear")(points);
        }

        // Translates all points in the edge using `dx` and `dy`.
        function translateEdge(e, dx, dy) {
            e.dagre.points.forEach(function (p) {
                p.x = Math.max(0, Math.min(svgBBox.width, p.x + dx));
                p.y = Math.max(0, Math.min(svgBBox.height, p.y + dy));
            });
        }

        // Now start laying things out
        var svg = svgElement ? d3.select(svgElement) : d3.select("svg");

        // lets remove all the old g elements
        if (svgElement) {
            $(svgElement).children("g").remove();
        }
        $(svg).children("g").remove();

        var svgGroup = svg.append("g").attr("transform", "translate(5, 5)");

        // `nodes` is center positioned for easy layout later
        var nodes = svgGroup.selectAll("g .node").data(states).enter().append("g").attr("class", "node").attr("data-cid", function (d) {
            return d.cid;
        }).attr("id", function (d) {
            return "node-" + d.label;
        });

        // lets add a tooltip
        nodes.append("title").text(function (d) {
            return d.tooltip || "";
        });

        var edges = svgGroup.selectAll("path .edge").data(transitions).enter().append("path").attr("class", "edge").attr("marker-end", "url(#arrowhead)");

        // Append rectangles to the nodes. We do this before laying out the text
        // because we want the text above the rectangle.
        var rects = nodes.append("rect").attr("rx", "4").attr("ry", "4").attr("filter", "url(#drop-shadow)").attr("class", function (d) {
            return d.type;
        });

        var images = nodes.append("image").attr("xlink:href", function (d) {
            return d.imageUrl;
        }).attr("x", -12).attr("y", -20).attr("height", 24).attr("width", 24);

        var counters = nodes.append("text").attr("text-anchor", "end").attr("class", "counter").attr("x", 0).attr("dy", 0).text(_counterFunction);

        // Append text
        var labels = nodes.append("text").attr("text-anchor", "middle").attr("x", 0);

        labels.append("tspan").attr("x", 0).attr("dy", 28).text(function (d) {
            return d.label;
        });

        var labelPadding = 12;
        var minLabelwidth = 80;

        labels.each(function (d) {
            var bbox = this.getBBox();
            d.bbox = bbox;
            if (bbox.width < minLabelwidth) {
                bbox.width = minLabelwidth;
            }
            d.width = bbox.width + 2 * nodePadding;
            d.height = bbox.height + 2 * nodePadding + labelPadding;
        });

        rects.attr("x", function (d) {
            return -(d.bbox.width / 2 + nodePadding);
        }).attr("y", function (d) {
            return -(d.bbox.height / 2 + nodePadding + (labelPadding / 2));
        }).attr("width", function (d) {
            return d.width;
        }).attr("height", function (d) {
            return d.height;
        });

        images.attr("x", function (d) {
            return -(d.bbox.width) / 2;
        });

        labels.attr("x", function (d) {
            return -d.bbox.width / 2;
        }).attr("y", function (d) {
            return -d.bbox.height / 2;
        });

        counters.attr("x", function (d) {
            var w = d.bbox.width;
            return w / 2;
        });

        // Create the layout and get the graph
        dagre.layout().nodeSep(50).edgeSep(10).rankSep(50).nodes(states).edges(transitions).debugLevel(1).run();

        nodes.attr("transform", function (d) {
            return 'translate(' + d.dagre.x + ',' + d.dagre.y + ')';
        });

        edges.attr('id', function (e) {
            return e.dagre.id;
        }).attr("d", function (e) {
            return spline(e);
        });

        // Resize the SVG element
        var svgNode = svg.node();
        if (svgNode) {
            var svgBBox = svgNode.getBBox();
            if (svgBBox) {
                svg.attr("width", svgBBox.width + 10);
                svg.attr("height", svgBBox.height + 10);
            }
        }

        // Drag handlers
        var nodeDrag = d3.behavior.drag().origin(function (d) {
            return d.pos ? { x: d.pos.x, y: d.pos.y } : { x: d.dagre.x, y: d.dagre.y };
        }).on('drag', function (d, i) {
            var prevX = d.dagre.x, prevY = d.dagre.y;

            // The node must be inside the SVG area
            d.dagre.x = Math.max(d.width / 2, Math.min(svgBBox.width - d.width / 2, d3.event.x));
            d.dagre.y = Math.max(d.height / 2, Math.min(svgBBox.height - d.height / 2, d3.event.y));
            d3.select(this).attr('transform', 'translate(' + d.dagre.x + ',' + d.dagre.y + ')');

            var dx = d.dagre.x - prevX, dy = d.dagre.y - prevY;

            // Edges position (inside SVG area)
            d.edges.forEach(function (e) {
                translateEdge(e, dx, dy);
                d3.select('#' + e.dagre.id).attr('d', spline(e));
            });
        });

        var edgeDrag = d3.behavior.drag().on('drag', function (d, i) {
            translateEdge(d, d3.event.dx, d3.event.dy);
            d3.select(this).attr('d', spline(d));
        });

        nodes.call(nodeDrag);
        edges.call(edgeDrag);

        return states;
    }
    Core.dagreLayoutGraph = dagreLayoutGraph;

    // TODO Export as a service
    function dagreUpdateGraphData(data) {
        var svg = d3.select("svg");
        svg.selectAll("text.counter").text(_counterFunction);

        // add tooltip
        svg.selectAll("g .node title").text(function (d) {
            return d.tooltip || "";
        });
        /*
        TODO can we reuse twitter bootstrap on an svg title?
        .each(function (d) {
        $(d).tooltip({
        'placement': "bottom"
        });
        });
        
        */
    }
    Core.dagreUpdateGraphData = dagreUpdateGraphData;

    function _counterFunction(d) {
        return d.counter || "";
    }
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    // NOTE - $route is brought in here to ensure the factory for that service
    // has been called, otherwise the ng-include directive doesn't show the partial
    // after a refresh until you click a top-level link.
    function ViewController($scope, $route, $location, layoutTree, layoutFull, viewRegistry) {
        findViewPartial();

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            findViewPartial();
        });

        function searchRegistry(path) {
            var answer = undefined;
            Object.extended(viewRegistry).keys(function (key, value) {
                if (!answer) {
                    if (key.startsWith("/") && key.endsWith("/")) {
                        // assume its a regex
                        var text = key.substring(1, key.length - 1);
                        try  {
                            var reg = new RegExp(text, "");
                            if (reg.exec(path)) {
                                answer = value;
                            }
                        } catch (e) {
                            console.log("Invalid RegExp " + text + " for viewRegistry value: " + value);
                        }
                    } else {
                        if (path.startsWith(key)) {
                            answer = value;
                        }
                    }
                }
            });

            //console.log("Searching for: " + path + " returning: ", answer);
            return answer;
        }

        function findViewPartial() {
            var answer = null;
            var hash = $location.search();
            var tab = hash['tab'];
            if (angular.isString(tab)) {
                answer = searchRegistry(tab);
            }
            if (!answer) {
                var path = $location.path();
                if (path) {
                    if (path.startsWith("/")) {
                        path = path.substring(1);
                    }
                    answer = searchRegistry(path);
                }
            }
            if (!answer) {
                answer = layoutTree;
            }
            $scope.viewPartial = answer;

            console.log("Using view partial: " + answer);
            return answer;
        }
    }
    Core.ViewController = ViewController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function NavBarController($scope, $location, workspace, $route, jolokia, localStorage) {
        $scope.hash = workspace.hash();
        $scope.topLevelTabs = [];
        $scope.subLevelTabs = workspace.subLevelTabs;
        $scope.currentPerspective = null;
        $scope.localStorage = localStorage;
        $scope.recentConnections = [];

        $scope.$watch('localStorage.recentConnections', function (newValue, oldValue) {
            $scope.recentConnections = Core.getRecentConnections(localStorage);
            Core.log.debug("recent containers: ", $scope.recentConnections);
        });

        $scope.openConnection = function (connection) {
            window.open(connection.url);
        };

        $scope.goHome = function () {
            window.open(".");
        };

        $scope.clearConnections = Core.clearConnections;

        $scope.perspectiveDetails = {
            perspective: null
        };

        $scope.topLevelTabs = function () {
            reloadPerspective();

            // TODO transform the top level tabs based on the current perspective
            // TODO watch for changes to workspace.topLevelTabs and for the current perspective
            return workspace.topLevelTabs;
        };

        $scope.$on('jmxTreeUpdated', function () {
            reloadPerspective();
        });

        $scope.$watch('workspace.topLevelTabs', function () {
            reloadPerspective();
        });

        $scope.validSelection = function (uri) {
            return workspace.validSelection(uri);
        };

        $scope.isValid = function (nav) {
            return nav && nav.isValid(workspace);
        };

        $scope.switchPerspective = function (perspective) {
            if (perspective.onSelect && angular.isFunction(perspective.onSelect)) {
                perspective.onSelect.apply();
                return;
            }
            var searchPerspectiveId = $location.search()[Perspective.perspectiveSearchId];
            if (perspective && ($scope.currentPerspective !== perspective || perspective.id !== searchPerspectiveId)) {
                Logger.debug("Changed the perspective to " + JSON.stringify(perspective) + " from search id " + searchPerspectiveId);
                if ($scope.currentPerspective) {
                    $scope.currentPerspective.lastPage = $location.url();
                }
                var pid = perspective.id;
                $location.search(Perspective.perspectiveSearchId, pid);
                Logger.debug("Setting perspective to " + pid);
                $scope.currentPerspective = perspective;
                reloadPerspective();
                $scope.topLevelTabs = Perspective.getTopLevelTabsForPerspective($location, workspace, jolokia, localStorage);

                // is any of the top level tabs marked as default?
                var defaultPlugin = Core.getDefaultPlugin(pid, workspace, jolokia, localStorage);
                var defaultTab;
                var path;
                if (defaultPlugin) {
                    $scope.topLevelTabs.forEach(function (tab) {
                        if (tab.id === defaultPlugin.id) {
                            defaultTab = tab;
                        }
                    });
                    if (defaultTab) {
                        path = Core.trimLeading(defaultTab.href(), "#");
                    }
                } else {
                    // if no default plugin configured, then select the last page as the active location
                    if (perspective.lastPage) {
                        path = Core.trimLeading(perspective.lastPage, "#");
                    }
                }

                if (path) {
                    // lets avoid any old paths with ?p=" inside
                    var idx = path.indexOf("?p=") || path.indexOf("&p=");
                    if (idx > 0) {
                        path = path.substring(0, idx);
                    }
                    var sep = (path.indexOf("?") >= 0) ? "&" : "?";
                    path += sep + "p=" + pid;
                    $location.url(path);
                }
            }
        };

        $scope.$watch('hash', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                Core.log.debug("hash changed from ", oldValue, " to ", newValue);
            }
        });

        // when we change the view/selection lets update the hash so links have the latest stuff
        $scope.$on('$routeChangeSuccess', function () {
            $scope.hash = workspace.hash();
            reloadPerspective();
        });

        // use includePerspective = false as default as that was the previous behavior
        $scope.link = function (nav, includePerspective) {
            if (typeof includePerspective === "undefined") { includePerspective = false; }
            var href;
            if (angular.isString(nav)) {
                href = nav;
            } else {
                href = nav.href();
            }
            var removeParams = ['tab', 'nid', 'chapter', 'pref', 'q'];
            if (!includePerspective) {
                if (href.indexOf("?p=") >= 0 || href.indexOf("&p=") >= 0) {
                    removeParams.push("p");
                }
            }
            return Core.createHref($location, href, removeParams);
        };

        $scope.fullScreenLink = function () {
            var href = "#" + $location.path() + "?tab=notree";
            return Core.createHref($location, href, ['tab']);
        };

        $scope.addToDashboardLink = function () {
            var href = "#" + $location.path() + workspace.hash();

            var answer = "#/dashboard/add?tab=dashboard&href=" + encodeURIComponent(href);

            if ($location.url().has("/jmx/charts")) {
                var size = {
                    size_x: 4,
                    size_y: 3
                };

                answer += "&size=" + encodeURIComponent(angular.toJson(size));
            }

            return answer;
        };

        $scope.isActive = function (nav) {
            if (angular.isString(nav))
                return workspace.isLinkActive(nav);
            var fn = nav.isActive;
            if (fn) {
                return fn(workspace);
            }
            return workspace.isLinkActive(nav.href());
        };

        $scope.isTopTabActive = function (nav) {
            if (angular.isString(nav))
                return workspace.isTopTabActive(nav);
            var fn = nav.isActive;
            if (fn) {
                return fn(workspace);
            }
            return workspace.isTopTabActive(nav.href());
        };

        $scope.activeLink = function () {
            var tabs = $scope.topLevelTabs();
            if (!tabs) {
                return "Loading...";
            }
            var tab = tabs.find(function (nav) {
                return $scope.isActive(nav);
            });
            return tab ? tab['content'] : "";
        };

        function reloadPerspective() {
            var perspectives = Perspective.getPerspectives($location, workspace, jolokia, localStorage);
            var currentId = Perspective.currentPerspectiveId($location, workspace, jolokia, localStorage);

            console.log("Reloading current perspective: " + currentId);

            // any tabs changed
            var newTopLevelTabs = Perspective.getTopLevelTabsForPerspective($location, workspace, jolokia, localStorage);
            var diff = newTopLevelTabs.subtract($scope.topLevelTabs);

            if (diff && diff.length > 0) {
                $scope.topLevelTabs = newTopLevelTabs;

                $scope.perspectiveId = currentId;
                $scope.perspectives = perspectives;
                $scope.perspectiveDetails.perspective = $scope.perspectives.find(function (p) {
                    return p['id'] === currentId;
                });

                console.log("Refreshing top level tabs for current perspective: " + currentId);

                // make sure to update the UI as the top level tabs changed
                Core.$apply($scope);
            }
        }
    }
    Core.NavBarController = NavBarController;
})(Core || (Core = {}));
var _this = this;
/**
* The main entry point for hawtio
*
* @module Core
* @main Core
*/
var Core;
(function (Core) {
    /**
    * Returns true if we are running inside a Chrome app or extension
    */
    function isChromeApp() {
        var answer = false;
        try  {
            answer = (chrome && chrome.app && chrome.extension) ? true : false;
        } catch (e) {
            answer = false;
        }

        //log.info("isChromeApp is: " + answer);
        return answer;
    }
    Core.isChromeApp = isChromeApp;

    /**
    * Name of plugin registered to hawtio's plugin loader and Angularjs module name
    *
    * @property pluginName
    * @for Core
    * @type String
    */
    Core.pluginName = 'hawtioCore';

    Core.templatePath = 'app/core/html/';
})(Core || (Core = {}));

// Add any other known possible jolokia URLs here
var jolokiaUrls = [
    "/jolokia"
];

var jolokiaUrl = getJolokiaUrl();
console.log("jolokiaUrl " + jolokiaUrl);

function getJolokiaUrl() {
    var query = hawtioPluginLoader.parseQueryString();
    var localMode = query['localMode'];
    if (localMode) {
        console.log("local mode so not using jolokia URL");
        jolokiaUrls = [];
        return null;
    }
    var uri = query['url'];
    if (angular.isArray(uri)) {
        uri = uri[0];
    }
    return uri ? decodeURIComponent(uri) : null;
}

if (!jolokiaUrl) {
    jolokiaUrl = jolokiaUrls.find(function (url) {
        var jqxhr = $.ajax(url, {
            async: false,
            username: 'public',
            password: 'biscuit'
        });
        return jqxhr.status === 200 || jqxhr.status === 401 || jqxhr.status === 403;
    });
}

// bootstrap plugin loader
hawtioPluginLoader.addUrl(url("/plugin"));

if (jolokiaUrl) {
    // TODO replace with a jolokia call so we use authentication headers
    //hawtioPluginLoader.addUrl("jolokia:" + jolokiaUrl + ":hawtio:type=plugin,name=*");
}

/*
interface IMyAppScope extends ng.IRootScopeService, ng.IScope {
lineCount: (value:any) => number;
params: ng.IRouteParamsService;
is: (type:any, value:any) => boolean;
empty: (value:any) => boolean;
log: (variable:string) => void;
alert: (text:string) => void;
}
*/
hawtioPluginLoader.addModule(Core.pluginName);

var hawtioCoreModule = angular.module(Core.pluginName, ['bootstrap', 'ngResource', 'ui', 'ui.bootstrap.dialog', 'hawtio-ui']).config(function ($routeProvider, $dialogProvider) {
    $dialogProvider.options({
        backdropFade: true,
        dialogFade: true
    });

    $routeProvider.when('/login', { templateUrl: Core.templatePath + 'login.html' }).when('/welcome', { templateUrl: Core.templatePath + 'welcome.html' }).when('/about', { templateUrl: Core.templatePath + 'about.html' }).when('/help', {
        redirectTo: '/help/index'
    }).when('/help/:topic/', { templateUrl: Core.templatePath + 'help.html' }).when('/help/:topic/:subtopic', { templateUrl: Core.templatePath + 'help.html' }).otherwise({ redirectTo: '/perspective/defaultPage' });
}).constant('layoutTree', Core.templatePath + 'layoutTree.html').constant('layoutFull', Core.templatePath + 'layoutFull.html').service('localStorage', function () {
    return Core.getLocalStorage();
}).factory('pageTitle', function () {
    var answer = new Core.PageTitle();
    return answer;
}).factory('viewRegistry', function () {
    return {};
}).factory('lastLocation', function () {
    return {};
}).factory('postLoginTasks', function () {
    return Core.postLoginTasks;
}).factory('preLogoutTasks', function () {
    return Core.preLogoutTasks;
}).factory('helpRegistry', function ($rootScope) {
    return new Core.HelpRegistry($rootScope);
}).factory('preferencesRegistry', function () {
    return new Core.PreferencesRegistry();
}).factory('jolokiaUrl', function () {
    return jolokiaUrl;
}).factory('jolokiaStatus', function () {
    return {
        xhr: null
    };
}).factory('jolokiaParams', function (jolokiaUrl) {
    return {
        url: jolokiaUrl,
        canonicalNaming: false,
        ignoreErrors: true,
        mimeType: 'application/json'
    };
}).factory('branding', function () {
    var branding = Themes.brandings['hawtio'].setFunc({});
    branding.logoClass = function () {
        if (branding.logoOnly) {
            return "without-text";
        } else {
            return "with-text";
        }
    };
    return branding;
}).factory('userDetails', function (jolokiaUrl, localStorage) {
    var answer = angular.fromJson(localStorage[jolokiaUrl]);
    if (!angular.isDefined(answer) && jolokiaUrl) {
        answer = {
            username: '',
            password: ''
        };

        Core.log.debug("No username set, checking if we have a session");

        // fetch the username if we've already got a session at the server
        var userUrl = jolokiaUrl.replace("jolokia", "user");
        $.ajax(userUrl, {
            type: "GET",
            success: function (response) {
                Core.log.debug("Got user response: ", response);
                Core.executePostLoginTasks();
                /*
                // We'll only touch these if they're not set
                if (response !== '' && response !== null) {
                answer.username = response;
                if (!('loginDetails' in answer)) {
                answer['loginDetails'] = {};
                }
                }
                */
            },
            error: function (xhr, textStatus, error) {
                Core.log.debug("Failed to get session username: ", error);
                Core.executePostLoginTasks();
                // silently ignore, we could be using the proxy
            }
        });

        return answer;
    } else {
        return answer;
    }
}).factory('jolokia', function ($location, localStorage, jolokiaStatus, $rootScope, userDetails, jolokiaParams) {
    // TODO - Maybe have separate URLs or even jolokia instances for loading plugins vs. application stuff
    // var jolokiaUrl = $location.search()['url'] || url("/jolokia");
    console.log("Jolokia URL is " + jolokiaUrl);
    if (jolokiaUrl) {
        var credentials = hawtioPluginLoader.getCredentials(jolokiaUrl);

        // pass basic auth credentials down to jolokia if set
        var username = null;
        var password = null;

        //var userDetails = angular.fromJson(localStorage[jolokiaUrl]);
        if (credentials.length === 2) {
            username = credentials[0];
            password = credentials[1];
            // TODO we should try avoid both permutations of username / userName :)
        } else if (angular.isDefined(userDetails) && angular.isDefined(userDetails.username) && angular.isDefined(userDetails.password)) {
            username = userDetails.username;
            password = userDetails.password;
        } else if (angular.isDefined(userDetails) && angular.isDefined(userDetails.userName) && angular.isDefined(userDetails.password)) {
            username = userDetails.userName;
            password = userDetails.password;
        } else {
            // lets see if they are passed in via request parameter...
            var search = hawtioPluginLoader.parseQueryString();
            username = search["_user"];
            password = search["_pwd"];
            if (angular.isArray(username))
                username = username[0];
            if (angular.isArray(password))
                password = password[0];
        }

        if (username && password) {
            /*
            TODO can't use this, sets the username/password in the URL on every request, plus jolokia passes them on to $.ajax() which causes a fatal exception in firefox
            jolokiaParams['username'] = username;
            jolokiaParams['password'] = password;
            */
            //console.log("Using user / pwd " + username + " / " + password);
            userDetails.username = username;
            userDetails.password = password;

            $.ajaxSetup({
                beforeSend: function (xhr) {
                    xhr.setRequestHeader('Authorization', Core.getBasicAuthHeader(userDetails.username, userDetails.password));
                }
            });

            var loginUrl = jolokiaUrl.replace("jolokia", "auth/login/");
            $.ajax(loginUrl, {
                type: "POST",
                success: function (response) {
                    if (response['credentials'] || response['principals']) {
                        userDetails.loginDetails = {
                            'credentials': response['credentials'],
                            'principals': response['principals']
                        };
                    } else {
                        var doc = Core.pathGet(response, ['children', 0, 'innerHTML']);

                        // hmm, maybe we got an XML document, let's log it just in case...
                        if (doc) {
                            Core.log.debug("Response is a document (ignoring this): ", doc);
                        }
                    }
                    Core.executePostLoginTasks();
                },
                error: function (xhr, textStatus, error) {
                    // silently ignore, we could be using the proxy
                    Core.executePostLoginTasks();
                }
            });
        }

        jolokiaParams['ajaxError'] = function (xhr, textStatus, error) {
            if (xhr.status === 401 || xhr.status === 403) {
                userDetails.username = null;
                userDetails.password = null;
            } else {
                jolokiaStatus.xhr = xhr;
                if (!xhr.responseText && error) {
                    xhr.responseText = error.stack;
                }
            }
            Core.$apply($rootScope);
        };

        var jolokia = new Jolokia(jolokiaParams);
        localStorage['url'] = jolokiaUrl;
        jolokia.stop();
        return jolokia;
    } else {
        // empty jolokia that returns nothing
        return {
            request: function () {
                return null;
            },
            register: function () {
                return null;
            },
            list: function () {
                return null;
            },
            search: function () {
                return null;
            },
            read: function () {
                return null;
            },
            execute: function () {
                return null;
            },
            start: function () {
                _this.running = true;
                return null;
            },
            stop: function () {
                _this.running = false;
                return null;
            },
            isRunning: function () {
                return _this.running;
            },
            jobs: function () {
                return [];
            }
        };
    }
}).factory('toastr', function () {
    var win = window;
    var answer = win.toastr;
    if (!answer) {
        // lets avoid any NPEs
        answer = {};
        win.toaster = answer;
    }
    return answer;
}).factory('xml2json', function ($window) {
    var jquery = $;
    return jquery.xml2json;
}).factory('workspace', function ($location, jmxTreeLazyLoadRegistry, $compile, $templateCache, localStorage, jolokia, jolokiaStatus, $rootScope, userDetails) {
    var answer = new Workspace(jolokia, jolokiaStatus, jmxTreeLazyLoadRegistry, $location, $compile, $templateCache, localStorage, $rootScope, userDetails);
    answer.loadTree();
    return answer;
}).filter("valueToHtml", function () {
    return Core.valueToHtml;
}).filter('humanize', function () {
    return humanizeValue;
}).filter('humanizeMs', function () {
    return Core.humanizeMilliseconds;
}).filter('maskPassword', function () {
    return Core.maskPassword;
}).directive('autofill', [
    '$timeout', function ($timeout) {
        return {
            restrict: "A",
            require: 'ngModel',
            link: function (scope, elem, attrs, ctrl) {
                var ngModel = attrs["ngModel"];
                if (ngModel) {
                    var log = Logger.get("Core");

                    function checkForDifference() {
                        // lets compare the current DOM node value with the model
                        // in case we can default it ourselves
                        var modelValue = scope.$eval(ngModel);
                        var value = elem.val();
                        if (value && !modelValue) {
                            Core.pathSet(scope, ngModel, value);
                            //log.info("autofill: Updated ngModel: " + ngModel + " original model value: " + modelValue + " UI value: " + value + " new value: " + scope.$eval(ngModel));
                        } else {
                            //log.info("Got invoked with ngModel: " + ngModel + " modelValue: " + modelValue + " value: " + value);
                            // lets try trigger input/change events just in case
                            // try both approaches just in case one doesn't work ;)
                            elem.trigger('input');
                            elem.trigger('change');
                            if (elem.length) {
                                var firstElem = $(elem[0]);
                                firstElem.trigger('input');
                                firstElem.trigger('change');
                            }
                        }
                    }

                    $timeout(checkForDifference, 200);
                    $timeout(checkForDifference, 800);
                    $timeout(checkForDifference, 1500);
                }
            }
        };
    }]).run(function ($rootScope, $routeParams, jolokia, workspace, localStorage, viewRegistry, layoutFull, helpRegistry, pageTitle, branding, toastr, userDetails, preferencesRegistry, postLoginTasks, preLogoutTasks) {
    postLoginTasks.addTask("ResetPreLogoutTasks", function () {
        preLogoutTasks.reset();
    });

    preLogoutTasks.addTask("ResetPostLoginTasks", function () {
        postLoginTasks.reset();
    });

    $.support.cors = true;

    /*
    * Count the number of lines in the given text
    */
    $rootScope.lineCount = lineCount;

    /*
    * Easy access to route params
    */
    $rootScope.params = $routeParams;

    /*
    * Wrapper for angular.isArray, isObject, etc checks for use in the view
    *
    * @param type {string} the name of the check (casing sensitive)
    * @param value {string} value to check
    */
    $rootScope.is = function (type, value) {
        return angular['is' + type](value);
    };

    /*
    * Wrapper for $.isEmptyObject()
    *
    * @param value  {mixed} Value to be tested
    * @return booleanean
    */
    $rootScope.empty = function (value) {
        return $.isEmptyObject(value);
    };

    /*
    * Initialize jolokia polling and add handler to change poll
    * frequency
    */
    $rootScope.$on('UpdateRate', function (event, rate) {
        jolokia.stop();
        if (rate > 0) {
            jolokia.start(rate);
        }
        Core.log.debug("Set update rate to: ", rate);
    });

    $rootScope.$emit('UpdateRate', localStorage['updateRate']);

    /*
    * Debugging Tools
    *
    * Allows you to execute debug functions from the view
    */
    // TODO Doesn't support vargs like it should
    $rootScope.log = function (variable) {
        console.log(variable);
    };
    $rootScope.alert = function (text) {
        alert(text);
    };

    viewRegistry['fullscreen'] = layoutFull;
    viewRegistry['notree'] = layoutFull;
    viewRegistry['help'] = layoutFull;
    viewRegistry['welcome'] = layoutFull;
    viewRegistry['preferences'] = layoutFull;
    viewRegistry['about'] = layoutFull;
    viewRegistry['login'] = layoutFull;
    viewRegistry['ui'] = layoutFull;

    helpRegistry.addUserDoc('index', 'app/core/doc/overview.md');
    helpRegistry.addUserDoc('preferences', 'app/core/doc/preferences.md');
    helpRegistry.addSubTopic('index', 'faq', 'app/core/doc/FAQ.md');
    helpRegistry.addSubTopic('index', 'changes', 'app/core/doc/CHANGES.md');
    helpRegistry.addSubTopic('index', 'developer', 'app/core/doc/developer.md');
    helpRegistry.addDevDoc('Core', 'app/core/doc/coreDeveloper.md');
    helpRegistry.addDevDoc('UI', '#/ui/developerPage');
    helpRegistry.addDevDoc('datatable', 'app/datatable/doc/developer.md');
    helpRegistry.addDevDoc('Force Graph', 'app/forcegraph/doc/developer.md');

    preferencesRegistry.addTab("Core", "app/core/html/corePreferences.html");
    preferencesRegistry.addTab("Plugins", "app/core/html/pluginPreferences.html");
    preferencesRegistry.addTab("Console Logging", "app/core/html/loggingPreferences.html");
    preferencesRegistry.addTab("Editor", "app/ui/html/editorPreferences.html");
    preferencesRegistry.addTab("Reset", "app/core/html/resetPreferences.html");

    //helpRegistry.discoverHelpFiles(hawtioPluginLoader.getModules());
    var opts = localStorage['CodeMirrorOptions'];
    if (opts) {
        opts = angular.fromJson(opts);
        CodeEditor.GlobalCodeMirrorOptions = angular.extend(CodeEditor.GlobalCodeMirrorOptions, opts);
    }

    toastr.options = {
        'closeButton': true,
        'showMethod': 'slideDown',
        'hideMethod': 'slideUp'
    };

    window['logInterceptors'].push(function (level, message) {
        if (level === "WARN") {
            notification('warning', message);
        }
        if (level === "ERROR") {
            notification('error', message);
        }
    });

    setTimeout(function () {
        $("#main-body").fadeIn(2000).after(function () {
            Core.log.info(branding.appName + " started");
            Core.$apply($rootScope);
            $(window).trigger('resize');
        });
    }, 500);
}).directive('noClick', function () {
    return function ($scope, $element, $attrs) {
        $element.click(function (event) {
            event.preventDefault();
        });
    };
}).directive('gridStyle', function ($window) {
    return new Core.GridStyle($window);
}).directive('logToggler', function (localStorage) {
    return {
        restrict: 'A',
        link: function ($scope, $element, $attr) {
            $element.click(function () {
                var log = $("#log-panel");
                var body = $('body');
                if (log.height() !== 0) {
                    localStorage['showLog'] = 'false';
                    log.css({ 'bottom': '110%' });
                    body.css({
                        'overflow-y': 'auto'
                    });
                } else {
                    localStorage['showLog'] = 'true';
                    log.css({ 'bottom': '50%' });
                    body.css({
                        'overflow-y': 'hidden'
                    });
                }
                return false;
            });
        }
    };
}).directive('hawtioFileUpload', function () {
    return new Core.FileUpload();
});

String.prototype.unescapeHTML = function () {
    var txt = document.createElement("textarea");
    txt.innerHTML = this;
    return txt.value;
};

// for chrome packaged apps lets enable chrome-extension pages
if (hawtioCoreModule && Core.isChromeApp()) {
    hawtioCoreModule.config([
        '$compileProvider',
        function ($compileProvider) {
            //$compileProvider.aHrefSanitizationWhitelist(/^\s*(https?|ftp|mailto|chrome-extension):/);
            $compileProvider.urlSanitizationWhitelist(/^\s*(https?|ftp|mailto|chrome-extension):/);
            // Angular before v1.2 uses $compileProvider.urlSanitizationWhitelist(...)
        }
    ]);
}

// enable bootstrap tooltips
$(function () {
    $("a[title]").tooltip({
        selector: '',
        delay: { show: 1000, hide: 100 }
    });
});

var adjustHeight = function () {
    var windowHeight = $(window).height();
    var headerHeight = $("#main-nav").height();
    var containerHeight = windowHeight - headerHeight;
    $("#main").css("min-height", "" + containerHeight + "px");
};

$(function () {
    hawtioPluginLoader.loadPlugins(function () {
        var doc = $(document);
        angular.bootstrap(doc, hawtioPluginLoader.getModules());
        $(document.documentElement).attr('xmlns:ng', "http://angularjs.org");
        $(document.documentElement).attr('ng-app', 'hawtioCore');
        adjustHeight();
        $(window).resize(adjustHeight);
    });
});
/**
* @module Core
*/
var Core;
(function (Core) {
    Core.username = null;
    Core.password = null;

    /**
    * Controller that handles the login page and actually logging in
    *
    * @method LoginController
    * @for Core
    * @static
    * @param $scope
    * @param jolokia
    * @param userDetails
    * @param jolokiaUrl
    * @param workspace
    * @param localStorage
    * @param branding
    */
    function LoginController($scope, jolokia, userDetails, jolokiaUrl, workspace, localStorage, branding, postLoginTasks) {
        jolokia.stop();

        $scope.entity = {
            username: '',
            password: ''
        };
        $scope.backstretch = $.backstretch(branding.loginBg);

        $scope.rememberMe = false;
        $scope.branding = branding;

        var details = angular.fromJson(localStorage[jolokiaUrl]);
        if (details) {
            $scope.entity.username = details['username'];
            $scope.entity.password = details['password'];
            $scope.rememberMe = details['rememberMe'];
        }

        $scope.$on('$routeChangeStart', function () {
            if ($scope.backstretch) {
                $scope.backstretch.destroy();
            }
        });

        jQuery(window).bind("beforeunload", function () {
            // auto logout if we should not remember me
            if (!userDetails.rememberMe) {
                console.log("Auto logging out as remember me is off");
                Core.logout(jolokiaUrl, userDetails, localStorage, $scope);
            }
        });

        $scope.doLogin = function () {
            if (jolokiaUrl) {
                var url = jolokiaUrl.replace("jolokia", "auth/login/");

                if ($scope.entity.username.trim() != '') {
                    $.ajax(url, {
                        type: "POST",
                        success: function (response) {
                            userDetails.username = $scope.entity.username;
                            userDetails.password = $scope.entity.password;
                            userDetails.rememberMe = $scope.rememberMe;
                            userDetails.loginDetails = response;

                            Core.username = $scope.entity.username;
                            Core.password = $scope.entity.password;
                            if ($scope.rememberMe) {
                                localStorage[jolokiaUrl] = angular.toJson(userDetails);
                            } else {
                                delete localStorage[jolokiaUrl];
                            }

                            jolokia.start();
                            workspace.loadTree();
                            Core.executePostLoginTasks();
                            Core.$apply($scope);
                        },
                        error: function (xhr, textStatus, error) {
                            switch (xhr.status) {
                                case 401:
                                    Core.notification('error', 'Failed to log in, ' + error);
                                    break;
                                case 403:
                                    Core.notification('error', 'Failed to log in, ' + error);
                                    break;
                                default:
                                    Core.notification('error', 'Failed to log in, ' + error);
                                    break;
                            }
                            Core.$apply($scope);
                        },
                        beforeSend: function (xhr) {
                            xhr.setRequestHeader('Authorization', Core.getBasicAuthHeader($scope.entity.username, $scope.entity.password));
                        }
                    });
                }
            }
        };
    }
    Core.LoginController = LoginController;
})(Core || (Core = {}));
/**
* module Core
*/
var Core;
(function (Core) {
    function PluginPreferences($scope, localStorage, $location, workspace, jolokia) {
        Core.initPreferenceScope($scope, localStorage, {
            'autoRefresh': {
                'value': true,
                'converter': Core.parseBooleanValue
            }
        });

        $scope.perspectiveId;
        $scope.perspectives = [];

        $scope.plugins = [];
        $scope.pluginDirty = false;

        $scope.pluginMoveUp = function (index) {
            $scope.pluginDirty = true;
            var tmp = $scope.plugins[index];
            $scope.plugins[index] = $scope.plugins[index - 1];
            $scope.plugins[index - 1] = tmp;
        };

        $scope.pluginMoveDown = function (index) {
            $scope.pluginDirty = true;
            var tmp = $scope.plugins[index];
            $scope.plugins[index] = $scope.plugins[index + 1];
            $scope.plugins[index + 1] = tmp;
        };

        $scope.pluginDisable = function (index) {
            $scope.pluginDirty = true;
            $scope.plugins[index].enabled = false;
            $scope.plugins[index].isDefault = false;
        };

        $scope.pluginEnable = function (index) {
            $scope.pluginDirty = true;
            $scope.plugins[index].enabled = true;
        };

        $scope.pluginDefault = function (index) {
            $scope.pluginDirty = true;
            $scope.plugins.forEach(function (p) {
                p.isDefault = false;
            });
            $scope.plugins[index].isDefault = true;
        };

        $scope.pluginApply = function () {
            $scope.pluginDirty = false;

            // set index before saving
            $scope.plugins.forEach(function (p, idx) {
                p.index = idx;
            });

            var json = angular.toJson($scope.plugins);
            if (json) {
                Core.log.debug("Saving plugin settings for perspective " + $scope.perspectiveId + " -> " + json);
                var id = "plugins-" + $scope.perspectiveId;
                localStorage[id] = json;
            }

            // force UI to update by reloading the page which works
            setTimeout(function () {
                window.location.reload();
            }, 10);
        };

        $scope.$watch('perspectiveId', function (newValue, oldValue) {
            if (newValue === oldValue) {
                return;
            }

            var perspective = Perspective.getPerspectiveById(newValue);
            if (perspective) {
                updateToPerspective(perspective);
                Core.$apply($scope);
            }
        });

        function updateToPerspective(perspective) {
            var plugins = Core.configuredPluginsForPerspectiveId(perspective.id, workspace, jolokia, localStorage);
            $scope.plugins = plugins;
            $scope.perspectiveId = perspective.id;
            Core.log.debug("Updated to perspective " + $scope.perspectiveId + " with " + plugins.length + " plugins");
        }

        // initialize the controller, and pick the 1st perspective
        $scope.perspectives = Perspective.getPerspectives($location, workspace, jolokia, localStorage);
        Core.log.debug("There are " + $scope.perspectives.length + " perspectives");

        // pick the current selected perspective
        var selectPerspective;
        var perspectiveId = Perspective.currentPerspectiveId($location, workspace, jolokia, localStorage);
        if (perspectiveId) {
            selectPerspective = $scope.perspectives.find(function (p) {
                return p.id === perspectiveId;
            });
        }
        if (!selectPerspective) {
            // just pick the 1st then
            selectPerspective = $scope.perspectives[0];
        }

        updateToPerspective(selectPerspective);

        // and force update the ui
        Core.$apply($scope);
    }
    Core.PluginPreferences = PluginPreferences;
    ;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function WelcomeController($scope, $location, branding, localStorage) {
        var log = Logger.get("Welcome");

        $scope.stopShowingWelcomePage = function () {
            log.debug("Stop showing welcome page");
            localStorage['showWelcomePage'] = false;

            $location.path("/");
        };

        // load the welcome.md file
        $.ajax({
            url: "app/core/doc/welcome.md",
            dataType: 'html',
            cache: false,
            success: function (data, textStatus, jqXHR) {
                $scope.html = "Unable to download welcome.md";
                if (angular.isDefined(data)) {
                    $scope.html = marked(data);
                    $scope.branding = branding;
                }
                Core.$apply($scope);
            },
            error: function (jqXHR, textStatus, errorThrown) {
                $scope.html = "Unable to download welcome.md";
                Core.$apply($scope);
            }
        });
    }
    Core.WelcomeController = WelcomeController;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    

    /**
    * @class Folder
    * @uses NodeSelection
    */
    var Folder = (function () {
        function Folder(title) {
            this.title = title;
            this.key = null;
            this.typeName = null;
            this.children = [];
            this.folderNames = [];
            this.domain = null;
            this.objectName = null;
            this.map = {};
            this.entries = {};
            this.addClass = null;
            this.parent = null;
            this.isLazy = false;
            this.icon = null;
            this.tooltip = null;
            this.entity = null;
            this.addClass = escapeTreeCssStyles(title);
        }
        Folder.prototype.get = function (key) {
            return this.map[key];
        };

        Folder.prototype.isFolder = function () {
            return this.children.length > 0;
        };

        /**
        * Navigates the given paths and returns the value there or null if no value could be found
        * @method navigate
        * @for Folder
        * @param {Array} paths
        * @return {NodeSelection}
        */
        Folder.prototype.navigate = function () {
            var paths = [];
            for (var _i = 0; _i < (arguments.length - 0); _i++) {
                paths[_i] = arguments[_i + 0];
            }
            var node = this;
            paths.forEach(function (path) {
                if (node) {
                    node = node.get(path);
                }
            });
            return node;
        };

        Folder.prototype.hasEntry = function (key, value) {
            var entries = this.entries;
            if (entries) {
                var actual = entries[key];
                return actual && value === actual;
            }
            return false;
        };

        Folder.prototype.parentHasEntry = function (key, value) {
            if (this.parent) {
                return this.parent.hasEntry(key, value);
            }
            return false;
        };

        Folder.prototype.ancestorHasEntry = function (key, value) {
            var parent = this.parent;
            while (parent) {
                if (parent.hasEntry(key, value))
                    return true;
                parent = parent.parent;
            }
            return false;
        };

        Folder.prototype.ancestorHasType = function (typeName) {
            var parent = this.parent;
            while (parent) {
                if (typeName === parent.typeName)
                    return true;
                parent = parent.parent;
            }
            return false;
        };

        Folder.prototype.getOrElse = function (key, defaultValue) {
            if (typeof defaultValue === "undefined") { defaultValue = new Folder(key); }
            var answer = this.map[key];
            if (!answer) {
                answer = defaultValue;
                this.map[key] = answer;
                this.children.push(answer);
                answer.parent = this;
                this.children = this.children.sortBy("title");
            }
            return answer;
        };

        Folder.prototype.sortChildren = function (recursive) {
            var children = this.children;
            if (children) {
                this.children = children.sortBy("title");
                if (recursive) {
                    angular.forEach(children, function (child) {
                        return child.sortChildren(recursive);
                    });
                }
            }
        };

        Folder.prototype.moveChild = function (child) {
            if (child && child.parent !== this) {
                child.detach();
                child.parent = this;
                this.children.push(child);
            }
        };

        /**
        * Removes this node from my parent if I have one
        * @method detach
        * @for Folder
        \   */
        Folder.prototype.detach = function () {
            var oldParent = this.parent;
            if (oldParent) {
                var oldParentChildren = oldParent.children;
                if (oldParentChildren) {
                    var idx = oldParentChildren.indexOf(this);
                    if (idx < 0) {
                        oldParent.children = oldParent.children.remove({ key: this.key });
                    } else {
                        oldParentChildren.splice(idx, 1);
                    }
                }
                this.parent = null;
            }
        };

        /**
        * Searches this folder and all its descendants for the first folder to match the filter
        * @method findDescendant
        * @for Folder
        * @param {Function} filter
        * @return {Folder}
        */
        Folder.prototype.findDescendant = function (filter) {
            if (filter(this)) {
                return this;
            }
            var answer = null;
            angular.forEach(this.children, function (child) {
                if (!answer) {
                    answer = child.findDescendant(filter);
                }
            });
            return answer;
        };
        return Folder;
    })();
    Core.Folder = Folder;
})(Core || (Core = {}));

;
var Folder = (function (_super) {
    __extends(Folder, _super);
    function Folder() {
        _super.apply(this, arguments);
    }
    return Folder;
})(Core.Folder);
;
/**
* @module Core
*/
var Core;
(function (Core) {
    function AboutController($scope, $location, jolokia, branding, localStorage) {
        var log = Logger.get("About");

        // load the about.md file
        $.ajax({
            url: "app/core/doc/about.md",
            dataType: 'html',
            cache: false,
            success: function (data, textStatus, jqXHR) {
                $scope.html = "Unable to download about.md";
                if (angular.isDefined(data)) {
                    $scope.html = marked(data);
                    $scope.branding = branding;
                    $scope.customBranding = branding.enabled;
                    try  {
                        $scope.hawtioVersion = jolokia.request({
                            type: "read",
                            mbean: "hawtio:type=About",
                            attribute: "HawtioVersion"
                        }).value;
                    } catch (Error) {
                        // ignore
                        $scope.hawtioVersion = "N/A";
                    }
                    $scope.jolokiaVersion = jolokia.version().agent;
                    $scope.serverProduct = jolokia.version().info.product;
                    $scope.serverVendor = jolokia.version().info.vendor;
                    $scope.serverVersion = jolokia.version().info.version;
                }
                Core.$apply($scope);
            },
            error: function (jqXHR, textStatus, errorThrown) {
                $scope.html = "Unable to download about.md";
                Core.$apply($scope);
            }
        });
    }
    Core.AboutController = AboutController;
})(Core || (Core = {}));
// TODO Get these functions and variables out of the global namespace
var _urlPrefix = null;

var numberTypeNames = {
    'byte': true,
    'short': true,
    'int': true,
    'long': true,
    'float': true,
    'double': true,
    'java.lang.byte': true,
    'java.lang.short': true,
    'java.lang.integer': true,
    'java.lang.long': true,
    'java.lang.float': true,
    'java.lang.double': true
};

/**
* Returns the number of lines in the given text
*
* @method lineCount
* @static
* @param {String} value
* @return {Number}
*
*/
function lineCount(value) {
    var rows = 0;
    if (value) {
        rows = 1;
        value.toString().each(/\n/, function () {
            return rows++;
        });
    }
    return rows;
}

function url(path) {
    if (path) {
        if (path.startsWith && path.startsWith("/")) {
            if (!_urlPrefix) {
                _urlPrefix = window.location.pathname || "";
                var idx = _urlPrefix.lastIndexOf("/");
                if (idx >= 0) {
                    _urlPrefix = _urlPrefix.substring(0, idx);
                }
            }
            if (_urlPrefix) {
                return _urlPrefix + path;
            }
        }
    }
    return path;
}

function safeNull(value) {
    if (typeof value === 'boolean') {
        return value;
    } else if (typeof value === 'number') {
        // return numbers as-is
        return value;
    }
    if (value) {
        return value;
    } else {
        return "";
    }
}

function safeNullAsString(value, type) {
    if (typeof value === 'boolean') {
        return "" + value;
    } else if (typeof value === 'number') {
        // return numbers as-is
        return "" + value;
    } else if (typeof value === 'string') {
        // its a string
        return "" + value;
    } else if (type === 'javax.management.openmbean.CompositeData' || type === '[Ljavax.management.openmbean.CompositeData;') {
        // composite data or composite data array, we just display as json
        // use json representation
        var data = angular.toJson(value, true);
        return data;
    } else if (type === 'javax.management.openmbean.TabularData') {
        // tabular data is a key/value structure so loop each field and convert to array we can
        // turn into a String
        var arr = [];
        for (var key in value) {
            var val = value[key];
            var line = "" + key + "=" + val;
            arr.push(line);
        }

        // sort array so the values is listed nicely
        arr = arr.sortBy(function (row) {
            return row.toString();
        });
        return arr.join("\n");
    } else if (angular.isArray(value)) {
        // join array with new line, and do not sort as the order in the array may matter
        return value.join("\n");
    } else if (value) {
        // force as string
        return "" + value;
    } else {
        return "";
    }
}

/**
* Converts the given value to an array of query arguments.
*
* If the value is null an empty array is returned.
* If the value is a non empty string then the string is split by commas
*
* @method toSearchArgumentArray
* @static
* @param {*} value
* @return {String[]}
*
*/
function toSearchArgumentArray(value) {
    if (value) {
        if (angular.isArray(value))
            return value;
        if (angular.isString(value))
            return value.split(',');
    }
    return [];
}

function folderMatchesPatterns(node, patterns) {
    if (node) {
        var folderNames = node.folderNames;
        if (folderNames) {
            return patterns.any(function (ignorePaths) {
                for (var i = 0; i < ignorePaths.length; i++) {
                    var folderName = folderNames[i];
                    var ignorePath = ignorePaths[i];
                    if (!folderName)
                        return false;
                    var idx = ignorePath.indexOf(folderName);
                    if (idx < 0) {
                        return false;
                    }
                }
                return true;
            });
        }
    }
    return false;
}

function scopeStoreJolokiaHandle($scope, jolokia, jolokiaHandle) {
    // TODO do we even need to store the jolokiaHandle in the scope?
    if (jolokiaHandle) {
        $scope.$on('$destroy', function () {
            closeHandle($scope, jolokia);
        });
        $scope.jolokiaHandle = jolokiaHandle;
    }
}

function closeHandle($scope, jolokia) {
    var jolokiaHandle = $scope.jolokiaHandle;
    if (jolokiaHandle) {
        //console.log('Closing the handle ' + jolokiaHandle);
        jolokia.unregister(jolokiaHandle);
        $scope.jolokiaHandle = null;
    }
}

/**
* Pass in null for the success function to switch to sync mode
*
* @method onSuccess
* @static
* @param {Function} Success callback function
* @param {Object} Options object to pass on to Jolokia request
* @return {Object} initialized options object
*/
function onSuccess(fn, options) {
    if (typeof options === "undefined") { options = {}; }
    options['mimeType'] = 'application/json';
    if (angular.isDefined(fn)) {
        options['success'] = fn;
    }
    if (!options['method']) {
        options['method'] = "POST";
    }
    options['canonicalNaming'] = false;
    options['canonicalProperties'] = false;
    if (!options['error']) {
        options['error'] = function (response) {
            Core.defaultJolokiaErrorHandler(response, options);
        };
    }
    return options;
}

function supportsLocalStorage() {
    try  {
        return 'localStorage' in window && window['localStorage'] !== null;
    } catch (e) {
        return false;
    }
}

function isNumberTypeName(typeName) {
    if (typeName) {
        var text = typeName.toString().toLowerCase();
        var flag = numberTypeNames[text];
        return flag;
    }
    return false;
}

function encodeMBeanPath(mbean) {
    return mbean.replace(/\//g, '!/').replace(':', '/').escapeURL();
}

function escapeMBeanPath(mbean) {
    return mbean.replace(/\//g, '!/').replace(':', '/');
}

function encodeMBean(mbean) {
    return mbean.replace(/\//g, '!/').escapeURL();
}

function escapeDots(text) {
    return text.replace(/\./g, '-');
}

/**
* Escapes all dots and 'span' text in the css style names to avoid clashing with bootstrap stuff
*
* @method escapeTreeCssStyles
* @static
* @param {String} text
* @return {String}
*/
function escapeTreeCssStyles(text) {
    return escapeDots(text).replace(/span/g, 'sp-an');
}

function showLogPanel() {
    var log = $("#log-panel");
    var body = $('body');
    localStorage['showLog'] = 'true';
    log.css({ 'bottom': '50%' });
    body.css({
        'overflow-y': 'hidden'
    });
}

/**
* Returns the CSS class for a log level based on if its info, warn, error etc.
*
* @method logLevelClass
* @static
* @param {String} level
* @return {String}
*/
function logLevelClass(level) {
    if (level) {
        var first = level[0];
        if (first === 'w' || first === "W") {
            return "warning";
        } else if (first === 'e' || first === "E") {
            return "error";
        } else if (first === 'i' || first === "I") {
            return "info";
        } else if (first === 'd' || first === "D") {
            // we have no debug css style
            return "";
        }
    }
    return "";
}

/**
* @module Core
*/
var Core;
(function (Core) {
    function parseMBean(mbean) {
        var answer = {};
        var parts = mbean.split(":");
        if (parts.length > 1) {
            answer['domain'] = parts.first();
            parts = parts.exclude(parts.first());
            parts = parts.join(":");
            answer['attributes'] = {};
            var nameValues = parts.split(",");
            nameValues.forEach(function (str) {
                var nameValue = str.split('=');
                var name = nameValue.first().trim();
                nameValue = nameValue.exclude(nameValue.first());
                answer['attributes'][name] = nameValue.join('=').trim();
            });
        }
        return answer;
    }
    Core.parseMBean = parseMBean;

    function executePostLoginTasks() {
        Core.log.debug("Executing post login tasks");
        Core.postLoginTasks.execute();
    }
    Core.executePostLoginTasks = executePostLoginTasks;

    function executePreLogoutTasks(onComplete) {
        Core.log.debug("Executing pre logout tasks");
        Core.preLogoutTasks.onComplete(onComplete);
        Core.preLogoutTasks.execute();
    }
    Core.executePreLogoutTasks = executePreLogoutTasks;

    /**
    * log out the current user
    * @for Core
    * @static
    * @method logout
    * @param {String} jolokiaUrl
    * @param {*} userDetails
    * @param {Object} localStorage
    * @param {Object} $scope
    * @param {Function} successCB
    * @param {Function} errorCB
    *
    */
    function logout(jolokiaUrl, userDetails, localStorage, $scope, successCB, errorCB) {
        if (typeof successCB === "undefined") { successCB = null; }
        if (typeof errorCB === "undefined") { errorCB = null; }
        if (jolokiaUrl) {
            var url = jolokiaUrl.replace("jolokia", "auth/logout/");

            Core.executePreLogoutTasks(function () {
                $.ajax(url, {
                    type: "POST",
                    success: function () {
                        userDetails.username = null;
                        userDetails.password = null;
                        userDetails.loginDetails = null;
                        userDetails.rememberMe = false;
                        localStorage[jolokiaUrl] = angular.toJson(userDetails);
                        if (successCB && angular.isFunction(successCB)) {
                            successCB();
                        }
                        Core.$apply($scope);
                    },
                    error: function (xhr, textStatus, error) {
                        switch (xhr.status) {
                            case 401:
                                Core.log.error('Failed to log out, ', error);
                                break;
                            case 403:
                                Core.log.error('Failed to log out, ', error);
                                break;
                            default:
                                Core.log.error('Failed to log out, ', error);
                                break;
                        }
                        if (errorCB && angular.isFunction(errorCB)) {
                            errorCB();
                        }
                        Core.$apply($scope);
                    }
                });
            });
        }
    }
    Core.logout = logout;

    Core.log = Logger.get("Core");

    /**
    * Creates a link by appending the current $location.search() hash to the given href link,
    * removing any required parameters from the link
    * @method createHref
    * @for Core
    * @static
    * @param {ng.ILocationService} $location
    * @param {String} href the link to have any $location.search() hash parameters appended
    * @param {Array} removeParams any parameters to be removed from the $location.search()
    * @return {Object} the link with any $location.search() parameters added
    */
    function createHref($location, href, removeParams) {
        if (typeof removeParams === "undefined") { removeParams = null; }
        var hashMap = angular.copy($location.search());

        // lets remove any top level nav bar related hash searches
        if (removeParams) {
            angular.forEach(removeParams, function (param) {
                return delete hashMap[param];
            });
        }
        var hash = Core.hashToString(hashMap);
        if (hash) {
            var prefix = (href.indexOf("?") >= 0) ? "&" : "?";
            href += prefix + hash;
        }
        return href;
    }
    Core.createHref = createHref;

    /**
    * Trims the leading prefix from a string if its present
    * @method trimLeading
    * @for Core
    * @static
    * @param {String} text
    * @param {String} prefix
    * @return {String}
    */
    function trimLeading(text, prefix) {
        if (text && prefix) {
            if (text.startsWith(prefix)) {
                return text.substring(prefix.length);
            }
        }
        return text;
    }
    Core.trimLeading = trimLeading;

    /**
    * Trims the trailing postfix from a string if its present
    * @method trimTrailing
    * @for Core
    * @static
    * @param {String} trim
    * @param {String} postfix
    * @return {String}
    */
    function trimTrailing(text, postfix) {
        if (text && postfix) {
            if (text.endsWith(postfix)) {
                return text.substring(0, text.length - postfix.length);
            }
        }
        return text;
    }
    Core.trimTrailing = trimTrailing;

    /**
    * Turns the given search hash into a URI style query string
    * @method hashToString
    * @for Core
    * @static
    * @param {Object} hash
    * @return {String}
    */
    function hashToString(hash) {
        var keyValuePairs = [];
        angular.forEach(hash, function (value, key) {
            keyValuePairs.push(key + "=" + value);
        });
        var params = keyValuePairs.join("&");
        return encodeURI(params);
    }
    Core.hashToString = hashToString;

    /**
    * Parses the given string of x=y&bar=foo into a hash
    * @method stringToHash
    * @for Core
    * @static
    * @param {String} hashAsString
    * @return {Object}
    */
    function stringToHash(hashAsString) {
        var entries = {};
        if (hashAsString) {
            var text = decodeURI(hashAsString);
            var items = text.split('&');
            angular.forEach(items, function (item) {
                var kv = item.split('=');
                var key = kv[0];
                var value = kv[1] || key;
                entries[key] = value;
            });
        }
        return entries;
    }
    Core.stringToHash = stringToHash;

    /**
    * Register a JMX operation to poll for changes
    * @method register
    * @for Core
    * @static
    * @return {Function} a zero argument function for unregistering  this registration
    * @param {*} jolokia
    * @param {*} scope
    * @param {Object} arguments
    * @param {Function} callback
    */
    function register(jolokia, scope, arguments, callback) {
        if (scope && !Core.isBlank(scope.name)) {
            Core.log.debug("Calling register from scope: ", scope.name);
        } else {
            Core.log.debug("Calling register from anonymous scope");
        }
        if (!angular.isDefined(scope.$jhandle) || !angular.isArray(scope.$jhandle)) {
            scope.$jhandle = [];
        }
        if (angular.isDefined(scope.$on)) {
            scope.$on('$destroy', function (event) {
                unregister(jolokia, scope);
            });
        }

        var handle = null;

        if (angular.isArray(arguments)) {
            if (arguments.length >= 1) {
                // TODO can't get this to compile in typescript :)
                //var args = [callback].concat(arguments);
                var args = [callback];
                angular.forEach(arguments, function (value) {
                    return args.push(value);
                });

                //var args = [callback];
                //args.push(arguments);
                var registerFn = jolokia.register;
                handle = registerFn.apply(jolokia, args);
                scope.$jhandle.push(handle);
                jolokia.request(arguments, callback);
            }
        } else {
            handle = jolokia.register(callback, arguments);
            scope.$jhandle.push(handle);
            jolokia.request(arguments, callback);
        }
        return function () {
            if (handle !== null) {
                scope.$jhandle.remove(handle);
                jolokia.unregister(handle);
            }
        };
    }
    Core.register = register;

    /**
    * Register a JMX operation to poll for changes using a jolokia search using the given mbean pattern
    * @method registerSearch
    * @for Core
    * @static
    * @paran {*} jolokia
    * @param {*} scope
    * @param {String} mbeanPattern
    * @param {Function} callback
    */
    function registerSearch(jolokia, scope, mbeanPattern, callback) {
        if (!angular.isDefined(scope.$jhandle) || !angular.isArray(scope.$jhandle)) {
            scope.$jhandle = [];
        }
        if (angular.isDefined(scope.$on)) {
            scope.$on('$destroy', function (event) {
                unregister(jolokia, scope);
            });
        }
        if (angular.isArray(arguments)) {
            if (arguments.length >= 1) {
                // TODO can't get this to compile in typescript :)
                //var args = [callback].concat(arguments);
                var args = [callback];
                angular.forEach(arguments, function (value) {
                    return args.push(value);
                });

                //var args = [callback];
                //args.push(arguments);
                var registerFn = jolokia.register;
                var handle = registerFn.apply(jolokia, args);
                scope.$jhandle.push(handle);
                jolokia.search(mbeanPattern, callback);
            }
        } else {
            var handle = jolokia.register(callback, arguments);
            scope.$jhandle.push(handle);
            jolokia.search(mbeanPattern, callback);
        }
    }
    Core.registerSearch = registerSearch;

    function unregister(jolokia, scope) {
        if (angular.isDefined(scope.$jhandle)) {
            scope.$jhandle.forEach(function (handle) {
                jolokia.unregister(handle);
            });
            delete scope.$jhandle;
        }
    }
    Core.unregister = unregister;

    /**
    * The default error handler which logs errors either using debug or log level logging based on the silent setting
    * @param response the response from a jolokia request
    */
    function defaultJolokiaErrorHandler(response, options) {
        if (typeof options === "undefined") { options = {}; }
        //alert("Jolokia request failed: " + response.error);
        var stacktrace = response.stacktrace;
        if (stacktrace) {
            var silent = options['silent'];
            if (!silent) {
                var operation = Core.pathGet(response, ['request', 'operation']) || "unknown";
                if (stacktrace.indexOf("javax.management.InstanceNotFoundException") >= 0 || stacktrace.indexOf("javax.management.AttributeNotFoundException") >= 0 || stacktrace.indexOf("java.lang.IllegalArgumentException: No operation") >= 0) {
                    // ignore these errors as they can happen on timing issues
                    // such as its been removed
                    // or if we run against older containers
                    Core.log.debug("Operation ", operation, " failed due to: ", response['error']);
                    Core.log.debug("Stack trace: ", Logger.formatStackTraceString(response['stacktrace']));
                } else {
                    Core.log.warn("Operation ", operation, " failed due to: ", response['error']);
                    Core.log.info("Stack trace: ", Logger.formatStackTraceString(response['stacktrace']));
                }
            } else {
                Core.log.debug("Operation ", operation, " failed due to: ", response['error']);
                Core.log.debug("Stack trace: ", Logger.formatStackTraceString(response['stacktrace']));
            }
        }
    }
    Core.defaultJolokiaErrorHandler = defaultJolokiaErrorHandler;

    /**
    * Logs any failed operation and stack traces
    */
    function logJolokiaStackTrace(response) {
        var stacktrace = response.stacktrace;
        if (stacktrace) {
            var operation = Core.pathGet(response, ['request', 'operation']) || "unknown";
            Core.log.info("Operation ", operation, " failed due to: ", response['error']);
            Core.log.info("Stack trace: ", Logger.formatStackTraceString(response['stacktrace']));
        }
    }
    Core.logJolokiaStackTrace = logJolokiaStackTrace;

    /**
    * Converts the given XML node to a string representation of the XML
    * @method xmlNodeToString
    * @for Core
    * @static
    * @param {Object} xmlNode
    * @return {Object}
    */
    function xmlNodeToString(xmlNode) {
        try  {
            // Gecko- and Webkit-based browsers (Firefox, Chrome), Opera.
            return (new XMLSerializer()).serializeToString(xmlNode);
        } catch (e) {
            try  {
                // Internet Explorer.
                return xmlNode.xml;
            } catch (e) {
                //Other browsers without XML Serializer
                console.log('WARNING: XMLSerializer not supported');
            }
        }
        return false;
    }
    Core.xmlNodeToString = xmlNodeToString;

    /**
    * Returns true if the given DOM node is a text node
    * @method isTextNode
    * @for Core
    * @static
    * @param {Object} node
    * @return {Boolean}
    */
    function isTextNode(node) {
        return node && node.nodeType === 3;
    }
    Core.isTextNode = isTextNode;

    /**
    * Returns the lowercase file extension of the given file name or returns the empty
    * string if the file does not have an extension
    * @method fileExtension
    * @for Core
    * @static
    * @param {String} name
    * @param {String} defaultValue
    * @return {String}
    */
    function fileExtension(name, defaultValue) {
        if (typeof defaultValue === "undefined") { defaultValue = ""; }
        var extension = defaultValue;
        if (name) {
            var idx = name.lastIndexOf(".");
            if (idx > 0) {
                extension = name.substring(idx + 1, name.length).toLowerCase();
            }
        }
        return extension;
    }
    Core.fileExtension = fileExtension;

    function getUUID() {
        var d = new Date();
        var ms = (d.getTime() * 1000) + d.getUTCMilliseconds();
        var random = Math.floor((1 + Math.random()) * 0x10000);
        return ms.toString(16) + random.toString(16);
    }
    Core.getUUID = getUUID;

    var _versionRegex = /[^\d]*(\d+)\.(\d+)(\.(\d+))?.*/;

    /**
    * Parses some text of the form "xxxx2.3.4xxxx"
    * to extract the version numbers as an array of numbers then returns an array of 2 or 3 numbers.
    *
    * Characters before the first digit are ignored as are characters after the last digit.
    * @method parseVersionNumbers
    * @for Core
    * @static
    * @param {String} text a maven like string containing a dash then numbers separated by dots
    * @return {Array}
    */
    function parseVersionNumbers(text) {
        if (text) {
            var m = text.match(_versionRegex);
            if (m && m.length > 4) {
                var m1 = m[1];
                var m2 = m[2];
                var m4 = m[4];
                if (angular.isDefined(m4)) {
                    return [parseInt(m1), parseInt(m2), parseInt(m4)];
                } else if (angular.isDefined(m2)) {
                    return [parseInt(m1), parseInt(m2)];
                } else if (angular.isDefined(m1)) {
                    return [parseInt(m1)];
                }
            }
        }
        return null;
    }
    Core.parseVersionNumbers = parseVersionNumbers;

    /**
    * Converts a version string with numbers and dots of the form "123.456.790" into a string
    * which is sortable as a string, by left padding each string between the dots to at least 4 characters
    * so things just sort as a string.
    *
    * @param text
    * @return {string} the sortable version string
    */
    function versionToSortableString(version, maxDigitsBetweenDots) {
        if (typeof maxDigitsBetweenDots === "undefined") { maxDigitsBetweenDots = 4; }
        return (version || "").split(".").map(function (x) {
            var length = x.length;
            return (length >= maxDigitsBetweenDots) ? x : x.padLeft(' ', maxDigitsBetweenDots - length);
        }).join(".");
    }
    Core.versionToSortableString = versionToSortableString;

    function time(message, fn) {
        var start = new Date().getTime();
        var answer = fn();
        var elapsed = new Date().getTime() - start;
        console.log(message + " " + elapsed);
        return answer;
    }
    Core.time = time;

    /**
    * Compares the 2 version arrays and returns -1 if v1 is less than v2 or 0 if they are equal or 1 if v1 is greater than v2
    * @method compareVersionNumberArrays
    * @for Core
    * @static
    * @param {Array} v1 an array of version numbers with the most significant version first (major, minor, patch).
    * @param {Array} v2
    * @return {Number}
    */
    function compareVersionNumberArrays(v1, v2) {
        if (v1 && !v2) {
            return 1;
        }
        if (!v1 && v2) {
            return -1;
        }
        if (v1 === v2) {
            return 0;
        }
        for (var i = 0; i < v1.length; i++) {
            var n1 = v1[i];
            if (i >= v2.length) {
                return 1;
            }
            var n2 = v2[i];
            if (!angular.isDefined(n1)) {
                return -1;
            }
            if (!angular.isDefined(n2)) {
                return 1;
            }
            if (n1 > n2) {
                return 1;
            } else if (n1 < n2) {
                return -1;
            }
        }
        return 0;
    }
    Core.compareVersionNumberArrays = compareVersionNumberArrays;

    /**
    * Helper function which converts objects into tables of key/value properties and
    * lists into a <ul> for each value.
    * @method valueToHtml
    * @for Core
    * @static
    * @param {any} value
    * @return {String}
    */
    function valueToHtml(value) {
        if (angular.isArray(value)) {
            var size = value.length;
            if (!size) {
                return "";
            } else if (size === 1) {
                return valueToHtml(value[0]);
            } else {
                var buffer = "<ul>";
                angular.forEach(value, function (childValue) {
                    buffer += "<li>" + valueToHtml(childValue) + "</li>";
                });
                return buffer + "</ul>";
            }
        } else if (angular.isObject(value)) {
            var buffer = "<table>";
            angular.forEach(value, function (childValue, key) {
                buffer += "<tr><td>" + key + "</td><td>" + valueToHtml(childValue) + "</td></tr>";
            });
            return buffer + "</table>";
        } else if (angular.isString(value)) {
            var uriPrefixes = ["http://", "https://", "file://", "mailto:"];
            var answer = value;
            angular.forEach(uriPrefixes, function (prefix) {
                if (answer.startsWith(prefix)) {
                    answer = "<a href='" + value + "'>" + value + "</a>";
                }
            });
            return answer;
        }
        return value;
    }
    Core.valueToHtml = valueToHtml;

    /**
    * If the string starts and ends with [] {} then try parse as JSON and return the parsed content or return null
    * if it does not appear to be JSON
    * @method tryParseJson
    * @for Core
    * @static
    * @param {String} text
    * @return {Object}
    */
    function tryParseJson(text) {
        text = text.trim();
        if ((text.startsWith("[") && text.endsWith("]")) || (text.startsWith("{") && text.endsWith("}"))) {
            try  {
                return JSON.parse(text);
            } catch (e) {
                // ignore
            }
        }
        return null;
    }
    Core.tryParseJson = tryParseJson;

    /**
    * Given values (n, "person") will return either "1 person" or "2 people" depending on if a plural
    * is required using the String.pluralize() function from sugarjs
    * @method maybePlural
    * @for Core
    * @static
    * @param {Number} count
    * @param {String} word
    * @return {String}
    */
    function maybePlural(count, word) {
        var pluralWord = (count === 1) ? word : word.pluralize();
        return "" + count + " " + pluralWord;
    }
    Core.maybePlural = maybePlural;

    /**
    * given a JMX ObjectName of the form <code>domain:key=value,another=something</code> then return the object
    * <code>{key: "value", another: "something"}</code>
    * @method objectNameProperties
    * @for Core
    * @static
    * @param {String} name
    * @return {Object}
    */
    function objectNameProperties(objectName) {
        var entries = {};
        if (objectName) {
            var idx = objectName.indexOf(":");
            if (idx > 0) {
                var path = objectName.substring(idx + 1);
                var items = path.split(',');
                angular.forEach(items, function (item) {
                    var kv = item.split('=');
                    var key = kv[0];
                    var value = kv[1] || key;
                    entries[key] = value;
                });
            }
        }
        return entries;
    }
    Core.objectNameProperties = objectNameProperties;

    function setPageTitle($document, title) {
        $document.attr('title', title.getTitleWithSeparator(' '));
    }
    Core.setPageTitle = setPageTitle;

    function setPageTitleWithTab($document, title, tab) {
        $document.attr('title', title.getTitleWithSeparator(' ') + " " + tab);
    }
    Core.setPageTitleWithTab = setPageTitleWithTab;

    /**
    * Returns the Folder object for the given domain name and type name or null if it can not be found
    * @method getMBeanTypeFolder
    * @for Core
    * @static
    * @param {Workspace} workspace
    * @param {String} domain
    * @param {String} typeName}
    * @return {Folder}
    */
    function getMBeanTypeFolder(workspace, domain, typeName) {
        if (workspace) {
            var mbeanTypesToDomain = workspace.mbeanTypesToDomain || {};
            var types = mbeanTypesToDomain[typeName] || {};
            var answer = types[domain];
            if (angular.isArray(answer) && answer.length) {
                return answer[0];
            }
            return answer;
        }
        return null;
    }
    Core.getMBeanTypeFolder = getMBeanTypeFolder;

    /**
    * Returns the JMX objectName for the given jmx domain and type name
    * @method getMBeanTypeObjectName
    * @for Core
    * @static
    * @param {Workspace} workspace
    * @param {String} domain
    * @param {String} typeName
    * @return {String}
    */
    function getMBeanTypeObjectName(workspace, domain, typeName) {
        var folder = Core.getMBeanTypeFolder(workspace, domain, typeName);
        return Core.pathGet(folder, ["objectName"]);
    }
    Core.getMBeanTypeObjectName = getMBeanTypeObjectName;

    /**
    * Removes dodgy characters from a value such as '/' or '.' so that it can be used as a DOM ID value
    * and used in jQuery / CSS selectors
    * @method toSafeDomID
    * @for Core
    * @static
    * @param {String} text
    * @return {String}
    */
    function toSafeDomID(text) {
        return text ? text.replace(/(\/|\.)/g, "_") : text;
    }
    Core.toSafeDomID = toSafeDomID;

    /**
    * Invokes the given function on each leaf node in the array of folders
    * @method forEachLeafFolder
    * @for Core
    * @static
    * @param {Array[Folder]} folders
    * @param {Function} fn
    */
    function forEachLeafFolder(folders, fn) {
        angular.forEach(folders, function (folder) {
            var children = folder["children"];
            if (angular.isArray(children) && children.length > 0) {
                forEachLeafFolder(children, fn);
            } else {
                fn(folder);
            }
        });
    }
    Core.forEachLeafFolder = forEachLeafFolder;

    function extractHashURL(url) {
        var parts = url.split('#');
        if (parts.length === 0) {
            return url;
        }
        var answer = parts[1];
        if (parts.length > 1) {
            var remaining = parts.last(parts.length - 2);
            remaining.forEach(function (part) {
                answer = answer + "#" + part;
            });
        }
        return answer;
    }
    Core.extractHashURL = extractHashURL;

    function getBasicAuthHeader(username, password) {
        var authInfo = username + ":" + password;
        authInfo = authInfo.encodeBase64();
        return "Basic " + authInfo;
    }
    Core.getBasicAuthHeader = getBasicAuthHeader;

    var httpRegex = new RegExp('^(https?):\/\/(([^:/?#]*)(?::([0-9]+))?)');

    /**
    * Breaks a URL up into a nice object
    * @method parseUrl
    * @for Core
    * @static
    * @param url
    * @returns object
    */
    function parseUrl(url) {
        if (Core.isBlank(url)) {
            return null;
        }

        var matches = url.match(httpRegex);

        if (matches === null) {
            return null;
        }

        //log.debug("matches: ", matches);
        var scheme = matches[1];
        var host = matches[3];
        var port = matches[4];

        var parts = null;
        if (!Core.isBlank(port)) {
            parts = url.split(port);
        } else {
            parts = url.split(host);
        }

        var path = parts[1];
        if (path && path.startsWith('/')) {
            path = path.slice(1, path.length);
        }

        //log.debug("parts: ", parts);
        return {
            scheme: scheme,
            host: host,
            port: port,
            path: path
        };
    }
    Core.parseUrl = parseUrl;

    var ConnectToServerOptions = (function () {
        function ConnectToServerOptions() {
            this.scheme = "http";
            this.useProxy = true;
        }
        return ConnectToServerOptions;
    })();
    Core.ConnectToServerOptions = ConnectToServerOptions;

    function getDocHeight() {
        var D = document;
        return Math.max(Math.max(D.body.scrollHeight, D.documentElement.scrollHeight), Math.max(D.body.offsetHeight, D.documentElement.offsetHeight), Math.max(D.body.clientHeight, D.documentElement.clientHeight));
    }
    Core.getDocHeight = getDocHeight;

    /**
    * If a URL is external to the current web application, then
    * replace the URL with the proxy servlet URL
    * @method useProxyIfExternal
    * @for Core
    * @static
    * @param {String} connectUrl
    * @return {String}
    */
    function useProxyIfExternal(connectUrl) {
        if (Core.isChromeApp()) {
            return connectUrl;
        }
        var host = window.location.host;
        if (!connectUrl.startsWith("http://" + host + "/") && !connectUrl.startsWith("https://" + host + "/")) {
            // lets remove the http stuff
            var idx = connectUrl.indexOf("://");
            if (idx > 0) {
                connectUrl = connectUrl.substring(idx + 3);
            }

            // lets replace the : with a /
            connectUrl = connectUrl.replace(":", "/");
            connectUrl = Core.trimLeading(connectUrl, "/");
            connectUrl = Core.trimTrailing(connectUrl, "/");
            connectUrl = url("/proxy/" + connectUrl);
        }
        return connectUrl;
    }
    Core.useProxyIfExternal = useProxyIfExternal;

    function getRecentConnections(localStorage) {
        if (Core.isBlank(localStorage['recentConnections'])) {
            Core.clearConnections();
        }
        return angular.fromJson(localStorage['recentConnections']);
    }
    Core.getRecentConnections = getRecentConnections;

    function addRecentConnection(localStorage, name, url) {
        var recent = getRecentConnections(localStorage);
        recent = recent.add({
            'name': name,
            'url': url
        }).unique(function (c) {
            return c.name;
        }).first(5);
        localStorage['recentConnections'] = angular.toJson(recent);
    }
    Core.addRecentConnection = addRecentConnection;

    function removeRecentConnection(localStorage, name) {
        var recent = getRecentConnections(localStorage);
        recent = recent.exclude(function (conn) {
            return conn.name === name;
        });
        localStorage['recentConnections'] = angular.toJson(recent);
    }
    Core.removeRecentConnection = removeRecentConnection;

    function clearConnections() {
        localStorage['recentConnections'] = '[]';
    }
    Core.clearConnections = clearConnections;

    function connectToServer(localStorage, options) {
        Core.log.debug("Connect to server, options: ", options);

        var connectUrl = options.jolokiaUrl;

        var userDetails = {
            username: options['userName'],
            password: options['password']
        };

        var json = angular.toJson(userDetails);
        if (connectUrl) {
            localStorage[connectUrl] = json;
        }
        var view = options.view;
        var full = "";
        var useProxy = options.useProxy && !Core.isChromeApp();
        if (connectUrl) {
            if (useProxy) {
                // lets remove the http stuff
                var idx = connectUrl.indexOf("://");
                if (idx > 0) {
                    connectUrl = connectUrl.substring(idx + 3);
                }

                // lets replace the : with a /
                connectUrl = connectUrl.replace(":", "/");
                connectUrl = Core.trimLeading(connectUrl, "/");
                connectUrl = Core.trimTrailing(connectUrl, "/");
                connectUrl = url("/proxy/" + connectUrl);
            } else {
                if (connectUrl.indexOf("://") < 0) {
                    connectUrl = options.scheme + "://" + connectUrl;
                }
            }
            console.log("going to server: " + connectUrl + " as user " + options.userName);
            localStorage[connectUrl] = json;

            full = "?url=" + encodeURIComponent(connectUrl);
            if (view) {
                full += "#" + view;
            }
        } else {
            var host = options.host || "localhost";
            var port = options.port;
            var path = Core.trimLeading(options.path || "jolokia", "/");
            path = Core.trimTrailing(path, "/");

            if (port > 0) {
                host += ":" + port;
            }
            var connectUrl = host + "/" + path;
            localStorage[connectUrl] = json;
            if (useProxy) {
                connectUrl = url("/proxy/" + connectUrl);
            } else {
                if (connectUrl.indexOf("://") < 0) {
                    connectUrl = options.scheme + "://" + connectUrl;
                }
            }
            console.log("going to server: " + connectUrl + " as user " + options.userName);
            localStorage[connectUrl] = json;

            full = "?url=" + encodeURIComponent(connectUrl);
            if (view) {
                full += "#" + view;
            }
        }
        if (full) {
            Core.log.info("Full URL is: " + full);
            Core.addRecentConnection(localStorage, options.name, full);
            window.open(full);
        }
    }
    Core.connectToServer = connectToServer;

    /**
    * Extracts the url of the target, eg usually http://localhost:port, but if we use fabric to proxy to another host,
    * then we return the url that we proxied too (eg the real target)
    *
    * @param {ng.ILocationService} $location
    * @param {String} scheme to force use a specific scheme, otherwise the scheme from location is used
    * @param {Number} port to force use a specific port number, otherwise the port from location is used
    */
    function extractTargetUrl($location, scheme, port) {
        if (angular.isUndefined(scheme)) {
            scheme = $location.scheme();
        }

        var host = $location.host();

        //  $location.search()['url']; does not work for some strange reason
        // var qUrl = $location.search()['url'];
        // if its a proxy request using hawtio-proxy servlet, then the url parameter
        // has the actual host/port
        var qUrl = $location.absUrl();
        var idx = qUrl.indexOf("url=");
        if (idx > 0) {
            qUrl = qUrl.substr(idx + 4);
            var value = decodeURIComponent(qUrl);
            if (value) {
                idx = value.indexOf("/proxy/");

                // after proxy we have host and optional port (if port is not 80)
                if (idx > 0) {
                    value = value.substr(idx + 7);
                    var data = value.split("/");
                    if (data.length >= 1) {
                        host = data[0];
                    }
                    if (angular.isUndefined(port) && data.length >= 2) {
                        var qPort = Core.parseIntValue(data[1], "port number");
                        if (qPort) {
                            port = qPort;
                        }
                    }
                }
            }
        }

        if (angular.isUndefined(port)) {
            port = $location.port();
        }

        var url = scheme + "://" + host;
        if (port != 80) {
            url += ":" + port;
        }
        return url;
    }
    Core.extractTargetUrl = extractTargetUrl;

    /**
    * Binds a $location.search() property to a model on a scope; so that its initialised correctly on startup
    * and its then watched so as the model changes, the $location.search() is updated to reflect its new value
    * @method bindModelToSearchParam
    * @for Core
    * @static
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @param {String} modelName
    * @param {String} paramName
    * @param {Object} initialValue
    */
    function bindModelToSearchParam($scope, $location, modelName, paramName, initialValue) {
        if (typeof initialValue === "undefined") { initialValue = null; }
        function currentValue() {
            return $location.search()[paramName] || initialValue;
        }

        var value = currentValue();
        Core.pathSet($scope, modelName, value);
        $scope.$watch(modelName, function () {
            var current = Core.pathGet($scope, modelName);
            if (current) {
                var params = $location.search();
                var old = currentValue();
                if (current !== old) {
                    $location.search(paramName, current);
                }
            } else {
                $location.search(paramName, null);
            }
        });
    }
    Core.bindModelToSearchParam = bindModelToSearchParam;

    /**
    * For controllers where reloading is disabled via "reloadOnSearch: false" on the registration; lets pick which
    * query parameters need to change to force the reload. We default to the JMX selection parameter 'nid'
    * @method reloadWhenParametersChange
    * @for Core
    * @static
    * @param {Object} $route
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @param {Array[String]} parameters
    */
    function reloadWhenParametersChange($route, $scope, $location, parameters) {
        if (typeof parameters === "undefined") { parameters = ["nid"]; }
        var initial = angular.copy($location.search());
        $scope.$on('$routeUpdate', function () {
            // lets check if any of the parameters changed
            var current = $location.search();
            var changed = [];
            angular.forEach(parameters, function (param) {
                if (current[param] !== initial[param]) {
                    changed.push(param);
                }
            });
            if (changed.length) {
                Core.log.info("Reloading page due to change to parameters: " + changed);
                $route.reload();
            }
        });
    }
    Core.reloadWhenParametersChange = reloadWhenParametersChange;

    /**
    * Creates a jolokia object for connecting to the container with the given remote jolokia URL,
    * username and password
    * @method createJolokia
    * @for Core
    * @static
    * @param {String} url
    * @param {String} username
    * @param {String} password
    * @return {Object}
    */
    function createJolokia(url, username, password) {
        var jolokiaParams = {
            url: url,
            username: username,
            password: password,
            canonicalNaming: false, ignoreErrors: true, mimeType: 'application/json'
        };
        return new Jolokia(jolokiaParams);
    }
    Core.createJolokia = createJolokia;

    /**
    * Returns a new function which ensures that the delegate function is only invoked at most once
    * within the given number of millseconds
    * @method throttled
    * @for Core
    * @static
    * @param {Function} fn the function to be invoked at most once within the given number of millis
    * @param {Number} millis the time window during which this function should only be called at most once
    * @return {Object}
    */
    function throttled(fn, millis) {
        var nextInvokeTime = 0;
        var lastAnswer = null;
        return function () {
            var now = Date.now();
            if (nextInvokeTime < now) {
                nextInvokeTime = now + millis;
                lastAnswer = fn();
            } else {
                Core.log.debug("Not invoking function as we did call " + (now - (nextInvokeTime - millis)) + " ms ago");
            }
            return lastAnswer;
        };
    }
    Core.throttled = throttled;

    /**
    * Attempts to parse the given JSON text and returns the JSON object structure or null.
    *Bad JSON is logged at info level.
    *
    * @param text a JSON formatted string
    * @param message description of the thing being parsed logged if its invalid
    */
    function parseJsonText(text, message) {
        if (typeof message === "undefined") { message = "JSON"; }
        var answer = null;
        try  {
            answer = angular.fromJson(text);
        } catch (e) {
            Core.log.info("Failed to parse " + message + " from: " + text + ". " + e);
        }
        return answer;
    }
    Core.parseJsonText = parseJsonText;

    /**
    * Returns the humanized markup of the given value
    */
    function humanizeValueHtml(value) {
        var formattedValue = "";
        if (value === true) {
            formattedValue = '<i class="icon-check"></i>';
        } else if (value === false) {
            formattedValue = '<i class="icon-check-empty"></i>';
        } else {
            formattedValue = Core.humanizeValue(value);
        }
        return formattedValue;
    }
    Core.humanizeValueHtml = humanizeValueHtml;

    /**
    * Gets a query value from the given url
    *
    * @param url  url
    * @param parameterName the uri parameter value to get
    * @returns {*}
    */
    function getQueryParameterValue(url, parameterName) {
        var parts;

        var query = (url || '').split('?');
        if (query && query.length > 0) {
            parts = query[1];
        } else {
            parts = '';
        }

        var vars = parts.split('&');
        for (var i = 0; i < vars.length; i++) {
            var pair = vars[i].split('=');
            if (decodeURIComponent(pair[0]) == parameterName) {
                return decodeURIComponent(pair[1]);
            }
        }

        // not found
        return null;
    }
    Core.getQueryParameterValue = getQueryParameterValue;

    /**
    * Creates a remote workspace given a remote jolokia for querying the JMX MBeans inside the jolokia
    * @param remoteJolokia
    * @param $location
    * @param localStorage
    * @return {Core.Workspace|Workspace}
    */
    function createRemoteWorkspace(remoteJolokia, $location, localStorage, $rootScope, $compile, $templateCache, userDetails) {
        if (typeof $rootScope === "undefined") { $rootScope = null; }
        if (typeof $compile === "undefined") { $compile = null; }
        if (typeof $templateCache === "undefined") { $templateCache = null; }
        if (typeof userDetails === "undefined") { userDetails = null; }
        // lets create a child workspace object for the remote container
        var jolokiaStatus = {
            xhr: null
        };

        // disable reload notifications
        var jmxTreeLazyLoadRegistry = Jmx.lazyLoaders;
        var profileWorkspace = new Core.Workspace(remoteJolokia, jolokiaStatus, jmxTreeLazyLoadRegistry, $location, $compile, $templateCache, localStorage, $rootScope, userDetails);

        Core.log.info("Loading the profile using jolokia: " + remoteJolokia);
        profileWorkspace.loadTree();
        return profileWorkspace;
    }
    Core.createRemoteWorkspace = createRemoteWorkspace;

    /**
    * Takes a value in ms and returns a human readable
    * duration
    * @param value
    */
    function humanizeMilliseconds(value) {
        if (!angular.isNumber(value)) {
            return "XXX";
        }

        var seconds = value / 1000;
        var years = Math.floor(seconds / 31536000);
        if (years) {
            return maybePlural(years, "year");
        }
        var days = Math.floor((seconds %= 31536000) / 86400);
        if (days) {
            return maybePlural(days, "day");
        }
        var hours = Math.floor((seconds %= 86400) / 3600);
        if (hours) {
            return maybePlural(hours, 'hour');
        }
        var minutes = Math.floor((seconds %= 3600) / 60);
        if (minutes) {
            return maybePlural(minutes, 'minute');
        }
        seconds = Math.floor(seconds % 60);
        if (seconds) {
            return maybePlural(seconds, 'second');
        }
        return value + " ms";
    }
    Core.humanizeMilliseconds = humanizeMilliseconds;

    function storeConnectionRegex(regexs, name, json) {
        if (!regexs.any(function (r) {
            r['name'] === name;
        })) {
            var regex = '';

            if (json['useProxy']) {
                regex = '/hawtio/proxy/';
            } else {
                regex = '//';
            }
            regex += json['host'] + ':' + json['port'] + '/' + json['path'];
            regexs.push({
                name: name,
                regex: regex.escapeURL(true),
                color: UI.colors.sample()
            });
            writeRegexs(regexs);
        }
    }
    Core.storeConnectionRegex = storeConnectionRegex;

    function getRegexs() {
        var regexs = [];
        try  {
            regexs = angular.fromJson(localStorage['regexs']);
        } catch (e) {
            // corrupted config
            delete localStorage['regexs'];
        }
        return regexs;
    }
    Core.getRegexs = getRegexs;

    function removeRegex(name) {
        var regexs = Core.getRegexs();
        var hasFunc = function (r) {
            return r['name'] === name;
        };
        if (regexs.any(hasFunc)) {
            regexs = regexs.exclude(hasFunc);
            Core.writeRegexs(regexs);
        }
    }
    Core.removeRegex = removeRegex;

    function writeRegexs(regexs) {
        localStorage['regexs'] = angular.toJson(regexs);
    }
    Core.writeRegexs = writeRegexs;

    function maskPassword(value) {
        if (value) {
            var text = value.toString();

            // we use the same patterns as in Apache Camel in its
            // org.apache.camel.util.URISupport.sanitizeUri
            var userInfoPattern = "(.*://.*:)(.*)(@)";
            value = value.replace(new RegExp(userInfoPattern, 'i'), "$1xxxxxx$3");
        }

        return value;
    }
    Core.maskPassword = maskPassword;

    /**
    * Match the given filter against the text, ignoring any case.
    * <p/>
    * This operation will regard as a match if either filter or text is null/undefined.
    * As its used for filtering out, unmatched.
    * <p/>
    *
    * @param text   the text
    * @param filter the filter
    * @return true if matched, false if not.
    */
    function matchFilterIgnoreCase(text, filter) {
        if (angular.isUndefined(text) || angular.isUndefined(filter)) {
            return true;
        }

        text = text.toString().trim().toLowerCase();
        filter = filter.toString().trim().toLowerCase();

        if (text.length === 0 || filter.length === 0) {
            return true;
        }

        return text.indexOf(filter) > -1;
    }
    Core.matchFilterIgnoreCase = matchFilterIgnoreCase;
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    /**
    * Controller that's attached to hawtio's drop-down console, mainly handles the
    * clipboard icon at the bottom-right of the console.
    *
    * @method ConsoleController
    * @for Core
    * @static
    * @param {*} $scope
    * @param {*} $element
    * @param {*} $templateCache
    */
    function ConsoleController($scope, $element, $templateCache) {
        $scope.setHandler = function (clip) {
            clip.addEventListener('mouseDown', function (client, args) {
                // this is apparently a global event handler for zero clipboard
                // so you have to make sure you're handling the right click event
                var icon = $element.find('.icon-copy');
                var icon2 = $element.find('.icon-trash');
                if (this !== icon.get(0) && this !== icon2.get(0)) {
                    return;
                }

                if (this == icon.get(0)) {
                    copyToClipboard();
                } else {
                    clearLogs();
                    Core.notification('info', "Cleared logging console23");
                }
                Core.$apply($scope);
            });

            function copyToClipboard() {
                var text = $templateCache.get("logClipboardTemplate").lines();
                text.removeAt(0);
                text.removeAt(text.length - 1);
                $element.find('#log-panel-statements').children().each(function (index, child) {
                    text.push('  <li>' + child.innerHTML + '</li>');
                });
                text.push('</ul>');
                clip.setText(text.join('\n'));
            }

            function clearLogs() {
                $element.find('#log-panel-statements').children().remove();
            }
        };
    }
    Core.ConsoleController = ConsoleController;

    /**
    * Outermost controller attached to almost the root of the document, handles
    * logging in and logging out, the PID/container indicator at the bottom right
    * of the window and the document title
    *
    * @method AppController
    * @for Core
    * @static
    * @param {*} $scope
    * @param {ng.ILocationService} $location
    * @param {Core.Workspace} workspace
    * @param {*} jolokiaStatus
    * @param {*} $document
    * @param {Core.PageTitle} pageTitle
    * @param {*} localStorage
    * @param {*} userDetails
    * @param {*} lastLocation
    * @param {*} jolokiaUrl
    * @param {*} branding
    */
    function AppController($scope, $location, workspace, jolokia, jolokiaStatus, $document, pageTitle, localStorage, userDetails, lastLocation, jolokiaUrl, branding) {
        if (!userDetails) {
            userDetails = {};
        }
        if (userDetails.username === null) {
            $location.url(defaultPage());
        }

        $scope.collapse = '';
        $scope.match = null;
        $scope.pageTitle = [];
        $scope.userDetails = userDetails;

        $scope.confirmLogout = false;
        $scope.connectionFailed = false;
        $scope.connectFailure = {};

        $scope.showPrefs = false;

        $scope.logoClass = function () {
            if (branding.logoOnly) {
                return "without-text";
            } else {
                return "with-text";
            }
        };

        setTimeout(function () {
            if ('showPrefs' in localStorage) {
                $scope.showPrefs = Core.parseBooleanValue(localStorage['showPrefs']);
                Core.$apply($scope);
            }
        }, 500);

        $scope.branding = branding;

        $scope.$watch('showPrefs', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                localStorage['showPrefs'] = newValue;
            }
        });

        $scope.hasMBeans = function () {
            return workspace.hasMBeans();
        };

        $scope.$watch('jolokiaStatus.xhr', function () {
            var failure = jolokiaStatus.xhr;
            $scope.connectionFailed = failure ? true : false;
            $scope.connectFailure.summaryMessage = null;
            if ($scope.connectionFailed) {
                $scope.connectFailure.status = failure.status;
                $scope.connectFailure.statusText = failure.statusText;
                var text = failure.responseText;
                if (text) {
                    try  {
                        var html = $(text);
                        var markup = html.find("body");
                        if (markup && markup.length) {
                            html = markup;
                        }

                        // lets tone down the size of the headers
                        html.each(function (idx, e) {
                            var name = e.localName;
                            if (name && name.startsWith("h")) {
                                $(e).addClass("ajaxError");
                            }
                        });
                        var container = $("<div></div>");
                        container.append(html);
                        $scope.connectFailure.summaryMessage = container.html();
                        console.log("Found HTML: " + $scope.connectFailure.summaryMessage);
                    } catch (e) {
                        if (text.indexOf('<') < 0) {
                            // lets assume its not html markup if it doesn't have a tag in it
                            $scope.connectFailure.summaryMessage = "<p>" + text + "</p>";
                        }
                    }
                }
            }
        });

        $scope.showPreferences = function () {
            $scope.showPrefs = true;
        };

        $scope.closePreferences = function () {
            $scope.showPrefs = false;
        };

        $scope.confirmConnectionFailed = function () {
            // I guess we should close the window now?
            window.close();
        };

        $scope.setPageTitle = function () {
            $scope.pageTitle = pageTitle.getTitleArrayExcluding([branding.appName]);
            var tab = workspace.getActiveTab();
            if (tab && tab.content) {
                Core.setPageTitleWithTab($document, pageTitle, tab.content);
            } else {
                Core.setPageTitle($document, pageTitle);
            }
        };

        $scope.setRegexIndicator = function () {
            try  {
                var regexs = angular.fromJson(localStorage['regexs']);
                if (regexs) {
                    regexs.reverse().each(function (regex) {
                        var r = new RegExp(regex.regex, 'g');
                        if (r.test($location.absUrl())) {
                            $scope.match = {
                                name: regex.name,
                                color: regex.color
                            };
                        }
                    });
                }
            } catch (e) {
                // ignore
            }
        };

        $scope.loggedIn = function () {
            return userDetails.username !== null && userDetails.username !== 'public';
        };

        $scope.showLogout = function () {
            return $scope.loggedIn() && angular.isDefined(userDetails.loginDetails);
        };

        $scope.logout = function () {
            $scope.confirmLogout = true;
        };

        $scope.getUsername = function () {
            if (userDetails.username && !userDetails.username.isBlank()) {
                return userDetails.username;
            } else {
                return 'user';
            }
        };

        $scope.doLogout = function () {
            $scope.confirmLogout = false;
            Core.logout(jolokiaUrl, userDetails, localStorage, $scope);
        };

        $scope.$watch(function () {
            return localStorage['regexs'];
        }, $scope.setRegexIndicator);

        $scope.maybeRedirect = function () {
            if (userDetails.username === null) {
                var currentUrl = $location.url();
                if (!currentUrl.startsWith('/login')) {
                    lastLocation.url = currentUrl;
                    $location.url('/login');
                }
            } else {
                if ($location.url().startsWith('/login')) {
                    var url = defaultPage();
                    if (angular.isDefined(lastLocation.url)) {
                        url = lastLocation.url;
                    }
                    $location.url(url);
                }
            }
        };

        $scope.$watch('userDetails', function (newValue, oldValue) {
            $scope.maybeRedirect();
        }, true);

        $scope.$on('hawtioOpenPrefs', function () {
            $scope.showPrefs = true;
        });

        $scope.$on('hawtioClosePrefs', function () {
            $scope.showPrefs = false;
        });

        $scope.$on('$routeChangeStart', function (event, args) {
            if ((!args.params || !args.params.pref) && $scope.showPrefs) {
                $scope.showPrefs = false;
            }
            $scope.maybeRedirect();
        });

        $scope.$on('$routeChangeSuccess', function () {
            $scope.setPageTitle();
            $scope.setRegexIndicator();
        });

        $scope.fullScreen = function () {
            if ($location.url().startsWith("/login")) {
                return branding.fullscreenLogin;
            }
            var tab = $location.search()['tab'];
            if (tab) {
                return tab === "fullscreen";
            }
            return false;
        };

        $scope.login = function () {
            return $location.url().startsWith("/login");
        };

        function defaultPage() {
            return Perspective.defaultPage($location, workspace, jolokia, localStorage);
        }
    }
    Core.AppController = AppController;
})(Core || (Core = {}));
var Core;
(function (Core) {
    var TasksImpl = (function () {
        function TasksImpl() {
            this.tasks = {};
            this.tasksExecuted = false;
            this._onComplete = null;
        }
        TasksImpl.prototype.addTask = function (name, task) {
            this.tasks[name] = task;
            if (this.tasksExecuted) {
                this.executeTask(name, task);
            }
        };

        TasksImpl.prototype.executeTask = function (name, task) {
            if (angular.isFunction(task)) {
                Core.log.debug("Executing task : ", name);
                try  {
                    task();
                } catch (error) {
                    Core.log.debug("Failed to execute task: ", name, " error: ", error);
                }
            }
        };

        TasksImpl.prototype.onComplete = function (cb) {
            this._onComplete = cb;
        };

        TasksImpl.prototype.execute = function () {
            var _this = this;
            if (this.tasksExecuted) {
                return;
            }
            angular.forEach(this.tasks, function (task, name) {
                _this.executeTask(name, task);
            });
            this.tasksExecuted = true;
            if (angular.isFunction(this._onComplete)) {
                this._onComplete();
            }
        };

        TasksImpl.prototype.reset = function () {
            this.tasksExecuted = false;
        };
        return TasksImpl;
    })();
    Core.TasksImpl = TasksImpl;

    Core.postLoginTasks = new Core.TasksImpl();
    Core.preLogoutTasks = new Core.TasksImpl();
})(Core || (Core = {}));
/**
* @module Core
*/
var Core;
(function (Core) {
    function ResetPreferences($scope, userDetails, jolokiaUrl, localStorage) {
        $scope.doReset = function () {
            Core.log.info("Resetting");

            var doReset = function () {
                localStorage.clear();
                setTimeout(function () {
                    window.location.reload();
                }, 10);
            };
            if (Core.isBlank(userDetails.username) && Core.isBlank(userDetails.password)) {
                doReset();
            } else {
                Core.logout(jolokiaUrl, userDetails, localStorage, $scope, doReset);
            }
        };
    }
    Core.ResetPreferences = ResetPreferences;
})(Core || (Core = {}));
var Forms;
(function (Forms) {
    var ResetForm = (function () {
        function ResetForm() {
            var _this = this;
            this.restrict = 'A';
            this.scope = true;
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        ResetForm.prototype.doLink = function (scope, element, attrs) {
            var el = $(element);

            var target = 'form[name=' + attrs['hawtioReset'] + ']';

            el.click(function () {
                var forms = $(target);
                for (var i = 0; i < forms.length; i++) {
                    forms[i].reset();
                }
                return false;
            });
        };
        return ResetForm;
    })();
    Forms.ResetForm = ResetForm;
})(Forms || (Forms = {}));
var Forms;
(function (Forms) {
    Forms.pluginName = 'hawtio-forms';
    var log = Logger.get("Forms");

    angular.module(Forms.pluginName, ['bootstrap', 'ngResource', 'hawtioCore', 'datatable', 'ui.bootstrap', 'ui.bootstrap.dialog', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/forms/test', { templateUrl: 'app/forms/html/test.html' }).when('/forms/testTable', { templateUrl: 'app/forms/html/testTable.html' });
    }).directive('simpleForm', function (workspace, $compile) {
        return new Forms.SimpleForm(workspace, $compile);
    }).directive('hawtioInputTable', function (workspace, $compile) {
        return new Forms.InputTable(workspace, $compile);
    }).directive('hawtioFormText', function (workspace, $compile) {
        return new Forms.TextInput(workspace, $compile);
    }).directive('hawtioFormPassword', function (workspace, $compile) {
        return new Forms.PasswordInput(workspace, $compile);
    }).directive('hawtioFormHidden', function (workspace, $compile) {
        return new Forms.HiddenText(workspace, $compile);
    }).directive('hawtioFormNumber', function (workspace, $compile) {
        return new Forms.NumberInput(workspace, $compile);
    }).directive('hawtioFormSelect', function (workspace, $compile) {
        return new Forms.SelectInput(workspace, $compile);
    }).directive('hawtioFormArray', function (workspace, $compile) {
        return new Forms.ArrayInput(workspace, $compile);
    }).directive('hawtioFormStringArray', function (workspace, $compile) {
        return new Forms.StringArrayInput(workspace, $compile);
    }).directive('hawtioFormCheckbox', function (workspace, $compile) {
        return new Forms.BooleanInput(workspace, $compile);
    }).directive('hawtioFormCustom', function (workspace, $compile) {
        return new Forms.CustomInput(workspace, $compile);
    }).directive('hawtioSubmit', function () {
        return new Forms.SubmitForm();
    }).directive('hawtioReset', function () {
        return new Forms.ResetForm();
    }).run(function (helpRegistry) {
        helpRegistry.addDevDoc("forms", 'app/forms/doc/developer.md');
    });

    hawtioPluginLoader.addModule(Forms.pluginName);
})(Forms || (Forms = {}));
var Forms;
(function (Forms) {
    var SubmitForm = (function () {
        function SubmitForm() {
            var _this = this;
            this.restrict = 'A';
            this.scope = true;
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        SubmitForm.prototype.doLink = function (scope, element, attrs) {
            var el = $(element);

            var target = 'form[name=' + attrs['hawtioSubmit'] + ']';

            el.click(function () {
                $(target).submit();
                return false;
            });
        };
        return SubmitForm;
    })();
    Forms.SubmitForm = SubmitForm;
})(Forms || (Forms = {}));
/**
* @module Forms
*/
var Forms;
(function (Forms) {
    /**
    * @class InputBaseConfig
    */
    var InputBaseConfig = (function () {
        function InputBaseConfig() {
            this.name = 'input';
            this.type = '';
            this.description = '';
            this._default = '';
            this.scope = null;
            // Can also be 'view'
            this.mode = 'edit';
            // the name of the full schema
            this.schemaName = "schema";
            this.controlgroupclass = 'control-group';
            this.controlclass = 'controls';
            this.labelclass = 'control-label';
            this.showtypes = 'false';
            /**
            * Custom template for custom form controls
            * @property
            * @type String
            */
            this.formtemplate = null;
            /**
            * the name of the attribute in the scope which is the data to be edited
            * @property
            * @type String
            */
            this.entity = 'entity';
            /**
            * the model expression to bind to. If omitted this defaults to entity + "." + name
            * @property
            * @type String
            */
            this.model = undefined;
        }
        InputBaseConfig.prototype.getEntity = function () {
            return this.entity || "entity";
        };

        InputBaseConfig.prototype.getMode = function () {
            return this.mode || "edit";
        };

        InputBaseConfig.prototype.isReadOnly = function () {
            return this.getMode() === "view";
        };
        return InputBaseConfig;
    })();
    Forms.InputBaseConfig = InputBaseConfig;

    var InputBase = (function () {
        function InputBase(workspace, $compile) {
            var _this = this;
            this.workspace = workspace;
            this.$compile = $compile;
            this.restrict = 'A';
            this.scope = true;
            this.replace = false;
            this.transclude = false;
            this.attributeName = '';
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        InputBase.prototype.doLink = function (scope, element, attrs) {
            var config = new InputBaseConfig;
            config = Forms.configure(config, null, attrs);
            config.scope = scope;
            config.schemaName = attrs["schema"] || "schema";

            var id = Forms.safeIdentifier(config.name);
            var group = this.getControlGroup(config, config, id);

            var modelName = config.model;
            if (!angular.isDefined(modelName)) {
                // TODO always use 2 way binding?
                modelName = config.getEntity() + "." + id;
            }

            // allow the prefix to be trimmed from the label
            var defaultLabel = id;
            if ("true" === attrs["ignorePrefixInLabel"]) {
                var idx = id.lastIndexOf('.');
                if (idx > 0) {
                    defaultLabel = id.substring(idx + 1);
                }
            }
            group.append(Forms.getLabel(config, config, attrs["title"] || humanizeValue(defaultLabel)));
            var controlDiv = Forms.getControlDiv(config);
            controlDiv.append(this.getInput(config, config, id, modelName));
            controlDiv.append(Forms.getHelpSpan(config, config, id));
            group.append(controlDiv);
            $(element).append(this.$compile(group)(scope));

            if (scope && modelName) {
                scope.$watch(modelName, onModelChange);
            }
            function onModelChange(newValue) {
                scope.$emit("hawtio.form.modelChange", modelName, newValue);
            }
        };

        InputBase.prototype.getControlGroup = function (config1, config2, id) {
            return Forms.getControlGroup(config1, config2, id);
        };

        InputBase.prototype.getInput = function (config, arg, id, modelName) {
            var rc = $('<span class="form-data"></span>');
            if (modelName) {
                rc.attr('ng-model', modelName);
                rc.append('{{' + modelName + '}}');
            }
            return rc;
        };
        return InputBase;
    })();
    Forms.InputBase = InputBase;

    var TextInput = (function (_super) {
        __extends(TextInput, _super);
        function TextInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
            this.type = "text";
        }
        /*public getControlGroup(config1, config2, id) {
        return super.getControlGroup(config1, config2, id);
        }*/
        TextInput.prototype.getInput = function (config, arg, id, modelName) {
            if (config.isReadOnly()) {
                return _super.prototype.getInput.call(this, config, arg, id, modelName);
            }
            var rc = $('<input type="' + this.type + '">');
            rc.attr('name', id);
            if (modelName) {
                rc.attr('ng-model', modelName);
            }
            if (config.isReadOnly()) {
                rc.attr('readonly', 'true');
            }
            var required = config.$attr["required"];
            if (required && required !== "false") {
                rc.attr('required', 'true');
            }
            return rc;
        };
        return TextInput;
    })(InputBase);
    Forms.TextInput = TextInput;

    var HiddenText = (function (_super) {
        __extends(HiddenText, _super);
        function HiddenText(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
            this.type = "hidden";
        }
        HiddenText.prototype.getControlGroup = function (config1, config2, id) {
            var group = _super.prototype.getControlGroup.call(this, config1, config2, id);
            group.css({ 'display': 'none' });
            return group;
        };

        HiddenText.prototype.getInput = function (config, arg, id, modelName) {
            var rc = _super.prototype.getInput.call(this, config, arg, id, modelName);
            rc.attr('readonly', 'true');
            return rc;
        };
        return HiddenText;
    })(TextInput);
    Forms.HiddenText = HiddenText;

    var PasswordInput = (function (_super) {
        __extends(PasswordInput, _super);
        function PasswordInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
            this.type = "password";
        }
        return PasswordInput;
    })(TextInput);
    Forms.PasswordInput = PasswordInput;

    var CustomInput = (function (_super) {
        __extends(CustomInput, _super);
        function CustomInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        CustomInput.prototype.getInput = function (config, arg, id, modelName) {
            var template = arg.formtemplate;
            template = Core.unescapeHtml(template);
            var rc = $(template);
            if (!rc.attr("name")) {
                rc.attr('name', id);
            }
            if (modelName) {
                rc.attr('ng-model', modelName);
            }
            if (config.isReadOnly()) {
                rc.attr('readonly', 'true');
            }
            return rc;
        };
        return CustomInput;
    })(InputBase);
    Forms.CustomInput = CustomInput;

    var SelectInput = (function (_super) {
        __extends(SelectInput, _super);
        function SelectInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        SelectInput.prototype.getInput = function (config, arg, id, modelName) {
            if (config.isReadOnly()) {
                return _super.prototype.getInput.call(this, config, arg, id, modelName);
            }

            // TODO calculate from input attributes...
            var required = true;

            // TODO we could configure the null option...
            var defaultOption = required ? "" : '<option value=""></option>';
            var rc = $('<select>' + defaultOption + '</select>');
            rc.attr('name', id);

            var scope = config.scope;
            var data = config.data;
            if (data && scope) {
                // this is a big ugly - would be nice to expose this a bit easier...
                // maybe nested objects should expose the model easily...
                var fullSchema = scope[config.schemaName];
                var model = scope[data];

                // now we need to keep walking the model to find the enum values
                var paths = id.split(".");
                var property = null;
                angular.forEach(paths, function (path) {
                    property = Core.pathGet(model, ["properties", path]);
                    var typeName = Core.pathGet(property, ["type"]);
                    var alias = Forms.lookupDefinition(typeName, fullSchema);
                    if (alias) {
                        model = alias;
                    }
                });
                var values = Core.pathGet(property, ["enum"]);
                scope["$selectValues"] = values;
                rc.attr("ng-options", "value for value in $selectValues");
            }
            if (modelName) {
                rc.attr('ng-model', modelName);
            }
            if (config.isReadOnly()) {
                rc.attr('readonly', 'true');
            }
            return rc;
        };
        return SelectInput;
    })(InputBase);
    Forms.SelectInput = SelectInput;

    var NumberInput = (function (_super) {
        __extends(NumberInput, _super);
        function NumberInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        NumberInput.prototype.getInput = function (config, arg, id, modelName) {
            if (config.isReadOnly()) {
                return _super.prototype.getInput.call(this, config, arg, id, modelName);
            }
            var rc = $('<input type="number">');
            rc.attr('name', id);

            if (angular.isDefined(arg.def)) {
                rc.attr('value', arg.def);
            }

            if (angular.isDefined(arg.minimum)) {
                rc.attr('min', arg.minimum);
            }

            if (angular.isDefined(arg.maximum)) {
                rc.attr('max', arg.maximum);
            }

            if (modelName) {
                rc.attr('ng-model', modelName);
            }
            if (config.isReadOnly()) {
                rc.attr('readonly', 'true');
            }

            // lets coerce any string values to numbers so that they work properly with the UI
            var scope = config.scope;
            if (scope) {
                function onModelChange() {
                    var value = Core.pathGet(scope, modelName);
                    if (value && angular.isString(value)) {
                        var numberValue = Number(value);
                        Core.pathSet(scope, modelName, numberValue);
                    }
                }
                scope.$watch(modelName, onModelChange);
                onModelChange();
            }
            return rc;
        };
        return NumberInput;
    })(InputBase);
    Forms.NumberInput = NumberInput;

    /**
    * Generates a list of strings which can be added / edited / removed
    * @class StringArrayInput
    */
    var StringArrayInput = (function (_super) {
        __extends(StringArrayInput, _super);
        function StringArrayInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        StringArrayInput.prototype.getInput = function (config, arg, id, modelName) {
            var rowScopeName = "_" + id;
            var ngRepeat = rowScopeName + ' in ' + modelName;

            var readOnlyWidget = '{{' + rowScopeName + '}}';
            if (config.isReadOnly()) {
                return $('<ul><li ng-repeat="' + rowScopeName + ' in ' + modelName + '">' + readOnlyWidget + '</li></ul>');
            } else {
                // TODO there should be an easier way to find the property / schema!
                var scope = config.scope;
                var fallbackSchemaName = (arg.$attr || {})["schema"] || "schema";
                var schema = scope[config.schemaName] || scope[fallbackSchemaName] || {};
                var properties = schema.properties || {};
                var arrayProperty = properties[id] || {};

                // lets refer to the property of the item, rather than the array
                var property = arrayProperty["items"] || {};
                var propTypeName = property.type;
                var ignorePrefixInLabel = true;
                var configScopeName = null;

                // lets create an empty array if its not yet set
                var value = Core.pathGet(scope, modelName);
                if (!value) {
                    Core.pathSet(scope, modelName, []);
                }

                var methodPrefix = "_form_stringArray" + rowScopeName + "_";
                var itemKeys = methodPrefix + "keys";
                var addMethod = methodPrefix + "add";
                var removeMethod = methodPrefix + "remove";

                // we maintain a separate object of all the keys (indices) of the array
                // and use that to lookup the values
                function updateKeys() {
                    var value = Core.pathGet(scope, modelName);
                    scope[itemKeys] = value ? Object.keys(value) : [];
                    scope.$emit("hawtio.form.modelChange", modelName, value);
                }

                updateKeys();

                scope[addMethod] = function () {
                    var value = Core.pathGet(scope, modelName) || [];
                    value.push("");
                    Core.pathSet(scope, modelName, value);
                    updateKeys();
                };
                scope[removeMethod] = function (idx) {
                    var value = Core.pathGet(scope, modelName) || [];
                    if (idx < value.length) {
                        value.splice(idx, 1);
                    }
                    Core.pathSet(scope, modelName, value);
                    updateKeys();
                };

                // the expression for an item value
                var itemId = modelName + "[" + rowScopeName + "]";
                var itemsConfig = {
                    model: itemId
                };
                var widget = Forms.createWidget(propTypeName, property, schema, itemsConfig, itemId, ignorePrefixInLabel, configScopeName, false);
                if (!widget) {
                    widget = $(readOnlyWidget);
                }
                var markup = $('<div style="white-space: nowrap" ng-repeat="' + rowScopeName + ' in ' + itemKeys + '"></div>');
                markup.append(widget);
                markup.append($('<a ng-click="' + removeMethod + '(' + rowScopeName + ')" title="Remove this value"><i class="red icon-remove"></i></a>'));
                markup.after($('<a ng-click="' + addMethod + '()" title="Add a new value"><i class="icon-plus"></i></a>'));
                return markup;
            }
        };
        return StringArrayInput;
    })(InputBase);
    Forms.StringArrayInput = StringArrayInput;

    var ArrayInput = (function (_super) {
        __extends(ArrayInput, _super);
        function ArrayInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        ArrayInput.prototype.doLink = function (scope, element, attrs) {
            var config = new InputBaseConfig;
            config = Forms.configure(config, null, attrs);

            var id = config.name;
            var dataName = attrs["data"] || "";
            var entityName = attrs["entity"] || config.entity;
            var schemaName = attrs["schema"] || config.schemaName;

            function renderRow(cell, type, data) {
                if (data) {
                    var description = data["description"];
                    if (!description) {
                        angular.forEach(data, function (value, key) {
                            if (value && !description) {
                                description = value;
                            }
                        });
                    }
                    return description;
                }
                return null;
            }

            // Had to fudge some of this
            // create a table UI!
            var tableConfigPaths = ["properties", id, "inputTable"];

            //var scope = config.scope;
            var tableConfig = null;
            Core.pathGet(scope, tableConfigPaths);

            // lets auto-create a default configuration if there is none
            if (!tableConfig) {
                // TODO ideally we should merge this config with whatever folks have hand-defined
                var tableConfigScopeName = tableConfigPaths.join(".");

                //var cellDescription = a["description"] || humanizeValue(id);
                var cellDescription = humanizeValue(id);
                tableConfig = {
                    formConfig: config,
                    title: cellDescription,
                    data: config.entity + "." + id,
                    displayFooter: false,
                    showFilter: false,
                    columnDefs: [
                        {
                            field: '_id',
                            displayName: cellDescription,
                            render: renderRow
                        }
                    ]
                };
                Core.pathSet(scope, tableConfigPaths, tableConfig);
            }
            var table = $('<div hawtio-input-table="' + tableConfigScopeName + '" data="' + dataName + '" property="' + id + '" entity="' + entityName + '" schema="' + schemaName + '"></div>');
            if (config.isReadOnly()) {
                table.attr("readonly", "true");
            }
            $(element).append(this.$compile(table)(scope));
        };
        return ArrayInput;
    })(InputBase);
    Forms.ArrayInput = ArrayInput;

    var BooleanInput = (function (_super) {
        __extends(BooleanInput, _super);
        function BooleanInput(workspace, $compile) {
            _super.call(this, workspace, $compile);
            this.workspace = workspace;
            this.$compile = $compile;
        }
        BooleanInput.prototype.getInput = function (config, arg, id, modelName) {
            var rc = $('<input class="hawtio-checkbox" type="checkbox">');
            rc.attr('name', id);

            if (config.isReadOnly()) {
                rc.attr('disabled', 'true');
            }
            if (modelName) {
                rc.attr('ng-model', modelName);
            }
            if (config.isReadOnly()) {
                rc.attr('readonly', 'true');
            }

            // lets coerce any string values to boolean so that they work properly with the UI
            var scope = config.scope;
            if (scope) {
                function onModelChange() {
                    var value = Core.pathGet(scope, modelName);
                    if (value && "true" === value) {
                        //console.log("coercing String to boolean for " + modelName);
                        Core.pathSet(scope, modelName, true);
                    }
                }
                scope.$watch(modelName, onModelChange);
                onModelChange();
            }
            return rc;
        };
        return BooleanInput;
    })(InputBase);
    Forms.BooleanInput = BooleanInput;
})(Forms || (Forms = {}));
/**
* @module Forms
*/
var Forms;
(function (Forms) {
    Forms.log = Logger.get("Forms");

    /**
    * Default any values in the schema on the entity if they are not already present
    * @method defaultValues
    * @param {any} entity
    * @param {any} schema
    */
    function defaultValues(entity, schema) {
        if (entity && schema) {
            angular.forEach(schema.properties, function (property, key) {
                var defaultValue = property.default;
                if (defaultValue && !entity[key]) {
                    console.log("===== defaulting value " + defaultValue + " into entity[" + key + "]");
                    entity[key] = defaultValue;
                }
            });
        }
    }
    Forms.defaultValues = defaultValues;

    /**
    * If the type name refers to an alias in the schemas definitions then perform the lookup and return the real type name
    * @method resolveTypeNAmeAlias
    * @param {String} type
    * @param {any} schema
    *
    */
    function resolveTypeNameAlias(type, schema) {
        if (type && schema) {
            var alias = lookupDefinition(type, schema);
            if (alias) {
                var realType = alias["type"];
                if (realType) {
                    type = realType;
                }
            }
        }
        return type;
    }
    Forms.resolveTypeNameAlias = resolveTypeNameAlias;

    /**
    * Walks the base class hierarchy checking if the given type is an instance of the given type name
    * @method isJsonType
    * @param {String} name
    * @param {any} schema
    * @param {String} typeName
    * @return {Boolean}
    */
    function isJsonType(name, schema, typeName) {
        var definition = lookupDefinition(name, schema);
        while (definition) {
            var extendsTypes = Core.pathGet(definition, ["extends", "type"]);
            if (extendsTypes) {
                if (typeName === extendsTypes) {
                    return true;
                } else {
                    definition = lookupDefinition(extendsTypes, schema);
                }
            } else {
                return false;
            }
        }
        return false;
    }
    Forms.isJsonType = isJsonType;

    /**
    * Removes any dodgy characters for a valid identifier in angularjs such as for '-' characters
    * which are replaced with '_'
    * @method safeIdentifier
    * @param {String} id
    * @return {String}
    */
    function safeIdentifier(id) {
        if (id) {
            return id.replace(/-/g, "_");
        }
        return id;
    }
    Forms.safeIdentifier = safeIdentifier;

    /**
    * Looks up the given type name in the schemas definitions
    * @method lookupDefinition
    * @param {String} name
    * @param {any} schema
    */
    function lookupDefinition(name, schema) {
        if (schema) {
            var defs = schema.definitions;
            if (defs) {
                var answer = defs[name];
                if (answer) {
                    var fullSchema = answer["fullSchema"];
                    if (fullSchema) {
                        return fullSchema;
                    }

                    // we may extend another, if so we need to copy in the base properties
                    var extendsTypes = Core.pathGet(answer, ["extends", "type"]);
                    if (extendsTypes) {
                        fullSchema = angular.copy(answer);
                        fullSchema.properties = fullSchema.properties || {};
                        if (!angular.isArray(extendsTypes)) {
                            extendsTypes = [extendsTypes];
                        }
                        angular.forEach(extendsTypes, function (extendType) {
                            if (angular.isString(extendType)) {
                                var extendDef = lookupDefinition(extendType, schema);
                                var properties = Core.pathGet(extendDef, ["properties"]);
                                if (properties) {
                                    angular.forEach(properties, function (property, key) {
                                        fullSchema.properties[key] = property;
                                    });
                                }
                            }
                        });
                        answer["fullSchema"] = fullSchema;
                        return fullSchema;
                    }
                }
                return answer;
            }
        }
        return null;
    }
    Forms.lookupDefinition = lookupDefinition;

    /**
    * For an array property, find the schema of the items which is either nested inside this property
    * in the 'items' property; or the type name is used to lookup in the schemas definitions
    * @method findArrayItemsSchema
    * @param {String} property
    * @param {any} schema
    */
    function findArrayItemsSchema(property, schema) {
        var items = null;
        if (property && schema) {
            items = property.items;
            if (items) {
                var typeName = items["type"];
                if (typeName) {
                    var definition = lookupDefinition(typeName, schema);
                    if (definition) {
                        return definition;
                    }
                }
            }

            // are we a json schema properties with a link to the schema doc?
            var additionalProperties = property.additionalProperties;
            if (additionalProperties) {
                if (additionalProperties["$ref"] === "#") {
                    return schema;
                }
            }
        }
        return items;
    }
    Forms.findArrayItemsSchema = findArrayItemsSchema;

    /**
    * Returns true if the given schema definition is an object
    * @method isObjectType
    * @param {any} definition
    */
    function isObjectType(definition) {
        var typeName = Core.pathGet(definition, "type");
        return typeName && "object" === typeName;
    }
    Forms.isObjectType = isObjectType;

    /**
    * Returns true if the given property represents a nested object or array of objects
    * @method isArrayOrNestedObject
    * @param {any} property
    * @param {any} schema
    */
    function isArrayOrNestedObject(property, schema) {
        if (property) {
            var propType = resolveTypeNameAlias(property["type"], schema);
            if (propType) {
                if (propType === "object" || propType === "array") {
                    return true;
                }
            }
        }
        return false;
    }
    Forms.isArrayOrNestedObject = isArrayOrNestedObject;

    function configure(config, scopeConfig, attrs) {
        if (angular.isDefined(scopeConfig)) {
            config = angular.extend(config, scopeConfig);
        }
        return angular.extend(config, attrs);
    }
    Forms.configure = configure;

    function getControlGroup(config, arg, id) {
        var rc = $('<div class="' + config.controlgroupclass + '"></div>');
        if (angular.isDefined(arg.description)) {
            rc.attr('title', arg.description);
        }

        // log.debug("getControlGroup, config:", config, " arg: ", arg, " id: ", id);
        if (config['properties'] && config['properties'][id]) {
            var elementConfig = config['properties'][id];

            // log.debug("elementConfig: ", elementConfig);
            if (elementConfig && 'control-attributes' in elementConfig) {
                angular.forEach(elementConfig['control-attributes'], function (value, key) {
                    rc.attr(key, value);
                });
            }
        }

        return rc;
    }
    Forms.getControlGroup = getControlGroup;

    function getLabel(config, arg, label) {
        return $('<label class="' + config.labelclass + '">' + label + ': </label>');
    }
    Forms.getLabel = getLabel;

    function getControlDiv(config) {
        return $('<div class="' + config.controlclass + '"></div>');
    }
    Forms.getControlDiv = getControlDiv;

    function getHelpSpan(config, arg, id) {
        var help = Core.pathGet(config.data, ['properties', id, 'help']);
        if (!Core.isBlank(help)) {
            return $('<span class="help-block">' + help + '</span>');
        } else {
            return $('<span class="help-block"></span>');
        }
    }
    Forms.getHelpSpan = getHelpSpan;
})(Forms || (Forms = {}));
var Forms;
(function (Forms) {
    var SimpleFormConfig = (function () {
        function SimpleFormConfig() {
            this.name = 'form';
            this.method = 'post';
            // the name of the attribute in the scope which is the data to be editted
            this.entity = 'entity';
            // the name of the full schema
            this.schemaName = 'schema';
            // set to 'view' or 'create' for different modes
            this.mode = 'edit';
            // the definition of the form
            this.data = {};
            this.json = undefined;
            // the scope
            this.scope = null;
            // the name to look up in the scope for the configuration data
            this.scopeName = null;
            this.properties = [];
            this.action = '';
            this.formclass = 'hawtio-form form-horizontal no-bottom-margin';
            this.controlgroupclass = 'control-group';
            this.controlclass = 'controls';
            this.labelclass = 'control-label';
            this.showtypes = 'false';
            this.onsubmit = 'onSubmit';
        }
        SimpleFormConfig.prototype.getMode = function () {
            return this.mode || "edit";
        };

        SimpleFormConfig.prototype.getEntity = function () {
            return this.entity || "entity";
        };

        SimpleFormConfig.prototype.isReadOnly = function () {
            return this.getMode() === "view";
        };
        return SimpleFormConfig;
    })();
    Forms.SimpleFormConfig = SimpleFormConfig;

    var SimpleForm = (function () {
        function SimpleForm(workspace, $compile) {
            var _this = this;
            this.workspace = workspace;
            this.$compile = $compile;
            this.restrict = 'A';
            this.scope = true;
            this.replace = true;
            this.transclude = true;
            this.attributeName = 'simpleForm';
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        SimpleForm.prototype.isReadOnly = function () {
            return false;
        };

        SimpleForm.prototype.doLink = function (scope, element, attrs) {
            var config = new SimpleFormConfig;

            var fullSchemaName = attrs["schema"];
            var fullSchema = fullSchemaName ? scope[fullSchemaName] : null;

            var compiledNode = null;
            var childScope = null;
            var tabs = null;
            var fieldset = null;
            var schema = null;
            var configScopeName = attrs[this.attributeName] || attrs["data"];

            var firstControl = null;

            var simple = this;
            scope.$watch(configScopeName, onWidgetDataChange);

            function onWidgetDataChange(scopeData) {
                if (scopeData) {
                    onScopeData(scopeData);
                }
            }

            function onScopeData(scopeData) {
                config = Forms.configure(config, scopeData, attrs);
                config.schemaName = fullSchemaName;
                config.scopeName = configScopeName;
                config.scope = scope;

                var entityName = config.getEntity();

                if (angular.isDefined(config.json)) {
                    config.data = $.parseJSON(config.json);
                } else {
                    config.data = scopeData;
                }

                var form = simple.createForm(config);
                fieldset = form.find('fieldset');
                schema = config.data;
                tabs = {
                    elements: {},
                    locations: {},
                    use: false
                };

                if (schema && angular.isDefined(schema.tabs)) {
                    tabs.use = true;
                    tabs['div'] = $('<div class="tabbable hawtio-form-tabs"></div>');

                    angular.forEach(schema.tabs, function (value, key) {
                        tabs.elements[key] = $('<div class="tab-pane" title="' + key + '"></div>');
                        tabs['div'].append(tabs.elements[key]);
                        value.forEach(function (val) {
                            tabs.locations[val] = key;
                        });
                    });

                    if (!tabs.locations['*']) {
                        tabs.locations['*'] = Object.extended(schema.tabs).keys()[0];
                    }
                }

                if (!tabs.use) {
                    fieldset.append('<div class="spacer"></div>');
                }

                if (schema) {
                    // if we're using tabs lets reorder the properties...
                    if (tabs.use) {
                        var tabKeyToIdPropObject = {};
                        angular.forEach(schema.properties, function (property, id) {
                            var tabkey = findTabOrderValue(id);
                            var array = tabKeyToIdPropObject[tabkey];
                            if (!array) {
                                array = [];
                                tabKeyToIdPropObject[tabkey] = array;
                            }
                            array.push({ id: id, property: property });
                        });

                        // now lets iterate through each tab...
                        angular.forEach(schema.tabs, function (value, key) {
                            value.forEach(function (val) {
                                var array = tabKeyToIdPropObject[val];
                                if (array) {
                                    angular.forEach(array, function (obj) {
                                        var id = obj.id;
                                        var property = obj.property;
                                        if (id && property) {
                                            addProperty(id, property);
                                        }
                                    });
                                }
                            });
                        });
                    } else {
                        angular.forEach(schema.properties, function (property, id) {
                            addProperty(id, property);
                        });
                    }
                }

                if (tabs.use) {
                    var tabDiv = tabs['div'];
                    var tabCount = Object.keys(tabs.elements).length;
                    if (tabCount < 2) {
                        // if we only have 1 tab lets extract the div contents of the tab
                        angular.forEach(tabDiv.children().children(), function (control) {
                            fieldset.append(control);
                        });
                    } else {
                        fieldset.append(tabDiv);
                    }
                }

                var findFunction = function (scope, func) {
                    if (angular.isDefined(scope[func]) && angular.isFunction(scope[func])) {
                        return scope;
                    }
                    if (angular.isDefined(scope.$parent) && scope.$parent !== null) {
                        return findFunction(scope.$parent, func);
                    } else {
                        return null;
                    }
                };

                var onSubmitFunc = config.onsubmit.replace('(', '').replace(')', '');
                var onSubmit = maybeGet(findFunction(scope, onSubmitFunc), onSubmitFunc);

                if (onSubmit === null) {
                    onSubmit = function (json, form) {
                        notification('error', 'No submit handler defined for form ' + form.get(0).name);
                    };
                }

                if (angular.isDefined(onSubmit)) {
                    form.submit(function () {
                        Forms.log.debug("child scope: ", childScope);
                        Forms.log.debug("form name: ", config);
                        if (childScope[config.name].$invalid) {
                            return false;
                        }
                        var entity = scope[entityName];
                        onSubmit(entity, form);
                        return false;
                    });
                }

                fieldset.append('<input type="submit" style="position: absolute; left: -9999px; width: 1px; height: 1px;">');

                // now lets try default an autofocus element onto the first item if we don't find any elements with an auto-focus
                var autoFocus = form.find("*[autofocus]");
                if (!autoFocus || !autoFocus.length) {
                    if (firstControl) {
                        console.log("No autofocus element, so lets add one!");
                        var input = firstControl.find("input").first() || firstControl.find("select").first();
                        if (input) {
                            input.attr("autofocus", "true");
                        }
                    }
                }

                if (compiledNode) {
                    $(compiledNode).remove();
                }
                if (childScope) {
                    childScope.$destroy();
                }
                childScope = scope.$new(false);

                compiledNode = simple.$compile(form)(childScope);

                // now lets expose the form object to the outer scope
                var formsScopeProperty = "forms";
                var forms = scope[formsScopeProperty];
                if (!forms) {
                    forms = {};
                    scope[formsScopeProperty] = forms;
                }
                var formName = config.name;
                if (formName) {
                    var formObject = childScope[formName];
                    if (formObject) {
                        forms[formName] = formObject;
                    }
                    var formScope = formName += "$scope";
                    forms[formScope] = childScope;
                }
                $(element).append(compiledNode);
            }

            function findTabKey(id) {
                var tabkey = tabs.locations[id];
                if (!tabkey) {
                    // lets try find a tab key using regular expressions
                    angular.forEach(tabs.locations, function (value, key) {
                        if (!tabkey && key !== "*" && id.match(key)) {
                            tabkey = value;
                        }
                    });
                }
                if (!tabkey) {
                    tabkey = tabs.locations['*'];
                }
                return tabkey;
            }

            function findTabOrderValue(id) {
                var answer = null;
                angular.forEach(schema.tabs, function (value, key) {
                    value.forEach(function (val) {
                        if (!answer && val !== "*" && id.match(val)) {
                            answer = val;
                        }
                    });
                });
                if (!answer) {
                    answer = '*';
                }
                return answer;
            }

            function addProperty(id, property, ignorePrefixInLabel) {
                if (typeof ignorePrefixInLabel === "undefined") { ignorePrefixInLabel = property.ignorePrefixInLabel; }
                // TODO should also support getting inputs from the template cache, maybe
                // for type="template"
                var propTypeName = property.type;

                // make sure we detect string as string
                if ("java.lang.String" === propTypeName) {
                    propTypeName = "string";
                }
                var propSchema = Forms.lookupDefinition(propTypeName, schema);
                if (!propSchema) {
                    propSchema = Forms.lookupDefinition(propTypeName, fullSchema);
                }

                // lets ignore fields marked as hidden from the generated form
                if (property.hidden) {
                    return;
                }
                var nestedProperties = null;
                if (!propSchema && "object" === propTypeName && property.properties) {
                    // if we've no type name but have nested properties on an object type use those
                    nestedProperties = property.properties;
                } else if (propSchema && Forms.isObjectType(propSchema)) {
                    // otherwise use the nested properties from the related schema type
                    //console.log("type name " + propTypeName + " has nested object type " + JSON.stringify(propSchema, null, "  "));
                    nestedProperties = propSchema.properties;
                }
                if (nestedProperties) {
                    angular.forEach(nestedProperties, function (childProp, childId) {
                        var newId = id + "." + childId;
                        addProperty(newId, childProp, property.ignorePrefixInLabel);
                    });
                } else {
                    var input = Forms.createWidget(propTypeName, property, schema, config, id, ignorePrefixInLabel, configScopeName);

                    if (tabs.use) {
                        var tabkey = findTabKey(id);
                        tabs.elements[tabkey].append(input);
                    } else {
                        fieldset.append(input);
                    }
                    if (!firstControl) {
                        firstControl = input;
                    }
                }
            }

            function maybeGet(scope, func) {
                if (scope !== null) {
                    return scope[func];
                }
                return null;
            }
        };

        SimpleForm.prototype.createForm = function (config) {
            var form = $('<form class="' + config.formclass + '" novalidate><fieldset></fieldset></form>');
            form.attr('name', config.name);
            form.attr('action', config.action);
            form.attr('method', config.method);
            form.find('fieldset').append(this.getLegend(config));
            return form;
        };

        SimpleForm.prototype.getLegend = function (config) {
            var description = Core.pathGet(config, "data.description");
            if (description) {
                return '<legend>' + description + '</legend>';
            }
            return '';
        };
        return SimpleForm;
    })();
    Forms.SimpleForm = SimpleForm;
})(Forms || (Forms = {}));
var Forms;
(function (Forms) {
    function FormTestController($scope, workspace) {
        $scope.editing = false;

        $scope.html = "text/html";
        $scope.javascript = "javascript";

        $scope.basicFormEx1Entity = {
            'key': 'Some key',
            'value': 'Some value'
        };
        $scope.basicFormEx1EntityString = angular.toJson($scope.basicFormEx1Entity, true);

        $scope.basicFormEx1Result = '';

        $scope.toggleEdit = function () {
            $scope.editing = !$scope.editing;
        };

        $scope.view = function () {
            if (!$scope.editing) {
                return "view";
            }
            return "edit";
        };

        $scope.basicFormEx1 = '<div simple-form name="some-form" action="#/forms/test" method="post" data="basicFormEx1SchemaObject" entity="basicFormEx1Entity" onSubmit="callThis()"></div>';

        $scope.toObject = function (str) {
            return angular.fromJson(str.replace("'", "\""));
        };

        $scope.fromObject = function (str) {
            return angular.toJson($scope[str], true);
        };

        //TODO - I totally did this backwards :-/
        $scope.basicFormEx1Schema = '' + '{\n' + '   "properties": {\n' + '     "key": {\n' + '       "description": "Argument key",\n' + '       "type": "java.lang.String"\n' + '     },\n' + '     "value": {\n' + '       "description": "Argument Value",\n' + '       "type": "java.lang.String"\n' + '     },\n' + '     "longArg": {\n' + '       "description": "Long argument",\n' + '       "type": "Long",\n' + '       "minimum": "5",\n' + '       "maximum": "10"\n' + '     },\n' + '     "intArg": {\n' + '       "description": "Int argument",\n' + '       "type": "Integer"\n' + '     },\n' + '     "objectArg": {\n' + '       "description": "some object",\n' + '       "type": "object"\n' + '     },\n' + '     "booleanArg": {\n' + '       "description": "Some boolean value",\n' + '       "type": "java.lang.Boolean"\n' + '     }\n' + '   },\n' + '   "description": "Show some stuff in a form",\n' + '   "type": "java.lang.String",\n' + '   "tabs": {\n' + '     "Tab One": ["key", "value"],\n' + '     "Tab Two": ["*"],\n' + '     "Tab Three": ["booleanArg"]\n' + '   }\n' + '}';

        $scope.basicFormEx1SchemaObject = $scope.toObject($scope.basicFormEx1Schema);

        $scope.updateSchema = function () {
            $scope.basicFormEx1SchemaObject = $scope.toObject($scope.basicFormEx1Schema);
        };

        $scope.updateEntity = function () {
            $scope.basicFormEx1Entity = angular.fromJson($scope.basicFormEx1EntityString);
        };

        $scope.hawtioResetEx = '<a class="btn" href="" hawtio-reset="some-form"><i class="icon-refresh"></i> Clear</a>';

        $scope.hawtioSubmitEx = '      <a class="btn" href="" hawtio-submit="some-form"><i class="icon-save"></i> Save</a>';

        $scope.callThis = function (json, form) {
            $scope.basicFormEx1Result = angular.toJson(json, true);
            notification('success', 'Form "' + form.get(0).name + '" submitted...');
            Core.$apply($scope);
        };

        $scope.config = {
            name: 'form-with-config-object',
            action: "/some/url",
            method: "post",
            data: 'setVMOption',
            showtypes: 'false'
        };

        $scope.cheese = {
            key: "keyABC",
            value: "valueDEF",
            intArg: 999
        };

        $scope.onCancel = function (form) {
            notification('success', 'Cancel clicked on form "' + form.get(0).name + '"');
        };

        $scope.onSubmit = function (json, form) {
            notification('success', 'Form "' + form.get(0).name + '" submitted... (well not really), data:' + JSON.stringify(json));
        };

        $scope.derp = function (json, form) {
            notification('error', 'derp with json ' + JSON.stringify(json));
        };

        $scope.inputTableSchema = {
            properties: {
                'id': {
                    description: 'Object ID',
                    type: 'java.lang.String'
                }
            },
            description: 'Some objects'
        };

        $scope.inputTableData = [
            { id: "object1", name: 'foo' },
            { id: "object2", name: 'bar' }
        ];

        $scope.inputTableConfig = {
            data: 'inputTableData',
            displayFooter: false,
            showFilter: false,
            columnDefs: [
                {
                    field: 'id',
                    displayName: 'ID'
                },
                {
                    field: 'name',
                    displayName: 'Name'
                }
            ]
        };
    }
    Forms.FormTestController = FormTestController;
})(Forms || (Forms = {}));
/**
* @module Forms
*/
var Forms;
(function (Forms) {
    /**
    * Create a DOM widget tree for the given set of form configuration data.
    *
    * This will include either the standard AngularJS widgets or custom widgets
    * @method createWidget
    * @param {String} propTypeName
    * @param {any} property
    * @param {any} schema
    * @param {any} config
    * @param {String} id
    * @param {Boolean ignorePrefixInLabel
    * @param {String} configScoepName
    * @param {Boolean} wrapInGroup
    */
    function createWidget(propTypeName, property, schema, config, id, ignorePrefixInLabel, configScopeName, wrapInGroup) {
        if (typeof wrapInGroup === "undefined") { wrapInGroup = true; }
        var input = null;
        var group = null;

        function copyElementAttributes(element, propertyName) {
            var propertyAttributes = property[propertyName];
            if (propertyAttributes) {
                angular.forEach(propertyAttributes, function (value, key) {
                    if (angular.isString(value)) {
                        element.attr(key, value);
                    }
                });
            }
        }
        function copyAttributes() {
            copyElementAttributes(input, "input-attributes");
            angular.forEach(property, function (value, key) {
                if (angular.isString(value) && key.indexOf("$") < 0 && key !== "type") {
                    var html = Core.escapeHtml(value);
                    input.attr(key, html);
                }
            });
        }

        // lets try to create standard widget markup by default
        // as they work better than the hawtio wrappers when inside forms...
        var options = {
            valueConverter: null
        };
        var safeId = Forms.safeIdentifier(id);

        var inputMarkup = createStandardWidgetMarkup(propTypeName, property, schema, config, options, safeId);

        // Note if for whatever reason we need to go back to the old way of using hawtio directives for standard
        // angularjs directives, just clear inputMarker to null here ;)
        // inputMarkup = null;
        if (inputMarkup) {
            input = $(inputMarkup);

            copyAttributes();

            id = safeId;

            var modelName = config.model || Core.pathGet(property, ["input-attributes", "ng-model"]);
            if (!modelName) {
                modelName = config.getEntity() + "." + id;
            }
            input.attr("ng-model", modelName);

            input.attr('name', id);

            try  {
                if (config.isReadOnly()) {
                    input.attr('readonly', 'true');
                }
            } catch (e) {
                // ignore missing read only function
            }
            var title = property.tooltip || property.label;
            if (title) {
                input.attr('title', title);
            }

            // allow the prefix to be trimmed from the label if enabled
            var defaultLabel = id;
            if (ignorePrefixInLabel || property.ignorePrefixInLabel) {
                var idx = id.lastIndexOf('.');
                if (idx > 0) {
                    defaultLabel = id.substring(idx + 1);
                }
            }

            // figure out which things to not wrap in a group and label etc...
            if (input.attr("type") !== "hidden" && wrapInGroup) {
                group = this.getControlGroup(config, config, id);
                var labelElement = Forms.getLabel(config, config, property.title || property.label || humanizeValue(defaultLabel));
                if (title) {
                    labelElement.attr('title', title);
                }
                group.append(labelElement);
                copyElementAttributes(labelElement, "label-attributes");

                var controlDiv = Forms.getControlDiv(config);
                controlDiv.append(input);
                controlDiv.append(Forms.getHelpSpan(config, config, id));

                group.append(controlDiv);

                // allow control level directives, such as ng-show / ng-hide
                copyElementAttributes(controlDiv, "control-attributes");
                copyElementAttributes(group, "control-group-attributes");

                var scope = config.scope;
                if (scope && modelName) {
                    var onModelChange = function (newValue) {
                        scope.$emit("hawtio.form.modelChange", modelName, newValue);
                    };
                    var fn = onModelChange;

                    // allow custom converters
                    var converterFn = options.valueConverter;
                    if (converterFn) {
                        fn = function () {
                            converterFn(scope, modelName);
                            var newValue = Core.pathGet(scope, modelName);
                            onModelChange(newValue);
                        };
                    }
                    scope.$watch(modelName, fn);
                }
            }
        } else {
            input = $('<div></div>');
            input.attr(Forms.normalize(propTypeName, property, schema), '');

            copyAttributes();

            input.attr('entity', config.getEntity());
            input.attr('mode', config.getMode());

            var fullSchemaName = config.schemaName;
            if (fullSchemaName) {
                input.attr('schema', fullSchemaName);
            }

            if (configScopeName) {
                input.attr('data', configScopeName);
            }

            if (ignorePrefixInLabel || property.ignorePrefixInLabel) {
                input.attr('ignore-prefix-in-label', true);
            }
            input.attr('name', id);
        }

        var label = property.label;
        if (label) {
            input.attr('title', label);
        }

        // TODO check for id in the schema["required"] array too!
        // as required can be specified either via either of these approaches
        /*
        var schema = {
        required: ["foo", "bar"],
        properties: {
        something: {
        required: true,
        type: "string"
        }
        }
        }
        */
        if (property.required) {
            // don't mark checkboxes as required
            if (input[0].localName === "input" && input.attr("type") === "checkbox") {
                // lets not set required on a checkbox, it doesn't make any sense ;)
            } else {
                input.attr('required', 'true');
            }
        }
        return group ? group : input;
    }
    Forms.createWidget = createWidget;

    /**
    * Lets try create the standard angular JS widgets markup
    * @method createStandardWidgetMarkup
    * @param {String} propTypeName
    * @param {any} property
    * @param {any} schema
    * @param {any} config
    * @param {any} options
    * @param {String} id
    */
    function createStandardWidgetMarkup(propTypeName, property, schema, config, options, id) {
        // lets try use standard widgets first...
        var type = Forms.resolveTypeNameAlias(propTypeName, schema);
        if (!type) {
            return '<input type="text"/>';
        }
        var custom = Core.pathGet(property, ["formTemplate"]);
        if (custom) {
            return null;
        }
        var inputElement = Core.pathGet(property, ["input-element"]);
        if (inputElement) {
            return "<" + inputElement + "></" + inputElement + ">";
        }
        var enumValues = Core.pathGet(property, ["enum"]);
        if (enumValues) {
            var required = true;
            var valuesScopeName = null;
            var attributes = "";
            if (enumValues) {
                // calculate from input attributes...
                var scope = config.scope;
                var data = config.data;
                if (data && scope) {
                    // this is a big ugly - would be nice to expose this a bit easier...
                    // maybe nested objects should expose the model easily...
                    var fullSchema = scope[config.schemaName];
                    var model = angular.isString(data) ? scope[data] : data;

                    // now we need to keep walking the model to find the enum values
                    var paths = id.split(".");
                    var property = null;
                    angular.forEach(paths, function (path) {
                        property = Core.pathGet(model, ["properties", path]);
                        var typeName = Core.pathGet(property, ["type"]);
                        var alias = Forms.lookupDefinition(typeName, fullSchema);
                        if (alias) {
                            model = alias;
                        }
                    });
                    var values = Core.pathGet(property, ["enum"]);
                    valuesScopeName = "$values_" + id.replace(/\./g, "_");
                    scope[valuesScopeName] = values;
                }
            }
            if (valuesScopeName) {
                attributes += ' ng-options="value for value in ' + valuesScopeName + '"';
            }
            var defaultOption = required ? "" : '<option value=""></option>';
            return '<select' + attributes + '>' + defaultOption + '</select>';
        }

        if (angular.isArray(type)) {
            // TODO union of tabbed forms such as Marshal / Unmarshal in camel...
            return null;
        }
        if (!angular.isString(type)) {
            return null;
        }
        var defaultValueConverter = null;
        var defaultValue = property.default;
        if (defaultValue) {
            // lets add a default value
            defaultValueConverter = function (scope, modelName) {
                var value = Core.pathGet(scope, modelName);
                if (!value) {
                    Core.pathSet(scope, modelName, property.default);
                }
            };
            options.valueConverter = defaultValueConverter;
        }

        function getModelValueOrDefault(scope, modelName) {
            var value = Core.pathGet(scope, modelName);
            if (!value) {
                var defaultValue = property.default;
                if (defaultValue) {
                    value = defaultValue;
                    Core.pathSet(scope, modelName, value);
                }
            }
            return value;
        }

        switch (type.toLowerCase()) {
            case "int":
            case "integer":
            case "long":
            case "short":
            case "java.lang.integer":
            case "java.lang.long":
            case "float":
            case "double":
            case "java.lang.float":
            case "java.lang.double":
                // lets add a value conversion watcher...
                options.valueConverter = function (scope, modelName) {
                    var value = getModelValueOrDefault(scope, modelName);
                    if (value && angular.isString(value)) {
                        var numberValue = Number(value);
                        Core.pathSet(scope, modelName, numberValue);
                    }
                };
                return '<input type="number"/>';

            case "array":
            case "java.lang.array":
            case "java.lang.iterable":
            case "java.util.list":
            case "java.util.collection":
            case "java.util.iterator":
            case "java.util.set":
            case "object[]":
                // TODO hack for now - objects should not really use the table, thats only really for arrays...
                /*
                case "object":
                case "java.lang.object":
                */
                //return "hawtio-form-array";
                return null;

            case "boolean":
            case "bool":
            case "java.lang.boolean":
                // lets add a value conversion watcher...
                options.valueConverter = function (scope, modelName) {
                    var value = getModelValueOrDefault(scope, modelName);
                    if (value && "true" === value) {
                        //console.log("coercing String to boolean for " + modelName);
                        Core.pathSet(scope, modelName, true);
                    }
                };
                return '<input type="checkbox"/>';

            case "password":
                return '<input type="password"/>';

            case "hidden":
                return '<input type="hidden"/>';
            default:
                // lets check if this name is an alias to a definition in the schema
                return '<input type="text"/>';
        }
    }
    Forms.createStandardWidgetMarkup = createStandardWidgetMarkup;

    function normalize(type, property, schema) {
        type = Forms.resolveTypeNameAlias(type, schema);
        if (!type) {
            return "hawtio-form-text";
        }
        var custom = Core.pathGet(property, ["formTemplate"]);
        if (custom) {
            return "hawtio-form-custom";
        }
        var enumValues = Core.pathGet(property, ["enum"]);
        if (enumValues) {
            // TODO could use different kinds of radio / combo box
            return "hawtio-form-select";
        }

        if (angular.isArray(type)) {
            // TODO union of tabbed forms such as Marshal / Unmarshal in camel...
            return null;
        }
        if (!angular.isString(type)) {
            try  {
                console.log("Unsupported JSON schema type value " + JSON.stringify(type));
            } catch (e) {
                console.log("Unsupported JSON schema type value " + type);
            }
            return null;
        }
        switch (type.toLowerCase()) {
            case "int":
            case "integer":
            case "long":
            case "short":
            case "java.lang.integer":
            case "java.lang.long":
            case "float":
            case "double":
            case "java.lang.float":
            case "java.lang.double":
                return "hawtio-form-number";

            case "array":
            case "java.lang.array":
            case "java.lang.iterable":
            case "java.util.list":
            case "java.util.collection":
            case "java.util.iterator":
            case "java.util.set":
            case "object[]":
                // TODO hack for now - objects should not really use the table, thats only really for arrays...
                /*
                case "object":
                case "java.lang.object":
                */
                var items = property.items;
                if (items) {
                    var typeName = items.type;
                    if (typeName && typeName === "string") {
                        return "hawtio-form-string-array";
                    }
                }
                return "hawtio-form-array";
            case "boolean":
            case "bool":
            case "java.lang.boolean":
                return "hawtio-form-checkbox";
            case "password":
                return "hawtio-form-password";
            case "hidden":
                return "hawtio-form-hidden";
            default:
                // lets check if this name is an alias to a definition in the schema
                return "hawtio-form-text";
        }
    }
    Forms.normalize = normalize;
})(Forms || (Forms = {}));
var Forms;
(function (Forms) {
    var InputTableConfig = (function () {
        function InputTableConfig() {
            this.name = 'form';
            this.method = 'post';
            // the name of the attribute in the scope which is the data to be editted
            this.entity = 'entity';
            // the name of the attribute in the scope which is the table configuration
            this.tableConfig = 'tableConfig';
            // set to 'view' or 'create' for different modes
            this.mode = 'edit';
            // the definition of the form
            this.data = {};
            this.json = undefined;
            this.properties = [];
            this.action = '';
            this.tableclass = 'table table-striped inputTable';
            this.controlgroupclass = 'control-group';
            this.controlclass = 'controls pull-right';
            this.labelclass = 'control-label';
            this.showtypes = 'true';
            this.removeicon = 'icon-remove';
            this.editicon = 'icon-edit';
            this.addicon = 'icon-plus';
            this.removetext = 'Remove';
            this.edittext = 'Edit';
            this.addtext = 'Add';
            this.onadd = 'onadd';
            this.onedit = 'onedit';
            this.onremove = 'onRemove';
        }
        // TODO - add toggles to turn off add or edit buttons
        InputTableConfig.prototype.getTableConfig = function () {
            return this.tableConfig || "tableConfig";
        };
        return InputTableConfig;
    })();
    Forms.InputTableConfig = InputTableConfig;

    var InputTable = (function () {
        function InputTable(workspace, $compile) {
            var _this = this;
            this.workspace = workspace;
            this.$compile = $compile;
            this.restrict = 'A';
            this.scope = true;
            this.replace = true;
            this.transclude = true;
            this.attributeName = 'hawtioInputTable';
            // necessary to ensure 'this' is this object <sigh>
            this.link = function (scope, element, attrs) {
                return _this.doLink(scope, element, attrs);
            };
        }
        InputTable.prototype.doLink = function (scope, element, attrs) {
            var _this = this;
            var config = new InputTableConfig;

            var configName = attrs[this.attributeName];
            var tableConfig = Core.pathGet(scope, configName);
            config = Forms.configure(config, tableConfig, attrs);

            var entityName = attrs["entity"] || config.data || "entity";
            var propertyName = attrs["property"] || "arrayData";
            var entityPath = entityName + "." + propertyName;

            // TODO better name?
            var tableName = config["title"] || entityName;

            if (angular.isDefined(config.json)) {
                config.data = $.parseJSON(config.json);
            } else {
                config.data = scope[config.data];
            }

            scope.selectedItems = [];

            var div = $("<div></div>");

            // TODO lets ensure we have some default columns in the column configuration?
            var tableConfig = Core.pathGet(scope, configName);
            if (!tableConfig) {
                console.log("No table configuration for table " + tableName);
            } else {
                tableConfig["selectedItems"] = scope.selectedItems;
            }

            var table = this.createTable(config, configName);

            var group = this.getControlGroup(config, {}, "");
            var controlDiv = this.getControlDiv(config);
            controlDiv.addClass('btn-group');
            group.append(controlDiv);

            function updateData(action) {
                var data = Core.pathGet(scope, entityPath);

                // lets coerce the data to an array if its empty or an object
                if (!data) {
                    data = [];
                }
                if (!angular.isArray(data) && data) {
                    data = [data];
                }
                data = action(data);
                Core.pathSet(scope, entityPath, data);

                // TODO for some reason this doesn't notify the underlying hawtio-datatable that the table has changed
                // so lets force it with a notify...
                scope.$emit("hawtio.datatable." + entityPath, data);
                Core.$apply(scope);
            }

            function removeSelected(data) {
                angular.forEach(scope.selectedItems, function (selected) {
                    var id = selected["_id"];
                    if (angular.isArray(data)) {
                        data = data.remove(function (value) {
                            return Object.equal(value, selected);
                        });
                        delete selected["_id"];
                        data = data.remove(function (value) {
                            return Object.equal(value, selected);
                        });
                    } else {
                        delete selected["_id"];
                        if (id) {
                            delete data[id];
                        } else {
                            // lets iterate for the value
                            var found = false;
                            angular.forEach(data, function (value, key) {
                                if (!found && (Object.equal(value, selected))) {
                                    console.log("Found row to delete! " + key);
                                    delete data[key];
                                    found = true;
                                }
                            });
                            if (!found) {
                                console.log("Could not find " + JSON.stringify(selected) + " in " + JSON.stringify(data));
                            }
                        }
                    }
                });
                return data;
            }

            var add = null;
            var edit = null;
            var remove = null;
            var addDialog = null;
            var editDialog = null;
            var readOnly = attrs["readonly"];
            if (!readOnly) {
                var property = null;
                var dataName = attrs["data"];
                var dataModel = dataName ? Core.pathGet(scope, dataName) : null;
                var schemaName = attrs["schema"] || dataName;
                var schema = schemaName ? Core.pathGet(scope, schemaName) : null;
                if (propertyName && dataModel) {
                    property = Core.pathGet(dataModel, ["properties", propertyName]);
                }

                add = this.getAddButton(config);

                scope.addDialogOptions = {
                    backdropFade: true,
                    dialogFade: true
                };
                scope.showAddDialog = false;

                scope.openAddDialog = function () {
                    // lets lazily create the add dialog
                    scope.addEntity = {};
                    scope.addFormConfig = Forms.findArrayItemsSchema(property, schema);

                    var childDataModelName = "addFormConfig";
                    if (!addDialog) {
                        var title = "Add " + tableName;
                        addDialog = $('<div modal="showAddDialog" close="closeAddDialog()" options="addDialogOptions">\n' + '<div class="modal-header"><h4>' + title + '</h4></div>\n' + '<div class="modal-body"><div simple-form="addFormConfig" entity="addEntity" data="' + childDataModelName + '" schema="' + schemaName + '"></div></div>\n' + '<div class="modal-footer">' + '<button class="btn btn-primary add" type="button" ng-click="addAndCloseDialog()">Add</button>' + '<button class="btn btn-warning cancel" type="button" ng-click="closeAddDialog()">Cancel</button>' + '</div></div>');
                        div.append(addDialog);
                        _this.$compile(addDialog)(scope);
                    }
                    scope.showAddDialog = true;
                    Core.$apply(scope);
                };

                scope.closeAddDialog = function () {
                    scope.showAddDialog = false;
                    scope.addEntity = {};
                };

                scope.addAndCloseDialog = function () {
                    var newData = scope.addEntity;
                    Forms.log.info("About to add the new entity " + JSON.stringify(newData));
                    if (newData) {
                        updateData(function (data) {
                            // TODO deal with non arrays
                            data.push(newData);
                            return data;
                        });
                    }
                    scope.closeAddDialog();
                };

                edit = this.getEditButton(config);

                scope.editDialogOptions = {
                    backdropFade: true,
                    dialogFade: true
                };
                scope.showEditDialog = false;

                scope.openEditDialog = function () {
                    var selected = scope.selectedItems;

                    // lets make a deep copy for the value being edited
                    var editObject = {};
                    if (selected && selected.length) {
                        angular.copy(selected[0], editObject);
                    }
                    scope.editEntity = editObject;
                    scope.editFormConfig = Forms.findArrayItemsSchema(property, schema);

                    // lets lazily create the edit dialog
                    if (!editDialog) {
                        var title = "Edit " + tableName;
                        editDialog = $('<div modal="showEditDialog" close="closeEditDialog()" options="editDialogOptions">\n' + '<div class="modal-header"><h4>' + title + '</h4></div>\n' + '<div class="modal-body"><div simple-form="editFormConfig" entity="editEntity"></div></div>\n' + '<div class="modal-footer">' + '<button class="btn btn-primary save" type="button" ng-click="editAndCloseDialog()">Save</button>' + '<button class="btn btn-warning cancel" type="button" ng-click="closeEditDialog()">Cancel</button>' + '</div></div>');
                        div.append(editDialog);
                        _this.$compile(editDialog)(scope);
                    }
                    scope.showEditDialog = true;
                    Core.$apply(scope);
                };

                scope.closeEditDialog = function () {
                    scope.showEditDialog = false;
                    scope.editEntity = {};
                };

                scope.editAndCloseDialog = function () {
                    var newData = scope.editEntity;
                    console.log("About to edit the new entity " + JSON.stringify(newData));
                    if (newData) {
                        updateData(function (data) {
                            data = removeSelected(data);

                            // TODO deal with non arrays
                            data.push(newData);
                            return data;
                        });
                    }
                    scope.closeEditDialog();
                };

                remove = this.getRemoveButton(config);
            }

            var findFunction = function (scope, func) {
                if (angular.isDefined(scope[func]) && angular.isFunction(scope[func])) {
                    return scope;
                }
                if (angular.isDefined(scope.$parent) && scope.$parent !== null) {
                    return findFunction(scope.$parent, func);
                } else {
                    return null;
                }
            };

            function maybeGet(scope, func) {
                if (scope !== null) {
                    return scope[func];
                }
                return null;
            }

            var onRemoveFunc = config.onremove.replace('(', '').replace(')', '');
            var onEditFunc = config.onedit.replace('(', '').replace(')', '');
            var onAddFunc = config.onadd.replace('(', '').replace(')', '');

            var onRemove = maybeGet(findFunction(scope, onRemoveFunc), onRemoveFunc);
            var onEdit = maybeGet(findFunction(scope, onEditFunc), onEditFunc);
            var onAdd = maybeGet(findFunction(scope, onAddFunc), onAddFunc);

            if (onRemove === null) {
                onRemove = function () {
                    updateData(function (data) {
                        return removeSelected(data);
                    });
                };
            }
            if (onEdit === null) {
                onEdit = function () {
                    scope.openEditDialog();
                };
            }
            if (onAdd === null) {
                onAdd = function (form) {
                    scope.openAddDialog();
                };
            }
            if (add) {
                add.click(function (event) {
                    onAdd();
                    return false;
                });
                controlDiv.append(add);
            }
            if (edit) {
                edit.click(function (event) {
                    onEdit();
                    return false;
                });
                controlDiv.append(edit);
            }
            if (remove) {
                remove.click(function (event) {
                    onRemove();
                    return false;
                });
                controlDiv.append(remove);
            }

            $(div).append(group);
            $(div).append(table);
            $(element).append(div);

            // compile the template
            this.$compile(div)(scope);
        };

        InputTable.prototype.getAddButton = function (config) {
            return $('<button type="button" class="btn add"><i class="' + config.addicon + '"></i> ' + config.addtext + '</button>');
        };

        InputTable.prototype.getEditButton = function (config) {
            return $('<button type="button" class="btn edit" ng-disabled="!selectedItems.length"><i class="' + config.editicon + '"></i> ' + config.edittext + '</button>');
        };

        InputTable.prototype.getRemoveButton = function (config) {
            return $('<button type="remove" class="btn remove" ng-disabled="!selectedItems.length"><i class="' + config.removeicon + '"></i> ' + config.removetext + '</button>');
        };

        InputTable.prototype.createTable = function (config, tableConfig) {
            //var tableType = "hawtio-datatable";
            var tableType = "hawtio-simple-table";
            var table = $('<div class="' + config.tableclass + '" ' + tableType + '="' + tableConfig + '">');

            //table.find('fieldset').append(this.getLegend(config));
            return table;
        };

        InputTable.prototype.getLegend = function (config) {
            var description = Core.pathGet(config, "data.description");
            if (description) {
                return '<legend>' + config.data.description + '</legend>';
            }
            return '';
        };

        InputTable.prototype.getControlGroup = function (config, arg, id) {
            var rc = $('<div class="' + config.controlgroupclass + '"></div>');
            if (angular.isDefined(arg.description)) {
                rc.attr('title', arg.description);
            }
            return rc;
        };

        InputTable.prototype.getControlDiv = function (config) {
            return $('<div class="' + config.controlclass + '"></div>');
        };

        InputTable.prototype.getHelpSpan = function (config, arg, id) {
            var rc = $('<span class="help-block"></span>');
            if (angular.isDefined(arg.type) && config.showtypes !== 'false') {
                rc.append('Type: ' + arg.type);
            }
            return rc;
        };
        return InputTable;
    })();
    Forms.InputTable = InputTable;
})(Forms || (Forms = {}));
/**
* @module Tree
* @main Tree
*/
var Tree;
(function (Tree) {
    Tree.pluginName = 'tree';
    Tree.log = Logger.get("Tree");

    function expandAll(el) {
        treeAction(el, true);
    }
    Tree.expandAll = expandAll;

    function contractAll(el) {
        treeAction(el, false);
    }
    Tree.contractAll = contractAll;

    function treeAction(el, expand) {
        $(el).dynatree("getRoot").visit(function (node) {
            node.expand(expand);
        });
    }

    /**
    * @function sanitize
    * @param tree
    *
    * Use to HTML escape all entries in a tree before passing it
    * over to the dynatree plugin to avoid cross site scripting
    * issues.
    *
    */
    function sanitize(tree) {
        if (!tree) {
            return;
        }
        if (angular.isArray(tree)) {
            tree.forEach(function (folder) {
                Tree.sanitize(folder);
            });
        }
        var title = tree['title'];
        if (title) {
            tree['title'] = title.unescapeHTML(true).escapeHTML();
        }
        if (tree.children) {
            Tree.sanitize(tree.children);
        }
    }
    Tree.sanitize = sanitize;

    angular.module(Tree.pluginName, ['bootstrap', 'ngResource', 'hawtioCore']).directive('hawtioTree', function (workspace, $timeout, $location) {
        // return the directive link function. (compile function not needed)
        return function (scope, element, attrs) {
            var tree = null;
            var data = null;
            var widget = null;
            var timeoutId = null;
            var onSelectFn = lookupFunction("onselect");
            var onDragStartFn = lookupFunction("ondragstart");
            var onDragEnterFn = lookupFunction("ondragenter");
            var onDropFn = lookupFunction("ondrop");

            function lookupFunction(attrName) {
                var answer = null;
                var fnName = attrs[attrName];
                if (fnName) {
                    answer = Core.pathGet(scope, fnName);
                    if (!angular.isFunction(answer)) {
                        answer = null;
                    }
                }
                return answer;
            }

            // watch the expression, and update the UI on change.
            var data = attrs.hawtioTree;
            var queryParam = data;

            scope.$watch(data, onWidgetDataChange);

            // lets add a separate event so we can force updates
            // if we find cases where the delta logic doesn't work
            scope.$on("hawtio.tree." + data, function (args) {
                var value = Core.pathGet(scope, data);
                onWidgetDataChange(value);
            });

            // listen on DOM destroy (removal) event, and cancel the next UI update
            // to prevent updating ofter the DOM element was removed.
            element.bind('$destroy', function () {
                $timeout.cancel(timeoutId);
            });

            updateLater(); // kick off the UI update process.

            // used to update the UI
            function updateWidget() {
                // console.log("updating the grid!!");
                Core.$applyNowOrLater(scope);
            }

            function onWidgetDataChange(value) {
                tree = value;
                if (tree) {
                    Tree.sanitize(tree);
                }
                if (tree && !widget) {
                    // lets find a child table element
                    // or lets add one if there's not one already
                    var treeElement = $(element);
                    var children = Core.asArray(tree);
                    var hideRoot = attrs["hideroot"];
                    if ("true" === hideRoot) {
                        children = tree['children'];
                    }
                    var config = {
                        clickFolderMode: 3,
                        /*
                        * The event handler called when a different node in the tree is selected
                        */
                        onActivate: function (node) {
                            var data = node.data;
                            if (onSelectFn) {
                                onSelectFn(data, node);
                            } else {
                                workspace.updateSelectionNode(data);
                            }
                            Core.$apply(scope);
                        },
                        /*
                        onLazyRead: function(treeNode) {
                        var folder = treeNode.data;
                        var plugin = null;
                        if (folder) {
                        plugin = Jmx.findLazyLoadingFunction(workspace, folder);
                        }
                        if (plugin) {
                        console.log("Lazy loading folder " + folder.title);
                        var oldChildren = folder.childen;
                        plugin(workspace, folder, () => {
                        treeNode.setLazyNodeStatus(DTNodeStatus_Ok);
                        var newChildren = folder.children;
                        if (newChildren !== oldChildren) {
                        treeNode.removeChildren();
                        angular.forEach(newChildren, newChild => {
                        treeNode.addChild(newChild);
                        });
                        }
                        });
                        } else {
                        treeNode.setLazyNodeStatus(DTNodeStatus_Ok);
                        }
                        },
                        */
                        onClick: function (node, event) {
                            if (event["metaKey"]) {
                                event.preventDefault();
                                var url = $location.absUrl();
                                if (node && node.data) {
                                    var key = node.data["key"];
                                    if (key) {
                                        var hash = $location.search();
                                        hash[queryParam] = key;

                                        // TODO this could maybe be a generic helper function?
                                        // lets trim after the ?
                                        var idx = url.indexOf('?');
                                        if (idx <= 0) {
                                            url += "?";
                                        } else {
                                            url = url.substring(0, idx + 1);
                                        }
                                        url += $.param(hash);
                                    }
                                }
                                window.open(url, '_blank');
                                window.focus();
                                return false;
                            }
                            return true;
                        },
                        persist: false,
                        debugLevel: 0,
                        children: children,
                        dnd: {
                            onDragStart: onDragStartFn ? onDragStartFn : function (node) {
                                /* This function MUST be defined to enable dragging for the tree.
                                *  Return false to cancel dragging of node.
                                */
                                console.log("onDragStart!");
                                return true;
                            },
                            onDragEnter: onDragEnterFn ? onDragEnterFn : function (node, sourceNode) {
                                console.log("onDragEnter!");
                                return true;
                            },
                            onDrop: onDropFn ? onDropFn : function (node, sourceNode, hitMode) {
                                console.log("onDrop!");

                                /* This function MUST be defined to enable dropping of items on
                                *  the tree.
                                */
                                sourceNode.move(node, hitMode);
                                return true;
                            }
                        }
                    };
                    if (!onDropFn && !onDragEnterFn && !onDragStartFn) {
                        delete config["dnd"];
                    }
                    widget = treeElement.dynatree(config);

                    var activatedNode = false;
                    var activateNodeName = attrs["activatenodes"];
                    if (activateNodeName) {
                        var values = scope[activateNodeName];
                        var tree = treeElement.dynatree("getTree");
                        if (values && tree) {
                            angular.forEach(Core.asArray(values), function (value) {
                                //tree.selectKey(value, true);
                                tree.activateKey(value);
                                activatedNode = true;
                            });
                        }
                    }
                    var root = treeElement.dynatree("getRoot");
                    if (root) {
                        var onRootName = attrs["onroot"];
                        if (onRootName) {
                            var fn = scope[onRootName];
                            if (fn) {
                                fn(root);
                            }
                        }

                        // select and activate first child if we have not activated any others
                        if (!activatedNode) {
                            var children = root['getChildren']();
                            if (children && children.length) {
                                var child = children[0];
                                child.expand(true);
                                child.activate(true);
                            }
                        }
                    }
                }
                updateWidget();
            }

            // schedule update in one second
            function updateLater() {
                // save the timeoutId for canceling
                timeoutId = $timeout(function () {
                    updateWidget(); // update DOM
                }, 300);
            }
        };
    }).run(function (helpRegistry) {
        helpRegistry.addDevDoc(Tree.pluginName, 'app/tree/doc/developer.md');
    });

    hawtioPluginLoader.addModule(Tree.pluginName);
})(Tree || (Tree = {}));
var ActiveMQ;
(function (ActiveMQ) {
    function JobSchedulerController($scope, workspace, jolokia) {
        $scope.refresh = loadTable;

        $scope.jobs = [];
        $scope.deleteJobsDialog = new UI.Dialog();

        $scope.gridOptions = {
            selectedItems: [],
            data: 'jobs',
            displayFooter: false,
            showFilter: false,
            showColumnMenu: true,
            enableColumnResize: true,
            enableColumnReordering: true,
            filterOptions: {
                filterText: ''
            },
            selectWithCheckboxOnly: true,
            showSelectionCheckbox: true,
            maintainColumnRatios: false,
            columnDefs: [
                {
                    field: 'jobId',
                    displayName: 'Job ID',
                    width: '25%'
                },
                {
                    field: 'cronEntry',
                    displayName: 'Cron Entry',
                    width: '10%'
                },
                {
                    field: 'delay',
                    displayName: 'Delay',
                    width: '5%'
                },
                {
                    field: 'repeat',
                    displayName: 'repeat',
                    width: '5%'
                },
                {
                    field: 'period',
                    displayName: 'period',
                    width: '5%'
                },
                {
                    field: 'start',
                    displayName: 'Start',
                    width: '25%'
                },
                {
                    field: 'next',
                    displayName: 'Next',
                    width: '25%'
                }
            ]
        };

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid())
                return;

            // lets defer execution as we may not have the selection just yet
            setTimeout(loadTable, 50);
        });

        function loadTable() {
            var selection = workspace.selection;
            if (selection) {
                var mbean = selection.objectName;
                if (mbean) {
                    jolokia.request({ type: 'read', mbean: mbean, attribute: "AllJobs" }, onSuccess(populateTable));
                }
            }
            Core.$apply($scope);
        }

        function populateTable(response) {
            var data = response.value;
            if (!angular.isArray(data)) {
                $scope.jobs = [];
                angular.forEach(data, function (value, idx) {
                    $scope.jobs.push(value);
                });
            } else {
                $scope.jobs = data;
            }
            Core.$apply($scope);
        }

        $scope.deleteJobs = function () {
            var selection = workspace.selection;
            var mbean = selection.objectName;
            if (mbean && selection) {
                var selectedItems = $scope.gridOptions.selectedItems;
                $scope.message = "Deleted " + Core.maybePlural(selectedItems.length, "job");
                var operation = "removeJob(java.lang.String)";
                angular.forEach(selectedItems, function (item, idx) {
                    var id = item.jobId;
                    if (id) {
                        var callback = (idx + 1 < selectedItems.length) ? intermediateResult : operationSuccess;
                        jolokia.execute(mbean, operation, id, onSuccess(callback));
                    }
                });
            }
        };

        function intermediateResult() {
        }

        function operationSuccess() {
            $scope.gridOptions.selectedItems.splice(0);
            notification("success", $scope.message);
            setTimeout(loadTable, 50);
        }
    }
    ActiveMQ.JobSchedulerController = JobSchedulerController;
})(ActiveMQ || (ActiveMQ = {}));
var ActiveMQ;
(function (ActiveMQ) {
    function DestinationController($scope, workspace, jolokia) {
        $scope.workspace = workspace;
        $scope.message = "";
        $scope.queueType = 'true';

        $scope.deleteDialog = false;
        $scope.purgeDialog = false;

        updateQueueType();

        function updateQueueType() {
            $scope.destinationTypeName = $scope.queueType ? "Queue" : "Topic";
        }

        $scope.$watch('queueType', function () {
            updateQueueType();
        });

        $scope.$watch('workspace.selection', function () {
            workspace.moveIfViewInvalid();
        });

        function operationSuccess() {
            $scope.destinationName = "";
            $scope.workspace.operationCounter += 1;
            Core.$apply($scope);
            notification("success", $scope.message);
            $scope.workspace.loadTree();
        }

        function deleteSuccess() {
            // lets set the selection to the parent
            workspace.removeAndSelectParentNode();
            $scope.workspace.operationCounter += 1;
            Core.$apply($scope);
            notification("success", $scope.message);
            $scope.workspace.loadTree();
        }

        function getBrokerMBean(jolokia) {
            var mbean = null;
            var selection = workspace.selection;
            if (selection && ActiveMQ.isBroker(workspace) && selection.objectName) {
                return selection.objectName;
            }
            var folderNames = selection.folderNames;

            //if (selection && jolokia && folderNames && folderNames.length > 1) {
            var parent = selection ? selection.parent : null;
            if (selection && parent && jolokia && folderNames && folderNames.length > 1) {
                mbean = parent.objectName;

                // we might be a destination, so lets try one more parent
                if (!mbean && parent) {
                    mbean = parent.parent.objectName;
                }
                if (!mbean) {
                    mbean = "" + folderNames[0] + ":BrokerName=" + folderNames[1] + ",Type=Broker";
                }
            }
            return mbean;
        }

        $scope.createDestination = function (name, isQueue) {
            var mbean = getBrokerMBean(jolokia);
            if (mbean) {
                var operation;
                if (isQueue) {
                    operation = "addQueue(java.lang.String)";
                    $scope.message = "Created queue " + name;
                } else {
                    operation = "addTopic(java.lang.String)";
                    $scope.message = "Created topic " + name;
                }
                if (mbean) {
                    jolokia.execute(mbean, operation, name, onSuccess(operationSuccess));
                } else {
                    notification("error", "Could not find the Broker MBean!");
                }
            }
        };

        $scope.deleteDestination = function () {
            var mbean = getBrokerMBean(jolokia);
            var selection = workspace.selection;
            var entries = selection.entries;
            if (mbean && selection && jolokia && entries) {
                var domain = selection.domain;
                var name = entries["Destination"] || entries["destinationName"] || selection.title;
                name = name.unescapeHTML();
                var isQueue = "Topic" !== (entries["Type"] || entries["destinationType"]);
                var operation;
                if (isQueue) {
                    operation = "removeQueue(java.lang.String)";
                    $scope.message = "Deleted queue " + name;
                } else {
                    operation = "removeTopic(java.lang.String)";
                    $scope.message = "Deleted topic " + name;
                }
                jolokia.execute(mbean, operation, name, onSuccess(deleteSuccess));
            }
        };

        $scope.purgeDestination = function () {
            var mbean = workspace.getSelectedMBeanName();
            var selection = workspace.selection;
            var entries = selection.entries;
            if (mbean && selection && jolokia && entries) {
                var name = entries["Destination"] || entries["destinationName"] || selection.title;
                name = name.unescapeHTML();
                var operation = "purge()";
                $scope.message = "Purged queue " + name;
                jolokia.execute(mbean, operation, onSuccess(operationSuccess));
            }
        };

        $scope.name = function () {
            var selection = workspace.selection;
            if (selection) {
                return selection.title;
            }
            return null;
        };
    }
    ActiveMQ.DestinationController = DestinationController;
})(ActiveMQ || (ActiveMQ = {}));
var ActiveMQ;
(function (ActiveMQ) {
    ActiveMQ.log = Logger.get("activemq");
    ActiveMQ.jmxDomain = 'org.apache.activemq';

    function getSelectionQueuesFolder(workspace) {
        function findQueuesFolder(node) {
            if (node) {
                if (node.title === "Queues" || node.title === "Queue") {
                    return node;
                }
                var parent = node.parent;
                if (parent) {
                    return findQueuesFolder(parent);
                }
            }
            return null;
        }

        var selection = workspace.selection;
        if (selection) {
            return findQueuesFolder(selection);
        }
        return null;
    }
    ActiveMQ.getSelectionQueuesFolder = getSelectionQueuesFolder;

    function getSelectionTopicsFolder(workspace) {
        function findTopicsFolder(node) {
            var answer = null;
            if (node) {
                if (node.title === "Topics" || node.title === "Topic") {
                    answer = node;
                }

                if (answer === null) {
                    angular.forEach(node.children, function (child) {
                        if (child.title === "Topics" || child.title === "Topic") {
                            answer = child;
                        }
                    });
                }
            }
            return answer;
        }

        var selection = workspace.selection;
        if (selection) {
            return findTopicsFolder(selection);
        }
        return null;
    }
    ActiveMQ.getSelectionTopicsFolder = getSelectionTopicsFolder;
})(ActiveMQ || (ActiveMQ = {}));
var ActiveMQ;
(function (ActiveMQ) {
    function TreeHeaderController($scope) {
        $scope.expandAll = function () {
            Tree.expandAll("#activemqtree");
        };

        $scope.contractAll = function () {
            Tree.contractAll("#activemqtree");
        };
    }
    ActiveMQ.TreeHeaderController = TreeHeaderController;

    function TreeController($scope, $location, workspace, localStorage) {
        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        $scope.$watch('workspace.tree', function () {
            reloadTree();
        });

        $scope.$on('jmxTreeUpdated', function () {
            reloadTree();
        });

        function reloadTree() {
            ActiveMQ.log.debug("workspace tree has changed, lets reload the activemq tree");

            var children = [];
            var tree = workspace.tree;
            if (tree) {
                var domainName = "org.apache.activemq";
                var folder = tree.get(domainName);
                if (folder) {
                    children = folder.children;
                }
                if (children.length) {
                    var firstChild = children[0];

                    // the children could be AMQ 5.7 style broker name folder with the actual MBean in the children
                    // along with folders for the Queues etc...
                    if (!firstChild.typeName && firstChild.children.length < 4) {
                        // lets avoid the top level folder
                        var answer = [];
                        angular.forEach(children, function (child) {
                            answer = answer.concat(child.children);
                        });
                        children = answer;
                    }
                }

                // filter out advisory topics
                children.forEach(function (broker) {
                    var grandChildren = broker.children;
                    if (grandChildren) {
                        Tree.sanitize(grandChildren);
                        var idx = grandChildren.findIndex(function (n) {
                            return n.title === "Topic";
                        });
                        if (idx > 0) {
                            var old = grandChildren[idx];

                            // we need to store all topics the first time on the workspace
                            // so we have access to them later if the user changes the filter in the preferences
                            var key = "ActiveMQ-allTopics-" + broker.title;
                            var allTopics = workspace.mapData[key];
                            if (angular.isUndefined(allTopics)) {
                                var allTopics = old.children.clone();
                                workspace.mapData[key] = allTopics;
                            }

                            var filter = Core.parseBooleanValue(localStorage["activemqFilterAdvisoryTopics"]);
                            if (filter) {
                                if (old && old.children) {
                                    var filteredTopics = old.children.filter(function (c) {
                                        return !c.title.startsWith("ActiveMQ.Advisory");
                                    });
                                    old.children = filteredTopics;
                                }
                            } else if (allTopics) {
                                old.children = allTopics;
                            }
                        }
                    }
                });

                var treeElement = $("#activemqtree");
                Jmx.enableTree($scope, $location, workspace, treeElement, children, true);

                // lets do this asynchronously to avoid Error: $digest already in progress
                setTimeout(updateSelectionFromURL, 50);
            }
        }

        function updateSelectionFromURL() {
            Jmx.updateTreeSelectionFromURLAndAutoSelect($location, $("#activemqtree"), function (first) {
                // use function to auto select the queue folder on the 1st broker
                var queues = first.getChildren()[0];
                if (queues && queues.data.title === 'Queue') {
                    first = queues;
                    first.expand(true);
                    return first;
                }
                return null;
            }, true);
        }
    }
    ActiveMQ.TreeController = TreeController;
})(ActiveMQ || (ActiveMQ = {}));
/**
* @module ActiveMQ
*/
var ActiveMQ;
(function (ActiveMQ) {
    function PreferencesController($scope, localStorage, userDetails, $rootScope) {
        Core.initPreferenceScope($scope, localStorage, {
            'activemqUserName': {
                'value': userDetails.username
            },
            'activemqPassword': {
                'value': userDetails.password
            },
            'activemqBrowseBytesMessages': {
                'value': 1,
                'converter': parseInt,
                'formatter': function (value) {
                    return "" + value;
                }
            },
            'activemqFilterAdvisoryTopics': {
                'value': false,
                'converter': Core.parseBooleanValue,
                'post': function (newValue) {
                    $rootScope.$broadcast('jmxTreeUpdated');
                }
            }
        });
    }
    ActiveMQ.PreferencesController = PreferencesController;
})(ActiveMQ || (ActiveMQ = {}));
var ActiveMQ;
(function (ActiveMQ) {
    function BrokerDiagramController($scope, $compile, $location, localStorage, jolokia, workspace) {
        Fabric.initScope($scope, $location, jolokia, workspace);

        var isFmc = Fabric.isFMCContainer(workspace);
        $scope.isFmc = isFmc;

        $scope.selectedNode = null;

        var defaultFlags = {
            panel: true,
            popup: false,
            label: true,
            group: false,
            profile: false,
            slave: false,
            broker: isFmc,
            network: true,
            container: false,
            queue: true,
            topic: true,
            consumer: true,
            producer: true
        };

        $scope.viewSettings = {};

        $scope.shapeSize = {
            broker: 20,
            queue: 14,
            topic: 14
        };

        var redrawGraph = Core.throttled(doRedrawGraph, 1000);

        var graphBuilder = new ForceGraph.GraphBuilder();

        Core.bindModelToSearchParam($scope, $location, "searchFilter", "q", "");

        angular.forEach(defaultFlags, function (defaultValue, key) {
            var modelName = "viewSettings." + key;

            // bind model values to search params...
            function currentValue() {
                var answer = $location.search()[paramName] || defaultValue;
                return answer === "false" ? false : answer;
            }

            var paramName = key;
            var value = currentValue();
            Core.pathSet($scope, modelName, value);

            $scope.$watch(modelName, function () {
                var current = Core.pathGet($scope, modelName);
                var old = currentValue();
                if (current !== old) {
                    var defaultValue = defaultFlags[key];
                    if (current !== defaultValue) {
                        if (!current) {
                            current = "false";
                        }
                        $location.search(paramName, current);
                    } else {
                        $location.search(paramName, null);
                    }
                }
                redrawGraph();
            });
        });

        $scope.connectToBroker = function () {
            var selectedNode = $scope.selectedNode;
            if (selectedNode) {
                var container = selectedNode["brokerContainer"] || selectedNode;
                connectToBroker(container, selectedNode["brokerName"]);
            }
        };

        function connectToBroker(container, brokerName, postfix) {
            if (typeof postfix === "undefined") { postfix = null; }
            if (isFmc && container.jolokia !== jolokia) {
                Fabric.connectToBroker($scope, container, postfix);
            } else {
                var view = "/jmx/attributes?tab=activemq";
                if (!postfix) {
                    if (brokerName) {
                        // lets default to the broker view
                        postfix = "nid=root-org.apache.activemq-Broker-" + brokerName;
                    }
                }
                if (postfix) {
                    view += "&" + postfix;
                }
                ActiveMQ.log.info("Opening view " + view);
                var path = url("/#" + view);
                window.open(path, '_destination');
                window.focus();
                //$location.path(view);
            }
        }

        $scope.connectToDestination = function () {
            var selectedNode = $scope.selectedNode;
            if (selectedNode) {
                var container = selectedNode["brokerContainer"] || selectedNode;
                var brokerName = selectedNode["brokerName"];
                var destinationType = selectedNode["destinationType"] || selectedNode["typeLabel"];
                var destinationName = selectedNode["destinationName"];
                var postfix = null;
                if (brokerName && destinationType && destinationName) {
                    postfix = "nid=root-org.apache.activemq-Broker-" + brokerName + "-" + destinationType + "-" + destinationName;
                }
                connectToBroker(container, brokerName, postfix);
            }
        };

        $scope.$on('$destroy', function (event) {
            stopOldJolokia();
        });

        function stopOldJolokia() {
            var oldJolokia = $scope.selectedNodeJolokia;
            if (oldJolokia && oldJolokia !== jolokia) {
                oldJolokia.stop();
            }
        }

        $scope.$watch("selectedNode", function (newValue, oldValue) {
            // lets cancel any previously registered thingy
            if ($scope.unregisterFn) {
                $scope.unregisterFn();
                $scope.unregisterFn = null;
            }
            var node = $scope.selectedNode;
            if (node) {
                var mbean = node.objectName;
                var brokerContainer = node.brokerContainer || {};
                var nodeJolokia = node.jolokia || brokerContainer.jolokia || jolokia;
                if (nodeJolokia !== $scope.selectedNodeJolokia) {
                    stopOldJolokia();
                    $scope.selectedNodeJolokia = nodeJolokia;
                    if (nodeJolokia !== jolokia) {
                        var rate = Core.parseIntValue(localStorage['updateRate'] || "2000", "update rate");
                        if (rate) {
                            nodeJolokia.start(rate);
                        }
                    }
                }
                var dummyResponse = { value: node.panelProperties || {} };
                if (mbean && nodeJolokia) {
                    $scope.unregisterFn = Core.register(nodeJolokia, $scope, {
                        type: 'read', mbean: mbean
                    }, onSuccess(renderNodeAttributes, { error: function (response) {
                            // probably we've got a wrong mbean name?
                            // so lets render at least
                            renderNodeAttributes(dummyResponse);
                            Core.defaultJolokiaErrorHandler(response);
                        } }));
                } else {
                    renderNodeAttributes(dummyResponse);
                }
            }
        });

        function getDestinationTypeName(attributes) {
            var prefix = attributes["DestinationTemporary"] ? "Temporary " : "";
            return prefix + (attributes["DestinationTopic"] ? "Topic" : "Queue");
        }

        var ignoreNodeAttributes = [
            "Broker", "BrokerId", "BrokerName", "Connection",
            "DestinationName", "DestinationQueue", "DestinationTemporary", "DestinationTopic"
        ];

        var ignoreNodeAttributesByType = {
            producer: ["Producer", "ProducerId"],
            queue: ["Name", "MessageGroups", "MessageGroupType", "Subscriptions"],
            topic: ["Name", "Subscriptions"],
            broker: ["DataDirectory", "DurableTopicSubscriptions", "DynamicDestinationProducers", "InactiveDurableToppicSubscribers"]
        };

        var brokerShowProperties = [
            "AverageMessageSize", "BrokerId", "JobSchedulerStorePercentUsage",
            "Slave", "MemoryPercentUsage", "StorePercentUsage", "TempPercentUsage"];
        var onlyShowAttributesByType = {
            broker: brokerShowProperties,
            brokerSlave: brokerShowProperties
        };

        function renderNodeAttributes(response) {
            var properties = [];
            if (response) {
                var value = response.value || {};
                $scope.selectedNodeAttributes = value;
                var selectedNode = $scope.selectedNode || {};
                var brokerContainer = selectedNode['brokerContainer'] || {};
                var nodeType = selectedNode["type"];
                var brokerName = selectedNode["brokerName"];
                var containerId = selectedNode["container"] || brokerContainer["container"];
                var group = selectedNode["group"] || brokerContainer["group"];
                var jolokiaUrl = selectedNode["jolokiaUrl"] || brokerContainer["jolokiaUrl"];
                var profile = selectedNode["profile"] || brokerContainer["profile"];
                var version = selectedNode["version"] || brokerContainer["version"];

                var isBroker = nodeType && nodeType.startsWith("broker");
                var ignoreKeys = ignoreNodeAttributes.concat(ignoreNodeAttributesByType[nodeType] || []);
                var onlyShowKeys = onlyShowAttributesByType[nodeType];

                angular.forEach(value, function (v, k) {
                    if (onlyShowKeys ? onlyShowKeys.indexOf(k) >= 0 : ignoreKeys.indexOf(k) < 0) {
                        var formattedValue = Core.humanizeValueHtml(v);
                        properties.push({ key: humanizeValue(k), value: formattedValue });
                    }
                });
                properties = properties.sortBy("key");

                var brokerProperty = null;
                if (brokerName) {
                    var brokerHtml = '<a target="broker" ng-click="connectToBroker()">' + '<img title="Apache ActiveMQ" src="img/icons/messagebroker.svg"> ' + brokerName + '</a>';
                    if (version && profile) {
                        var brokerLink = Fabric.brokerConfigLink(workspace, jolokia, localStorage, version, profile, brokerName);
                        if (brokerLink) {
                            brokerHtml += ' <a title="configuration settings" target="brokerConfig" href="' + brokerLink + '"><i class="icon-tasks"></i></a>';
                        }
                    }
                    var html = $compile(brokerHtml)($scope);
                    brokerProperty = { key: "Broker", value: html };
                    if (!isBroker) {
                        properties.splice(0, 0, brokerProperty);
                    }
                }

                if (containerId) {
                    var containerModel = "selectedNode" + (selectedNode['brokerContainer'] ? ".brokerContainer" : "");
                    properties.splice(0, 0, { key: "Container", value: $compile('<div fabric-container-link="' + containerModel + '"></div>')($scope) });
                }

                var destinationName = value["DestinationName"] || selectedNode["destinationName"];
                if (destinationName && (nodeType !== "queue" && nodeType !== "topic")) {
                    var destinationTypeName = getDestinationTypeName(value);
                    var html = createDestinationLink(destinationName, destinationTypeName);
                    properties.splice(0, 0, { key: destinationTypeName, value: html });
                }

                var typeLabel = selectedNode["typeLabel"];
                var name = selectedNode["name"] || selectedNode["id"] || selectedNode['objectName'];
                if (typeLabel) {
                    var html = name;
                    if (nodeType === "queue" || nodeType === "topic") {
                        html = createDestinationLink(name, nodeType);
                    }
                    var typeProperty = { key: typeLabel, value: html };
                    if (isBroker && brokerProperty) {
                        typeProperty = brokerProperty;
                    }
                    properties.splice(0, 0, typeProperty);
                }
            }
            $scope.selectedNodeProperties = properties;
            Core.$apply($scope);
        }

        /**
        * Generates the HTML for a link to the destination
        */
        function createDestinationLink(destinationName, destinationType) {
            if (typeof destinationType === "undefined") { destinationType = "queue"; }
            return $compile('<a target="destination" title="' + destinationName + '" ng-click="connectToDestination()">' + destinationName + '</a>')($scope);
        }

        $scope.$watch("searchFilter", function (newValue, oldValue) {
            redrawGraph();
        });

        if (isFmc) {
            Core.register(jolokia, $scope, { type: 'exec', mbean: Fabric.mqManagerMBean, operation: "loadBrokerStatus()" }, onSuccess(onBrokerData));
        } else {
            // lets just use the current stuff from the workspace
            $scope.$watch('workspace.tree', function () {
                redrawGraph();
            });

            $scope.$on('jmxTreeUpdated', function () {
                redrawGraph();
            });
        }

        function onBrokerData(response) {
            if (response) {
                var responseJson = angular.toJson(response.value);
                if ($scope.responseJson === responseJson) {
                    return;
                }

                $scope.responseJson = responseJson;

                $scope.brokers = response.value;
                doRedrawGraph();
            }
        }

        function redrawFabricBrokers() {
            var containersToDelete = $scope.activeContainers || {};
            $scope.activeContainers = {};

            angular.forEach($scope.brokers, function (brokerStatus) {
                // only query master brokers which are provisioned correctly
                brokerStatus.validContainer = brokerStatus.alive && brokerStatus.master && brokerStatus.provisionStatus === "success";

                // don't use type field so we can use it for the node types..
                renameTypeProperty(brokerStatus);

                //log.info("Broker status: " + angular.toJson(brokerStatus, true));
                var groupId = brokerStatus.group;
                var profileId = brokerStatus.profile;
                var brokerId = brokerStatus.brokerName;
                var containerId = brokerStatus.container;
                var versionId = brokerStatus.version || "1.0";

                var group = getOrAddNode("group", groupId, brokerStatus, function () {
                    return {
                        /*
                        navUrl: ,
                        image: {
                        url: "/hawtio/img/icons/osgi/bundle.png",
                        width: 32,
                        height:32
                        },
                        */
                        typeLabel: "Broker Group",
                        popup: {
                            title: "Broker Group: " + groupId,
                            content: "<p>" + groupId + "</p>"
                        }
                    };
                });

                var profile = getOrAddNode("profile", profileId, brokerStatus, function () {
                    return {
                        typeLabel: "Profile",
                        popup: {
                            title: "Profile: " + profileId,
                            content: "<p>" + profileId + "</p>"
                        }
                    };
                });

                // TODO do we need to create a physical broker node per container and logical broker maybe?
                var container = null;
                if (containerId) {
                    container = getOrAddNode("container", containerId, brokerStatus, function () {
                        return {
                            containerId: containerId,
                            typeLabel: "Container",
                            popup: {
                                title: "Container: " + containerId,
                                content: "<p>" + containerId + " version: " + versionId + "</p>"
                            }
                        };
                    });
                }

                var master = brokerStatus.master;
                var broker = getOrAddBroker(master, brokerId, groupId, containerId, container, brokerStatus);
                if (container && container.validContainer) {
                    var key = container.containerId;
                    $scope.activeContainers[key] = container;
                    delete containersToDelete[key];
                }

                // add the links...
                if ($scope.viewSettings.group) {
                    if ($scope.viewSettings.profile) {
                        addLink(group, profile, "group");
                        addLink(profile, broker, "broker");
                    } else {
                        addLink(group, broker, "group");
                    }
                } else {
                    if ($scope.viewSettings.profile) {
                        addLink(profile, broker, "broker");
                    }
                }

                if (container) {
                    if ((master || $scope.viewSettings.slave) && $scope.viewSettings.container) {
                        addLink(broker, container, "container");
                        container.destinationLinkNode = container;
                    } else {
                        container.destinationLinkNode = broker;
                    }
                }
            });
            redrawActiveContainers();
        }

        function redrawLocalBroker() {
            var container = {
                jolokia: jolokia
            };
            var containerId = "local";
            $scope.activeContainers = {
                containerId: container
            };

            if ($scope.viewSettings.broker) {
                jolokia.search("org.apache.activemq:type=Broker,brokerName=*", onSuccess(function (response) {
                    angular.forEach(response, function (objectName) {
                        var details = Core.parseMBean(objectName);
                        if (details) {
                            var properties = details['attributes'];
                            ActiveMQ.log.info("Got broker: " + objectName + " on container: " + containerId + " properties: " + angular.toJson(properties, true));
                            if (properties) {
                                var master = true;
                                var brokerId = properties["brokerName"] || "unknown";
                                var groupId = "";
                                var broker = getOrAddBroker(master, brokerId, groupId, containerId, container, properties);
                            }
                        }
                    });
                    redrawActiveContainers();
                }));
            } else {
                redrawActiveContainers();
            }
        }

        function redrawActiveContainers() {
            // TODO delete any nodes from dead containers in containersToDelete
            angular.forEach($scope.activeContainers, function (container, id) {
                var containerJolokia = container.jolokia;
                if (containerJolokia) {
                    onContainerJolokia(containerJolokia, container, id);
                } else {
                    Fabric.containerJolokia(jolokia, id, function (containerJolokia) {
                        return onContainerJolokia(containerJolokia, container, id);
                    });
                }
            });
            $scope.graph = graphBuilder.buildGraph();
            Core.$apply($scope);
        }

        function doRedrawGraph() {
            graphBuilder = new ForceGraph.GraphBuilder();
            if (isFmc) {
                redrawFabricBrokers();
            } else {
                redrawLocalBroker();
            }
        }

        function brokerNameMarkup(brokerName) {
            return brokerName ? "<p></p>broker: " + brokerName + "</p>" : "";
        }

        function matchesDestinationName(destinationName, typeName) {
            if (destinationName) {
                var selection = workspace.selection;
                if (selection && selection.domain === ActiveMQ.jmxDomain) {
                    var type = selection.entries["destinationType"];
                    if (type) {
                        if ((type === "Queue" && typeName === "topic") || (type === "Topic" && typeName === "queue")) {
                            return false;
                        }
                    }
                    var destName = selection.entries["destinationName"];
                    if (destName) {
                        if (destName !== destinationName)
                            return false;
                    }
                }
                ActiveMQ.log.info("selection: " + selection);

                // TODO if the current selection is a destination...
                return !$scope.searchFilter || destinationName.indexOf($scope.searchFilter) >= 0;
            }
            return false;
        }

        function onContainerJolokia(containerJolokia, container, id) {
            if (containerJolokia) {
                container.jolokia = containerJolokia;

                function getOrAddDestination(properties) {
                    var typeName = properties.destType;
                    var brokerName = properties.brokerName;
                    var destinationName = properties.destinationName;
                    if (!matchesDestinationName(destinationName, typeName)) {
                        return null;
                    }

                    // should we be filtering this destination out
                    var hideFlag = "topic" === typeName ? $scope.viewSettings.topic : $scope.viewSettings.queue;
                    if (!hideFlag) {
                        return null;
                    }
                    var destination = getOrAddNode(typeName, destinationName, properties, function () {
                        var destinationTypeName = properties.destinationType || "Queue";
                        var objectName = "";
                        if (brokerName) {
                            // lets ignore temp topic stuff as there's no mbean for these
                            if (!destinationName.startsWith("ActiveMQ.Advisory.TempQueue_ActiveMQ.Advisory.TempTopic")) {
                                objectName = "org.apache.activemq:type=Broker,brokerName=" + brokerName + ",destinationType=" + destinationTypeName + ",destinationName=" + destinationName;
                            }
                        }
                        var answer = {
                            typeLabel: destinationTypeName,
                            brokerContainer: container,
                            objectName: objectName,
                            jolokia: containerJolokia,
                            popup: {
                                title: destinationTypeName + ": " + destinationName,
                                content: brokerNameMarkup(properties.brokerName)
                            }
                        };
                        if (!brokerName) {
                            containerJolokia.search("org.apache.activemq:destinationType=" + destinationTypeName + ",destinationName=" + destinationName + ",*", onSuccess(function (response) {
                                ActiveMQ.log.info("Found destination mbean: " + response);
                                if (response && response.length) {
                                    answer.objectName = response[0];
                                }
                            }));
                        }
                        return answer;
                    });
                    if (destination && $scope.viewSettings.broker && brokerName) {
                        addLinkIds(brokerNodeId(brokerName), destination["id"], "destination");
                    }
                    return destination;
                }

                // find networks
                var brokerId = container.brokerName;
                if (brokerId && $scope.viewSettings.network && $scope.viewSettings.broker) {
                    containerJolokia.request({ type: "read", mbean: "org.apache.activemq:connector=networkConnectors,*" }, onSuccess(function (response) {
                        angular.forEach(response.value, function (properties, objectName) {
                            var details = Core.parseMBean(objectName);
                            var attributes = details['attributes'];
                            if (properties) {
                                configureDestinationProperties(properties);
                                var remoteBrokerId = properties.RemoteBrokerName;
                                if (remoteBrokerId) {
                                    addLinkIds(brokerNodeId(brokerId), brokerNodeId(remoteBrokerId), "network");
                                }
                            }
                        });
                        graphModelUpdated();
                    }));
                }

                // find consumers
                if ($scope.viewSettings.consumer) {
                    containerJolokia.search("org.apache.activemq:endpoint=Consumer,*", onSuccess(function (response) {
                        angular.forEach(response, function (objectName) {
                            //log.info("Got consumer: " + objectName + " on container: " + id);
                            var details = Core.parseMBean(objectName);
                            if (details) {
                                var properties = details['attributes'];
                                if (properties) {
                                    configureDestinationProperties(properties);
                                    var consumerId = properties.consumerId;
                                    if (consumerId) {
                                        var destination = getOrAddDestination(properties);
                                        if (destination) {
                                            addLink(container.destinationLinkNode, destination, "destination");
                                            var consumer = getOrAddNode("consumer", consumerId, properties, function () {
                                                return {
                                                    typeLabel: "Consumer",
                                                    brokerContainer: container,
                                                    objectName: objectName,
                                                    jolokia: containerJolokia,
                                                    popup: {
                                                        title: "Consumer: " + consumerId,
                                                        content: "<p>client: " + (properties.clientId || "") + "</p> " + brokerNameMarkup(properties.brokerName)
                                                    }
                                                };
                                            });
                                            addLink(destination, consumer, "consumer");
                                        }
                                    }
                                }
                            }
                        });
                        graphModelUpdated();
                    }));
                }

                // find producers
                if ($scope.viewSettings.producer) {
                    containerJolokia.search("org.apache.activemq:endpoint=Producer,*", onSuccess(function (response) {
                        angular.forEach(response, function (objectName) {
                            var details = Core.parseMBean(objectName);
                            if (details) {
                                var properties = details['attributes'];
                                if (properties) {
                                    configureDestinationProperties(properties);
                                    var producerId = properties.producerId;
                                    if (producerId) {
                                        var destination = getOrAddDestination(properties);
                                        if (destination) {
                                            addLink(container.destinationLinkNode, destination, "destination");
                                            var producer = getOrAddNode("producer", producerId, properties, function () {
                                                return {
                                                    typeLabel: "Producer",
                                                    brokerContainer: container,
                                                    objectName: objectName,
                                                    jolokia: containerJolokia,
                                                    popup: {
                                                        title: "Producer: " + producerId,
                                                        content: "<p>client: " + (properties.clientId || "") + "</p> " + brokerNameMarkup(properties.brokerName)
                                                    }
                                                };
                                            });
                                            addLink(producer, destination, "producer");
                                        }
                                        graphModelUpdated();
                                    }
                                }
                            }
                        });
                        graphModelUpdated();
                    }));
                }

                // find dynamic producers
                if ($scope.viewSettings.producer) {
                    containerJolokia.request({ type: "read", mbean: "org.apache.activemq:endpoint=dynamicProducer,*" }, onSuccess(function (response) {
                        angular.forEach(response.value, function (mbeanValues, objectName) {
                            var details = Core.parseMBean(objectName);
                            var attributes = details['attributes'];
                            var properties = {};
                            angular.forEach(attributes, function (value, key) {
                                properties[key] = value;
                            });
                            angular.forEach(mbeanValues, function (value, key) {
                                properties[key] = value;
                            });
                            configureDestinationProperties(properties);
                            properties['destinationName'] = properties['DestinationName'];
                            var producerId = properties["producerId"] || properties["ProducerId"];
                            if (properties["DestinationTemporary"] || properties["DestinationTopc"]) {
                                properties["destType"] = "topic";
                            }
                            var destination = getOrAddDestination(properties);
                            if (producerId && destination) {
                                addLink(container.destinationLinkNode, destination, "destination");
                                var producer = getOrAddNode("producer", producerId, properties, function () {
                                    return {
                                        typeLabel: "Producer (Dynamic)",
                                        brokerContainer: container,
                                        objectName: objectName,
                                        jolokia: containerJolokia,
                                        popup: {
                                            title: "Producer (Dynamic): " + producerId,
                                            content: "<p>client: " + (properties['ClientId'] || "") + "</p> " + brokerNameMarkup(properties['brokerName'])
                                        }
                                    };
                                });
                                addLink(producer, destination, "producer");
                            }
                        });
                        graphModelUpdated();
                    }));
                }
            }
        }

        function graphModelUpdated() {
            $scope.graph = graphBuilder.buildGraph();
            Core.$apply($scope);
        }

        function getOrAddBroker(master, brokerId, groupId, containerId, container, brokerStatus) {
            var broker = null;
            var brokerFlag = master ? $scope.viewSettings.broker : $scope.viewSettings.slave;
            if (brokerFlag) {
                broker = getOrAddNode("broker", brokerId + (master ? "" : ":slave"), brokerStatus, function () {
                    return {
                        type: master ? "broker" : "brokerSlave",
                        typeLabel: master ? "Broker" : "Slave Broker",
                        popup: {
                            title: (master ? "Master" : "Slave") + " Broker: " + brokerId,
                            content: "<p>Container: " + containerId + "</p> <p>Group: " + groupId + "</p>"
                        }
                    };
                });
                if (master) {
                    if (!broker['objectName']) {
                        // lets try guess the mbean name
                        broker['objectName'] = "org.apache.activemq:type=Broker,brokerName=" + brokerId;
                        ActiveMQ.log.info("Guessed broker mbean: " + broker['objectName']);
                    }
                    if (!broker['brokerContainer'] && container) {
                        broker['brokerContainer'] = container;
                    }
                }
            }
            return broker;
        }

        function getOrAddNode(typeName, id, properties, createFn) {
            var node = null;
            if (id) {
                var nodeId = typeName + ":" + id;
                node = graphBuilder.getNode(nodeId);
                if (!node) {
                    var nodeValues = createFn();
                    node = angular.copy(properties);
                    angular.forEach(nodeValues, function (value, key) {
                        return node[key] = value;
                    });

                    node['id'] = nodeId;
                    if (!node['type']) {
                        node['type'] = typeName;
                    }
                    if (!node['name']) {
                        node['name'] = id;
                    }
                    if (node) {
                        var size = $scope.shapeSize[typeName];
                        if (size && !node['size']) {
                            node['size'] = size;
                        }
                        if (!node['summary']) {
                            node['summary'] = node['popup'] || "";
                        }
                        if (!$scope.viewSettings.popup) {
                            delete node['popup'];
                        }
                        if (!$scope.viewSettings.label) {
                            delete node['name'];
                        }

                        // lets not add nodes which are defined as being disabled
                        var enabled = $scope.viewSettings[typeName];
                        if (enabled || !angular.isDefined(enabled)) {
                            //log.info("Adding node " + nodeId + " of type + " + typeName);
                            graphBuilder.addNode(node);
                        } else {
                            //log.info("Ignoring node " + nodeId + " of type + " + typeName);
                        }
                    }
                }
            }
            return node;
        }

        function addLink(object1, object2, linkType) {
            if (object1 && object2) {
                addLinkIds(object1.id, object2.id, linkType);
            }
        }

        function addLinkIds(id1, id2, linkType) {
            if (id1 && id2) {
                graphBuilder.addLink(id1, id2, linkType);
            }
        }

        function brokerNodeId(brokerId) {
            return brokerId ? "broker:" + brokerId : null;
        }

        /**
        * Avoid the JMX type property clashing with the ForceGraph type property; used for associating css classes with nodes on the graph
        *
        * @param properties
        */
        function renameTypeProperty(properties) {
            properties.mbeanType = properties['type'];
            delete properties['type'];
        }

        function configureDestinationProperties(properties) {
            renameTypeProperty(properties);
            var destinationType = properties.destinationType || "Queue";
            var typeName = destinationType.toLowerCase();
            properties.isQueue = !typeName.startsWith("t");
            properties['destType'] = typeName;
        }
    }
    ActiveMQ.BrokerDiagramController = BrokerDiagramController;
})(ActiveMQ || (ActiveMQ = {}));
/**
* @module ActiveMQ
* @main ActiveMQ
*/
var ActiveMQ;
(function (ActiveMQ) {
    var pluginName = 'activemq';

    angular.module(pluginName, ['bootstrap', 'ngResource', 'ui.bootstrap.dialog', 'hawtioCore', 'camel', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/activemq/browseQueue', { templateUrl: 'app/activemq/html/browseQueue.html' }).when('/activemq/diagram', { templateUrl: 'app/activemq/html/brokerDiagram.html', reloadOnSearch: false }).when('/activemq/createDestination', { templateUrl: 'app/activemq/html/createDestination.html' }).when('/activemq/createQueue', { templateUrl: 'app/activemq/html/createQueue.html' }).when('/activemq/createTopic', { templateUrl: 'app/activemq/html/createTopic.html' }).when('/activemq/deleteQueue', { templateUrl: 'app/activemq/html/deleteQueue.html' }).when('/activemq/deleteTopic', { templateUrl: 'app/activemq/html/deleteTopic.html' }).when('/activemq/sendMessage', { templateUrl: 'app/camel/html/sendMessage.html' }).when('/activemq/durableSubscribers', { templateUrl: 'app/activemq/html/durableSubscribers.html' }).when('/activemq/jobs', { templateUrl: 'app/activemq/html/jobs.html' });
    }).run(function ($location, workspace, viewRegistry, helpRegistry, preferencesRegistry) {
        viewRegistry['activemq'] = 'app/activemq/html/layoutActiveMQTree.html';
        helpRegistry.addUserDoc('activemq', 'app/activemq/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("org.apache.activemq");
        });

        preferencesRegistry.addTab("ActiveMQ", "app/activemq/html/preferences.html", function () {
            return workspace.treeContainsDomainAndProperties("org.apache.activemq");
        });

        workspace.addTreePostProcessor(postProcessTree);

        // register default attribute views
        var attributes = workspace.attributeColumnDefs;
        attributes[ActiveMQ.jmxDomain + "/Broker/folder"] = [
            { field: 'BrokerName', displayName: 'Name', width: "**" },
            { field: 'TotalProducerCount', displayName: 'Producer #' },
            { field: 'TotalConsumerCount', displayName: 'Consumer #' },
            { field: 'StorePercentUsage', displayName: 'Store %' },
            { field: 'TempPercentUsage', displayName: 'Temp %' },
            { field: 'MemoryPercentUsage', displayName: 'Memory %' },
            { field: 'TotalEnqueueCount', displayName: 'Enqueue #' },
            { field: 'TotalDequeueCount', displayName: 'Dequeue #' }
        ];
        attributes[ActiveMQ.jmxDomain + "/Queue/folder"] = [
            { field: 'Name', displayName: 'Name', width: "***" },
            { field: 'QueueSize', displayName: 'Queue Size' },
            { field: 'ProducerCount', displayName: 'Producer #' },
            { field: 'ConsumerCount', displayName: 'Consumer #' },
            { field: 'EnqueueCount', displayName: 'Enqueue #' },
            { field: 'DequeueCount', displayName: 'Dequeue #' },
            { field: 'MemoryPercentUsage', displayName: 'Memory %' },
            { field: 'DispatchCount', displayName: 'Dispatch #', visible: false }
        ];
        attributes[ActiveMQ.jmxDomain + "/Topic/folder"] = [
            { field: 'Name', displayName: 'Name', width: "****" },
            { field: 'ProducerCount', displayName: 'Producer #' },
            { field: 'ConsumerCount', displayName: 'Consumer #' },
            { field: 'EnqueueCount', displayName: 'Enqueue #' },
            { field: 'DequeueCount', displayName: 'Dequeue #' },
            { field: 'MemoryPercentUsage', displayName: 'Memory %' },
            { field: 'DispatchCount', displayName: 'Dispatch #', visible: false }
        ];
        attributes[ActiveMQ.jmxDomain + "/Consumer/folder"] = [
            { field: 'ConnectionId', displayName: 'Name', width: "**" },
            { field: 'PrefetchSize', displayName: 'Prefetch Size' },
            { field: 'Priority', displayName: 'Priority' },
            { field: 'DispatchedQueueSize', displayName: 'Dispatched Queue #' },
            { field: 'SlowConsumer', displayName: 'Slow ?' },
            { field: 'Retroactive', displayName: 'Retroactive' },
            { field: 'Selector', displayName: 'Selector' }
        ];
        attributes[ActiveMQ.jmxDomain + "/networkConnectors/folder"] = [
            { field: 'Name', displayName: 'Name', width: "**" },
            { field: 'UserName', displayName: 'User Name' },
            { field: 'PrefetchSize', displayName: 'Prefetch Size' },
            { field: 'ConduitSubscriptions', displayName: 'Conduit Subscriptions?' },
            { field: 'Duplex', displayName: 'Duplex' },
            { field: 'DynamicOnly', displayName: 'Dynamic Only' }
        ];
        attributes[ActiveMQ.jmxDomain + "/PersistenceAdapter/folder"] = [
            { field: 'IndexDirectory', displayName: 'Index Directory', width: "**" },
            { field: 'LogDirectory', displayName: 'Log Directory', width: "**" }
        ];

        workspace.topLevelTabs.push({
            id: "activemq",
            content: "ActiveMQ",
            title: "Manage your ActiveMQ message brokers",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("org.apache.activemq");
            },
            href: function () {
                return "#/jmx/attributes?tab=activemq";
            },
            isActive: function () {
                return workspace.isTopTabActive("activemq");
            }
        });

        // add sub level tabs
        workspace.subLevelTabs.push({
            content: '<i class="icon-envelope"></i> Browse',
            title: "Browse the messages on the queue",
            isValid: function (workspace) {
                return isQueue(workspace);
            },
            href: function () {
                return "#/activemq/browseQueue";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-pencil"></i> Send',
            title: "Send a message to this destination",
            isValid: function (workspace) {
                return isQueue(workspace) || isTopic(workspace);
            },
            href: function () {
                return "#/activemq/sendMessage";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-picture"></i> Diagram',
            title: "View a diagram of the producers, destinations and consumers",
            isValid: function (workspace) {
                return workspace.isTopTabActive("activemq") || workspace.selectionHasDomain(ActiveMQ.jmxDomain);
            },
            href: function () {
                return "#/activemq/diagram";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-plus"></i> Create',
            title: "Create a new destination",
            isValid: function (workspace) {
                return isBroker(workspace);
            },
            href: function () {
                return "#/activemq/createDestination";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-plus"></i> Create',
            title: "Create a new queue",
            isValid: function (workspace) {
                return isQueuesFolder(workspace);
            },
            href: function () {
                return "#/activemq/createQueue";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-plus"></i> Create',
            title: "Create a new topic",
            isValid: function (workspace) {
                return isTopicsFolder(workspace);
            },
            href: function () {
                return "#/activemq/createTopic";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-remove"></i> Delete Topic',
            title: "Delete this topic",
            isValid: function (workspace) {
                return isTopic(workspace);
            },
            href: function () {
                return "#/activemq/deleteTopic";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-remove"></i> Delete',
            title: "Delete or purge this queue",
            isValid: function (workspace) {
                return isQueue(workspace);
            },
            href: function () {
                return "#/activemq/deleteQueue";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-list"></i> Durable Subscribers',
            title: "Manage durable subscribers",
            isValid: function (workspace) {
                return isBroker(workspace);
            },
            href: function () {
                return "#/activemq/durableSubscribers";
            }
        });

        workspace.subLevelTabs.push({
            content: '<i class="icon-list"></i> Jobs',
            title: "Manage jobs",
            isValid: function (workspace) {
                return isJobScheduler(workspace);
            },
            href: function () {
                return "#/activemq/jobs";
            }
        });

        function postProcessTree(tree) {
            var activemq = tree.get("org.apache.activemq");
            setConsumerType(activemq);

            // lets move queue and topic as first children within brokers
            if (activemq) {
                angular.forEach(activemq.children, function (broker) {
                    angular.forEach(broker.children, function (child) {
                        // lets move Topic/Queue to the front.
                        var grandChildren = child.children;
                        if (grandChildren) {
                            var names = ["Topic", "Queue"];
                            angular.forEach(names, function (name) {
                                var idx = grandChildren.findIndex(function (n) {
                                    return n.title === name;
                                });
                                if (idx > 0) {
                                    var old = grandChildren[idx];
                                    grandChildren.splice(idx, 1);
                                    grandChildren.splice(0, 0, old);
                                }
                            });
                        }
                    });
                });
            }
        }

        function setConsumerType(node) {
            if (node) {
                var parent = node.parent;
                var entries = node.entries;
                if (parent && !parent.typeName && entries) {
                    var endpoint = entries["endpoint"];
                    if (endpoint === "Consumer" || endpoint === "Producer") {
                        //console.log("Setting the typeName on " + parent.title + " to " + endpoint);
                        parent.typeName = endpoint;
                    }
                    var connectorName = entries["connectorName"];
                    if (connectorName && !node.icon) {
                        // lets default a connector icon
                        node.icon = url("/img/icons/activemq/connector.png");
                    }
                }
                angular.forEach(node.children, function (child) {
                    return setConsumerType(child);
                });
            }
        }
    });

    hawtioPluginLoader.addModule(pluginName);

    function isQueue(workspace) {
        //return workspace.selectionHasDomainAndType(jmxDomain, 'Queue');
        return workspace.hasDomainAndProperties(ActiveMQ.jmxDomain, { 'destinationType': 'Queue' }, 4) || workspace.selectionHasDomainAndType(ActiveMQ.jmxDomain, 'Queue');
    }
    ActiveMQ.isQueue = isQueue;

    function isTopic(workspace) {
        //return workspace.selectionHasDomainAndType(jmxDomain, 'Topic');
        return workspace.hasDomainAndProperties(ActiveMQ.jmxDomain, { 'destinationType': 'Topic' }, 4) || workspace.selectionHasDomainAndType(ActiveMQ.jmxDomain, 'Topic');
    }
    ActiveMQ.isTopic = isTopic;

    function isQueuesFolder(workspace) {
        return workspace.selectionHasDomainAndLastFolderName(ActiveMQ.jmxDomain, 'Queue');
    }
    ActiveMQ.isQueuesFolder = isQueuesFolder;

    function isTopicsFolder(workspace) {
        return workspace.selectionHasDomainAndLastFolderName(ActiveMQ.jmxDomain, 'Topic');
    }
    ActiveMQ.isTopicsFolder = isTopicsFolder;

    function isJobScheduler(workspace) {
        return workspace.hasDomainAndProperties(ActiveMQ.jmxDomain, { 'service': 'JobScheduler' }, 4);
    }
    ActiveMQ.isJobScheduler = isJobScheduler;

    function isBroker(workspace) {
        if (workspace.selectionHasDomainAndType(ActiveMQ.jmxDomain, 'Broker')) {
            var parent = Core.pathGet(workspace, ["selection", "parent"]);
            return !(parent && parent.ancestorHasType('Broker'));
        }
        return false;
    }
    ActiveMQ.isBroker = isBroker;
})(ActiveMQ || (ActiveMQ = {}));
var ActiveMQ;
(function (ActiveMQ) {
    function DurableSubscriberController($scope, workspace, jolokia) {
        $scope.refresh = loadTable;

        $scope.durableSubscribers = [];

        $scope.tempData = [];

        $scope.createSubscriberDialog = new UI.Dialog();
        $scope.deleteSubscriberDialog = new UI.Dialog();
        $scope.showSubscriberDialog = new UI.Dialog();

        $scope.topicName = '';
        $scope.clientId = '';
        $scope.subscriberName = '';
        $scope.subSelector = '';

        $scope.gridOptions = {
            selectedItems: [],
            data: 'durableSubscribers',
            displayFooter: false,
            showFilter: false,
            showColumnMenu: true,
            enableCellSelection: false,
            enableColumnResize: true,
            enableColumnReordering: true,
            selectWithCheckboxOnly: false,
            showSelectionCheckbox: false,
            multiSelect: false,
            displaySelectionCheckbox: false,
            filterOptions: {
                filterText: ''
            },
            maintainColumnRatios: false,
            columnDefs: [
                {
                    field: 'destinationName',
                    displayName: 'Topic',
                    width: '30%'
                },
                {
                    field: 'clientId',
                    displayName: 'Client ID',
                    width: '30%'
                },
                {
                    field: 'consumerId',
                    displayName: 'Consumer ID',
                    cellTemplate: '<div class="ngCellText"><span ng-hide="row.entity.status != \'Offline\'">{{row.entity.consumerId}}</span><a ng-show="row.entity.status != \'Offline\'" ng-click="openSubscriberDialog(row)">{{row.entity.consumerId}}</a></div>',
                    width: '30%'
                },
                {
                    field: 'status',
                    displayName: 'Status',
                    width: '10%'
                }
            ]
        };

        $scope.doCreateSubscriber = function (clientId, subscriberName, topicName, subSelector) {
            $scope.createSubscriberDialog.close();
            $scope.clientId = clientId;
            $scope.subscriberName = subscriberName;
            $scope.topicName = topicName;
            $scope.subSelector = subSelector;
            if (Core.isBlank($scope.subSelector)) {
                $scope.subSelector = null;
            }
            var mbean = getBrokerMBean(jolokia);
            if (mbean) {
                jolokia.execute(mbean, "createDurableSubscriber(java.lang.String, java.lang.String, java.lang.String, java.lang.String)", $scope.clientId, $scope.subscriberName, $scope.topicName, $scope.subSelector, onSuccess(function () {
                    notification('success', "Created durable subscriber " + clientId);
                    $scope.clientId = '';
                    $scope.subscriberName = '';
                    $scope.topicName = '';
                    $scope.subSelector = '';
                    loadTable();
                }));
            } else {
                notification("error", "Could not find the Broker MBean!");
            }
        };

        $scope.deleteSubscribers = function () {
            var mbean = $scope.gridOptions.selectedItems[0]._id;
            jolokia.execute(mbean, "destroy()", onSuccess(function () {
                $scope.showSubscriberDialog.close();
                notification('success', "Deleted durable subscriber");
                loadTable();
                $scope.gridOptions.selectedItems = [];
            }));
        };

        $scope.openSubscriberDialog = function (subscriber) {
            jolokia.request({ type: "read", mbean: subscriber.entity._id }, onSuccess(function (response) {
                $scope.showSubscriberDialog.subscriber = response.value;
                $scope.showSubscriberDialog.subscriber.Status = subscriber.entity.status;
                console.log("Subscriber is now " + $scope.showSubscriberDialog.subscriber);
                Core.$apply($scope);

                // now lets start opening the dialog
                setTimeout(function () {
                    $scope.showSubscriberDialog.open();
                    Core.$apply($scope);
                }, 100);
            }));
        };

        $scope.topicNames = function (completionText) {
            var topicsFolder = ActiveMQ.getSelectionTopicsFolder(workspace);
            return (topicsFolder) ? topicsFolder.children.map(function (n) {
                return n.title;
            }) : [];
        };

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid())
                return;

            // lets defer execution as we may not have the selection just yet
            setTimeout(loadTable, 50);
        });

        function loadTable() {
            var mbean = getBrokerMBean(jolokia);
            if (mbean) {
                $scope.durableSubscribers = [];
                jolokia.request({ type: "read", mbean: mbean, attribute: ["DurableTopicSubscribers"] }, onSuccess(function (response) {
                    return populateTable(response, "DurableTopicSubscribers", "Active");
                }));
                jolokia.request({ type: "read", mbean: mbean, attribute: ["InactiveDurableTopicSubscribers"] }, onSuccess(function (response) {
                    return populateTable(response, "InactiveDurableTopicSubscribers", "Offline");
                }));
            }
        }

        function populateTable(response, attr, status) {
            var data = response.value;
            ActiveMQ.log.debug("Got data: ", data);
            $scope.durableSubscribers.push.apply($scope.durableSubscribers, data[attr].map(function (o) {
                var objectName = o["objectName"];
                var entries = Core.objectNameProperties(objectName);
                if (!('objectName' in o)) {
                    if ('canonicalName' in o) {
                        objectName = o['canonicalName'];
                    }
                    entries = Object.extended(o['keyPropertyList']).clone();
                }

                entries["_id"] = objectName;
                entries["status"] = status;
                return entries;
            }));

            Core.$apply($scope);
        }

        function getBrokerMBean(jolokia) {
            var mbean = null;
            var selection = workspace.selection;
            if (selection && ActiveMQ.isBroker(workspace) && selection.objectName) {
                return selection.objectName;
            }
            var folderNames = selection.folderNames;

            //if (selection && jolokia && folderNames && folderNames.length > 1) {
            var parent = selection ? selection.parent : null;
            if (selection && parent && jolokia && folderNames && folderNames.length > 1) {
                mbean = parent.objectName;

                // we might be a destination, so lets try one more parent
                if (!mbean && parent) {
                    mbean = parent.parent.objectName;
                }
                if (!mbean) {
                    mbean = "" + folderNames[0] + ":BrokerName=" + folderNames[1] + ",Type=Broker";
                }
            }
            return mbean;
        }
    }
    ActiveMQ.DurableSubscriberController = DurableSubscriberController;
})(ActiveMQ || (ActiveMQ = {}));
var ActiveMQ;
(function (ActiveMQ) {
    function BrowseQueueController($scope, workspace, jolokia, localStorage) {
        $scope.searchText = '';

        $scope.messages = [];
        $scope.headers = {};
        $scope.mode = 'text';

        $scope.deleteDialog = false;
        $scope.moveDialog = false;

        $scope.gridOptions = {
            selectedItems: [],
            data: 'messages',
            displayFooter: false,
            showFilter: false,
            showColumnMenu: true,
            enableColumnResize: true,
            enableColumnReordering: true,
            filterOptions: {
                filterText: ''
            },
            selectWithCheckboxOnly: true,
            showSelectionCheckbox: true,
            maintainColumnRatios: false,
            columnDefs: [
                {
                    field: 'JMSMessageID',
                    displayName: 'Message ID',
                    cellTemplate: '<div class="ngCellText"><a ng-click="openMessageDialog(row)">{{row.entity.JMSMessageID}}</a></div>',
                    // for ng-grid
                    width: '34%'
                },
                {
                    field: 'JMSType',
                    displayName: 'Type',
                    width: '10%'
                },
                {
                    field: 'JMSPriority',
                    displayName: 'Priority',
                    width: '7%'
                },
                {
                    field: 'JMSTimestamp',
                    displayName: 'Timestamp',
                    width: '19%'
                },
                {
                    field: 'JMSExpiration',
                    displayName: 'Expires',
                    width: '10%'
                },
                {
                    field: 'JMSReplyTo',
                    displayName: 'Reply To',
                    width: '10%'
                },
                {
                    field: 'JMSCorrelationID',
                    displayName: 'Correlation ID',
                    width: '10%'
                }
            ]
        };

        $scope.showMessageDetails = false;

        var ignoreColumns = ["PropertiesText", "BodyPreview", "Text"];
        var flattenColumns = [
            "BooleanProperties", "ByteProperties", "ShortProperties", "IntProperties", "LongProperties", "FloatProperties",
            "DoubleProperties", "StringProperties"];

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid())
                return;

            // lets defer execution as we may not have the selection just yet
            setTimeout(loadTable, 50);
        });

        $scope.openMessageDialog = function (message) {
            var idx = Core.pathGet(message, ["rowIndex"]);
            $scope.selectRowIndex(idx);
            if ($scope.row) {
                $scope.mode = CodeEditor.detectTextFormat($scope.row.Text);
                $scope.showMessageDetails = true;
            }
        };

        $scope.refresh = loadTable;

        $scope.selectRowIndex = function (idx) {
            $scope.rowIndex = idx;
            var selected = $scope.gridOptions.selectedItems;
            selected.splice(0, selected.length);
            if (idx >= 0 && idx < $scope.messages.length) {
                $scope.row = $scope.messages[idx];
                if ($scope.row) {
                    selected.push($scope.row);
                }
            } else {
                $scope.row = null;
            }
        };

        $scope.moveMessages = function () {
            var selection = workspace.selection;
            var mbean = selection.objectName;
            if (mbean && selection) {
                var selectedItems = $scope.gridOptions.selectedItems;
                $scope.message = "Moved " + Core.maybePlural(selectedItems.length, "message" + " to " + $scope.queueName);
                var operation = "moveMessageTo(java.lang.String, java.lang.String)";
                angular.forEach(selectedItems, function (item, idx) {
                    var id = item.JMSMessageID;
                    if (id) {
                        var callback = (idx + 1 < selectedItems.length) ? intermediateResult : moveSuccess;
                        jolokia.execute(mbean, operation, id, $scope.queueName, onSuccess(callback));
                    }
                });
            }
        };

        $scope.deleteMessages = function () {
            var selection = workspace.selection;
            var mbean = selection.objectName;
            if (mbean && selection) {
                var selectedItems = $scope.gridOptions.selectedItems;
                $scope.message = "Deleted " + Core.maybePlural(selectedItems.length, "message");
                var operation = "removeMessage(java.lang.String)";
                angular.forEach(selectedItems, function (item, idx) {
                    var id = item.JMSMessageID;
                    if (id) {
                        var callback = (idx + 1 < selectedItems.length) ? intermediateResult : operationSuccess;
                        jolokia.execute(mbean, operation, id, onSuccess(callback));
                    }
                });
            }
        };

        $scope.retryMessages = function () {
            var selection = workspace.selection;
            var mbean = selection.objectName;
            if (mbean && selection) {
                var selectedItems = $scope.gridOptions.selectedItems;
                $scope.message = "Retry " + Core.maybePlural(selectedItems.length, "message");
                var operation = "retryMessage(java.lang.String)";
                angular.forEach(selectedItems, function (item, idx) {
                    var id = item.JMSMessageID;
                    if (id) {
                        var callback = (idx + 1 < selectedItems.length) ? intermediateResult : operationSuccess;
                        jolokia.execute(mbean, operation, id, onSuccess(callback));
                    }
                });
            }
        };

        $scope.queueNames = function (completionText) {
            var queuesFolder = ActiveMQ.getSelectionQueuesFolder(workspace);
            return (queuesFolder) ? queuesFolder.children.map(function (n) {
                return n.title;
            }) : [];
        };

        function populateTable(response) {
            var data = response.value;
            if (!angular.isArray(data)) {
                $scope.messages = [];
                angular.forEach(data, function (value, idx) {
                    $scope.messages.push(value);
                });
            } else {
                $scope.messages = data;
            }
            angular.forEach($scope.messages, function (message) {
                message.headerHtml = createHeaderHtml(message);
                message.bodyText = createBodyText(message);
            });
            Core.$apply($scope);
        }

        /*
        * For some reason using ng-repeat in the modal dialog doesn't work so lets
        * just create the HTML in code :)
        */
        function createBodyText(message) {
            if (message.Text) {
                var body = message.Text;
                var lenTxt = "" + body.length;
                message.textMode = "text (" + lenTxt + " chars)";
                return body;
            } else if (message.BodyPreview) {
                var code = Core.parseIntValue(localStorage["activemqBrowseBytesMessages"] || "1", "browse bytes messages");
                var body;

                message.textMode = "bytes (turned off)";
                if (code != 99) {
                    var bytesArr = [];
                    var textArr = [];
                    message.BodyPreview.forEach(function (b) {
                        if (code === 1 || code === 2) {
                            // text
                            textArr.push(String.fromCharCode(b));
                        }
                        if (code === 1 || code === 4) {
                            // hex and must be 2 digit so they space out evenly
                            var s = b.toString(16);
                            if (s.length === 1) {
                                s = "0" + s;
                            }
                            bytesArr.push(s);
                        } else {
                            // just show as is without spacing out, as that is usually more used for hex than decimal
                            var s = b.toString(10);
                            bytesArr.push(s);
                        }
                    });

                    var bytesData = bytesArr.join(" ");
                    var textData = textArr.join("");

                    if (code === 1 || code === 2) {
                        // bytes and text
                        var len = message.BodyPreview.length;
                        var lenTxt = "" + textArr.length;
                        body = "bytes:\n" + bytesData + "\n\ntext:\n" + textData;
                        message.textMode = "bytes (" + len + " bytes) and text (" + lenTxt + " chars)";
                    } else {
                        // bytes only
                        var len = message.BodyPreview.length;
                        body = bytesData;
                        message.textMode = "bytes (" + len + " bytes)";
                    }
                }
                return body;
            } else {
                message.textMode = "unsupported";
                return "Unsupported message body type which cannot be displayed by hawtio";
            }
        }

        /*
        * For some reason using ng-repeat in the modal dialog doesn't work so lets
        * just create the HTML in code :)
        */
        function createHeaderHtml(message) {
            var headers = createHeaders(message);
            var properties = createProperties(message);
            var headerKeys = Object.extended(headers).keys();

            function sort(a, b) {
                if (a > b)
                    return 1;
                if (a < b)
                    return -1;
                return 0;
            }

            var propertiesKeys = Object.extended(properties).keys().sort(sort);

            var jmsHeaders = headerKeys.filter(function (key) {
                return key.startsWith("JMS");
            }).sort(sort);

            var remaining = headerKeys.subtract(jmsHeaders, propertiesKeys).sort(sort);

            var buffer = [];

            function appendHeader(key) {
                var value = headers[key];
                if (value === null) {
                    value = '';
                }

                buffer.push('<tr><td class="propertyName"><span class="green">Header</span> - ' + key + '</td><td class="property-value">' + value + '</td></tr>');
            }

            function appendProperty(key) {
                var value = properties[key];
                if (value === null) {
                    value = '';
                }

                buffer.push('<tr><td class="propertyName">' + key + '</td><td class="property-value">' + value + '</td></tr>');
            }

            jmsHeaders.forEach(appendHeader);
            remaining.forEach(appendHeader);
            propertiesKeys.forEach(appendProperty);
            return buffer.join("\n");
        }

        function createHeaders(row) {
            ActiveMQ.log.debug("headers: ", row);
            var answer = {};
            angular.forEach(row, function (value, key) {
                if (!ignoreColumns.any(key) && !flattenColumns.any(key)) {
                    answer[Core.escapeHtml(key)] = Core.escapeHtml(value);
                }
            });
            return answer;
        }

        function createProperties(row) {
            ActiveMQ.log.debug("properties: ", row);
            var answer = {};
            angular.forEach(row, function (value, key) {
                if (!ignoreColumns.any(key) && flattenColumns.any(key)) {
                    angular.forEach(value, function (v2, k2) {
                        answer['<span class="green">' + key.replace('Properties', ' Property') + '</span> - ' + Core.escapeHtml(k2)] = Core.escapeHtml(v2);
                    });
                }
            });
            return answer;
        }

        function loadTable() {
            var selection = workspace.selection;
            if (selection) {
                var mbean = selection.objectName;
                if (mbean) {
                    $scope.dlq = false;
                    jolokia.getAttribute(mbean, "DLQ", onSuccess(onDlq, { silent: true }));
                    jolokia.request({ type: 'exec', mbean: mbean, operation: 'browse()' }, onSuccess(populateTable));
                }
            }
        }

        function onDlq(response) {
            $scope.dlq = response;
            Core.$apply($scope);
        }

        function intermediateResult() {
        }

        function operationSuccess() {
            $scope.messageDialog = false;
            $scope.gridOptions.selectedItems.splice(0);
            notification("success", $scope.message);
            setTimeout(loadTable, 50);
        }

        function moveSuccess() {
            operationSuccess();
            workspace.loadTree();
        }
    }
    ActiveMQ.BrowseQueueController = BrowseQueueController;
})(ActiveMQ || (ActiveMQ = {}));
/**
* @module Site
*/
var Site;
(function (Site) {
    function IndexController($scope, $location) {
        $scope.slideInterval = 5000;
    }
    Site.IndexController = IndexController;
})(Site || (Site = {}));
/**
* @module Site
*/
var Site;
(function (Site) {
    function PageController($scope, $routeParams, $location, $compile, $http, fileExtensionTypeRegistry) {
        var log = Logger.get("Site");
        var pageId = $routeParams["page"];
        if (!pageId) {
            pageId = "README.md";
            /*
            $location.path("/site/doc/index.md");
            return;
            */
        }

        if (!pageId.startsWith("/") && pageId.indexOf(":/") < 0 && pageId.indexOf("app/site/") < 0) {
            // lets assume the page is relative to app/site/
            pageId = "app/site/" + pageId;
        }
        $scope.pageId = pageId;
        $scope.pageFolder = pageId.substring(0, pageId.lastIndexOf('/') + 1);

        log.info("Loading page '" + $scope.pageId + "'");

        $scope.getContents = function (filename, cb) {
            var fullPath = $scope.pageFolder + filename;
            log.info("Loading the contents of: " + fullPath);
            $http.get(fullPath).success(cb).error(function () {
                return cb(" ");
            });
        };

        $http.get($scope.pageId).success(onResults);

        function onResults(contents, status, headers, config) {
            $scope.contents = contents;
            $scope.html = contents;

            var format = Wiki.fileFormat($scope.pageId, fileExtensionTypeRegistry) || "markdown";
            if ("markdown" === format) {
                // lets convert it to HTML
                $scope.html = contents ? marked(contents) : "";
            } else if (format && format.startsWith("html")) {
                $scope.html = contents;
            } else {
                // TODO?
            }
            $compile($scope.html)($scope);
            Core.$apply($scope);
        }
    }
    Site.PageController = PageController;
})(Site || (Site = {}));
/**
* @module Site
*/
var Site;
(function (Site) {
    Site.sitePluginEnabled = false;

    function isSiteNavBarValid() {
        return Site.sitePluginEnabled;
    }
    Site.isSiteNavBarValid = isSiteNavBarValid;
})(Site || (Site = {}));
/**
* @module Site
* @main Site
*/
var Site;
(function (Site) {
    var pluginName = 'site';

    angular.module(pluginName, ['bootstrap', 'ngResource', 'ngGrid', 'datatable', 'hawtioCore', 'hawtio-ui']).config(function ($routeProvider) {
        $routeProvider.when('/site', { templateUrl: 'app/site/html/index.html' }).when('/site/', { templateUrl: 'app/site/html/index.html' }).when('/site/book/*page', { templateUrl: 'app/site/html/book.html', reloadOnSearch: false }).when('/site/*page', { templateUrl: 'app/site/html/page.html' });
    }).run(function ($location, workspace, viewRegistry, layoutFull, helpRegistry) {
        viewRegistry[pluginName] = layoutFull;

        workspace.topLevelTabs.push({
            id: "site",
            content: "Site",
            title: "View the documentation for Hawtio",
            isValid: function (workspace) {
                return false;
            },
            href: function () {
                return "#/site";
            }
        });
        /*
        helpRegistry.addUserDoc('log', 'app/log/doc/help.md', () => {
        return workspace.treeContainsDomainAndProperties('org.fusesource.insight', {type: 'LogQuery'});
        });
        
        */
    });

    hawtioPluginLoader.addModule(pluginName);
})(Site || (Site = {}));
var Tomcat;
(function (Tomcat) {
    function TomcatController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | tomcatIconClass}}"></i></div>';
        var urlTemplate = '<div class="ngCellText" title="{{row.getProperty(col.field)}}">' + '<a ng-href="{{row.getProperty(col.field)}}" target="_blank">{{row.getProperty(col.field)}}</a>' + '</div>';

        $scope.uninstallDialog = new UI.Dialog();

        $scope.httpPort;
        $scope.httpScheme = "http";

        $scope.webapps = [];
        $scope.selected = [];

        var columnDefsTomcat5 = [
            {
                field: 'state',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'path',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'url',
                displayName: 'Url',
                cellTemplate: urlTemplate,
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'startTime',
                displayName: 'Start Time',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        var columnDefsTomcat6 = [
            {
                field: 'stateName',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'path',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'url',
                displayName: 'Url',
                cellTemplate: urlTemplate,
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'startTime',
                displayName: 'Start Time',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        var columnDefsTomcat7 = [
            {
                field: 'stateName',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'path',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'displayName',
                displayName: 'Display Name',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'url',
                displayName: 'Url',
                cellTemplate: urlTemplate,
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'startTime',
                displayName: 'Start Time',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'webapps',
            displayFooter: true,
            selectedItems: $scope.selected,
            selectWithCheckboxOnly: true,
            filterOptions: {
                filterText: ''
            }
        };

        /*        function extractHttpPort(response) {
        var obj = response;
        if (obj) {
        angular.forEach(obj, function (key, value) {
        var mbean = key;
        jolokia.request({type: "read", mbean: mbean, attribute: ["port", "scheme", "protocol"]}, onSuccess(onHttpPort));
        });
        }
        }
        
        function onHttpPort(response) {
        // we only need the HTTP protocol
        var obj = response.value;
        if (obj && obj.protocol && obj.protocol.toString().startsWith("HTTP")) {
        $scope.httpPort = obj.port;
        $scope.httpScheme = obj.scheme;
        }
        }
        */
        function render(response) {
            response = Tomcat.filerTomcatOrCatalina(response);

            $scope.webapps = [];
            $scope.mbeanIndex = {};
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    var mbean = obj.mbean;

                    // compute the url for the webapp, and we want to use http as scheme
                    var hostname = Core.extractTargetUrl($location, $scope.httpScheme, $scope.httpPort);
                    obj.url = hostname + obj['path'];

                    if (mbean) {
                        // format the start time as readable date format
                        obj.startTime = Tomcat.millisToDateFormat(obj.startTime);

                        var idx = $scope.mbeanIndex[mbean];
                        if (angular.isDefined(idx)) {
                            $scope.webapps[mbean] = obj;
                        } else {
                            $scope.mbeanIndex[mbean] = $scope.webapps.length;
                            $scope.webapps.push(obj);
                        }

                        // ensure web page is updated
                        Core.$apply($scope);
                    }
                }
            }

            angular.forEach(response, function (value, key) {
                var mbean = value;
                if (Tomcat.isTomcat5($scope.tomcatServerVersion)) {
                    jolokia.request({
                        type: "read", mbean: mbean,
                        attribute: ["path", "state", "startTime"] }, onSuccess(onAttributes));
                } else if (Tomcat.isTomcat6($scope.tomcatServerVersion)) {
                    jolokia.request({
                        type: "read", mbean: mbean,
                        attribute: ["path", "stateName", "startTime"] }, onSuccess(onAttributes));
                } else {
                    jolokia.request({
                        type: "read", mbean: mbean,
                        attribute: ["displayName", "path", "stateName", "startTime"] }, onSuccess(onAttributes));
                }
            });
            Core.$apply($scope);
        }
        ;

        // function to control the web applications
        $scope.controlWebApps = function (op) {
            // grab id of mbean names to control
            var mbeanNames = $scope.selected.map(function (b) {
                return b.mbean;
            });
            if (!angular.isArray(mbeanNames)) {
                mbeanNames = [mbeanNames];
            }

            // execute operation on each mbean
            var lastIndex = (mbeanNames.length || 1) - 1;
            angular.forEach(mbeanNames, function (mbean, idx) {
                var onResponse = (idx >= lastIndex) ? $scope.onLastResponse : $scope.onResponse;
                jolokia.request({
                    type: 'exec',
                    mbean: mbean,
                    operation: op,
                    arguments: null
                }, onSuccess(onResponse, { error: onResponse }));
            });
        };

        $scope.stop = function () {
            $scope.controlWebApps('stop');
        };

        $scope.start = function () {
            $scope.controlWebApps('start');
        };

        $scope.reload = function () {
            $scope.controlWebApps('reload');
        };

        $scope.uninstall = function () {
            $scope.controlWebApps('destroy');
            $scope.uninstallDialog.close();
        };

        // function to trigger reloading page
        $scope.onLastResponse = function (response) {
            $scope.onResponse(response);

            // we only want to force updating the data on the last response
            loadData();
        };

        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
        };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading tomcat webapp data...");

            // must load connectors first, before showing applications, so we do this call synchronously
            var connectors = jolokia.search("Catalina:type=Connector,*");
            if (connectors) {
                var found = false;
                angular.forEach(connectors, function (key, value) {
                    var mbean = key;
                    if (!found) {
                        var data = jolokia.request({ type: "read", mbean: mbean, attribute: ["port", "scheme", "protocol"] });
                        if (data && data.value && data.value.protocol && data.value.protocol.toString().toLowerCase().startsWith("http")) {
                            found = true;
                            $scope.httpPort = data.value.port;
                            $scope.httpScheme = data.value.scheme;
                        }
                    }
                });
            }
            jolokia.search("*:j2eeType=WebModule,*", onSuccess(render));
        }

        // grab server information once
        $scope.tomcatServerVersion = "";

        var servers = jolokia.search("*:type=Server");
        servers = Tomcat.filerTomcatOrCatalina(servers);
        if (servers && servers.length === 1) {
            $scope.tomcatServerVersion = jolokia.getAttribute(servers[0], "serverInfo");
        } else {
            console.log("Cannot find Tomcat server or there was more than one server. response is: " + servers);
        }

        // the columns shown in the applications view depends on the Tomcat version in use
        if (Tomcat.isTomcat5($scope.tomcatServerVersion)) {
            console.log("Using Tomcat 5");
            $scope.gridOptions.columnDefs = columnDefsTomcat5;
        } else if (Tomcat.isTomcat6($scope.tomcatServerVersion)) {
            console.log("Using Tomcat 6");
            $scope.gridOptions.columnDefs = columnDefsTomcat6;
        } else {
            console.log("Using Tomcat 7");
            $scope.gridOptions.columnDefs = columnDefsTomcat7;
        }
    }
    Tomcat.TomcatController = TomcatController;
})(Tomcat || (Tomcat = {}));
var Tomcat;
(function (Tomcat) {
    function filerTomcatOrCatalina(response) {
        if (response) {
            // Tomcat can have mbean server names with Catalina or Tomcat
            response = response.filter(function (name) {
                return name.startsWith("Catalina") || name.startsWith("Tomcat");
            });
        }
        return response;
    }
    Tomcat.filerTomcatOrCatalina = filerTomcatOrCatalina;
    ;

    function iconClass(state) {
        if (state) {
            switch (state.toString().toLowerCase()) {
                case '1':
                    return "green icon-play-circle";
                case 'started':
                    return "green icon-play-circle";
                case '0':
                    return "orange icon-off";
                case 'stopped':
                    return "orange icon-off";
            }
        }

        // Tomcat 5 uses 0 for stopped
        if (angular.isNumber(state)) {
            if (state.toString() === '0') {
                return "orange icon-off";
            }
        }

        return "icon-question-sign";
    }
    Tomcat.iconClass = iconClass;

    function millisToDateFormat(time) {
        if (time) {
            var date = new Date(time);
            return date.toLocaleDateString() + " " + date.toLocaleTimeString();
        } else {
            return "";
        }
    }
    Tomcat.millisToDateFormat = millisToDateFormat;

    function isTomcat5(name) {
        return name.toString().indexOf("Apache Tomcat/5") !== -1;
    }
    Tomcat.isTomcat5 = isTomcat5;

    function isTomcat6(name) {
        return name.toString().indexOf("Apache Tomcat/6") !== -1;
    }
    Tomcat.isTomcat6 = isTomcat6;
})(Tomcat || (Tomcat = {}));
var Tomcat;
(function (Tomcat) {
    function ConnectorsController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | tomcatIconClass}}"></i></div>';

        $scope.connectors = [];
        $scope.selected = [];

        var columnDefs = [
            {
                field: 'stateName',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'port',
                displayName: 'Port',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'scheme',
                displayName: 'Scheme',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'protocol',
                displayName: 'Protocol',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'secure',
                displayName: 'Secure',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'connectionLinger',
                displayName: 'Connection Linger',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'connectionTimeout',
                displayName: 'Connection Timeout',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'keepAliveTimeout',
                displayName: 'Keep Alive Timeout',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'minSpareThreads',
                displayName: 'Minimum Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxThreads',
                displayName: 'Maximum Threads',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'connectors',
            displayFooter: true,
            selectedItems: $scope.selected,
            selectWithCheckboxOnly: true,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            }
        };

        function render(response) {
            response = Tomcat.filerTomcatOrCatalina(response);

            $scope.connectors = [];
            $scope.selected.length = 0;

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    $scope.connectors.push(obj);
                    Core.$apply($scope);
                }
            }

            // create structure for each response
            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({
                    type: "read", mbean: mbean, attribute: [
                        "scheme", "port", "protocol", "secure",
                        "connectionLinger", "connectionTimeout", "keepAliveTimeout", "minSpareThreads", "maxThreads", "stateName"] }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        // function to control the connectors
        $scope.controlConnector = function (op) {
            // grab id of mbean names to control
            var ids = $scope.selected.map(function (b) {
                return b.mbean;
            });
            if (!angular.isArray(ids)) {
                ids = [ids];
            }

            // execute operation on each mbean
            ids.forEach(function (id) {
                jolokia.request({
                    type: 'exec',
                    mbean: id,
                    operation: op,
                    arguments: null
                }, onSuccess($scope.onResponse, { error: $scope.onResponse }));
            });
        };

        $scope.stop = function () {
            $scope.controlConnector('stop');
        };

        $scope.start = function () {
            $scope.controlConnector('start');
        };

        $scope.destroy = function () {
            $scope.controlConnector('destroy');
        };

        // function to trigger reloading page
        $scope.onResponse = function (response) {
            //console.log("got response: " + response);
            loadData();
        };

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading tomcat connector data...");
            jolokia.search("Catalina:type=Connector,*", onSuccess(render));
        }
    }
    Tomcat.ConnectorsController = ConnectorsController;
})(Tomcat || (Tomcat = {}));
var Tomcat;
(function (Tomcat) {
    function SessionsController($scope, $location, workspace, jolokia) {
        var stateTemplate = '<div class="ngCellText pagination-centered" title="{{row.getProperty(col.field)}}"><i class="{{row.getProperty(col.field) | tomcatIconClass}}"></i></div>';

        $scope.sessions = [];
        $scope.search = "";

        var columnDefs = [
            {
                field: 'stateName',
                displayName: 'State',
                cellTemplate: stateTemplate,
                width: 56,
                minWidth: 56,
                maxWidth: 56,
                resizable: false
            },
            {
                field: 'path',
                displayName: 'Context-Path',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'activeSessions',
                displayName: 'Active Sessions',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'expiredSessions',
                displayName: 'Expired Sessions',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'rejectedSessions',
                displayName: 'Rejected Sessions',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxActive',
                displayName: 'Max Active Sessions',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxActiveSessions',
                displayName: 'Max Active Sessions Allowed',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'maxInactiveInterval',
                displayName: 'Max Inactive Interval',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'sessionCounter',
                displayName: 'Session Counter',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'sessionCreateRate',
                displayName: 'Session Create Rate',
                cellFilter: null,
                width: "*",
                resizable: true
            },
            {
                field: 'sessionExpireRate',
                displayName: 'Session Expire Rate',
                cellFilter: null,
                width: "*",
                resizable: true
            }
        ];

        $scope.gridOptions = {
            data: 'sessions',
            displayFooter: false,
            displaySelectionCheckbox: false,
            canSelectRows: false,
            columnDefs: columnDefs,
            filterOptions: {
                filterText: ''
            }
        };

        function render(response) {
            response = Tomcat.filerTomcatOrCatalina(response);

            $scope.sessions = [];

            function onAttributes(response) {
                var obj = response.value;
                if (obj) {
                    obj.mbean = response.request.mbean;
                    var mbean = obj.mbean;
                    if (mbean) {
                        // the context path is part of the mbean name
                        // grab the 2nd part of the mbean that has context=/name
                        var context = mbean.toString().split(",")[1];
                        if (context) {
                            if (context.toString().indexOf("path=") !== -1) {
                                // and remove the leading path=/ from the name (Tomcat 5 or 6)
                                obj.path = context.toString().substr(5);
                            } else {
                                // and remove the leading context=/ from the name (Tomcat 7)
                                obj.path = context.toString().substr(9);
                            }
                        } else {
                            obj.path = "";
                        }

                        $scope.sessions.push(obj);
                        Core.$apply($scope);
                    }
                }
            }

            angular.forEach(response, function (value, key) {
                var mbean = value;
                jolokia.request({ type: "read", mbean: mbean }, onSuccess(onAttributes));
            });
            Core.$apply($scope);
        }
        ;

        $scope.$on('jmxTreeUpdated', reloadFunction);
        $scope.$watch('workspace.tree', reloadFunction);

        function reloadFunction() {
            // if the JMX tree is reloaded its probably because a new MBean has been added or removed
            // so lets reload, asynchronously just in case
            setTimeout(loadData, 50);
        }

        function loadData() {
            console.log("Loading tomcat session data...");
            jolokia.search("*:type=Manager,*", onSuccess(render));
        }
    }
    Tomcat.SessionsController = SessionsController;
})(Tomcat || (Tomcat = {}));
/**
* @module Tomcat
* @main Tomcat
*/
var Tomcat;
(function (Tomcat) {
    var pluginName = 'tomcat';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'ui.bootstrap.dialog', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/tomcat/server', { templateUrl: 'app/tomcat/html/server.html' }).when('/tomcat/applications', { templateUrl: 'app/tomcat/html/applications.html' }).when('/tomcat/connectors', { templateUrl: 'app/tomcat/html/connectors.html' }).when('/tomcat/sessions', { templateUrl: 'app/tomcat/html/sessions.html' });
    }).filter('tomcatIconClass', function () {
        return Tomcat.iconClass;
    }).run(function ($location, workspace, viewRegistry, helpRegistry) {
        viewRegistry['tomcat'] = "app/tomcat/html/layoutTomcatTabs.html";
        helpRegistry.addUserDoc('tomcat', 'app/tomcat/doc/help.md', function () {
            return workspace.treeContainsDomainAndProperties("Tomcat") || workspace.treeContainsDomainAndProperties("Catalina");
        });

        workspace.topLevelTabs.push({
            id: "tomcat",
            content: "Tomcat",
            title: "Manage your Tomcat container",
            isValid: function (workspace) {
                return workspace.treeContainsDomainAndProperties("Tomcat") || workspace.treeContainsDomainAndProperties("Catalina");
            },
            href: function () {
                return "#/tomcat/applications";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("tomcat");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Tomcat || (Tomcat = {}));
var Camin;
(function (Camin) {
    var Gantt = (function () {
        function Gantt() {
            this.resources = [];
            this.tasks = [];
            this.links = [];
        }
        Gantt.prototype.resource = function (data) {
            var resource = new Resource(data, this.resources.length);
            this.resources.push(resource);
            return resource;
        };

        Gantt.prototype.task = function (resource, start, stop, data) {
            var task = resource.task(start, stop, data);
            this.tasks.push(task);
            return task;
        };

        Gantt.prototype.link = function (start, taskA, stop, taskB, data) {
            var link = new Link(start, taskA, stop, taskB, data);
            this.links.push(link);
            return link;
        };

        Gantt.prototype.layout = function () {
            for (var i = 0; i < this.resources.length; i++) {
                this.resources[i].layout();
                this.start = this.start ? Math.min(this.start, this.resources[i].start) : this.resources[i].start;
                this.stop = this.stop ? Math.max(this.stop, this.resources[i].stop) : this.resources[i].stop;
            }
            for (var i = 0; i < this.links.length; i++) {
                this.start = this.start ? Math.min(this.start, this.links[i].start) : this.links[i].start;
                this.stop = this.stop ? Math.max(this.stop, this.links[i].stop) : this.links[i].stop;
            }
        };

        Gantt.prototype.taskByData = function (data) {
            for (var i = 0; i < this.tasks.length; i++) {
                if (this.tasks[i].data === data) {
                    return this.tasks[i];
                }
            }
            return undefined;
        };
        return Gantt;
    })();
    Camin.Gantt = Gantt;

    var Resource = (function () {
        function Resource(data, index) {
            this.tasks = [];
            this.index = index;
            this.data = data;
        }
        Resource.prototype.task = function (start, stop, data) {
            var task = new Task(start, stop, data, this);
            this.tasks.push(task);
            return task;
        };

        Resource.prototype.layout = function () {
            this.tasks.sort(function (ta, tb) {
                return ta.start - tb.start;
            });
            var bands = [];
            for (var i = 0; i < this.tasks.length; i++) {
                this.start = this.start ? Math.min(this.start, this.tasks[i].start) : this.tasks[i].start;
                this.stop = this.stop ? Math.max(this.stop, this.tasks[i].stop) : this.tasks[i].stop;
                for (var j = 0; j < bands.length; j++) {
                    if (bands[j] < this.tasks[i].start) {
                        bands[j] = this.tasks[i].stop;
                        this.tasks[i].index = j;
                        break;
                    }
                }
                if (!this.tasks[i].index) {
                    var index = bands.length;
                    this.tasks[i].index = index;
                    bands[index] = this.tasks[i].stop;
                }
            }
            for (var i = 0; i < this.tasks.length; i++) {
                this.tasks[i].max = bands.length;
            }
        };
        return Resource;
    })();
    Camin.Resource = Resource;

    var Task = (function () {
        function Task(start, stop, data, resource) {
            this.start = start;
            this.stop = stop;
            this.data = data;
            this.resource = resource;
        }
        return Task;
    })();
    Camin.Task = Task;

    var Link = (function () {
        function Link(start, taskA, stop, taskB, data) {
            this.start = start;
            this.stop = stop;
            this.taskA = taskA;
            this.taskB = taskB;
            this.data = data;
        }
        return Link;
    })();
    Camin.Link = Link;
})(Camin || (Camin = {}));
var Camin;
(function (Camin) {
    var Sequence = (function () {
        function Sequence() {
            this.endpoints = [];
            this.execs = [];
            this.calls = [];
        }
        Sequence.prototype.endpoint = function (url, routeId, contextId, host) {
            for (var i = 0; i < this.endpoints.length; i++) {
                if (this.endpoints[i].url === url && this.endpoints[i].routeId === routeId && this.endpoints[i].contextId === contextId && this.endpoints[i].host === host) {
                    return this.endpoints[i];
                }
            }
            var endpoint = new Endpoint(url, routeId, contextId, host);
            this.endpoints.push(endpoint);
            return endpoint;
        };

        Sequence.prototype.exec = function (exchangeId, endpoint, start, stop) {
            var exec = new Execution(exchangeId, endpoint, start, stop);
            this.execs.push(exec);
            return exec;
        };

        Sequence.prototype.call = function (callId, execA, execB, start, stop) {
            var call = new Call(callId, execA, execB, start, stop);
            this.calls.push(call);
            return call;
        };

        Sequence.prototype.start = function () {
            var start;
            for (var i = 0; i < this.execs.length; i++) {
                start = start ? Math.min(start, this.execs[i].start) : this.execs[i].start;
            }
            for (var i = 0; i < this.calls.length; i++) {
                start = start ? Math.min(start, this.calls[i].start) : this.calls[i].start;
            }
            return start;
        };

        Sequence.prototype.stop = function () {
            var stop;
            for (var i = 0; i < this.execs.length; i++) {
                stop = stop ? Math.max(stop, this.execs[i].stop) : this.execs[i].stop;
            }
            for (var i = 0; i < this.calls.length; i++) {
                stop = stop ? Math.max(stop, this.calls[i].stop) : this.calls[i].stop;
            }
            return stop;
        };
        return Sequence;
    })();
    Camin.Sequence = Sequence;

    var Endpoint = (function () {
        function Endpoint(url, routeId, contextId, host) {
            this.url = url;
            this.routeId = routeId;
            this.contextId = contextId;
            this.host = host;
        }
        return Endpoint;
    })();
    Camin.Endpoint = Endpoint;

    var Execution = (function () {
        function Execution(exchangeId, endpoint, start, stop) {
            this.exchangeId = exchangeId;
            this.endpoint = endpoint;
            this.start = start;
            this.stop = stop;
        }
        return Execution;
    })();
    Camin.Execution = Execution;

    var Call = (function () {
        function Call(callId, execA, execB, start, stop) {
            this.callId = callId;
            this.execA = execA;
            this.execB = execB;
            this.start = start;
            this.stop = stop;
        }
        return Call;
    })();
    Camin.Call = Call;
})(Camin || (Camin = {}));
/**
* Camel Insight Plugin
*
* @module Camin
* @main Camin
*/
var Camin;
(function (Camin) {
    var pluginName = 'camin';
    angular.module(pluginName, ['bootstrap', 'ngResource', 'ngGrid', 'hawtioCore']).config(function ($routeProvider) {
        $routeProvider.when('/camin', { templateUrl: 'app/camin/html/camin.html' }).when('/camin/:exchangeId', { templateUrl: 'app/camin/html/camin.html' });
    }).run(function (workspace, viewRegistry, helpRegistry) {
        viewRegistry["camin"] = "app/camin/html/layoutCamin.html";
        helpRegistry.addUserDoc('camin', 'app/camin/doc/help.md', function () {
            return Fabric.hasFabric(workspace);
        });

        workspace.topLevelTabs.push({
            id: "camin",
            content: "Camel",
            title: "Insight into Camel",
            isValid: function (workspace) {
                return Fabric.hasFabric(workspace);
            },
            href: function () {
                return "#/camin";
            },
            isActive: function (workspace) {
                return workspace.isLinkActive("camin");
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Camin || (Camin = {}));
var Camin;
(function (Camin) {
    function Controller($scope, jolokia, localStorage, $routeParams) {
        $scope.query = "";
        $scope.result = "";
        $scope.breadcrumbs = [];

        $scope.onQueryChange = function () {
            $scope.result = "Querying exchanges related to " + $scope.query;
            $scope.breadcrumbs = [$scope.query];
            request();
        };

        var request = function () {
            var queryStr = "exchange.id:\"" + $scope.breadcrumbs.join("\" or exchange.id:\"") + "\" or " + "exchange.in.headers.ExtendedBreadcrumb:\"" + $scope.breadcrumbs.join("\" or exchange.in.headers.ExtendedBreadcrumb:\"") + "\" or " + "exchange.out.headers.ExtendedBreadcrumb:\"" + $scope.breadcrumbs.join("\" or exchange.out.headers.ExtendedBreadcrumb:\"") + "\"";
            var query = {
                "query": { "query_string": { "query": queryStr } },
                "fields": ["exchange.id", "exchange.in.headers.ExtendedBreadcrumb", "exchange.out.headers.ExtendedBreadcrumb"],
                "from": 0,
                "size": 1000
            };
            var jreq = {
                type: 'exec',
                mbean: 'org.elasticsearch:service=restjmx',
                operation: 'exec',
                arguments: ['POST', '/_all/camel/_search', angular.toJson(query)] };
            jolokia.request(jreq, {
                method: 'POST',
                error: function (response) {
                    $scope.result = $scope.result + "<br/>" + "Error: " + angular.toJson(response);
                },
                success: function (response) {
                    var data = jQuery.parseJSON(response.value);
                    var oldsize = $scope.breadcrumbs.length;
                    for (var i = 0; i < data['hits']['hits'].length; i++) {
                        var fields = data['hits']['hits'][i].fields;
                        var concat = function (breadcrumbs) {
                            if (breadcrumbs) {
                                if (typeof breadcrumbs === 'string') {
                                    breadcrumbs = [breadcrumbs];
                                }
                                for (var j = 0; j < breadcrumbs.length; j++) {
                                    var id = breadcrumbs[j];
                                    if ($scope.breadcrumbs.indexOf(id) < 0) {
                                        $scope.breadcrumbs.push(id);
                                    }
                                }
                            }
                        };
                        concat(fields["exchange.in.headers.ExtendedBreadcrumb"]);
                        concat(fields["exchange.out.headers.ExtendedBreadcrumb"]);
                    }
                    $scope.result = $scope.result + "<br/>" + "Found " + data.hits.total + " ids";
                    if (oldsize != $scope.breadcrumbs.length) {
                        request();
                    } else {
                        var ids = [];
                        for (var i = 0; i < data['hits']['hits'].length; i++) {
                            var id = data['hits']['hits'][i].fields["exchange.id"];
                            if (ids.indexOf(id) < 0) {
                                ids.push(id);
                            }
                        }
                        var queryStr = "exchange.id:\"" + ids.join("\" or exchange.id:\"") + "\"";
                        $scope.result = $scope.result + "<br/>" + query;
                        var query = {
                            "query": { "query_string": { "query": queryStr } },
                            "from": 0,
                            "size": 1000,
                            "sort": ["timestamp"]
                        };
                        var jreq = {
                            type: 'exec',
                            mbean: 'org.elasticsearch:service=restjmx',
                            operation: 'exec',
                            arguments: ['POST', '/_all/camel/_search', angular.toJson(query)] };
                        jolokia.request(jreq, {
                            method: 'POST',
                            error: function (response) {
                                $scope.result = $scope.result + "<br/>" + "Error: " + angular.toJson(response);
                            },
                            success: function (response) {
                                var data = jQuery.parseJSON(response.value);
                                $scope.result = $scope.result + "<br/>" + "Found " + data['hits']['total'] + " exchanges";
                                var events = [];
                                for (var i = 0; i < data['hits']['hits'].length; i++) {
                                    var e = data['hits']['hits'][i]._source;
                                    events.push(e);
                                }
                                draw(events);
                            }
                        });
                    }
                } });
        };

        var isoDate = function (date) {
            var timestamp, struct, minutesOffset = 0;
            var numericKeys = [1, 4, 5, 6, 7, 10, 11];

            // ES5 §15.9.4.2 states that the string should attempt to be parsed as a Date Time String Format string
            // before falling back to any implementation-specific date parsing, so that’s what we do, even if native
            // implementations could be faster
            //              1 YYYY                2 MM       3 DD           4 HH    5 mm       6 ss        7 msec        8 Z 9 ±    10 tzHH    11 tzmm
            if ((struct = /^(\d{4}|[+\-]\d{6})(?:-(\d{2})(?:-(\d{2}))?)?(?:T(\d{2}):(\d{2})(?::(\d{2})(?:\.(\d{3}))?)?(?:(Z)|([+\-])(\d{2})(?::(\d{2}))?)?)?$/.exec(date))) {
                for (var i = 0, k; (k = numericKeys[i]); ++i) {
                    struct[k] = +struct[k] || 0;
                }

                // allow undefined days and months
                struct[2] = (+struct[2] || 1) - 1;
                struct[3] = +struct[3] || 1;

                if (struct[8] !== 'Z' && struct[9] !== undefined) {
                    minutesOffset = struct[10] * 60 + struct[11];

                    if (struct[9] === '+') {
                        minutesOffset = 0 - minutesOffset;
                    }
                }

                timestamp = Date.UTC(struct[1], struct[2], struct[3], struct[4], struct[5] + minutesOffset, struct[6], struct[7]);
            } else {
                timestamp = Date.parse(date);
            }

            return timestamp;
        };

        var buildSequence = function (events) {
            var sequence = new Camin.Sequence();
            var exchangeToExec = {};

            // Sort events
            events = events.sort(function (a, b) {
                return isoDate(a.timestamp) - isoDate(b.timestamp);
            });

            for (var i = 0; i < events.length; i++) {
                if (events[i].event === 'Created') {
                    var evtCreated = events[i];
                    var evtCompleted = null;
                    for (var j = 0; j < events.length; j++) {
                        if (events[j].event === 'Completed' && evtCreated.exchange.id === events[j].exchange.id) {
                            evtCompleted = events[j];
                            break;
                        }
                    }
                    if (evtCompleted === null) {
                        console.log('Could not find matching Completed exchange for ' + evtCreated.exchange.id);
                        continue;
                    }

                    // We use the completed event here because the created event may miss the routeId information
                    var endpoint = sequence.endpoint(evtCompleted.exchange.fromEndpoint, evtCompleted.exchange.routeId, evtCompleted.exchange.contextId, evtCompleted.host);
                    var exec = sequence.exec(evtCreated.exchange.id, endpoint, isoDate(evtCreated.timestamp), isoDate(evtCompleted.timestamp));
                    exchangeToExec[evtCreated.exchange.id] = exec;
                }
            }

            // Extract calls
            var calls = {};
            for (var i = 0; i < events.length; i++) {
                if (events[i].event === 'Sending' && events[i].exchange.in && events[i].exchange.in.headers) {
                    var callId = events[i].exchange.in.headers.AuditCallId;
                    if (callId && calls[callId] === undefined) {
                        var evtSending = events[i];
                        var evtSent = null;
                        var evtCreated = null;
                        for (var j = 0; j < events.length; j++) {
                            if (events[j].event === 'Sent' && evtSending.exchange.id === events[j].exchange.id && events[j].exchange.in.headers.AuditCallId === callId) {
                                evtSent = events[j];
                                break;
                            }
                        }
                        for (var j = 0; j < events.length; j++) {
                            if (events[j].event === 'Created' && evtSending.exchange.id !== events[j].exchange.id && events[j].exchange.in.headers.AuditCallId === callId) {
                                evtCreated = events[j];
                                break;
                            }
                        }
                        var execA = exchangeToExec[evtSending.exchange.id];
                        var execB = evtCreated ? exchangeToExec[evtCreated.exchange.id] : null;
                        if (evtSent !== null && evtCreated !== null && execA !== null && execB != null) {
                            var call = sequence.call(callId, execA, execB, isoDate(evtSending.timestamp), isoDate(evtSent.timestamp));
                            calls[callId] = call;
                        } else {
                            console.log("Could not find Execution for exchange " + evtSending.exchange.id);
                        }
                    }
                }
            }
            return sequence;
        };

        var buildDiagram = function (sequence) {
            var diagram = new Camin.Diagram();
            var actors = {};
            var signals = [];
            var base = sequence.start();
            for (var i = 0; i < sequence.endpoints.length; i++) {
                var actor = diagram.actor("ep" + i);
                var ep = sequence.endpoints[i];
                var key = ep.url + "|" + ep.routeId + "|" + ep.contextId + "|" + ep.host;
                actors[key] = actor;
            }
            for (var i = 0; i < sequence.calls.length; i++) {
                var call = sequence.calls[i];
                if (call.execB) {
                    var epA = call.execA.endpoint;
                    var keyA = epA.url + "|" + epA.routeId + "|" + epA.contextId + "|" + epA.host;
                    var epB = call.execB.endpoint;
                    var keyB = epB.url + "|" + epB.routeId + "|" + epB.contextId + "|" + epB.host;
                    var actorA = actors[keyA];
                    var actorB = actors[keyB];
                    var start1 = call.start - base;
                    var stop1 = call.execB.start - base;
                    var start2 = call.execB.stop - base;
                    var stop2 = call.stop - base;
                    signals.push({
                        actorA: actorA,
                        actorB: actorB,
                        message: start1 + "ms - " + stop1 + "ms",
                        timestamp: start1 });
                    signals.push({
                        actorA: actorB,
                        actorB: actorA,
                        message: start2 + "ms - " + stop2 + "ms",
                        timestamp: start2 });
                }
            }
            signals = signals.sort(function (a, b) {
                return a.timestamp - b.timestamp;
            });
            for (var i = 0; i < signals.length; i++) {
                diagram.signal(signals[i].actorA, signals[i].actorB, signals[i].message);
            }
            return diagram;
        };

        var buildGantt = function (sequence) {
            var gantt = new Camin.Gantt();
            for (var i = 0; i < sequence.endpoints.length; i++) {
                var endpoint = sequence.endpoints[i];
                var resource = gantt.resource(endpoint);
                for (var j = 0; j < sequence.execs.length; j++) {
                    var exec = sequence.execs[j];
                    if (exec.endpoint === endpoint) {
                        gantt.task(resource, exec.start, exec.stop, exec);
                    }
                }
            }
            for (var i = 0; i < sequence.calls.length; i++) {
                var call = sequence.calls[i];
                if (call.execB) {
                    var taskA = gantt.taskByData(call.execA);
                    var taskB = gantt.taskByData(call.execB);
                    gantt.link(call.start, taskA, call.stop, taskB, call);
                }
            }
            gantt.layout();
            return gantt;
        };

        var eventTypeValue = { "Created": 0, "Sending": 1, "Sent": 2, "Completed": 3 };

        var draw = function (events) {
            $scope.definition = "";

            events = events.sort(function (a, b) {
                return isoDate(a.timestamp) - isoDate(b.timestamp);
            });
            console.log(events);

            var sequence = buildSequence(events);
            console.log(sequence);

            var gantt = buildGantt(sequence);
            console.log(gantt);
            $('#gantt').html('');
            drawGantt('#gantt', gantt);

            var diagram = buildDiagram(sequence);
            console.log(diagram);
            $('#diagram').html('');
            drawDiagram('#diagram', diagram);
        };

        var drawDiagram = function (container, diagram) {
            var arrow_size = 10;
            var margin = 10;
            var actor_width = 100;
            var actor_margin = 30;
            var actor_height = 40;
            var signal_height = 30;
            var actor_font = 20;
            var signal_font = 14;
            var width = diagram.actors.length * (actor_width + actor_margin * 2);
            var height = (diagram.signals.length + 1) * signal_height + actor_height * 2 + margin * 2;

            var svg = d3.select(container).append('svg').attr('width', width + 2 * margin).attr('height', height + 2 * margin);
            var g = svg.append('g').attr('text-anchor', 'middle');
            for (var i = 0; i < diagram.actors.length; i++) {
                var actor = diagram.actors[i];
                var gu = g.append('g').attr('transform', 'translate(' + (i * (actor_width + actor_margin * 2) + actor_margin) + ',' + actor_height + ')');
                gu.append('rect').attr('width', actor_width).attr('height', actor_height).attr('stroke', '#000').attr('stroke-width', '2').attr('fill', '#FFFFFF');
                gu.append('text').attr('x', actor_width / 2).attr('y', actor_height / 2).attr('stroke-width', '0').attr('dominant-baseline', 'middle').attr('font-size', actor_font).text(actor.name);

                g.append('line').attr('x1', i * (actor_width + actor_margin * 2) + actor_width / 2 + actor_margin).attr('y1', actor_height * 2).attr('x2', i * (actor_width + actor_margin * 2) + actor_width / 2 + actor_margin).attr('y2', height - actor_height).attr('stroke', '#000').attr('stroke-width', '2');

                var gu = g.append('g').attr('transform', 'translate(' + (i * (actor_width + actor_margin * 2) + actor_margin) + ',' + (height - actor_height) + ')');
                gu.append('rect').attr('width', actor_width).attr('height', actor_height).attr('stroke', '#000').attr('stroke-width', '2').attr('fill', 'white');
                gu.append('text').attr('x', actor_width / 2).attr('y', actor_height / 2).attr('stroke-width', '0').attr('dominant-baseline', 'middle').attr('font-size', actor_font).text(actor.name);
            }
            for (var i = 0; i < diagram.signals.length; i++) {
                var x;
                var y;
                var length;
                var direction;
                var text;

                x = diagram.signals[i].actorA.index * (actor_width + actor_margin * 2) + actor_width / 2 + actor_margin;
                y = (i + 1) * signal_height + actor_height * 2;
                length = Math.abs(diagram.signals[i].actorA.index - diagram.signals[i].actorB.index) * (actor_width + actor_margin * 2);
                direction = diagram.signals[i].actorB.index > diagram.signals[i].actorA.index ? +1 : -1;
                text = diagram.signals[i].message;

                var gu = g.append('g').attr('transform', 'translate(' + x + ',' + y + ')').attr('stroke-width', '2');
                gu.append('rect').attr('x', Math.min(3, length * direction + 3)).attr('y', '-16').attr('width', Math.abs((length - 6) * direction)).attr('height', '19').attr('stroke', 'white').attr('stroke-width', '0').attr('fill', 'white');
                gu.append('line').attr('x1', 0).attr('y1', 0).attr('x2', length * direction).attr('y2', 0).attr('stroke', '#000').attr('stroke-width', '2');
                gu.append('line').attr('x1', length * direction - arrow_size * direction).attr('y1', -arrow_size).attr('x2', length * direction).attr('y2', 0).attr('stroke', '#000').attr('stroke-width', '2');
                gu.append('line').attr('x1', length * direction).attr('y1', 0).attr('x2', length * direction - arrow_size * direction).attr('y2', arrow_size).attr('stroke', '#000').attr('stroke-width', '2');
                gu.append('text').attr('x', length * direction / 2).attr('y', -8).attr('stroke-width', '0').attr('dominant-baseline', 'middle').attr('font-size', signal_font).text(text);
            }
        };

        var drawGantt = function (container, gantt) {
            var lineHeight = 35;
            var lineMargin = 3;
            var arrowWidth = 4;

            var width = 800;
            var height = lineHeight * gantt.resources.length;
            var margin = {
                top: 20,
                right: 40,
                bottom: 20,
                left: 250
            };

            var begin = gantt.start;
            var end = gantt.stop;

            var x = d3.scale.linear().domain([begin - (end - begin) * 0.1, end + (end - begin) * 0.1]).range([0, width]);
            var yt = function (t) {
                return t.resource.index * lineHeight + lineMargin + t.index * (lineHeight - 2 * lineMargin) / (t.max + 1);
            };
            var ht = function (t) {
                return 2 * (lineHeight - 2 * lineMargin) / (t.max + 1);
            };

            var svg = d3.select(container).append('svg').attr('width', width + margin.left + margin.right).attr('height', height + margin.top + margin.bottom);

            var text = svg.append('g').attr('width', width).attr('height', height).attr('transform', 'translate(0,' + margin.top + ')').selectAll('text').data(gantt.resources).enter();
            text.append('text').attr('x', 0).attr('y', function (r) {
                return r.index * lineHeight + lineHeight / 2;
            }).attr('dy', '-0.2em').attr('text-anchor', 'start').text(function (r) {
                var endpoint = r.data;
                var text = endpoint.url;
                if (text.indexOf("Endpoint[") == 0) {
                    text = text.substring(9, text.length - 1);
                }
                return text;
            });
            text.append('text').attr('x', 0).attr('y', function (r) {
                return r.index * lineHeight + lineHeight / 2;
            }).attr('dy', '0.8em').attr('text-anchor', 'start').text(function (r) {
                var endpoint = r.data;
                return endpoint.host + "/" + endpoint.contextId + "/" + endpoint.routeId;
            });

            var g = svg.append('g').attr('width', width).attr('height', height).attr('transform', 'translate(' + margin.left + ',' + margin.top + ')');

            g.append('g').attr('width', width).attr('height', height).selectAll('rect').data(gantt.tasks).enter().append('rect').attr('rx', lineMargin * 2).attr('ry', lineMargin * 2).attr('x', function (t) {
                return x(t.start);
            }).attr('y', yt).attr('height', ht).attr('width', function (t) {
                return x(t.stop) - x(t.start);
            }).attr('stroke', '#000000').attr('stroke-width', '2').attr('fill', function (t) {
                return d3.hsl(Math.random() * 360, 0.8, 0.8).toString();
            });

            var lines = g.append('g').attr('width', width).attr('height', height).attr('stroke', '#404040').attr('stroke-width', '2').selectAll('line').data(gantt.links).enter();
            lines.append('line').attr('x1', function (l) {
                return x(l.start);
            }).attr('y1', function (l) {
                return yt(l.taskA) + ht(l.taskA);
            }).attr('x2', function (l) {
                return x(l.start);
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.start);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.taskB.start);
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.taskB.start);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.taskB.start) - arrowWidth;
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2 - arrowWidth;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.taskB.start);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.taskB.start) - arrowWidth;
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2 + arrowWidth;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.taskB.stop);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.stop);
            }).attr('y2', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.stop);
            }).attr('y1', function (l) {
                return yt(l.taskB) + ht(l.taskB) / 2;
            }).attr('x2', function (l) {
                return x(l.stop);
            }).attr('y2', function (l) {
                return yt(l.taskA) + ht(l.taskA);
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.stop);
            }).attr('y1', function (l) {
                return yt(l.taskA) + ht(l.taskA);
            }).attr('x2', function (l) {
                return x(l.stop) - arrowWidth;
            }).attr('y2', function (l) {
                return yt(l.taskA) + ht(l.taskA) + arrowWidth;
            });
            lines.append('line').attr('x1', function (l) {
                return x(l.stop);
            }).attr('y1', function (l) {
                return yt(l.taskA) + ht(l.taskA);
            }).attr('x2', function (l) {
                return x(l.stop) + arrowWidth;
            }).attr('y2', function (l) {
                return yt(l.taskA) + ht(l.taskA) + arrowWidth;
            });
        };

        if ($routeParams["exchangeId"]) {
            $scope.query = $routeParams["exchangeId"];
            $scope.onQueryChange();
        }
    }
    Camin.Controller = Controller;
})(Camin || (Camin = {}));
var Camin;
(function (Camin) {
    var Diagram = (function () {
        function Diagram() {
            this.actors = [];
            this.signals = [];
        }
        Diagram.prototype.actor = function (name) {
            for (var i = 0; i < this.actors.length; i++) {
                if (this.actors[i].name === name) {
                    return this.actors[i];
                }
            }
            var actor = new Actor(name, this.actors.length);
            this.actors.push(actor);
            return actor;
        };

        Diagram.prototype.signal = function (actorA, actorB, message) {
            var signal = new Signal(actorA, actorB, message);
            this.signals.push(signal);
            return signal;
        };
        return Diagram;
    })();
    Camin.Diagram = Diagram;

    var Actor = (function () {
        function Actor(name, index) {
            this.name = name;
            this.index = index;
        }
        return Actor;
    })();
    Camin.Actor = Actor;

    var Signal = (function () {
        function Signal(actorA, actorB, message) {
            this.actorA = actorA;
            this.actorB = actorB;
            this.message = message;
        }
        Signal.prototype.isSelf = function () {
            return this.actorA.index === this.actorB.index;
        };
        return Signal;
    })();
    Camin.Signal = Signal;
})(Camin || (Camin = {}));
/**
* @module Jmx
* @main Jmx
*/
var Jmx;
(function (Jmx) {
    var pluginName = 'jmx';

    Jmx.currentProcessId = '';

    angular.module(pluginName, ['bootstrap', 'ui.bootstrap', 'ui.bootstrap.modal', 'ngResource', 'datatable', 'hawtioCore', 'hawtio-ui', 'hawtioRbac']).config(function ($routeProvider) {
        $routeProvider.when('/jmx/attributes', { templateUrl: 'app/jmx/html/attributes.html' }).when('/jmx/operations', { templateUrl: 'app/jmx/html/operations.html' }).when('/jmx/charts', { templateUrl: 'app/jmx/html/charts.html' }).when('/jmx/chartEdit', { templateUrl: 'app/jmx/html/chartEdit.html' }).when('/jmx/help/:tabName', { templateUrl: 'app/core/html/help.html' }).when('/jmx/widget/donut', { templateUrl: 'app/jmx/html/donutChart.html' }).when('/jmx/widget/area', { templateUrl: 'app/jmx/html/areaChart.html' });
    }).factory('jmxTreeLazyLoadRegistry', function () {
        return Jmx.lazyLoaders;
    }).factory('jmxWidgetTypes', function () {
        return Jmx.jmxWidgetTypes;
    }).factory('jmxWidgets', function () {
        return Jmx.jmxWidgets;
    }).run(function ($location, workspace, viewRegistry, layoutTree, jolokia, pageTitle, helpRegistry) {
        viewRegistry['jmx'] = layoutTree;
        helpRegistry.addUserDoc('jmx', 'app/jmx/doc/help.md');

        pageTitle.addTitleElement(function () {
            if (Jmx.currentProcessId === '') {
                try  {
                    Jmx.currentProcessId = jolokia.getAttribute('java.lang:type=Runtime', 'Name');
                } catch (e) {
                    // ignore
                }
                if (Jmx.currentProcessId && Jmx.currentProcessId.has("@")) {
                    Jmx.currentProcessId = "pid:" + Jmx.currentProcessId.split("@")[0];
                }
            }
            return Jmx.currentProcessId;
        });

        workspace.topLevelTabs.push({
            id: "jmx",
            content: "JMX",
            title: "View the JMX MBeans in this process",
            isValid: function (workspace) {
                return workspace.hasMBeans();
            },
            href: function () {
                return "#/jmx/attributes";
            },
            isActive: function (workspace) {
                return workspace.isTopTabActive("jmx");
            }
        });

        workspace.subLevelTabs.push({
            content: '<i class="icon-list"></i> Attributes',
            title: "View the attribute values on your selection",
            isValid: function (workspace) {
                return true;
            },
            href: function () {
                return "#/jmx/attributes";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-leaf"></i> Operations',
            title: "Execute operations on your selection",
            isValid: function (workspace) {
                return true;
            },
            href: function () {
                return "#/jmx/operations";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-bar-chart"></i> Chart',
            title: "View a chart of the metrics on your selection",
            isValid: function (workspace) {
                return true;
            },
            href: function () {
                return "#/jmx/charts";
            }
        });
        workspace.subLevelTabs.push({
            content: '<i class="icon-cog"></i> Edit Chart',
            title: "Edit the chart configuration",
            isValid: function (workspace) {
                return workspace.isLinkActive("jmx/chart");
            },
            href: function () {
                return "#/jmx/chartEdit";
            }
        });
    });

    hawtioPluginLoader.addModule(pluginName);
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function TreeHeaderController($scope) {
        $scope.expandAll = function () {
            Tree.expandAll("#jmxtree");
        };

        $scope.contractAll = function () {
            Tree.contractAll("#jmxtree");
        };
    }
    Jmx.TreeHeaderController = TreeHeaderController;

    function MBeansController($scope, $location, workspace) {
        $scope.num = 1;

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateSelectionFromURL, 50);
        });

        $scope.select = function (node) {
            $scope.workspace.updateSelectionNode(node);
            Core.$apply($scope);
        };

        function updateSelectionFromURL() {
            Jmx.updateTreeSelectionFromURL($location, $("#jmxtree"));
        }

        $scope.populateTree = function () {
            var treeElement = $("#jmxtree");
            $scope.tree = workspace.tree;
            Jmx.enableTree($scope, $location, workspace, treeElement, $scope.tree.children, true);
            setTimeout(updateSelectionFromURL, 50);
        };

        $scope.$on('jmxTreeUpdated', $scope.populateTree);

        $scope.populateTree();
    }
    Jmx.MBeansController = MBeansController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function ChartController($scope, $element, $location, workspace, localStorage, jolokiaUrl, jolokiaParams) {
        $scope.metrics = [];
        $scope.updateRate = 1000; //parseInt(localStorage['updateRate']);

        $scope.context = null;
        $scope.jolokia = null;
        $scope.charts = null;

        $scope.reset = function () {
            if ($scope.context) {
                $scope.context.stop();
                $scope.context = null;
            }
            if ($scope.jolokia) {
                $scope.jolokia.stop();
                $scope.jolokia = null;
            }
            if ($scope.charts) {
                $scope.charts.empty();
                $scope.charts = null;
            }
        };

        $scope.$on('$destroy', function () {
            try  {
                $scope.deregRouteChange();
            } catch (error) {
                // ignore
            }
            try  {
                $scope.dereg();
            } catch (error) {
                // ignore
            }
            $scope.reset();
        });

        $scope.errorMessage = function () {
            if ($scope.updateRate === 0) {
                return "updateRate";
            }

            if ($scope.metrics.length === 0) {
                return "metrics";
            }
        };

        var doRender = Core.throttled(render, 200);

        $scope.deregRouteChange = $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            doRender();
        });
        $scope.dereg = $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid())
                return;
            doRender();
        });

        doRender();

        function render() {
            var node = workspace.selection;
            if (!angular.isDefined(node) || !angular.isDefined($scope.updateRate) || $scope.updateRate === 0) {
                // Called render too early, let's retry
                setTimeout(doRender, 500);
                Core.$apply($scope);
                return;
            }
            var width = 594;
            var charts = $element.find('#charts');
            if (charts) {
                width = charts.width();
            } else {
                // Called render too early, let's retry
                setTimeout(doRender, 500);
                Core.$apply($scope);
                return;
            }

            // clear out any existing context
            $scope.reset();

            $scope.charts = charts;
            $scope.jolokia = new Jolokia(jolokiaParams);
            $scope.jolokia.start($scope.updateRate);

            var mbean = node.objectName;
            $scope.metrics = [];

            var context = cubism.context().serverDelay($scope.updateRate).clientDelay($scope.updateRate).step($scope.updateRate).size(width);

            $scope.context = context;
            $scope.jolokiaContext = context.jolokia($scope.jolokia);
            var search = $location.search();
            var attributeNames = toSearchArgumentArray(search["att"]);

            if (mbean) {
                // TODO make generic as we can cache them; they rarely ever change
                // lets get the attributes for this mbean
                // we need to escape the mbean path for list
                var listKey = encodeMBeanPath(mbean);

                //console.log("Looking up mbeankey: " + listKey);
                var meta = $scope.jolokia.list(listKey);
                if (meta) {
                    var attributes = meta.attr;
                    if (attributes) {
                        var foundNames = [];
                        for (var key in attributes) {
                            var value = attributes[key];
                            if (value) {
                                var typeName = value['type'];
                                if (isNumberTypeName(typeName)) {
                                    foundNames.push(key);
                                }
                            }
                        }

                        // lets filter the attributes
                        // if we find none then the att search attribute is invalid
                        // so lets discard the filter - as it must be for some other mbean
                        if (attributeNames.length) {
                            var filtered = foundNames.filter(function (key) {
                                return attributeNames.indexOf(key) >= 0;
                            });
                            if (filtered.length) {
                                foundNames = filtered;
                            }
                        }
                        angular.forEach(foundNames, function (key) {
                            var metric = $scope.jolokiaContext.metric({
                                type: 'read',
                                mbean: mbean,
                                attribute: key
                            }, humanizeValue(key));
                            if (metric) {
                                $scope.metrics.push(metric);
                            }
                        });
                    }
                }
            } else {
                // lets try pull out the attributes and elements from the URI and use those to chart
                var elementNames = toSearchArgumentArray(search["el"]);
                if (attributeNames && attributeNames.length && elementNames && elementNames.length) {
                    // first lets map the element names to mbean names to keep the URI small
                    var mbeans = {};
                    elementNames.forEach(function (elementName) {
                        var child = node.get(elementName);
                        if (!child && node.children) {
                            child = node.children.find(function (n) {
                                return elementName === n["title"];
                            });
                        }
                        if (child) {
                            var mbean = child.objectName;
                            if (mbean) {
                                mbeans[elementName] = mbean;
                            }
                        }
                    });

                    // lets create the metrics
                    attributeNames.forEach(function (key) {
                        angular.forEach(mbeans, function (mbean, name) {
                            var attributeTitle = humanizeValue(key);

                            // for now lets always be verbose
                            var title = name + ": " + attributeTitle;

                            var metric = $scope.jolokiaContext.metric({
                                type: 'read',
                                mbean: mbean,
                                attribute: key
                            }, title);
                            if (metric) {
                                $scope.metrics.push(metric);
                            }
                        });
                    });
                }

                // if we've children and none of the query arguments matched any metrics
                // lets redirect back to the edit view
                if (node.children.length && !$scope.metrics.length) {
                    // lets forward to the chart selection UI if we have some children; they may have
                    // chartable attributes
                    $location.path("jmx/chartEdit");
                }
            }

            if ($scope.metrics.length > 0) {
                var d3Selection = d3.select(charts.get(0));
                var axisEl = d3Selection.selectAll(".axis");

                var bail = false;

                axisEl.data(["top", "bottom"]).enter().append("div").attr("class", function (d) {
                    return d + " axis";
                }).each(function (d) {
                    if (bail) {
                        return;
                    }
                    try  {
                        d3.select(this).call(context.axis().ticks(12).orient(d));
                    } catch (error) {
                        // still rendering at not the right time...
                        // log.debug("error: ", error);
                        if (!bail) {
                            bail = true;
                        }
                    }
                });

                if (bail) {
                    $scope.reset();
                    setTimeout(doRender, 500);
                    Core.$apply($scope);
                    return;
                }

                d3Selection.append("div").attr("class", "rule").call(context.rule());

                context.on("focus", function (i) {
                    try  {
                        d3Selection.selectAll(".value").style("right", i === null ? null : context.size() - i + "px");
                    } catch (error) {
                        Jmx.log.info("error: ", error);
                    }
                });

                $scope.metrics.forEach(function (metric) {
                    d3Selection.call(function (div) {
                        div.append("div").data([metric]).attr("class", "horizon").call(context.horizon());
                    });
                });
            } else {
                $scope.reset();
            }

            Core.$apply($scope);
        }
        ;
    }
    Jmx.ChartController = ChartController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function ChartEditController($scope, $location, workspace, jolokia) {
        $scope.selectedAttributes = [];
        $scope.selectedMBeans = [];
        $scope.metrics = {};
        $scope.mbeans = {};

        // TODO move this function to $routeScope
        $scope.size = function (value) {
            if (angular.isObject(value)) {
                return Object.size(value);
            } else if (angular.isArray(value)) {
                return value.length;
            } else
                return 1;
        };

        $scope.canViewChart = function () {
            return $scope.selectedAttributes.length && $scope.selectedMBeans.length && $scope.size($scope.mbeans) > 0 && $scope.size($scope.metrics) > 0;
        };

        $scope.showAttributes = function () {
            return $scope.canViewChart() && $scope.size($scope.metrics) > 1;
        };

        $scope.showElements = function () {
            return $scope.canViewChart() && $scope.size($scope.mbeans) > 1;
        };

        $scope.viewChart = function () {
            // lets add the attributes and mbeans into the URL so we can navigate back to the charts view
            var search = $location.search();

            // if we have selected all attributes, then lets just remove the attribute
            if ($scope.selectedAttributes.length === $scope.size($scope.metrics)) {
                delete search["att"];
            } else {
                search["att"] = $scope.selectedAttributes;
            }

            // if we are on an mbean with no children lets discard an unnecessary parameter
            if ($scope.selectedMBeans.length === $scope.size($scope.mbeans) && $scope.size($scope.mbeans) === 1) {
                delete search["el"];
            } else {
                search["el"] = $scope.selectedMBeans;
            }
            $location.search(search);
            $location.path("jmx/charts");
        };

        $scope.$watch('workspace.selection', render);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(render, 50);
        });

        function render() {
            var node = workspace.selection;
            if (!angular.isDefined(node)) {
                return;
            }

            $scope.selectedAttributes = [];
            $scope.selectedMBeans = [];
            $scope.metrics = {};
            $scope.mbeans = {};
            var mbeanCounter = 0;
            var resultCounter = 0;

            // lets iterate through all the children if the current node is not an mbean
            var children = node.children;
            if (!children || !children.length || node.objectName) {
                children = [node];
            }
            if (children) {
                children.forEach(function (mbeanNode) {
                    var mbean = mbeanNode.objectName;
                    var name = mbeanNode.title;
                    if (name && mbean) {
                        mbeanCounter++;
                        $scope.mbeans[name] = name;

                        // we need to escape the mbean path for list
                        var listKey = escapeMBeanPath(mbean);

                        //var listKey = encodeMBeanPath(mbean);
                        jolokia.list(listKey, onSuccess(function (meta) {
                            var attributes = meta.attr;
                            if (attributes) {
                                for (var key in attributes) {
                                    var value = attributes[key];
                                    if (value) {
                                        var typeName = value['type'];
                                        if (isNumberTypeName(typeName)) {
                                            if (!$scope.metrics[key]) {
                                                //console.log("Number attribute " + key + " for " + mbean);
                                                $scope.metrics[key] = key;
                                            }
                                        }
                                    }
                                }
                                if (++resultCounter >= mbeanCounter) {
                                    // TODO do we need to sort just in case?
                                    // lets look in the search URI to default the selections
                                    var search = $location.search();
                                    var attributeNames = toSearchArgumentArray(search["att"]);
                                    var elementNames = toSearchArgumentArray(search["el"]);
                                    if (attributeNames && attributeNames.length) {
                                        attributeNames.forEach(function (name) {
                                            if ($scope.metrics[name]) {
                                                $scope.selectedAttributes.push(name);
                                            }
                                        });
                                    }
                                    if (elementNames && elementNames.length) {
                                        elementNames.forEach(function (name) {
                                            if ($scope.mbeans[name]) {
                                                $scope.selectedMBeans.push(name);
                                            }
                                        });
                                    }

                                    // default selections if there are none
                                    if ($scope.selectedMBeans.length < 1) {
                                        $scope.selectedMBeans = Object.keys($scope.mbeans);
                                    }
                                    if ($scope.selectedAttributes.length < 1) {
                                        var attrKeys = Object.keys($scope.metrics).sort();
                                        if ($scope.selectedMBeans.length > 1) {
                                            $scope.selectedAttributes = [attrKeys.first()];
                                        } else {
                                            $scope.selectedAttributes = attrKeys;
                                        }
                                    }

                                    // lets update the sizes using jquery as it seems AngularJS doesn't support it
                                    $("#attributes").attr("size", Object.size($scope.metrics));
                                    $("#mbeans").attr("size", Object.size($scope.mbeans));
                                    Core.$apply($scope);
                                }
                            }
                        }));
                    }
                });
            }
        }
    }
    Jmx.ChartEditController = ChartEditController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    // IOperationControllerScope
    function OperationController($scope, workspace, jolokia, $timeout) {
        $scope.item = $scope.selectedOperation;
        $scope.title = $scope.item.humanReadable;
        $scope.desc = $scope.item.desc;
        $scope.operationResult = '';
        $scope.executeIcon = "icon-ok";
        $scope.mode = "text";
        $scope.entity = {};
        $scope.formConfig = {
            properties: {},
            description: $scope.objectName + "::" + $scope.item.name
        };

        $scope.item.args.forEach(function (arg) {
            $scope.formConfig.properties[arg.name] = {
                type: arg.type,
                tooltip: arg.desc,
                help: "Type: " + arg.type
            };
        });

        $timeout(function () {
            $("html, body").animate({ scrollTop: 0 }, "medium");
        }, 250);

        var sanitize = function (args) {
            if (args) {
                args.forEach(function (arg) {
                    switch (arg.type) {
                        case "int":
                        case "long":
                            arg.formType = "number";
                            break;
                        default:
                            arg.formType = "text";
                    }
                });
            }

            return args;
        };

        $scope.args = sanitize($scope.item.args);

        $scope.dump = function (data) {
            console.log(data);
        };

        $scope.ok = function () {
            $scope.operationResult = '';
        };

        $scope.reset = function () {
            $scope.entity = {};
        };

        $scope.close = function () {
            $scope.$parent.showInvoke = false;
        };

        $scope.handleResponse = function (response) {
            $scope.executeIcon = "icon-ok";
            $scope.operationStatus = "success";

            if (response === null || 'null' === response) {
                $scope.operationResult = "Operation Succeeded!";
            } else if (typeof response === 'string') {
                $scope.operationResult = response;
            } else {
                $scope.operationResult = angular.toJson(response, true);
            }

            $scope.mode = CodeEditor.detectTextFormat($scope.operationResult);

            Core.$apply($scope);
        };

        $scope.onSubmit = function (json, form) {
            Jmx.log.debug("onSubmit: json:", json, " form: ", form);
            Jmx.log.debug("$scope.item.args: ", $scope.item.args);
            angular.forEach(json, function (value, key) {
                $scope.item.args.find(function (arg) {
                    return arg['name'] === key;
                }).value = value;
            });
            $scope.execute();
        };

        $scope.execute = function () {
            var node = workspace.selection;

            if (!node) {
                return;
            }

            var objectName = node.objectName;

            if (!objectName) {
                return;
            }

            var args = [objectName, $scope.item.name];
            if ($scope.item.args) {
                $scope.item.args.forEach(function (arg) {
                    args.push(arg.value);
                });
            }

            args.push(onSuccess($scope.handleResponse, {
                error: function (response) {
                    $scope.executeIcon = "icon-ok";
                    $scope.operationStatus = "error";
                    var error = response.error;
                    $scope.operationResult = error;
                    var stacktrace = response.stacktrace;
                    if (stacktrace) {
                        $scope.operationResult = stacktrace;
                    }
                    Core.$apply($scope);
                }
            }));

            $scope.executeIcon = "icon-spinner icon-spin";
            var fn = jolokia.execute;
            fn.apply(jolokia, args);
        };
    }
    Jmx.OperationController = OperationController;

    function OperationsController($scope, workspace, jolokia, rbacACLMBean, $templateCache) {
        $scope.operations = {};
        $scope.objectName = '';
        $scope.methodFilter = '';
        $scope.workspace = workspace;
        $scope.selectedOperation = null;
        $scope.showInvoke = false;
        $scope.template = "";

        $scope.invokeOp = function (operation) {
            if (!$scope.canInvoke(operation)) {
                return;
            }
            $scope.selectedOperation = operation;
            $scope.showInvoke = true;
        };

        $scope.getJson = function (operation) {
            return angular.toJson(operation, true);
        };

        $scope.cancel = function () {
            $scope.selectedOperation = null;
            $scope.showInvoke = false;
        };

        $scope.$watch('showInvoke', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if (newValue) {
                    $scope.template = $templateCache.get("operationTemplate");
                } else {
                    $scope.template = "";
                }
            }
        });

        var fetch = Core.throttled(function () {
            var node = workspace.selection;
            if (!node) {
                return;
            }

            $scope.objectName = node.objectName;
            if (!$scope.objectName) {
                return;
            }

            jolokia.request({
                type: 'list',
                path: escapeMBeanPath($scope.objectName)
            }, onSuccess(render));
        }, 500);

        function getArgs(args) {
            return "(" + args.map(function (arg) {
                return arg.type;
            }).join() + ")";
        }

        function sanitize(value) {
            for (var item in value) {
                item = "" + item;
                value[item].name = item;
                value[item].humanReadable = humanizeValue(item);
            }
            return value;
        }

        $scope.isOperationsEmpty = function () {
            return $.isEmptyObject($scope.operations);
        };

        $scope.doFilter = function (item) {
            if (Core.isBlank($scope.methodFilter)) {
                return true;
            }
            if (item.name.toLowerCase().has($scope.methodFilter.toLowerCase()) || item.humanReadable.toLowerCase().has($scope.methodFilter.toLowerCase())) {
                return true;
            }
            return false;
        };

        $scope.canInvoke = function (operation) {
            if (!('canInvoke' in operation)) {
                return true;
            } else {
                return operation['canInvoke'];
            }
        };

        $scope.getClass = function (operation) {
            if ($scope.canInvoke(operation)) {
                return 'can-invoke';
            } else {
                return 'cant-invoke';
            }
        };

        $scope.$watch('workspace.selection', function (newValue, oldValue) {
            if (!workspace.selection || workspace.moveIfViewInvalid()) {
                return;
            }
            fetch();
        });

        function fetchPermissions(objectName, operations) {
            var map = {};
            map[objectName] = [];

            angular.forEach(operations, function (value, key) {
                map[objectName].push(value.name);
            });

            jolokia.request({
                type: 'exec',
                mbean: rbacACLMBean,
                operation: 'canInvoke(java.util.Map)',
                arguments: [map]
            }, onSuccess(function (response) {
                var map = response.value;
                angular.forEach(map[objectName], function (value, key) {
                    operations[key]['canInvoke'] = value['CanInvoke'];
                });

                //log.debug("Got operations: ", $scope.operations);
                Core.$apply($scope);
            }, {
                error: function (response) {
                    // silently ignore
                    Core.$apply($scope);
                }
            }));
        }

        function render(response) {
            var ops = response.value.op;
            var answer = {};

            angular.forEach(ops, function (value, key) {
                if (angular.isArray(value)) {
                    angular.forEach(value, function (value, index) {
                        answer[key + getArgs(value.args)] = value;
                    });
                } else {
                    answer[key + getArgs(value.args)] = value;
                }
            });
            $scope.operations = sanitize(answer);
            if (Core.isBlank(rbacACLMBean) || $scope.isOperationsEmpty()) {
                Core.$apply($scope);
                return;
            } else {
                fetchPermissions($scope.objectName, $scope.operations);
            }
        }
    }
    Jmx.OperationsController = OperationsController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function DonutChartController($scope, $routeParams, jolokia, $templateCache) {
        /*
        console.log("routeParams: ", $routeParams);
        
        
        // using multiple attributes
        $scope.mbean = "java.lang:type=OperatingSystem";
        $scope.total = "MaxFileDescriptorCount";
        $scope.terms = "OpenFileDescriptorCount";
        */
        // using a single attribute with multiple paths
        /*
        $scope.mbean = "java.lang:type=Memory";
        $scope.total = "Max";
        $scope.attribute = "HeapMemoryUsage";
        $scope.terms = "Used";
        */
        $scope.mbean = $routeParams['mbean'];
        $scope.total = $routeParams['total'];
        $scope.attribute = $routeParams['attribute'];
        $scope.terms = $routeParams['terms'];

        $scope.remainder = $routeParams['remaining'];

        $scope.template = "";
        $scope.termsArray = $scope.terms.split(",");

        $scope.data = {
            total: 0,
            terms: []
        };

        if (!$scope.attribute) {
            $scope.reqs = [{ type: 'read', mbean: $scope.mbean, attribute: $scope.total }];

            $scope.termsArray.forEach(function (term) {
                $scope.reqs.push({ type: 'read', mbean: $scope.mbean, attribute: term });
                $scope.data.terms.push({
                    term: term,
                    count: 0
                });
            });
        } else {
            var terms = $scope.termsArray.include($scope.total);
            $scope.reqs = [{ type: 'read', mbean: $scope.mbean, attribute: $scope.attribute, paths: terms.join(",") }];

            $scope.termsArray.forEach(function (term) {
                $scope.data.terms.push({
                    term: term,
                    count: 0
                });
            });
        }

        if ($scope.remainder && $scope.remainder !== "-") {
            $scope.data.terms.push({
                term: $scope.remainder,
                count: 0
            });
        }

        /*
        $scope.data = {
        total: 100,
        terms: [{
        term: "One",
        count: 25
        }, {
        term: "Two",
        count: 75
        }]
        };
        */
        $scope.render = function (response) {
            //console.log("got: ", response);
            var freeTerm = null;
            if ($scope.remainder && $scope.remainder !== "-") {
                freeTerm = $scope.data.terms.find(function (term) {
                    return term.term === $scope.remainder;
                });
            }

            if (!$scope.attribute) {
                if (response.request.attribute === $scope.total) {
                    $scope.data.total = response.value;
                } else {
                    var term = $scope.data.terms.find(function (term) {
                        return term.term === response.request.attribute;
                    });
                    if (term) {
                        term.count = response.value;
                    }

                    if (freeTerm) {
                        freeTerm.count = $scope.data.total;
                        $scope.data.terms.forEach(function (term) {
                            if (term.term !== $scope.remainder) {
                                freeTerm.count = freeTerm.count - term.count;
                            }
                        });
                    }
                }
            } else {
                if (response.request.attribute === $scope.attribute) {
                    $scope.data.total = response.value[$scope.total.toLowerCase()];

                    $scope.data.terms.forEach(function (term) {
                        if (term.term !== $scope.remainder) {
                            term.count = response.value[term.term.toLowerCase()];
                        }
                    });

                    if (freeTerm) {
                        freeTerm.count = $scope.data.total;
                        $scope.data.terms.forEach(function (term) {
                            if (term.term !== $scope.remainder) {
                                freeTerm.count = freeTerm.count - term.count;
                            }
                        });
                    }
                }
            }
            if ($scope.template === "") {
                $scope.template = $templateCache.get("donut");
            }

            // console.log("Data: ", $scope.data);
            $scope.data = Object.clone($scope.data);
            Core.$apply($scope);
        };

        Core.register(jolokia, $scope, $scope.reqs, onSuccess($scope.render));
    }
    Jmx.DonutChartController = DonutChartController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function createDashboardLink(widgetType, widget) {
        var href = "#" + widgetType.route;
        var routeParams = angular.toJson(widget);
        var title = widget.title;
        var size = angular.toJson({
            size_x: widgetType.size_x,
            size_y: widgetType.size_y
        });

        return "/dashboard/add?tab=dashboard" + "&href=" + encodeURIComponent(href) + "&size=" + encodeURIComponent(size) + "&title=" + encodeURIComponent(title) + "&routeParams=" + encodeURIComponent(routeParams);
    }
    Jmx.createDashboardLink = createDashboardLink;

    function getWidgetType(widget) {
        return Jmx.jmxWidgetTypes.find(function (type) {
            return type.type === widget.type;
        });
    }
    Jmx.getWidgetType = getWidgetType;

    Jmx.jmxWidgetTypes = [
        {
            type: "donut",
            icon: "icon-smile",
            route: "/jmx/widget/donut",
            size_x: 2,
            size_y: 2,
            title: "Add Donut chart to Dashboard"
        },
        {
            type: "area",
            icon: "icon-bar-chart",
            route: "/jmx/widget/area",
            size_x: 4,
            size_y: 2,
            title: "Add Area chart to Dashboard"
        }
    ];

    Jmx.jmxWidgets = [
        {
            type: "donut",
            title: "Java Heap Memory",
            mbean: "java.lang:type=Memory",
            attribute: "HeapMemoryUsage",
            total: "Max",
            terms: "Used",
            remaining: "Free"
        },
        {
            type: "donut",
            title: "Java Non Heap Memory",
            mbean: "java.lang:type=Memory",
            attribute: "NonHeapMemoryUsage",
            total: "Max",
            terms: "Used",
            remaining: "Free"
        },
        {
            type: "donut",
            title: "File Descriptor Usage",
            mbean: "java.lang:type=OperatingSystem",
            total: "MaxFileDescriptorCount",
            terms: "OpenFileDescriptorCount",
            remaining: "Free"
        },
        {
            type: "donut",
            title: "Loaded Classes",
            mbean: "java.lang:type=ClassLoading",
            total: "TotalLoadedClassCount",
            terms: "LoadedClassCount,UnloadedClassCount",
            remaining: "-"
        },
        {
            type: "donut",
            title: "Swap Size",
            mbean: "java.lang:type=OperatingSystem",
            total: "TotalSwapSpaceSize",
            terms: "FreeSwapSpaceSize",
            remaining: "Used Swap"
        },
        {
            type: "area",
            title: "Process CPU Time",
            mbean: "java.lang:type=OperatingSystem",
            attribute: "ProcessCpuTime"
        },
        {
            type: "area",
            title: "Process CPU Load",
            mbean: "java.lang:type=OperatingSystem",
            attribute: "ProcessCpuLoad"
        },
        {
            type: "area",
            title: "System CPU Load",
            mbean: "java.lang:type=OperatingSystem",
            attribute: "SystemCpuLoad"
        },
        {
            type: "area",
            title: "System CPU Time",
            mbean: "java.lang:type=OperatingSystem",
            attribute: "SystemCpuTime"
        }
    ];
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    Jmx.propertiesColumnDefs = [
        {
            field: 'name',
            displayName: 'Property',
            width: "27%",
            cellTemplate: '<div class="ngCellText" title="{{row.entity.attrDesc}}" ' + 'data-placement="bottom"><div ng-show="!inDashboard" class="inline" compile="row.entity.getDashboardWidgets()"></div>{{row.entity.name}}</div>' },
        {
            field: 'value',
            displayName: 'Value',
            width: "70%",
            cellTemplate: '<div class="ngCellText" ng-click="row.entity.onViewAttribute()" title="{{row.entity.tooltip}}" ng-bind-html-unsafe="row.entity.summary"></div>'
        }
    ];

    Jmx.foldersColumnDefs = [
        {
            displayName: 'Name',
            cellTemplate: '<div class="ngCellText"><a href="{{row.entity.folderHref(row)}}"><i class="{{row.entity.folderIconClass(row)}}"></i> {{row.getProperty("title")}}</a></div>'
        }
    ];

    function AttributesController($scope, $element, $location, workspace, jolokia, jmxWidgets, jmxWidgetTypes, $templateCache) {
        $scope.searchText = '';
        $scope.selectedItems = [];

        $scope.lastKey = null;
        $scope.attributesInfoCache = {};

        $scope.entity = {};
        $scope.attributeSchema = {};
        $scope.gridData = [];
        $scope.attributes = "";

        $scope.$watch('gridData.length', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                if (newValue > 0) {
                    $scope.attributes = $templateCache.get('gridTemplate');
                } else {
                    $scope.attributes = "";
                }
            }
        });

        var attributeSchemaBasic = {
            properties: {
                'key': {
                    description: 'Key',
                    tooltip: 'Attribute key',
                    type: 'string',
                    readOnly: 'true'
                },
                'description': {
                    description: 'Description',
                    tooltip: 'Attribute description',
                    type: 'string',
                    formTemplate: "<textarea class='input-xlarge' rows='2' readonly='true'></textarea>"
                },
                'type': {
                    description: 'Type',
                    tooltip: 'Attribute type',
                    type: 'string',
                    readOnly: 'true'
                }
            }
        };

        $scope.gridOptions = {
            scope: $scope,
            selectedItems: [],
            showFilter: false,
            canSelectRows: false,
            enableRowSelection: false,
            enableRowClickSelection: false,
            keepLastSelected: false,
            multiSelect: true,
            showColumnMenu: true,
            displaySelectionCheckbox: false,
            filterOptions: {
                filterText: ''
            },
            // TODO disabled for now as it causes https://github.com/hawtio/hawtio/issues/262
            //sortInfo: { field: 'name', direction: 'asc'},
            data: 'gridData',
            columnDefs: Jmx.propertiesColumnDefs
        };

        $scope.$watch('gridOptions.selectedItems', function (newValue, oldValue) {
            if (newValue !== oldValue) {
                Jmx.log.debug("Selected items: ", newValue);
                $scope.selectedItems = newValue;
            }
        }, true);

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            // clear selection if we clicked the jmx nav bar button
            // otherwise we may show data from Camel/ActiveMQ or other plugins that
            // reuse the JMX plugin for showing tables (#884)
            var currentUrl = $location.url();
            if (currentUrl.endsWith("/jmx/attributes")) {
                Jmx.log.debug("Reset selection in JMX plugin");
                workspace.selection = null;
                $scope.lastKey = null;
            }

            setTimeout(updateTableContents, 50);
        });

        $scope.$watch('workspace.selection', function () {
            if (workspace.moveIfViewInvalid()) {
                Core.unregister(jolokia, $scope);
                return;
            }
            setTimeout(function () {
                $scope.gridData = [];
                Core.$apply($scope);
                setTimeout(function () {
                    updateTableContents();
                }, 10);
            }, 10);
        });

        $scope.hasWidget = function (row) {
            return true;
        };

        $scope.onCancelAttribute = function () {
            // clear entity
            $scope.entity = {};
        };

        $scope.onUpdateAttribute = function () {
            var value = $scope.entity["attrValueEdit"];
            var key = $scope.entity["key"];

            // clear entity
            $scope.entity = {};

            // TODO: check if value changed
            // update the attribute on the mbean
            var mbean = workspace.getSelectedMBeanName();
            if (mbean) {
                jolokia.setAttribute(mbean, key, value, onSuccess(function (response) {
                    notification("success", "Updated attribute " + key);
                }));
            }
        };

        $scope.onViewAttribute = function (row) {
            if (!row.summary) {
                return;
            }

            // create entity and populate it with data from the selected row
            $scope.entity = {};
            $scope.entity["key"] = row.key;
            $scope.entity["description"] = row.attrDesc;
            $scope.entity["type"] = row.type;
            $scope.entity["rw"] = row.rw;
            var type = asJsonSchemaType(row.type, row.key);
            var readOnly = !row.rw;

            // calculate a textare with X number of rows that usually fit the value to display
            var len = row.summary.length;
            var rows = (len / 40) + 1;
            if (rows > 10) {
                // cap at most 10 rows to not make the dialog too large
                rows = 10;
            }

            if (readOnly) {
                // if the value is empty its a &nbsp; as we need this for the table to allow us to click on the empty row
                if (row.summary === '&nbsp;') {
                    $scope.entity["attrValueView"] = '';
                } else {
                    $scope.entity["attrValueView"] = row.summary;
                }

                // clone from the basic schema to the new schema we create on-the-fly
                // this is needed as the dialog have problems if reusing the schema, and changing the schema afterwards
                // so its safer to create a new schema according to our needs
                $scope.attributeSchemaView = {};
                for (var i in attributeSchemaBasic) {
                    $scope.attributeSchemaView[i] = attributeSchemaBasic[i];
                }

                // and add the new attrValue which is dynamic computed
                $scope.attributeSchemaView.properties.attrValueView = {
                    description: 'Value',
                    label: "Value",
                    tooltip: 'Attribute value',
                    type: 'string',
                    formTemplate: "<textarea class='input-xlarge' rows='" + rows + "' readonly='true'></textarea>"
                };

                // just to be safe, then delete not needed part of the schema
                if ($scope.attributeSchemaView) {
                    delete $scope.attributeSchemaView.properties.attrValueEdit;
                }
            } else {
                // if the value is empty its a &nbsp; as we need this for the table to allow us to click on the empty row
                if (row.summary === '&nbsp;') {
                    $scope.entity["attrValueEdit"] = '';
                } else {
                    $scope.entity["attrValueEdit"] = row.summary;
                }

                // clone from the basic schema to the new schema we create on-the-fly
                // this is needed as the dialog have problems if reusing the schema, and changing the schema afterwards
                // so its safer to create a new schema according to our needs
                $scope.attributeSchemaEdit = {};
                for (var i in attributeSchemaBasic) {
                    $scope.attributeSchemaEdit[i] = attributeSchemaBasic[i];
                }

                // and add the new attrValue which is dynamic computed
                $scope.attributeSchemaEdit.properties.attrValueEdit = {
                    description: 'Value',
                    label: "Value",
                    tooltip: 'Attribute value',
                    type: 'string',
                    formTemplate: "<textarea class='input-xlarge' rows='" + rows + "'></textarea>"
                };

                // just to be safe, then delete not needed part of the schema
                if ($scope.attributeSchemaEdit) {
                    delete $scope.attributeSchemaEdit.properties.attrValueView;
                }
            }

            $scope.showAttributeDialog = true;
        };

        $scope.getDashboardWidgets = function (row) {
            var mbean = workspace.getSelectedMBeanName();
            if (!mbean) {
                return '';
            }
            var potentialCandidates = jmxWidgets.filter(function (widget) {
                return mbean === widget.mbean;
            });

            if (potentialCandidates.isEmpty()) {
                return '';
            }

            potentialCandidates = potentialCandidates.filter(function (widget) {
                return widget.attribute === row.key || widget.total === row.key;
            });

            if (potentialCandidates.isEmpty()) {
                return '';
            }

            row.addChartToDashboard = function (type) {
                $scope.addChartToDashboard(row, type);
            };

            var rc = [];
            potentialCandidates.forEach(function (widget) {
                var widgetType = Jmx.getWidgetType(widget);
                rc.push("<i class=\"" + widgetType['icon'] + " clickable\" title=\"" + widgetType['title'] + "\" ng-click=\"row.entity.addChartToDashboard('" + widgetType['type'] + "')\"></i>");
            });
            return rc.join() + "&nbsp;";
        };

        $scope.addChartToDashboard = function (row, widgetType) {
            var mbean = workspace.getSelectedMBeanName();
            var candidates = jmxWidgets.filter(function (widget) {
                return mbean === widget.mbean;
            });

            candidates = candidates.filter(function (widget) {
                return widget.attribute === row.key || widget.total === row.key;
            });

            candidates = candidates.filter(function (widget) {
                return widget.type === widgetType;
            });

            // hmmm, we really should only have one result...
            var widget = candidates.first();
            var type = Jmx.getWidgetType(widget);

            //console.log("widgetType: ", type, " widget: ", widget);
            $location.url(Jmx.createDashboardLink(type, widget));
        };

        /*
        * Returns the toolBar template HTML to use for the current selection
        */
        $scope.toolBarTemplate = function () {
            // lets lookup the list of helpers by domain
            var answer = Jmx.getAttributeToolBar(workspace.selection);

            // TODO - maybe there's a better way to determine when to enable selections
            /*
            if (answer.startsWith("app/camel") && workspace.selection.children.length > 0) {
            $scope.selectToggle.setSelect(true);
            } else {
            $scope.selectToggle.setSelect(false);
            }
            */
            return answer;
        };

        $scope.invokeSelectedMBeans = function (operationName, completeFunction) {
            if (typeof completeFunction === "undefined") { completeFunction = null; }
            var queries = [];
            angular.forEach($scope.selectedItems || [], function (item) {
                var mbean = item["_id"];
                if (mbean) {
                    var opName = operationName;
                    if (angular.isFunction(operationName)) {
                        opName = operationName(item);
                    }

                    //console.log("Invoking operation " + opName + " on " + mbean);
                    queries.push({ type: "exec", operation: opName, mbean: mbean });
                }
            });
            if (queries.length) {
                var callback = function () {
                    if (completeFunction) {
                        completeFunction();
                    } else {
                        operationComplete();
                    }
                };
                jolokia.request(queries, onSuccess(callback, { error: callback }));
            }
        };

        $scope.folderHref = function (row) {
            if (!row.getProperty) {
                return "";
            }
            var key = row.getProperty("key");
            if (key) {
                return Core.createHref($location, "#" + $location.path() + "?nid=" + key, ["nid"]);
            } else {
                return "";
            }
        };

        $scope.folderIconClass = function (row) {
            // TODO lets ignore the classes property for now
            // as we don't have an easy way to know if there is an icon defined for an icon or not
            // and we want to make sure there always is an icon shown
            /*
            var classes = (row.getProperty("addClass") || "").trim();
            if (classes) {
            return classes;
            }
            */
            if (!row.getProperty) {
                return "";
            }
            return row.getProperty("objectName") ? "icon-cog" : "icon-folder-close";
        };

        function operationComplete() {
            updateTableContents();
        }

        function updateTableContents() {
            // lets clear any previous queries just in case!
            Core.unregister(jolokia, $scope);

            $scope.gridData = [];
            $scope.mbeanIndex = null;
            var mbean = workspace.getSelectedMBeanName();
            var request = null;
            var node = workspace.selection;
            if (node === null || angular.isUndefined(node) || node.key !== $scope.lastKey) {
                // cache attributes info, so we know if the attribute is read-only or read-write, and also the attribute description
                $scope.attributesInfoCache = null;
                if (mbean) {
                    var asQuery = function (node) {
                        var path = escapeMBeanPath(node);
                        var query = {
                            type: "LIST",
                            method: "post",
                            path: path,
                            ignoreErrors: true
                        };
                        return query;
                    };
                    var infoQuery = asQuery(mbean);
                    jolokia.request(infoQuery, onSuccess(function (response) {
                        $scope.attributesInfoCache = response.value;
                        Jmx.log.debug("Updated attributes info cache for mbean " + mbean);
                    }));
                }
            }

            if (mbean) {
                request = { type: 'read', mbean: mbean };
                if (node.key !== $scope.lastKey) {
                    $scope.gridOptions.columnDefs = Jmx.propertiesColumnDefs;
                    $scope.gridOptions.enableRowClickSelection = false;
                }
            } else if (node) {
                if (node.key !== $scope.lastKey) {
                    $scope.gridOptions.columnDefs = [];
                    $scope.gridOptions.enableRowClickSelection = true;
                }

                // lets query each child's details
                var children = node.children;
                if (children) {
                    var childNodes = children.map(function (child) {
                        return child.objectName;
                    });
                    var mbeans = childNodes.filter(function (mbean) {
                        return mbean;
                    });
                    if (mbeans) {
                        var typeNames = Jmx.getUniqueTypeNames(children);
                        if (typeNames.length <= 1) {
                            var query = mbeans.map(function (mbean) {
                                return { type: "READ", mbean: mbean, ignoreErrors: true };
                            });
                            if (query.length > 0) {
                                request = query;

                                // deal with multiple results
                                $scope.mbeanIndex = {};
                                $scope.mbeanRowCounter = 0;
                                $scope.mbeanCount = mbeans.length;
                                //$scope.columnDefs = [];
                            }
                        } else {
                            console.log("Too many type names " + typeNames);
                        }
                    }
                }
            }

            //var callback = onSuccess(render, { error: render });
            var callback = onSuccess(render);
            if (request) {
                $scope.request = request;
                Core.register(jolokia, $scope, request, callback);
            } else if (node) {
                if (node.key !== $scope.lastKey) {
                    $scope.gridOptions.columnDefs = Jmx.foldersColumnDefs;
                    $scope.gridOptions.enableRowClickSelection = true;
                }
                $scope.gridData = node.children;
                addHandlerFunctions($scope.gridData);
            }
            if (node) {
                $scope.lastKey = node.key;
            }
            Core.$apply($scope);
        }

        function render(response) {
            var data = response.value;
            var mbeanIndex = $scope.mbeanIndex;
            var mbean = response.request['mbean'];

            if (mbean) {
                // lets store the mbean in the row for later
                data["_id"] = mbean;
            }
            if (mbeanIndex) {
                if (mbean) {
                    var idx = mbeanIndex[mbean];
                    if (!angular.isDefined(idx)) {
                        idx = $scope.mbeanRowCounter;
                        mbeanIndex[mbean] = idx;
                        $scope.mbeanRowCounter += 1;
                    }
                    if (idx === 0) {
                        // this is to force the table to repaint
                        $scope.selectedIndices = $scope.selectedItems.map(function (item) {
                            return $scope.gridData.indexOf(item);
                        });
                        $scope.gridData = [];

                        if (!$scope.gridOptions.columnDefs.length) {
                            // lets update the column definitions based on any configured defaults
                            var key = workspace.selectionConfigKey();
                            var defaultDefs = workspace.attributeColumnDefs[key] || [];
                            var defaultSize = defaultDefs.length;
                            var map = {};
                            angular.forEach(defaultDefs, function (value, key) {
                                var field = value.field;
                                if (field) {
                                    map[field] = value;
                                }
                            });

                            var extraDefs = [];
                            angular.forEach(data, function (value, key) {
                                if (includePropertyValue(key, value)) {
                                    if (!map[key]) {
                                        extraDefs.push({
                                            field: key,
                                            displayName: key === '_id' ? 'Object name' : humanizeValue(key),
                                            visible: defaultSize === 0
                                        });
                                    }
                                }
                            });

                            // the additional columns (which are not pre-configured), should be sorted
                            // so the column menu has a nice sorted list instead of random ordering
                            extraDefs = extraDefs.sort(function (def, def2) {
                                // make sure _id is last
                                if (def.field.startsWith('_')) {
                                    return 1;
                                } else if (def2.field.startsWith('_')) {
                                    return -1;
                                }
                                return def.field.localeCompare(def2.field);
                            });
                            extraDefs.forEach(function (e) {
                                defaultDefs.push(e);
                            });

                            $scope.gridOptions.columnDefs = defaultDefs;
                            $scope.gridOptions.enableRowClickSelection = true;
                        }
                    }

                    // assume 1 row of data per mbean
                    $scope.gridData[idx] = data;
                    addHandlerFunctions($scope.gridData);

                    var count = $scope.mbeanCount;
                    if (!count || idx + 1 >= count) {
                        // only cause a refresh on the last row
                        var newSelections = $scope.selectedIndices.map(function (idx) {
                            return $scope.gridData[idx];
                        }).filter(function (row) {
                            return row;
                        });
                        $scope.selectedItems.splice(0, $scope.selectedItems.length);
                        $scope.selectedItems.push.apply($scope.selectedItems, newSelections);

                        //console.log("Would have selected " + JSON.stringify($scope.selectedItems));
                        Core.$apply($scope);
                    }
                    // if the last row, then fire an event
                } else {
                    console.log("No mbean name in request " + JSON.stringify(response.request));
                }
            } else {
                $scope.gridOptions.columnDefs = Jmx.propertiesColumnDefs;
                $scope.gridOptions.enableRowClickSelection = false;
                var showAllAttributes = true;
                if (angular.isObject(data)) {
                    var properties = [];
                    angular.forEach(data, function (value, key) {
                        if (showAllAttributes || includePropertyValue(key, value)) {
                            // always skip keys which start with _
                            if (!key.startsWith("_")) {
                                // lets format the ObjectName nicely dealing with objects with
                                // nested object names or arrays of object names
                                if (key === "ObjectName") {
                                    value = unwrapObjectName(value);
                                }

                                // lets unwrap any arrays of object names
                                if (angular.isArray(value)) {
                                    value = value.map(function (v) {
                                        return unwrapObjectName(v);
                                    });
                                }

                                // the value must be string as the sorting/filtering of the table relies on that
                                var type = lookupAttributeType(key);
                                var data = { key: key, name: humanizeValue(key), value: safeNullAsString(value, type) };

                                generateSummaryAndDetail(key, data);
                                properties.push(data);
                            }
                        }
                    });
                    if (!properties.any(function (p) {
                        return p['key'] === 'ObjectName';
                    })) {
                        var objectName = {
                            key: "ObjectName",
                            name: "Object Name",
                            value: mbean
                        };
                        generateSummaryAndDetail(objectName.key, objectName);
                        properties.push(objectName);
                    }
                    properties = properties.sortBy("name");
                    $scope.selectedItems = [data];
                    data = properties;
                }
                $scope.gridData = data;
                addHandlerFunctions($scope.gridData);
                Core.$apply($scope);
            }
        }

        function addHandlerFunctions(data) {
            data.forEach(function (item) {
                item['inDashboard'] = $scope.inDashboard;
                item['getDashboardWidgets'] = function () {
                    return $scope.getDashboardWidgets(item);
                };
                item['onViewAttribute'] = function () {
                    $scope.onViewAttribute(item);
                };
                item['folderIconClass'] = function (row) {
                    return $scope.folderIconClass(row);
                };
                item['folderHref'] = function (row) {
                    return $scope.folderHref(row);
                };
            });
        }

        function unwrapObjectName(value) {
            if (!angular.isObject(value)) {
                return value;
            }
            var keys = Object.keys(value);
            if (keys.length === 1 && keys[0] === "objectName") {
                return value["objectName"];
            }
            return value;
        }

        function generateSummaryAndDetail(key, data) {
            var value = data.value;
            if (!angular.isArray(value) && angular.isObject(value)) {
                var detailHtml = "<table class='table table-striped'>";
                var summary = "";
                var object = value;
                var keys = Object.keys(value).sort();
                angular.forEach(keys, function (key) {
                    var value = object[key];
                    detailHtml += "<tr><td>" + humanizeValue(key) + "</td><td>" + value + "</td></tr>";
                    summary += "" + humanizeValue(key) + ": " + value + "  ";
                });
                detailHtml += "</table>";
                data.summary = summary;
                data.detailHtml = detailHtml;
                data.tooltip = summary;
            } else {
                var text = value;

                // if the text is empty then use a no-break-space so the table allows us to click on the row,
                // otherwise if the text is empty, then you cannot click on the row
                if (text === '') {
                    text = '&nbsp;';
                    data.tooltip = "";
                } else {
                    data.tooltip = text;
                }
                data.summary = "" + text + "";
                data.detailHtml = "<pre>" + text + "</pre>";
                if (angular.isArray(value)) {
                    var html = "<ul>";
                    angular.forEach(value, function (item) {
                        html += "<li>" + item + "</li>";
                    });
                    html += "</ul>";
                    data.detailHtml = html;
                }
            }

            // enrich the data with information if the attribute is read-only/read-write, and the JMX attribute description (if any)
            data.rw = false;
            data.attrDesc = data.name;
            data.type = "string";
            if ($scope.attributesInfoCache != null && 'attr' in $scope.attributesInfoCache) {
                var info = $scope.attributesInfoCache.attr[key];
                if (angular.isDefined(info)) {
                    data.rw = info.rw;
                    data.attrDesc = info.desc;
                    data.type = info.type;
                }
            }
        }

        function lookupAttributeType(key) {
            if ($scope.attributesInfoCache != null && 'attr' in $scope.attributesInfoCache) {
                var info = $scope.attributesInfoCache.attr[key];
                if (angular.isDefined(info)) {
                    return info.type;
                }
            }
            return null;
        }

        function includePropertyValue(key, value) {
            return !angular.isObject(value);
        }

        function asJsonSchemaType(typeName, id) {
            if (typeName) {
                var lower = typeName.toLowerCase();
                if (lower.startsWith("int") || lower === "long" || lower === "short" || lower === "byte" || lower.endsWith("int")) {
                    return "integer";
                }
                if (lower === "double" || lower === "float" || lower === "bigdecimal") {
                    return "number";
                }
                if (lower === "boolean" || lower === "java.lang.boolean") {
                    return "boolean";
                }
                if (lower === "string" || lower === "java.lang.String") {
                    return "string";
                }
            }

            // fallback as string
            return "string";
        }
    }
    Jmx.AttributesController = AttributesController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function AttributeController($scope, jolokia) {
        $scope.init = function (mbean, attribute) {
            $scope.mbean = mbean;
            $scope.attribute = attribute;

            if (angular.isDefined($scope.mbean) && angular.isDefined($scope.attribute)) {
                Core.register(jolokia, $scope, {
                    type: 'read', mbean: $scope.mbean, attribute: $scope.attribute
                }, onSuccess(render));
            }
        };

        function render(response) {
            if (!Object.equal($scope.data, response.value)) {
                $scope.data = safeNull(response.value);
                Core.$apply($scope);
            }
        }
    }
    Jmx.AttributeController = AttributeController;

    function AttributeChartController($scope, jolokia, $document) {
        $scope.init = function (mbean, attribute) {
            $scope.mbean = mbean;
            $scope.attribute = attribute;

            if (angular.isDefined($scope.mbean) && angular.isDefined($scope.attribute)) {
                Core.register(jolokia, $scope, {
                    type: 'read', mbean: $scope.mbean, attribute: $scope.attribute
                }, onSuccess(render));
            }
        };

        function render(response) {
            if (!angular.isDefined($scope.chart)) {
                $scope.chart = $($document.find("#" + $scope.attribute)[0]);
                if ($scope.chart) {
                    $scope.width = $scope.chart.width();
                }
            }

            if (!angular.isDefined($scope.context)) {
                console.log("Got: ", response);

                $scope.context = cubism.context().serverDelay(0).clientDelay(0).step(1000).size($scope.width);
                $scope.jcontext = $scope.context.jolokia(jolokia);

                $scope.metrics = [];

                Object.extended(response.value).keys(function (key, value) {
                    $scope.metrics.push($scope.jcontext.metric({
                        type: 'read',
                        mbean: $scope.mbean,
                        attribute: $scope.attribute,
                        path: key
                    }, $scope.attribute));
                });

                d3.select("#" + $scope.attribute).call(function (div) {
                    div.append("div").data($scope.metrics).call($scope.context.horizon());
                });

                // let cubism take over at this point...
                Core.unregister(jolokia, $scope);
                Core.$apply($scope);
            }
        }
    }
    Jmx.AttributeChartController = AttributeChartController;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    Jmx.log = Logger.get("JMX");

    var attributesToolBars = {};

    Jmx.lazyLoaders = {};

    function findLazyLoadingFunction(workspace, folder) {
        var factories = workspace.jmxTreeLazyLoadRegistry[folder.domain];
        var lazyFunction = null;
        if (factories && factories.length) {
            angular.forEach(factories, function (customLoader) {
                if (!lazyFunction) {
                    lazyFunction = customLoader(folder);
                }
            });
        }
        return lazyFunction;
    }
    Jmx.findLazyLoadingFunction = findLazyLoadingFunction;

    function registerLazyLoadHandler(domain, lazyLoaderFactory) {
        if (!Jmx.lazyLoaders) {
            Jmx.lazyLoaders = {};
        }
        var array = Jmx.lazyLoaders[domain];
        if (!array) {
            array = [];
            Jmx.lazyLoaders[domain] = array;
        }
        array.push(lazyLoaderFactory);
    }
    Jmx.registerLazyLoadHandler = registerLazyLoadHandler;

    function unregisterLazyLoadHandler(domain, lazyLoaderFactory) {
        if (Jmx.lazyLoaders) {
            var array = Jmx.lazyLoaders[domain];
            if (array) {
                array.remove(lazyLoaderFactory);
            }
        }
    }
    Jmx.unregisterLazyLoadHandler = unregisterLazyLoadHandler;

    /**
    * Registers a toolbar template for the given plugin name, jmxDomain.
    * @method addAttributeToolBar
    * @for Jmx
    * @param {String} pluginName used so that we can later on remove this function when the plugin is removed
    * @param {String} jmxDomain the JMX domain to avoid having to evaluate too many functions on each selection
    * @param {Function} fn the function used to decide which attributes tool bar should be used for the given select
    */
    function addAttributeToolBar(pluginName, jmxDomain, fn) {
        var array = attributesToolBars[jmxDomain];
        if (!array) {
            array = [];
            attributesToolBars[jmxDomain] = array;
        }
        array.push(fn);
    }
    Jmx.addAttributeToolBar = addAttributeToolBar;

    /**
    * Try find a custom toolbar HTML template for the given selection or returns the default value
    * @method getAttributeToolbar
    * @for Jmx
    * @param {Core.NodeSelection} node
    * @param {String} defaultValue
    */
    function getAttributeToolBar(node, defaultValue) {
        if (typeof defaultValue === "undefined") { defaultValue = "app/jmx/html/attributeToolBar.html"; }
        var answer = null;
        var jmxDomain = (node) ? node.domain : null;
        if (jmxDomain) {
            var array = attributesToolBars[jmxDomain];
            if (array) {
                for (var idx in array) {
                    var fn = array[idx];
                    answer = fn(node);
                    if (answer)
                        break;
                }
            }
        }
        return (answer) ? answer : defaultValue;
    }
    Jmx.getAttributeToolBar = getAttributeToolBar;

    function updateTreeSelectionFromURL($location, treeElement, activateIfNoneSelected) {
        if (typeof activateIfNoneSelected === "undefined") { activateIfNoneSelected = false; }
        updateTreeSelectionFromURLAndAutoSelect($location, treeElement, null, activateIfNoneSelected);
    }
    Jmx.updateTreeSelectionFromURL = updateTreeSelectionFromURL;

    function updateTreeSelectionFromURLAndAutoSelect($location, treeElement, autoSelect, activateIfNoneSelected) {
        if (typeof activateIfNoneSelected === "undefined") { activateIfNoneSelected = false; }
        var dtree = treeElement.dynatree("getTree");
        if (dtree) {
            var node = null;
            var key = $location.search()['nid'];
            if (key) {
                try  {
                    node = dtree.activateKey(key);
                } catch (e) {
                    // tree not visible we suspect!
                }
            }
            if (node) {
                node.expand(true);
            } else {
                if (!treeElement.dynatree("getActiveNode")) {
                    // lets expand the first node
                    var root = treeElement.dynatree("getRoot");
                    var children = root ? root.getChildren() : null;
                    if (children && children.length) {
                        var first = children[0];
                        first.expand(true);

                        // invoke any auto select function, and use its result as new first, if any returned
                        if (autoSelect) {
                            var result = autoSelect(first);
                            if (result) {
                                first = result;
                            }
                        }
                        if (activateIfNoneSelected) {
                            first.expand();
                            first.activate();
                        }
                    } else {
                        /*
                        causes NPE :)
                        
                        var first = children[0];
                        first.expand(true);
                        if (activateIfNoneSelected) {
                        first.activate();
                        }
                        */
                    }
                }
            }
        }
    }
    Jmx.updateTreeSelectionFromURLAndAutoSelect = updateTreeSelectionFromURLAndAutoSelect;

    function getUniqueTypeNames(children) {
        var typeNameMap = {};
        angular.forEach(children, function (mbean) {
            var typeName = mbean.typeName;
            if (typeName) {
                typeNameMap[typeName] = mbean;
            }
        });

        // only query if all the typenames are the same
        var typeNames = Object.keys(typeNameMap);
        return typeNames;
    }
    Jmx.getUniqueTypeNames = getUniqueTypeNames;

    function enableTree($scope, $location, workspace, treeElement, children, redraw, onActivateFn) {
        if (typeof redraw === "undefined") { redraw = false; }
        if (typeof onActivateFn === "undefined") { onActivateFn = null; }
        //$scope.workspace = workspace;
        if (treeElement.length) {
            if (!onActivateFn) {
                onActivateFn = function (node) {
                    var data = node.data;

                    //$scope.select(data);
                    workspace.updateSelectionNode(data);
                    Core.$apply($scope);
                };
            }
            workspace.treeElement = treeElement;
            treeElement.dynatree({
                /*
                * The event handler called when a different node in the tree is selected
                */
                onActivate: onActivateFn,
                onLazyRead: function (treeNode) {
                    var folder = treeNode.data;
                    var plugin = null;
                    if (folder) {
                        plugin = Jmx.findLazyLoadingFunction(workspace, folder);
                    }
                    if (plugin) {
                        console.log("Lazy loading folder " + folder.title);
                        var oldChildren = folder.childen;
                        plugin(workspace, folder, function () {
                            treeNode.setLazyNodeStatus(DTNodeStatus_Ok);
                            var newChildren = folder.children;
                            if (newChildren !== oldChildren) {
                                treeNode.removeChildren();
                                angular.forEach(newChildren, function (newChild) {
                                    treeNode.addChild(newChild);
                                });
                            }
                        });
                    } else {
                        treeNode.setLazyNodeStatus(DTNodeStatus_Ok);
                    }
                },
                onClick: function (node, event) {
                    if (event["metaKey"]) {
                        event.preventDefault();
                        var url = $location.absUrl();
                        if (node && node.data) {
                            var key = node.data["key"];
                            if (key) {
                                var hash = $location.search();
                                hash["nid"] = key;

                                // TODO this could maybe be a generic helper function?
                                // lets trim after the ?
                                var idx = url.indexOf('?');
                                if (idx <= 0) {
                                    url += "?";
                                } else {
                                    url = url.substring(0, idx + 1);
                                }
                                url += $.param(hash);
                            }
                        }
                        window.open(url, '_blank');
                        window.focus();
                        return false;
                    }
                    return true;
                },
                persist: false,
                debugLevel: 0,
                //children: $scope.workspace.tree.children
                children: children
            });

            if (redraw) {
                workspace.redrawTree();
            }
        }
    }
    Jmx.enableTree = enableTree;
})(Jmx || (Jmx = {}));
/**
* @module Jmx
*/
var Jmx;
(function (Jmx) {
    function AreaChartController($scope, $routeParams, jolokia, $templateCache, localStorage, $element) {
        $scope.mbean = $routeParams['mbean'];
        $scope.attribute = $routeParams['attribute'];
        $scope.duration = localStorage['updateRate'];

        $scope.width = 308;
        $scope.height = 296;

        $scope.template = "";

        $scope.entries = [];

        $scope.data = {
            entries: $scope.entries
        };

        $scope.req = [{ type: 'read', mbean: $scope.mbean, attribute: $scope.attribute }];

        $scope.render = function (response) {
            $scope.entries.push({
                time: response.timestamp,
                count: response.value
            });

            $scope.entries = $scope.entries.last(15);

            if ($scope.template === "") {
                $scope.template = $templateCache.get("areaChart");
            }

            $scope.data = {
                _type: "date_histogram",
                entries: $scope.entries
            };

            Core.$apply($scope);
        };

        Core.register(jolokia, $scope, $scope.req, onSuccess($scope.render));
    }
    Jmx.AreaChartController = AreaChartController;
})(Jmx || (Jmx = {}));
/**
* @module Wiki
*/
var Wiki;
(function (Wiki) {
    function CamelCanvasController($scope, $element, workspace, jolokia, wikiRepository, $templateCache, $interpolate) {
        $scope.addDialog = new UI.Dialog();
        $scope.propertiesDialog = new UI.Dialog();
        $scope.modified = false;
        $scope.camelIgnoreIdForLabel = Camel.ignoreIdForLabel(localStorage);
        $scope.camelMaximumLabelWidth = Camel.maximumLabelWidth(localStorage);
        $scope.camelMaximumTraceOrDebugBodyLength = Camel.maximumTraceOrDebugBodyLength(localStorage);

        $scope.forms = {};

        $scope.nodeTemplate = $interpolate($templateCache.get("nodeTemplate"));

        $scope.$watch("camelContextTree", function () {
            var tree = $scope.camelContextTree;
            $scope.rootFolder = tree;

            // now we've got cid values in the tree and DOM, lets create an index so we can bind the DOM to the tree model
            $scope.folders = Camel.addFoldersToIndex($scope.rootFolder);

            var doc = Core.pathGet(tree, ["xmlDocument"]);
            if (doc) {
                $scope.doc = doc;
                reloadRouteIds();
                onRouteSelectionChanged();
            }
        });

        $scope.addAndCloseDialog = function () {
            var nodeModel = $scope.selectedNodeModel();
            if (nodeModel) {
                addNewNode(nodeModel);
            }
            $scope.addDialog.close();
        };

        $scope.removeNode = function () {
            var folder = getSelectedOrRouteFolder();
            if (folder) {
                var nodeName = Camel.getFolderCamelNodeId(folder);
                folder.detach();
                if ("route" === nodeName) {
                    // lets also clear the selected route node
                    $scope.selectedRouteId = null;
                }
                updateSelection(null);
                treeModified();
            }
        };

        $scope.doLayout = function () {
            $scope.drawnRouteId = null;
            onRouteSelectionChanged();
        };

        function isRouteOrNode() {
            return !$scope.selectedFolder;
        }

        $scope.getDeleteTitle = function () {
            if (isRouteOrNode()) {
                return "Delete this route";
            }
            return "Delete this node";
        };

        $scope.getDeleteTarget = function () {
            if (isRouteOrNode()) {
                return "Route";
            }
            return "Node";
        };

        $scope.isFormDirty = function () {
            Wiki.log.debug("endpointForm: ", $scope.endpointForm);
            if ($scope.endpointForm.$dirty) {
                return true;
            }
            if (!$scope.forms['formEditor']) {
                return false;
            } else {
                return $scope.forms['formEditor']['$dirty'];
            }
        };

        /* TODO
        $scope.resetForms = () => {
        
        }
        */
        /*
        * Converts a path and a set of endpoint parameters into a URI we can then use to store in the XML
        */
        function createEndpointURI(endpointScheme, slashesText, endpointPath, endpointParameters) {
            console.log("scheme " + endpointScheme + " path " + endpointPath + " parameters " + endpointParameters);

            // now lets create the new URI from the path and parameters
            // TODO should we use JMX for this?
            var uri = ((endpointScheme) ? endpointScheme + ":" + slashesText : "") + (endpointPath ? endpointPath : "");
            var paramText = Core.hashToString(endpointParameters);
            if (paramText) {
                uri += "?" + paramText;
            }
            return uri;
        }

        $scope.updateProperties = function () {
            Wiki.log.info("old URI is " + $scope.nodeData.uri);
            var uri = createEndpointURI($scope.endpointScheme, ($scope.endpointPathHasSlashes ? "//" : ""), $scope.endpointPath, $scope.endpointParameters);
            Wiki.log.info("new URI is " + uri);
            if (uri) {
                $scope.nodeData.uri = uri;
            }

            var key = null;
            var selectedFolder = $scope.selectedFolder;
            if (selectedFolder) {
                key = selectedFolder.key;

                // lets delete the current selected node's div so its updated with the new template values
                var elements = $element.find(".canvas").find("[id='" + key + "']").first().remove();
            }

            treeModified();

            if (key) {
                updateSelection(key);
            }

            if ($scope.isFormDirty()) {
                $scope.endpointForm.$setPristine();
                if ($scope.forms['formEditor']) {
                    $scope.forms['formEditor'].$setPristine();
                }
            }

            Core.$apply($scope);
        };

        $scope.save = function () {
            // generate the new XML
            if ($scope.rootFolder) {
                var xmlNode = Camel.generateXmlFromFolder($scope.rootFolder);
                if (xmlNode) {
                    var text = Core.xmlNodeToString(xmlNode);
                    if (text) {
                        var decoded = decodeURIComponent(text);
                        Wiki.log.debug("Saving xml decoded: " + decoded);

                        // lets save the file...
                        var commitMessage = $scope.commitMessage || "Updated page " + $scope.pageId;
                        wikiRepository.putPage($scope.branch, $scope.pageId, decoded, commitMessage, function (status) {
                            Wiki.onComplete(status);
                            notification("success", "Saved " + $scope.pageId);
                            $scope.modified = false;
                            goToView();
                            Core.$apply($scope);
                        });
                    }
                }
            }
        };

        $scope.cancel = function () {
            console.log("cancelling...");
            // TODO show dialog if folks are about to lose changes...
        };

        $scope.$watch("selectedRouteId", onRouteSelectionChanged);

        function goToView() {
            // TODO lets navigate to the view if we have a separate view one day :)
            /*
            if ($scope.breadcrumbs && $scope.breadcrumbs.length > 1) {
            var viewLink = $scope.breadcrumbs[$scope.breadcrumbs.length - 2];
            console.log("goToView has found view " + viewLink);
            var path = Core.trimLeading(viewLink, "#");
            $location.path(path);
            } else {
            console.log("goToView has no breadcrumbs!");
            }
            */
        }

        function addNewNode(nodeModel) {
            var doc = $scope.doc || document;
            var parentFolder = $scope.selectedFolder || $scope.rootFolder;
            var key = nodeModel["_id"];
            if (!key) {
                console.log("WARNING: no id for model " + JSON.stringify(nodeModel));
            } else {
                var treeNode = $scope.selectedFolder;
                if (key === "route") {
                    // lets add to the root of the tree
                    treeNode = $scope.rootFolder;
                } else {
                    if (!treeNode) {
                        // lets select the last route - and create a new route if need be
                        var root = $scope.rootFolder;
                        var children = root.children;
                        if (!children || !children.length) {
                            addNewNode(Camel.getCamelSchema("route"));
                            children = root.children;
                        }
                        if (children && children.length) {
                            treeNode = getRouteFolder($scope.rootFolder, $scope.selectedRouteId) || children[children.length - 1];
                        } else {
                            console.log("Could not add a new route to the empty tree!");
                            return;
                        }
                    }

                    // if the parent folder likes to act as a pipeline, then add
                    // after the parent, rather than as a child
                    var parentTypeName = Camel.getFolderCamelNodeId(treeNode);
                    if (!Camel.acceptOutput(parentTypeName)) {
                        treeNode = treeNode.parent || treeNode;
                    }
                }
                if (treeNode) {
                    var node = doc.createElement(key);
                    parentFolder = treeNode;
                    var addedNode = Camel.addRouteChild(parentFolder, node);

                    // TODO add the schema here for an element??
                    // or default the data or something
                    var nodeData = {};
                    if (key === "endpoint" && $scope.endpointConfig) {
                        var key = $scope.endpointConfig.key;
                        if (key) {
                            nodeData["uri"] = key + ":";
                        }
                    }
                    addedNode["camelNodeData"] = nodeData;
                    addedNode["endpointConfig"] = $scope.endpointConfig;

                    if (key === "route") {
                        // lets generate a new routeId and switch to it
                        var count = $scope.routeIds.length;
                        var nodeId = null;
                        while (true) {
                            nodeId = "route" + (++count);
                            if (!$scope.routeIds.find(nodeId)) {
                                break;
                            }
                        }
                        addedNode["routeXmlNode"].setAttribute("id", nodeId);
                        $scope.selectedRouteId = nodeId;
                    }
                }
            }
            treeModified();
        }

        function treeModified(reposition) {
            if (typeof reposition === "undefined") { reposition = true; }
            // lets recreate the XML model from the update Folder tree
            var newDoc = Camel.generateXmlFromFolder($scope.rootFolder);
            var tree = Camel.loadCamelTree(newDoc, $scope.pageId);
            if (tree) {
                $scope.rootFolder = tree;
                $scope.doc = Core.pathGet(tree, ["xmlDocument"]);
            }
            $scope.modified = true;
            reloadRouteIds();
            $scope.doLayout();
            Core.$apply($scope);
        }

        function reloadRouteIds() {
            $scope.routeIds = [];
            var doc = $($scope.doc);
            $scope.camelSelectionDetails.selectedCamelContextId = doc.find("camelContext").attr("id");
            doc.find("route").each(function (idx, route) {
                var id = route.getAttribute("id");
                if (id) {
                    $scope.routeIds.push(id);
                }
            });
        }

        function onRouteSelectionChanged() {
            if ($scope.doc) {
                if (!$scope.selectedRouteId && $scope.routeIds && $scope.routeIds.length) {
                    $scope.selectedRouteId = $scope.routeIds[0];
                }
                if ($scope.selectedRouteId && $scope.selectedRouteId !== $scope.drawnRouteId) {
                    var nodes = [];
                    var links = [];
                    Camel.loadRouteXmlNodes($scope, $scope.doc, $scope.selectedRouteId, nodes, links, getWidth());
                    updateSelection($scope.selectedRouteId);

                    // now we've got cid values in the tree and DOM, lets create an index so we can bind the DOM to the tree model
                    $scope.folders = Camel.addFoldersToIndex($scope.rootFolder);
                    showGraph(nodes, links);
                    $scope.drawnRouteId = $scope.selectedRouteId;
                }
                $scope.camelSelectionDetails.selectedRouteId = $scope.selectedRouteId;
            }
        }

        function showGraph(nodes, links) {
            layoutGraph(nodes, links);
        }

        function getNodeId(node) {
            if (angular.isNumber(node)) {
                var idx = node;
                node = $scope.nodeStates[idx];
                if (!node) {
                    console.log("Cant find node at " + idx);
                    return "node-" + idx;
                }
            }
            return node.cid || "node-" + node.id;
        }

        function getSelectedOrRouteFolder() {
            return $scope.selectedFolder || getRouteFolder($scope.rootFolder, $scope.selectedRouteId);
        }

        function getContainerElement() {
            var rootElement = $element;
            var containerElement = rootElement.find(".canvas");
            if (!containerElement || !containerElement.length)
                containerElement = rootElement;
            return containerElement;
        }

        // context menu (right click) on any component.
        /* TODO disabling this for now just so I can look at elements easily in the dev tools
        jsPlumb.bind("contextmenu", function (component, originalEvent) {
        alert("context menu on component " + component.id);
        originalEvent.preventDefault();
        return false;
        });
        */
        /*
        function clearCanvasLayout(jsPlumb, containerElement) {
        try {
        containerElement.empty();
        jsPlumb.reset();
        } catch (e) {
        // ignore errors
        }
        return jsPlumb;
        }
        */
        // configure canvas layout and styles
        var endpointStyle = ["Dot", { radius: 4, cssClass: 'camel-canvas-endpoint' }];
        var hoverPaintStyle = { strokeStyle: "red", lineWidth: 3 };

        //var labelStyles: any[] = [ "Label", { label:"FOO", id:"label" }];
        var labelStyles = ["Label"];
        var arrowStyles = [
            "Arrow", {
                location: 1,
                id: "arrow",
                length: 8,
                width: 8,
                foldback: 0.8
            }];
        var connectorStyle = ["StateMachine", { curviness: 10, proximityLimit: 50 }];

        jsPlumb.importDefaults({
            Endpoint: endpointStyle,
            HoverPaintStyle: hoverPaintStyle,
            ConnectionOverlays: [
                arrowStyles,
                labelStyles
            ]
        });

        $scope.$on('$destroy', function () {
            jsPlumb.reset();
        });

        // double click on any connection
        jsPlumb.bind("dblclick", function (connection, originalEvent) {
            if (jsPlumb.isSuspendDrawing()) {
                return;
            }
            alert("double click on connection from " + connection.sourceId + " to " + connection.targetId);
        });

        jsPlumb.bind('connection', function (info, evt) {
            if (jsPlumb.isSuspendDrawing()) {
                return;
            }

            //log.debug("Connection event: ", info);
            Wiki.log.debug("Creating connection from ", info.source.get(0).id, " to ", info.target.get(0).id);
            var link = getLink(info);
            var source = $scope.folders[link.source];
            var target = $scope.folders[link.target];
            source.moveChild(target);
            treeModified();
        });

        jsPlumb.bind('connectionDetached', function (info, evt) {
            if (jsPlumb.isSuspendDrawing()) {
                return;
            }

            //log.debug("Connection detach event: ", info);
            Wiki.log.debug("Detaching connection from ", info.source.get(0).id, " to ", info.target.get(0).id);
            var link = getLink(info);
            var source = $scope.folders[link.source];
            var target = $scope.folders[link.target];
            // TODO orphan target folder without actually deleting it
        });

        // lets delete connections on click
        jsPlumb.bind("click", function (c) {
            if (jsPlumb.isSuspendDrawing()) {
                return;
            }
            jsPlumb.detach(c);
        });

        function layoutGraph(nodes, links) {
            var transitions = [];
            var states = Core.createGraphStates(nodes, links, transitions);

            Wiki.log.debug("links: ", links);
            Wiki.log.debug("transitions: ", transitions);

            $scope.nodeStates = states;
            var containerElement = getContainerElement();

            jsPlumb.doWhileSuspended(function () {
                //set our container to some arbitrary initial size
                containerElement.css({
                    'width': '800px',
                    'height': '800px',
                    'min-height': '800px',
                    'min-width': '800px'
                });
                var containerHeight = 0;
                var containerWidth = 0;

                containerElement.find('div.component').each(function (i, el) {
                    Wiki.log.debug("Checking: ", el, " ", i);
                    if (!states.any(function (node) {
                        return el.id === getNodeId(node);
                    })) {
                        Wiki.log.debug("Removing element: ", el.id);
                        jsPlumb.remove(el);
                    }
                });

                angular.forEach(states, function (node) {
                    Wiki.log.debug("node: ", node);
                    var id = getNodeId(node);
                    var div = containerElement.find('#' + id);

                    if (!div[0]) {
                        div = $($scope.nodeTemplate({
                            id: id,
                            node: node
                        }));
                        div.appendTo(containerElement);
                    }

                    // Make the node a jsplumb source
                    jsPlumb.makeSource(div, {
                        filter: "img.nodeIcon",
                        anchor: "Continuous",
                        connector: connectorStyle,
                        connectorStyle: { strokeStyle: "#666", lineWidth: 3 },
                        maxConnections: -1
                    });

                    // and also a jsplumb target
                    jsPlumb.makeTarget(div, {
                        dropOptions: { hoverClass: "dragHover" },
                        anchor: "Continuous"
                    });

                    jsPlumb.draggable(div, {
                        containment: '.camel-canvas'
                    });

                    // add event handlers to this node
                    div.click(function () {
                        var newFlag = !div.hasClass("selected");
                        containerElement.find('div.component').toggleClass("selected", false);
                        div.toggleClass("selected", newFlag);
                        var id = div.attr("id");
                        updateSelection(newFlag ? id : null);
                        Core.$apply($scope);
                    });

                    div.dblclick(function () {
                        var id = div.attr("id");
                        updateSelection(id);

                        //$scope.propertiesDialog.open();
                        Core.$apply($scope);
                    });

                    var height = div.height();
                    var width = div.width();
                    if (height || width) {
                        node.width = width;
                        node.height = height;
                        div.css({
                            'min-width': width,
                            'min-height': height
                        });
                    }
                });

                var edgeSep = 10;

                // Create the layout and get the buildGraph
                dagre.layout().nodeSep(100).edgeSep(edgeSep).rankSep(75).nodes(states).edges(transitions).debugLevel(1).run();

                angular.forEach(states, function (node) {
                    // position the node in the graph
                    var id = getNodeId(node);
                    var div = $("#" + id);
                    var divHeight = div.height();
                    var divWidth = div.width();
                    var leftOffset = node.dagre.x + divWidth;
                    var bottomOffset = node.dagre.y + divHeight;
                    if (containerHeight < bottomOffset) {
                        containerHeight = bottomOffset + edgeSep * 2;
                    }
                    if (containerWidth < leftOffset) {
                        containerWidth = leftOffset + edgeSep * 2;
                    }
                    div.css({ top: node.dagre.y, left: node.dagre.x });
                });

                // size the container to fit the graph
                containerElement.css({
                    'width': containerWidth,
                    'height': containerHeight,
                    'min-height': containerHeight,
                    'min-width': containerWidth
                });

                containerElement.dblclick(function () {
                    $scope.propertiesDialog.open();
                });

                jsPlumb.setSuspendEvents(true);

                // Detach all the current connections and reconnect everything based on the updated graph
                jsPlumb.detachEveryConnection({ fireEvent: false });

                angular.forEach(links, function (link) {
                    jsPlumb.connect({
                        source: getNodeId(link.source),
                        target: getNodeId(link.target)
                    });
                });
                jsPlumb.setSuspendEvents(false);
            });

            return states;
        }

        function getLink(info) {
            var sourceId = info.source.get(0).id;
            var targetId = info.target.get(0).id;
            return {
                source: sourceId,
                target: targetId
            };
        }

        function getNodeByCID(nodes, cid) {
            return nodes.find(function (node) {
                return node.cid === cid;
            });
        }

        /*
        * Updates the selection with the given folder or ID
        */
        function updateSelection(folderOrId) {
            var folder = null;
            if (angular.isString(folderOrId)) {
                var id = folderOrId;
                folder = (id && $scope.folders) ? $scope.folders[id] : null;
            } else {
                folder = folderOrId;
            }
            $scope.selectedFolder = folder;
            folder = getSelectedOrRouteFolder();
            $scope.nodeXmlNode = null;
            $scope.propertiesTemplate = null;
            if (folder) {
                var nodeName = Camel.getFolderCamelNodeId(folder);
                $scope.nodeData = Camel.getRouteFolderJSON(folder);
                $scope.nodeDataChangedFields = {};
                $scope.nodeModel = Camel.getCamelSchema(nodeName);
                if ($scope.nodeModel) {
                    $scope.propertiesTemplate = "app/wiki/html/camelPropertiesEdit.html";
                }
                $scope.selectedEndpoint = null;
                if ("endpoint" === nodeName) {
                    var uri = $scope.nodeData["uri"];
                    if (uri) {
                        // lets decompose the URI into scheme, path and parameters
                        var idx = uri.indexOf(":");
                        if (idx > 0) {
                            var endpointScheme = uri.substring(0, idx);
                            var endpointPath = uri.substring(idx + 1);

                            // for empty paths lets assume we need // on a URI
                            $scope.endpointPathHasSlashes = endpointPath ? false : true;
                            if (endpointPath.startsWith("//")) {
                                endpointPath = endpointPath.substring(2);
                                $scope.endpointPathHasSlashes = true;
                            }
                            idx = endpointPath.indexOf("?");
                            var endpointParameters = {};
                            if (idx > 0) {
                                var parameters = endpointPath.substring(idx + 1);
                                endpointPath = endpointPath.substring(0, idx);
                                endpointParameters = Core.stringToHash(parameters);
                            }

                            $scope.endpointScheme = endpointScheme;
                            $scope.endpointPath = endpointPath;
                            $scope.endpointParameters = endpointParameters;

                            console.log("endpoint " + endpointScheme + " path " + endpointPath + " and parameters " + JSON.stringify(endpointParameters));
                            $scope.loadEndpointSchema(endpointScheme);
                            $scope.selectedEndpoint = {
                                endpointScheme: endpointScheme,
                                endpointPath: endpointPath,
                                parameters: endpointParameters
                            };
                        }
                    }
                }
            }
        }

        function getWidth() {
            var canvasDiv = $($element);
            return canvasDiv.width();
        }

        function getFolderIdAttribute(route) {
            var id = null;
            if (route) {
                var xmlNode = route["routeXmlNode"];
                if (xmlNode) {
                    id = xmlNode.getAttribute("id");
                }
            }
            return id;
        }

        function getRouteFolder(tree, routeId) {
            var answer = null;
            if (tree) {
                angular.forEach(tree.children, function (route) {
                    if (!answer) {
                        var id = getFolderIdAttribute(route);
                        if (routeId === id) {
                            answer = route;
                        }
                    }
                });
            }
            return answer;
        }
        /*
        if (jsPlumb) {
        jsPlumb.bind("ready", setup);
        }
        
        function setup() {
        $scope.jsPlumbSetup = true;
        }
        */
    }
    Wiki.CamelCanvasController = CamelCanvasController;
})(Wiki || (Wiki = {}));
/**
* @module Wiki
*/
var Wiki;
(function (Wiki) {
    function GitPreferences($scope, localStorage, userDetails) {
        Core.initPreferenceScope($scope, localStorage, {
            'gitUserName': {
                'value': userDetails.username
            },
            'gitUserEmail': {
                'value': ''
            }
        });
    }
    Wiki.GitPreferences = GitPreferences;
})(Wiki || (Wiki = {}));
/**
* @module Wiki
*/
var Wiki;
(function (Wiki) {
    function HistoryController($scope, $location, $routeParams, $templateCache, workspace, marked, fileExtensionTypeRegistry, wikiRepository) {
        Wiki.initScope($scope, $routeParams, $location);
        $scope.selectedItems = [];

        // TODO we could configure this?
        $scope.dateFormat = 'EEE, MMM d, y : hh:mm:ss a';

        $scope.gridOptions = {
            data: 'logs',
            showFilter: false,
            selectedItems: $scope.selectedItems,
            showSelectionCheckbox: true,
            displaySelectionCheckbox: true,
            filterOptions: {
                filterText: ''
            },
            columnDefs: [
                {
                    field: 'commitHashText',
                    displayName: 'Change',
                    cellTemplate: $templateCache.get('changeCellTemplate.html'),
                    cellFilter: "",
                    width: "*"
                },
                {
                    field: 'date',
                    displayName: 'Modified',
                    cellFilter: "date: dateFormat",
                    width: "**"
                },
                {
                    field: 'author',
                    displayName: 'Author',
                    cellFilter: "",
                    width: "**"
                },
                {
                    field: 'shortMessage',
                    displayName: 'Message',
                    cellTemplate: '<div class="ngCellText" title="{{row.entity.shortMessage}}">{{row.entity.trimmedMessage}}</div>',
                    cellFilter: "",
                    width: "****"
                }
            ]
        };

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(updateView, 50);
        });

        $scope.$watch('workspace.tree', function () {
            if (!$scope.git && Git.getGitMBean(workspace)) {
                // lets do this asynchronously to avoid Error: $digest already in progress
                //console.log("Reloading the view as we now seem to have a git mbean!");
                setTimeout(updateView, 50);
            }
        });
        $scope.canRevert = function () {
            return $scope.selectedItems.length === 1 && $scope.selectedItems[0] !== $scope.logs[0];
        };

        $scope.revert = function () {
            if ($scope.selectedItems.length > 0) {
                var objectId = $scope.selectedItems[0].name;
                if (objectId) {
                    var commitMessage = "Reverting file " + $scope.pageId + " to previous version " + objectId;
                    wikiRepository.revertTo($scope.branch, objectId, $scope.pageId, commitMessage, function (result) {
                        Wiki.onComplete(result);

                        // now lets update the view
                        notification('success', "Successfully reverted " + $scope.pageId);
                        updateView();
                    });
                }
                $scope.selectedItems.splice(0, $scope.selectedItems.length);
            }
        };

        $scope.diff = function () {
            var defaultValue = " ";
            var objectId = defaultValue;
            if ($scope.selectedItems.length > 0) {
                objectId = $scope.selectedItems[0].name || defaultValue;
            }
            var baseObjectId = defaultValue;
            if ($scope.selectedItems.length > 1) {
                baseObjectId = $scope.selectedItems[1].name || defaultValue;

                // make the objectId (the one that will start with b/ path) always newer than baseObjectId
                if ($scope.selectedItems[0].date < $scope.selectedItems[1].date) {
                    var _ = baseObjectId;
                    baseObjectId = objectId;
                    objectId = _;
                }
            }
            var link = Wiki.startLink($scope.branch) + "/diff/" + $scope.pageId + "/" + objectId + "/" + baseObjectId;
            var path = Core.trimLeading(link, "#");
            $location.path(path);
        };

        updateView();

        function updateView() {
            var objectId = "";
            var limit = 0;

            $scope.git = wikiRepository.history($scope.branch, objectId, $scope.pageId, limit, function (logArray) {
                angular.forEach(logArray, function (log) {
                    // lets use the shorter hash for links by default
                    var commitId = log.commitHashText || log.name;
                    log.commitLink = Wiki.startLink($scope.branch) + "/commit/" + $scope.pageId + "/" + commitId;
                });
                $scope.logs = logArray;
                Core.$apply($scope);
            });
            Wiki.loadBranches(wikiRepository, $scope);
        }
    }
    Wiki.HistoryController = HistoryController;
})(Wiki || (Wiki = {}));
/**
* @module Wiki
*/
var Wiki;
(function (Wiki) {
    function goToLink(link, $timeout, $location) {
        var href = Core.trimLeading(link, "#");
        $timeout(function () {
            console.log("About to navigate to: " + href);
            $location.url(href);
        }, 100);
    }

    function ViewController($scope, $location, $routeParams, $route, $http, $timeout, workspace, marked, fileExtensionTypeRegistry, wikiRepository, $compile, $templateCache, jolokia) {
        var log = Logger.get("Wiki");

        Wiki.initScope($scope, $routeParams, $location);

        $scope.fabricTopLevel = "fabric/profiles/";

        $scope.versionId = $scope.branch;

        $scope.profileId = Fabric.pagePathToProfileId($scope.pageId);
        $scope.showProfileHeader = $scope.profileId && $scope.pageId.endsWith(Fabric.profileSuffix) ? true : false;

        $scope.operationCounter = 1;
        $scope.addDialog = new UI.Dialog();
        $scope.generateDialog = new UI.Dialog();
        $scope.renameDialog = new UI.Dialog();
        $scope.moveDialog = new UI.Dialog();
        $scope.deleteDialog = new UI.Dialog();
        $scope.isFile = false;
        $scope.rename = {
            newFileName: ""
        };
        $scope.move = {
            moveFolder: ""
        };
        $scope.createDocumentTree = Wiki.createWizardTree(workspace, $scope);

        $scope.createDocumentTreeActivations = ["camel-spring.xml", "ReadMe.md"];
        $scope.fileExists = {
            exists: false,
            name: ""
        };

        // bind filter model values to search params...
        Core.bindModelToSearchParam($scope, $location, "searchText", "q", "");

        // only reload the page if certain search parameters change
        Core.reloadWhenParametersChange($route, $scope, $location);

        $scope.gridOptions = {
            data: 'children',
            displayFooter: false,
            selectedItems: [],
            showSelectionCheckbox: true,
            enableSorting: false,
            useExternalSorting: true,
            columnDefs: [
                {
                    field: 'name',
                    displayName: 'Name',
                    cellTemplate: $templateCache.get('fileCellTemplate.html'),
                    headerCellTemplate: $templateCache.get('fileColumnTemplate.html')
                }
            ]
        };

        $scope.childActions = [];

        var maybeUpdateView = Core.throttled(updateView, 1000);

        $scope.$on('wikiBranchesUpdated', function () {
            updateView();
        });

        /*
        if (!$scope.nameOnly) {
        $scope.gridOptions.columnDefs.push({
        field: 'lastModified',
        displayName: 'Modified',
        cellFilter: "date:'EEE, MMM d, y : hh:mm:ss a'"
        });
        $scope.gridOptions.columnDefs.push({
        field: 'length',
        displayName: 'Size',
        cellFilter: "number"
        });
        }
        */
        $scope.createDashboardLink = function () {
            var href = '/wiki/branch/:branch/view/*page';
            var page = $routeParams['page'];
            var title = page ? page.split("/").last() : null;
            var size = angular.toJson({
                size_x: 2,
                size_y: 2
            });
            var answer = "#/dashboard/add?tab=dashboard" + "&href=" + encodeURIComponent(href) + "&size=" + encodeURIComponent(size) + "&routeParams=" + encodeURIComponent(angular.toJson($routeParams));
            if (title) {
                answer += "&title=" + encodeURIComponent(title);
            }
            return answer;
        };

        $scope.displayClass = function () {
            if (!$scope.children || $scope.children.length === 0) {
                return "";
            }
            return "span9";
        };

        $scope.parentLink = function () {
            var start = Wiki.startLink($scope.branch);
            var prefix = start + "/view";

            //console.log("pageId: ", $scope.pageId)
            var parts = $scope.pageId.split("/");

            //console.log("parts: ", parts);
            var path = "/" + parts.first(parts.length - 1).join("/");

            //console.log("path: ", path);
            return Core.createHref($location, prefix + path, []);
        };

        $scope.childLink = function (child) {
            var start = Wiki.startLink($scope.branch);
            var prefix = start + "/view";
            var postFix = "";
            var path = Wiki.encodePath(child.path);
            if (child.directory) {
                // if we are a folder with the same name as a form file, lets add a form param...
                var formPath = path + ".form";
                var children = $scope.children;
                if (children) {
                    var formFile = children.find(function (child) {
                        return child['path'] === formPath;
                    });
                    if (formFile) {
                        prefix = start + "/formTable";
                        postFix = "?form=" + formPath;
                    }
                }
            } else {
                var xmlNamespaces = child.xmlNamespaces;
                if (xmlNamespaces && xmlNamespaces.length) {
                    if (xmlNamespaces.any(function (ns) {
                        return Wiki.camelNamespaces.any(ns);
                    })) {
                        prefix = start + "/camel/canvas";
                    } else if (xmlNamespaces.any(function (ns) {
                        return Wiki.dozerNamespaces.any(ns);
                    })) {
                        prefix = start + "/dozer/mappings";
                    } else {
                        console.log("child " + path + " has namespaces " + xmlNamespaces);
                    }
                }
                if (child.path.endsWith(".form")) {
                    postFix = "?form=/";
                } else if (Wiki.isIndexPage(child.path)) {
                    // lets default to book view on index pages
                    prefix = start + "/book";
                }
            }
            return Core.createHref($location, prefix + path + postFix, ["form"]);
        };

        $scope.fileName = function (entity) {
            return Wiki.hideFineNameExtensions(entity.name);
        };

        $scope.fileClass = function (entity) {
            if (entity.name.has(".profile")) {
                return "green";
            }
            return "";
        };

        $scope.fileIconHtml = function (entity) {
            return Wiki.fileIconHtml(entity);
        };

        $scope.format = Wiki.fileFormat($scope.pageId, fileExtensionTypeRegistry);
        var options = {
            readOnly: true,
            mode: {
                name: $scope.format
            }
        };
        $scope.codeMirrorOptions = CodeEditor.createEditorSettings(options);

        $scope.editLink = function () {
            var pageName = ($scope.directory) ? $scope.readMePath : $scope.pageId;
            return (pageName) ? Wiki.editLink($scope.branch, pageName, $location) : null;
        };

        $scope.branchLink = function (branch) {
            if (branch) {
                return Wiki.branchLink(branch, $scope.pageId, $location);
            }
            return null;
        };

        $scope.historyLink = "#/wiki" + ($scope.branch ? "/branch/" + $scope.branch : "") + "/history/" + $scope.pageId;

        $scope.$watch('workspace.tree', function () {
            if (!$scope.git && Git.getGitMBean(workspace)) {
                // lets do this asynchronously to avoid Error: $digest already in progress
                //log.info("Reloading view as the tree changed and we have a git mbean now");
                setTimeout(maybeUpdateView, 50);
            }
        });

        /*
        // TODO this doesn't work for some reason!
        $scope.$on('jmxTreeUpdated', function () {
        console.log("view: jmx tree updated!");
        });
        */
        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            //log.info("Reloading view due to $routeChangeSuccess");
            setTimeout(maybeUpdateView, 50);
        });

        $scope.onSubmit = function (json, form) {
            notification("success", "Submitted form :" + form.get(0).name + " data: " + JSON.stringify(json));
        };

        $scope.onCancel = function (form) {
            notification("success", "Clicked cancel!");
        };

        $scope.onCreateDocumentSelect = function (node) {
            $scope.selectedCreateDocumentTemplate = node ? node.entity : null;
            $scope.selectedCreateDocumentTemplateRegex = $scope.selectedCreateDocumentTemplate.regex || /.*/;
            checkFileExists(getNewDocumentPath());
        };

        $scope.$watch("newDocumentName", function () {
            checkFileExists(getNewDocumentPath());
        });

        $scope.openAddDialog = function () {
            $scope.newDocumentName = null;
            $scope.addDialog.open();
        };

        $scope.addAndCloseDialog = function (fileName) {
            $scope.newDocumentName = fileName;
            var template = $scope.selectedCreateDocumentTemplate;
            var path = getNewDocumentPath();
            if (!template || !path) {
                return;
            }
            var name = Wiki.fileName(path);
            var fileName = name;
            var folder = Wiki.fileParent(path);
            var exemplar = template.exemplar;

            var commitMessage = "Created " + template.label;
            var exemplarUri = url("/app/wiki/exemplar/" + exemplar);

            if (template.folder) {
                notification("success", "Creating new folder " + name);

                wikiRepository.createDirectory($scope.branch, path, commitMessage, function (status) {
                    $scope.addDialog.close();
                    Core.$apply($scope);
                    var link = Wiki.viewLink($scope.branch, path, $location);
                    goToLink(link, $timeout, $location);
                });
            } else if (template.profile) {
                function toPath(profileName) {
                    var answer = "fabric/profiles/" + profileName;
                    answer = answer.replace(/-/g, "/");
                    answer = answer + ".profile";
                    return answer;
                }

                function toProfileName(path) {
                    var answer = path.replace(/^fabric\/profiles\//, "");
                    answer = answer.replace(/\//g, "-");
                    answer = answer.replace(/\.profile$/, "");
                    return answer;
                }

                // strip off any profile name in case the user creates a profile while looking at
                // another profile
                folder = folder.replace(/\/=?(\w*)\.profile$/, "");

                var concatenated = folder + "/" + name;

                var profileName = toProfileName(concatenated);
                var targetPath = toPath(profileName);

                $scope.addDialog.close();

                Fabric.createProfile(workspace.jolokia, $scope.branch, profileName, ['default'], function () {
                    notification('success', 'Created profile ' + profileName);
                    Core.$apply($scope);

                    Fabric.newConfigFile(workspace.jolokia, $scope.branch, profileName, 'ReadMe.md', function () {
                        notification('info', 'Created empty Readme.md in profile ' + profileName);
                        Core.$apply($scope);

                        var contents = "Here's an empty ReadMe.md for '" + profileName + "', please update!";

                        Fabric.saveConfigFile(workspace.jolokia, $scope.branch, profileName, 'ReadMe.md', contents.encodeBase64(), function () {
                            notification('info', 'Updated Readme.md in profile ' + profileName);
                            Core.$apply($scope);
                            var link = Wiki.viewLink($scope.branch, targetPath, $location);
                            goToLink(link, $timeout, $location);
                        }, function (response) {
                            notification('error', 'Failed to set ReadMe.md data in profile ' + profileName + ' due to ' + response.error);
                            Core.$apply($scope);
                        });
                    }, function (response) {
                        notification('error', 'Failed to create ReadMe.md in profile ' + profileName + ' due to ' + response.error);
                        Core.$apply($scope);
                    });
                }, function (response) {
                    notification('error', 'Failed to create profile ' + profileName + ' due to ' + response.error);
                    Core.$apply($scope);
                });
            } else if (template.version) {
                if (name === exemplar) {
                    name = '';
                }
                Fabric.doCreateVersion($scope, jolokia, $location, name);
            } else if (template.generated) {
                $scope.addDialog.close();

                var generateDialog = $scope.generateDialog;
                $scope.formSchema = template.generated.schema;
                $scope.formData = template.generated.form(workspace, $scope);
                $scope.generate = function () {
                    template.generated.generate(workspace, $scope.formData, function (contents) {
                        generateDialog.close();
                        wikiRepository.putPageBase64($scope.branch, path, contents, commitMessage, function (status) {
                            console.log("Created file " + name);
                            Wiki.onComplete(status);
                            $scope.generateDialog.close();
                            updateView();
                        });
                    }, function (error) {
                        generateDialog.close();
                        notification('error', error);
                    });
                };
                generateDialog.open();
            } else {
                notification("success", "Creating new document " + name);

                $http.get(exemplarUri).success(function (contents) {
                    // TODO lets check this page does not exist - if it does lets keep adding a new post fix...
                    wikiRepository.putPage($scope.branch, path, contents, commitMessage, function (status) {
                        console.log("Created file " + name);
                        Wiki.onComplete(status);

                        // lets navigate to the edit link
                        // load the directory and find the child item
                        $scope.git = wikiRepository.getPage($scope.branch, folder, $scope.objectId, function (details) {
                            // lets find the child entry so we can calculate its correct edit link
                            var link = null;
                            if (details && details.children) {
                                console.log("scanned the directory " + details.children.length + " children");
                                var child = details.children.find(function (c) {
                                    return c.name === fileName;
                                });
                                if (child) {
                                    link = $scope.childLink(child);
                                } else {
                                    console.log("Could not find name '" + fileName + "' in the list of file names " + JSON.stringify(details.children.map(function (c) {
                                        return c.name;
                                    })));
                                }
                            }
                            if (!link) {
                                console.log("WARNING: could not find the childLink so reverting to the wiki edit page!");
                                link = Wiki.editLink($scope.branch, path, $location);
                            }
                            $scope.addDialog.close();
                            Core.$apply($scope);
                            goToLink(link, $timeout, $location);
                        });
                    });
                });
            }
            $scope.addDialog.close();
        };

        $scope.openDeleteDialog = function () {
            if ($scope.gridOptions.selectedItems.length) {
                $scope.selectedFileHtml = "<ul>" + $scope.gridOptions.selectedItems.map(function (file) {
                    return "<li>" + file.name + "</li>";
                }).sort().join("") + "</ul>";
                $scope.deleteDialog.open();
            } else {
                console.log("No items selected right now! " + $scope.gridOptions.selectedItems);
            }
        };

        $scope.deleteAndCloseDialog = function () {
            var files = $scope.gridOptions.selectedItems;
            var fileCount = files.length;
            console.log("Deleting selection: " + files);
            angular.forEach(files, function (file, idx) {
                var path = $scope.pageId + "/" + file.name;
                console.log("About to delete " + path);
                $scope.git = wikiRepository.removePage($scope.branch, path, null, function (result) {
                    if (idx + 1 === fileCount) {
                        $scope.gridOptions.selectedItems.splice(0, fileCount);
                        var message = Core.maybePlural(fileCount, "document");
                        notification("success", "Deleted " + message);
                        Core.$apply($scope);
                        updateView();
                    }
                });
            });
            $scope.deleteDialog.close();
        };

        $scope.$watch("rename.newFileName", function () {
            // ignore errors if the file is the same as the rename file!
            var path = getRenameFilePath();
            if ($scope.originalRenameFilePath === path) {
                $scope.fileExists = { exsits: false, name: null };
            } else {
                checkFileExists(path);
            }
        });

        $scope.openRenameDialog = function () {
            var name = null;
            if ($scope.gridOptions.selectedItems.length) {
                var selected = $scope.gridOptions.selectedItems[0];
                name = selected.name;
            }
            if (name) {
                $scope.rename.newFileName = name;
                $scope.originalRenameFilePath = getRenameFilePath();
                $scope.renameDialog.open();
                $timeout(function () {
                    $('#renameFileName').focus();
                }, 50);
            } else {
                console.log("No items selected right now! " + $scope.gridOptions.selectedItems);
            }
        };

        $scope.renameAndCloseDialog = function () {
            if ($scope.gridOptions.selectedItems.length) {
                var selected = $scope.gridOptions.selectedItems[0];
                var newPath = getRenameFilePath();
                if (selected && newPath) {
                    var oldName = selected.name;
                    var newName = Wiki.fileName(newPath);
                    var oldPath = $scope.pageId + "/" + oldName;
                    console.log("About to rename file " + oldPath + " to " + newPath);
                    $scope.git = wikiRepository.rename($scope.branch, oldPath, newPath, null, function (result) {
                        notification("success", "Renamed file to  " + newName);
                        $scope.gridOptions.selectedItems.splice(0, 1);
                        $scope.renameDialog.close();
                        Core.$apply($scope);
                        updateView();
                    });
                }
            }
            $scope.renameDialog.close();
        };

        $scope.openMoveDialog = function () {
            if ($scope.gridOptions.selectedItems.length) {
                $scope.move.moveFolder = $scope.pageId;
                $scope.moveDialog.open();
                $timeout(function () {
                    $('#moveFolder').focus();
                }, 50);
            } else {
                console.log("No items selected right now! " + $scope.gridOptions.selectedItems);
            }
        };

        $scope.moveAndCloseDialog = function () {
            var files = $scope.gridOptions.selectedItems;
            var fileCount = files.length;
            var moveFolder = $scope.move.moveFolder;
            var oldFolder = $scope.pageId;
            if (moveFolder && fileCount && moveFolder !== oldFolder) {
                console.log("Moving " + fileCount + " file(s) to " + moveFolder);
                angular.forEach(files, function (file, idx) {
                    var oldPath = oldFolder + "/" + file.name;
                    var newPath = moveFolder + "/" + file.name;
                    console.log("About to move " + oldPath + " to " + newPath);
                    $scope.git = wikiRepository.rename($scope.branch, oldPath, newPath, null, function (result) {
                        if (idx + 1 === fileCount) {
                            $scope.gridOptions.selectedItems.splice(0, fileCount);
                            var message = Core.maybePlural(fileCount, "document");
                            notification("success", "Moved " + message + " to " + newPath);
                            $scope.moveDialog.close();
                            Core.$apply($scope);
                            updateView();
                        }
                    });
                });
            }
            $scope.moveDialog.close();
        };

        $scope.folderNames = function (text) {
            return wikiRepository.completePath($scope.branch, text, true, null);
        };

        setTimeout(maybeUpdateView, 50);

        function isDiffView() {
            var path = $location.path();
            return path && (path.startsWith("/wiki/diff") || path.startsWith("/wiki/branch/" + $scope.branch + "/diff"));
        }

        function updateView() {
            if (isDiffView()) {
                var baseObjectId = $routeParams["baseObjectId"];
                $scope.git = wikiRepository.diff($scope.objectId, baseObjectId, $scope.pageId, onFileDetails);
            } else {
                $scope.git = wikiRepository.getPage($scope.branch, $scope.pageId, $scope.objectId, onFileDetails);
            }
            Wiki.loadBranches(wikiRepository, $scope);
        }

        $scope.updateView = updateView;

        function viewContents(pageName, contents) {
            $scope.sourceView = null;

            var format = null;
            if (isDiffView()) {
                format = "diff";
            } else {
                format = Wiki.fileFormat(pageName, fileExtensionTypeRegistry) || $scope.format;
            }
            if ("markdown" === format) {
                // lets convert it to HTML
                $scope.html = contents ? marked(contents) : "";
                $scope.html = $compile($scope.html)($scope);
            } else if (format && format.startsWith("html")) {
                $scope.html = contents;
                $compile($scope.html)($scope);
            } else {
                var form = null;
                if (format && format === "javascript") {
                    form = $location.search()["form"];
                }
                $scope.source = contents;
                $scope.form = form;
                if (form) {
                    // now lets try load the form JSON so we can then render the form
                    $scope.sourceView = null;
                    if (form === "/") {
                        onFormSchema(_jsonSchema);
                    } else {
                        $scope.git = wikiRepository.getPage($scope.branch, form, $scope.objectId, function (details) {
                            onFormSchema(Wiki.parseJson(details.text));
                        });
                    }
                } else {
                    $scope.sourceView = "app/wiki/html/sourceView.html";
                }
            }
            Core.$apply($scope);
        }

        function onFormSchema(json) {
            $scope.formDefinition = json;
            if ($scope.source) {
                $scope.formEntity = Wiki.parseJson($scope.source);
            }
            $scope.sourceView = "app/wiki/html/formView.html";
            Core.$apply($scope);
        }

        function onFileDetails(details) {
            var contents = details.text;
            $scope.directory = details.directory;

            if (details && details.format) {
                $scope.format = details.format;
            } else {
                $scope.format = Wiki.fileFormat($scope.pageId, fileExtensionTypeRegistry);
            }
            $scope.codeMirrorOptions.mode.name = $scope.format;

            //console.log("format is '" + $scope.format + "'");
            $scope.children = null;

            if (details.directory) {
                var directories = details.children.filter(function (dir) {
                    return dir.directory && !dir.name.has(".profile");
                });
                var profiles = details.children.filter(function (dir) {
                    return dir.directory && dir.name.has(".profile");
                });
                var files = details.children.filter(function (file) {
                    return !file.directory;
                });

                directories = directories.sortBy(function (dir) {
                    return dir.name;
                });
                profiles = profiles.sortBy(function (dir) {
                    return dir.name;
                });

                files = files.sortBy(function (file) {
                    return file.name;
                }).sortBy(function (file) {
                    return file.name.split('.').last();
                });

                $scope.children = Array.create(directories, profiles, files);
            }

            $scope.html = null;
            $scope.source = null;
            $scope.readMePath = null;

            $scope.isFile = false;
            if ($scope.children) {
                // if we have a readme then lets render it...
                var item = $scope.children.find(function (info) {
                    var name = (info.name || "").toLowerCase();
                    var ext = Wiki.fileExtension(name);
                    return name && ext && ((name.startsWith("readme.") || name === "readme") || (name.startsWith("index.") || name === "index"));
                });
                if (item) {
                    var pageName = item.path;
                    $scope.readMePath = pageName;
                    wikiRepository.getPage($scope.branch, pageName, $scope.objectId, function (readmeDetails) {
                        viewContents(pageName, readmeDetails.text);
                    });
                }
            } else {
                var pageName = $scope.pageId;
                viewContents(pageName, contents);
                $scope.isFile = true;
            }
            Core.$apply($scope);
        }

        function checkFileExists(path) {
            $scope.operationCounter += 1;
            var counter = $scope.operationCounter;
            if (path) {
                wikiRepository.exists($scope.branch, path, function (result) {
                    // filter old results
                    if ($scope.operationCounter === counter) {
                        console.log("for path " + path + " got result " + result);
                        $scope.fileExists.exists = result ? true : false;
                        $scope.fileExists.name = result ? result.name : null;
                        Core.$apply($scope);
                    } else {
                        // console.log("Ignoring old results for " + path);
                    }
                });
            }
        }

        // Called by hawtio TOC directive...
        $scope.getContents = function (filename, cb) {
            var pageId = filename;
            if ($scope.directory) {
                pageId = $scope.pageId + '/' + filename;
            } else {
                var pathParts = $scope.pageId.split('/');
                pathParts = pathParts.remove(pathParts.last());
                pathParts.push(filename);
                pageId = pathParts.join('/');
            }
            log.debug("pageId: ", $scope.pageId);
            log.debug("branch: ", $scope.branch);
            log.debug("filename: ", filename);
            log.debug("using pageId: ", pageId);
            wikiRepository.getPage($scope.branch, pageId, undefined, function (data) {
                cb(data.text);
            });
        };

        function getNewDocumentPath() {
            var template = $scope.selectedCreateDocumentTemplate;
            if (!template) {
                console.log("No template selected.");
                return null;
            }
            var exemplar = template.exemplar || "";
            var name = $scope.newDocumentName || exemplar;

            if (name.indexOf('.') < 0) {
                // lets add the file extension from the exemplar
                var idx = exemplar.lastIndexOf(".");
                if (idx > 0) {
                    name += exemplar.substring(idx);
                }
            }

            // lets deal with directories in the name
            var folder = $scope.pageId;
            if ($scope.isFile) {
                // if we are a file lets discard the last part of the path
                var idx = folder.lastIndexOf("/");
                if (idx <= 0) {
                    folder = "";
                } else {
                    folder = folder.substring(0, idx);
                }
            }
            var fileName = name;
            var idx = name.lastIndexOf("/");
            if (idx > 0) {
                folder += "/" + name.substring(0, idx);
                name = name.substring(idx + 1);
            }
            folder = Core.trimLeading(folder, "/");
            return folder + (folder ? "/" : "") + name;
        }

        function getRenameFilePath() {
            var newFileName = $scope.rename.newFileName;
            return ($scope.pageId && newFileName) ? $scope.pageId + "/" + newFileName : null;
        }
    }
    Wiki.ViewController = ViewController;
})(Wiki || (Wiki = {}));
/**
* @module Wiki
*/
var Wiki;
(function (Wiki) {
    function NavBarController($scope, $location, $routeParams, workspace, wikiRepository) {
        Wiki.initScope($scope, $routeParams, $location);

        $scope.createLink = function () {
            var pageId = Wiki.pageId($routeParams, $location);
            return Wiki.createLink($scope.branch, pageId, $location, $scope);
        };

        $scope.startLink = Wiki.startLink($scope.branch);

        $scope.sourceLink = function () {
            var path = $location.path();
            var answer = null;
            angular.forEach(Wiki.customViewLinks($scope), function (link) {
                if (path.startsWith(link)) {
                    answer = Core.createHref($location, Wiki.startLink($scope.branch) + "/view" + path.substring(link.length));
                }
            });

            // remove the form parameter on view/edit links
            return (!answer && $location.search()["form"]) ? Core.createHref($location, "#" + path, ["form"]) : answer;
        };

        $scope.isActive = function (href) {
            if (!href) {
                return false;
            }
            return href.endsWith($routeParams['page']);
        };

        $scope.$on("$routeChangeSuccess", function (event, current, previous) {
            // lets do this asynchronously to avoid Error: $digest already in progress
            setTimeout(loadBreadcrumbs, 50);
        });

        loadBreadcrumbs();

        function switchFromViewToCustomLink(breadcrumb, link) {
            var href = breadcrumb.href;
            if (href) {
                breadcrumb.href = href.replace("wiki/view", link);
            }
        }

        function loadBreadcrumbs() {
            var start = Wiki.startLink($scope.branch);
            var href = start + "/view";
            $scope.breadcrumbs = [
                { href: href, name: "root" }
            ];
            var path = Wiki.pageId($routeParams, $location);
            var array = path ? path.split("/") : [];
            angular.forEach(array, function (name) {
                if (!name.startsWith("/") && !href.endsWith("/")) {
                    href += "/";
                }
                href += Wiki.encodePath(name);
                if (!name.isBlank()) {
                    $scope.breadcrumbs.push({ href: href, name: name });
                }
            });

            // lets swizzle the last one or two to be formTable views if the last or 2nd to last
            var loc = $location.path();
            if ($scope.breadcrumbs.length) {
                var last = $scope.breadcrumbs[$scope.breadcrumbs.length - 1];

                // possibly trim any required file extensions
                last.name = Wiki.hideFineNameExtensions(last.name);

                var swizzled = false;
                angular.forEach(Wiki.customViewLinks($scope), function (link) {
                    if (!swizzled && loc.startsWith(link)) {
                        // lets swizzle the view to the current link
                        switchFromViewToCustomLink($scope.breadcrumbs.last(), Core.trimLeading(link, "/"));
                        swizzled = true;
                    }
                });
                if (!swizzled && $location.search()["form"]) {
                    var lastName = $scope.breadcrumbs.last().name;
                    if (lastName && lastName.endsWith(".json")) {
                        // previous breadcrumb should be a formTable
                        switchFromViewToCustomLink($scope.breadcrumbs[$scope.breadcrumbs.length - 2], "wiki/formTable");
                    }
                }
            }

            /*
            if (loc.startsWith("/wiki/history") || loc.startsWith("/wiki/version")
            || loc.startsWith("/wiki/diff") || loc.startsWith("/wiki/commit")) {
            // lets add a history tab
            $scope.breadcrumbs.push({href: "#/wiki/history/" + path, name: "History"});
            } else if ($scope.branch) {
            var prefix ="/wiki/branch/" + $scope.branch;
            if (loc.startsWith(prefix + "/history") || loc.startsWith(prefix + "/version")
            || loc.startsWith(prefix + "/diff") || loc.startsWith(prefix + "/commit")) {
            // lets add a history tab
            $scope.breadcrumbs.push({href: "#/wiki/branch/" + $scope.branch + "/history/" + path, name: "History"});
            }
            }
            */
            var name = null;
            if (loc.startsWith("/wiki/version")) {
                // lets add a version tab
                name = ($routeParams["objectId"] || "").substring(0, 6) || "Version";
                $scope.breadcrumbs.push({ href: "#" + loc, name: name });
            }
            if (loc.startsWith("/wiki/diff")) {
                // lets add a version tab
                var v1 = ($routeParams["objectId"] || "").substring(0, 6);
                var v2 = ($routeParams["baseObjectId"] || "").substring(0, 6);
                name = "Diff";
                if (v1) {
                    if (v2) {
                        name += " " + v1 + " " + v2;
                    } else {
                        name += " " + v1;
                    }
                }
                $scope.breadcrumbs.push({ href: "#" + loc, name: name });
            }
            Core.$apply($scope);
        }
    }
    Wiki.NavBarController = NavBarController;
})(Wiki || (Wiki = {}));
/**
* @module Wiki
*/
var Wiki;
(function (Wiki) {
    function EditController($scope, $location,