package chat.rocket.persistence.realm.repositories

import android.os.Looper
import android.support.v4.util.Pair
import chat.rocket.core.SortDirection
import chat.rocket.core.models.Spotlight
import chat.rocket.core.repositories.SpotlightRepository
import chat.rocket.persistence.realm.RealmStore
import chat.rocket.persistence.realm.models.ddp.RealmSpotlight
import chat.rocket.persistence.realm.models.ddp.RealmSpotlight.Columns
import hu.akarnokd.rxjava.interop.RxJavaInterop
import io.reactivex.Flowable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.realm.Case
import io.realm.Realm
import io.realm.RealmResults
import io.realm.Sort
import java.util.ArrayList

class RealmSpotlightRepository(private val hostname: String) : RealmRepository(), SpotlightRepository {

    override fun getSuggestionsFor(term: String, direction: SortDirection, limit: Int): Flowable<List<Spotlight>> {
        return Flowable.defer { Flowable.using<RealmResults<RealmSpotlight>, Pair<Realm, Looper>>({
            Pair<Realm, Looper>(RealmStore.getRealm(hostname), Looper.myLooper())
        }, { pair -> RxJavaInterop.toV2Flowable<RealmResults<RealmSpotlight>>(pair.first.where(RealmSpotlight::class.java)
                .beginGroup()
                .like(Columns.NAME, "*$term*", Case.INSENSITIVE)
                .endGroup()
                .or()
                .beginGroup()
                .like(Columns.USERNAME, "*$term*", Case.INSENSITIVE)
                .endGroup()
                .findAllSorted(Columns.NAME, if (direction == SortDirection.ASC) Sort.ASCENDING else Sort.DESCENDING)
                .asObservable())
        }) { pair -> close(pair.first, pair.second) }
                .unsubscribeOn(AndroidSchedulers.from(Looper.myLooper()!!))
                .filter { realmSpotlightResults -> realmSpotlightResults.isLoaded && realmSpotlightResults.isValid }
                .map { realmSpotlightResults -> toList(safeSubList<RealmSpotlight>(realmSpotlightResults, 0, limit)) }
        }
    }

    private fun toList(realmSpotlightList: List<RealmSpotlight>): List<Spotlight> {
        val total = realmSpotlightList.size
        val spotlightList = ArrayList<Spotlight>(total)

        (0..total - 1).mapTo(spotlightList) {
            realmSpotlightList[it].asSpotlight()
        }

        return spotlightList
    }
}